# Workflow Core 1.7.0

* Various performance optimizations, any users of the EntityFramework persistence providers will have to update their persistence libraries to the latest version as well.
* Added `CancelCondition` to fluent builder API.

    ```
    .CancelCondition(data => <<expression>>, <<Continue after cancellation>>)

    ```

    This allows you to specify a condition under which any active step can be prematurely cancelled.
    For example, suppose you create a future scheduled task, but you want to cancel the future execution of this task if some condition becomes true.


    ```c#
    builder
        .StartWith(context => Console.WriteLine("Hello"))
        .Schedule(data => TimeSpan.FromSeconds(5)).Do(schedule => schedule
            .StartWith<DoSomething>()
            .Then<DoSomethingFurther>()
        )
        .CancelCondition(data => !data.SheduledTaskRequired)
        .Then(context => Console.WriteLine("Doing normal tasks"));
    ```

    You could also use this implement a parallel flow where once a single path completes, all the other paths are cancelled.

    ```c#
    .Parallel()
        .Do(then => then
            .StartWith<DoSomething>()
            .WaitFor("Approval", (data, context) => context.Workflow.IdNow)
        )
        .Do(then => then
            .StartWith<DoSomething>()
            .Delay(data => TimeSpan.FromDays(3))
            .Then<EscalateIssue>()
        )
    .Join()
        .CancelCondition(data => data.IsApproved, true)
    .Then<MoveAlong>();
    ```

* Deprecated `WorkflowCore.LockProviders.RedLock` in favour of `WorkflowCore.Providers.Redis`
* Create a new `WorkflowCore.Providers.Redis` library that includes providers for distributed locking, queues and event hubs.
    * Provides Queueing support backed by Redis.
    * Provides Distributed locking support backed by Redis.
    * Provides event hub support backed by Redis.

    This makes it possible to have a cluster of nodes processing your workflows.

    ## Installing

    Install the NuGet package "WorkflowCore.Providers.Redis"

    Using Nuget package console
        ```
        PM> Install-Package WorkflowCore.Providers.Redis
        ```
    Using .NET CLI
        ```
        dotnet add package WorkflowCore.Providers.Redis
        ```


    ## Usage

    Use the `IServiceCollection` extension methods when building your service provider
    * .UseRedisQueues
    * .UseRedisLocking
    * .UseRedisEventHub

    ```C#
    services.AddWorkflow(cfg =>
    {
        cfg.UseRedisLocking("localhost:6379");
        cfg.UseRedisQueues("localhost:6379", "my-app");
        cfg.UseRedisEventHub("localhost:6379", "my-channel")
    });
    ```
