// This file is part of CoreLibrary containing useful reusable utility
// classes.
//
// Copyright (C) 2014 to present, Duncan Crutchley
// Contact <dac1976github@outlook.com>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License and GNU Lesser General Public License
// for more details.
//
// You should have received a copy of the GNU General Public License
// and GNU Lesser General Public License along with this program. If
// not, see <http://www.gnu.org/licenses/>.

/*!
 * \file SimpleUdpSender.h
 * \brief File containing simple UDP sender class declaration.
 */

#ifndef SIMPLEUDPSENDER
#define SIMPLEUDPSENDER

#include "UdpTypedSender.h"

/*! \brief The core_lib namespace. */
namespace core_lib
{
/*! \brief The asio namespace. */
namespace asio
{
/*! \brief The udp namespace. */
namespace udp
{
/*! \brief A simplified UDP sender. */
class CORE_LIBRARY_DLL_SHARED_API SimpleUdpSender final
{
public:
    /*! \brief Default constructor - deleted. */
    SimpleUdpSender() = delete;
    /*!
     * \brief Initialisation constructor.
     * \param[in] ioContext - External boost IO context to manage ASIO.
     * \param[in] receiver - Connection object describing target receiver's address and port.
     * \param[in] sendOption - Socket send option to control the use of broadcasts/unicast.
     * \param[in] sendBufferSize - Socket send option to control send buffer size.
     *
     * Typically use this constructor when managing a bool of threads using an instance of
     * core_lib::asio::IoContextThreadGroup in your application to manage a pool of std::threads.
     * This means you can use a single thread pool and all ASIO operations will be exectued
     * using this thread pool managed by a single IO context. This is the recommended constructor.
     */
    SimpleUdpSender(boost_iocontext_t& ioContext, const defs::connection_t& receiver,
                    eUdpOption sendOption     = eUdpOption::broadcast,
                    size_t     sendBufferSize = DEFAULT_UDP_BUF_SIZE);
    /*!
     * \brief Initialisation constructor.
     * \param[in] receiver - Connection object describing target receiver's address and port.
     * \param[in] sendOption - Socket send option to control the use of broadcasts/unicast.
     * \param[in] sendBufferSize - Socket send option to control send buffer size.
     *
     * This constructor does not require an external IO context to run instead it creates
     * its own IO context object along with its own thread. For very simple cases this
     * version will be fine but in more performance and resource critical situations the
     * external IO context constructor is recommened.
     */
    explicit SimpleUdpSender(const defs::connection_t& receiver,
                             eUdpOption                sendOption     = eUdpOption::broadcast,
                             size_t                    sendBufferSize = DEFAULT_UDP_BUF_SIZE);
    /*! \brief Copy constructor - deleted. */
    SimpleUdpSender(const SimpleUdpSender&) = delete;
    /*! \brief Copy assignment operator - deleted. */
    SimpleUdpSender& operator=(const SimpleUdpSender&) = delete;
    /*! \brief Move constructor - deleted. */
    SimpleUdpSender(SimpleUdpSender&&) = delete;
    /*! \brief Move assignment operator - deleted. */
    SimpleUdpSender& operator=(SimpleUdpSender&&) = delete;
    /*! \brief Default destructor. */
    ~SimpleUdpSender() = default;
    /*!
     * \brief Retrieve receiver connection details.
     * \return - Connection object describing target receiver's address and port.
     */
    defs::connection_t ReceiverConnection() const;
    /*!
     * \brief Send a header-only message to the receiver.
     * \param[in] messageId - Unique message ID to insert into message header.
     * \param[in] responseAddress - (Optional) The address and port where the receiver should send
     * the response, the default value will mean the response address will point to this client
     * socket. \return Returns the success state of the send as a boolean.
     *
     * This method only sends a simple core_lib::asio::defs::MessageHeader
     * object to the server.
     */
    bool SendMessage(int32_t                   messageId,
                     const defs::connection_t& responseAddress = defs::NULL_CONNECTION);
    /*!
     * \brief Send a header plus message buffer to the receiver.
     * \param[in] message - The message buffer.
     * \param[in] messageId - Unique message ID to insert into message header.
     * \param[in] responseAddress - (Optional) The address and port where the receiver should send
     * the response, the default value will mean the response address will point to this client
     * socket.
     * \return Returns the success state of the send as a boolean.
     */
    bool SendMessage(const defs::char_buffer_t& message, int32_t messageId,
                     const defs::connection_t& responseAddress = defs::NULL_CONNECTION);
    /*!
     * \brief Send a full message to the server.
     * \param[in] message - The message of type T to send behind the header serialized to an
     * boost::serialization-compatible archive of type A. \param[in] messageId - Unique message ID
     * to insert into message header. \param[in] responseAddress - (Optional) The address and port
     * where the receiver should send the response, the default value will mean the response address
     * will point to this client socket. \return Returns the success state of the send as a boolean.
     *
     * This method uses the a core_lib::asio::defs::MessageHeader object as the header.
     */
    template <typename T, typename A = serialize::archives::out_port_bin_t>
    bool SendMessage(const T& message, int32_t messageId,
                     const defs::connection_t& responseAddress = defs::NULL_CONNECTION)
    {
        return m_udpTypedSender.SendMessage<T, A>(message, messageId, responseAddress);
    }
    /*!
     * \brief Send a message buffer to the receiver.
     * \param[in] message - The message buffer.
     * \return Returns the success state of the send as a boolean.
     */
    bool SendMessage(const defs::char_buffer_t& message);
    /*!
     * \brief Send a message buffer to the receiver.
     * \param[in] message - The message buffer pointer.
     * \param[in] message - The message buffer size in bytes.
     * \return Returns the success state of the send as a boolean.
     */
    bool SendMessage(const char* message, size_t length);

private:
    /*! \brief Default message builder object of type core_lib::asio::messages::MessageBuilder. */
    messages::MessageBuilder m_messageBuilder;
    /*! \brief Our actual typed UDP sender object. */
    UdpTypedSender<messages::MessageBuilder> m_udpTypedSender;
};

} // namespace udp
} // namespace asio
} // namespace core_lib

#endif // SIMPLEUDPSENDER
