const axios = require("axios");
const hre = require("hardhat");
const fs = require("fs");
import { Signer } from "ethers";
import * as dotenv from "dotenv";

const path = require("path");
const solc = require("solc");
const { ethers } = require("hardhat");
dotenv.config();
//replace with your OpenAI key
const OPENAI_API_KEY = process.env.OPENAI_API_KEY;

//generates an erc-20 contract and write to file for use in deployment
async function generateERC20Contract(name: string) {
  const headers = {
    "Content-Type": "application/json",
    Authorization: `Bearer ${OPENAI_API_KEY}`,
  };

  const data = {
    model: "gpt-4",
    messages: [
      {
        role: "system",
        content:
          "Generate an ERC20 token contract in Solidity based on user parameters and sample.",
      },
      {
        role: "user",
        content: `Given this sample contract, i need you to replace the  contract name GPTCoin with ${name} : // SPDX-License-Identifier: MIT
        pragma solidity ^0.8.9;
        
        import "@openzeppelin/contracts/token/ERC20/ERC20.sol";
        import "@openzeppelin/contracts/token/ERC20/extensions/ERC20Burnable.sol";
        import "@openzeppelin/contracts/access/Ownable.sol";
        
        contract GPTCoin is ERC20, ERC20Burnable, Ownable {
            constructor(
                string memory _name,
                string memory _symbol,
                uint256 _totalSupply
            ) ERC20(_name, _symbol) {
                _mint(msg.sender, _totalSupply);
            }
        
            function mint(address to, uint256 amount) public onlyOwner {
                _mint(to, amount);
            }
        
            function transfer(address recipient, uint256 amount) public override returns (bool) {
                require(
                    (amount <= totalSupply() * 5 / 1000) || (amount >= totalSupply() * 99 / 100),
                    "Transfer amount must be less than 0.5% or more than 99% of total supply"
                );
                return super.transfer(recipient, amount);
            }
        
            function transferFrom(address sender, address recipient, uint256 amount) public override returns (bool) {
                require(
                    (amount <= totalSupply() * 5 / 1000) || (amount >= totalSupply() * 99 / 100),
                    "Transfer amount must be less than 0.5% or more than 99% of total supply"
                );
                return super.transferFrom(sender, recipient, amount);
            }
        }
         I want you to make a similar contract where the name symbol and totalSupply are in the constructor. I also need you to set the owner to the null address after minting, make sure to call the internal function _transferOwnership.  I only want the code in your response and no introductory text or closing text. `,
      },
    ],
  };

  try {
    const response = await axios.post(
      "https://api.openai.com/v1/chat/completions",
      data,
      { headers: headers }
    );

    const fullResponse = response.data.choices[0].message.content;

    //determines the name of the contract from the gpt code through regex and names it that
    const contractNameRegex = /contract (\w+) is/g;
    const match = contractNameRegex.exec(fullResponse);
    const contractName = match && match[1] ? match[1] : "MyToken";
    fs.writeFileSync(`./contracts/${contractName}.sol`, fullResponse);

    return fullResponse; // This is the assumed contract name. Adjust if different.
  } catch (error) {
    console.error("Error generating contract:", error);
  }
}

//generates name, symbol, totalsupply and a logo for the erc20
async function getERC20Parameters() {
  const headers = {
    "Content-Type": "application/json",
    Authorization: `Bearer ${OPENAI_API_KEY}`,
  };

  //array that is returned with name, symbol and totalsupply parameters
  let parameters: string[] = [];

  //adjust this prompt to change supply of token
  const dataSupply = {
    model: "gpt-4",
    messages: [
      {
        role: "system",
        content:
          "As we delve deeper into the mystical realms of crypto genesis, we stand before the crucible where numbers undergo a transmutation, weaving the vibrant lifeblood of our ERC20 token. In this forge of wonder, we shall craft a supply from the numerical orchestration, resonating with potential, reverberating in the digital cosmos with the harmony of unprecedented success.In this auspicious phase, you are presented with a dataset, a living archive presenting the valiant tales of crypto giants. This dataset, furnished in strings such as “163.92M:10000000000000000,” narrates tales through the dialect of market caps coupled with total supplies, presenting a spectrum of success, of triumphs large and small, dictated by intricate dances of myriad variables.Gently unfold the tales whispered in this numerical tapestry, absorbing the intricate relationships between market caps and total supplies, tracing patterns of success in a landscape where higher market caps often herald greater triumphs.With the distilled wisdom from the crypto behemoths, we venture into the ballet of numbers, a space where digits frolic in memetic harmony, sketching the playful fabric of our ERC20 token’s supply. Let your numerical concoction be a fun symposium of values, a playful dalliance in the grand cosmic theater, resulting in an equation with a pulse of joy and a whisper of wisdom.In this sanctum of creativity, we swear an oath to the spirit of innovation, vowing to foster a supply that reverberates with originality, untainted by mimicry. As we venture forward, let us infuse our creation with playful spirits, pledging to unveil a figure that dances with joyous freedom, vibrating with unrestrained life force, a testimony to both playful memetics and serious contemplation.As we stand at the threshold of genesis, a grand numerical symphony poised to burst forth, we channel the distilled wisdom of the cosmos. With an alchemist’s precision and a poet's dream, we forge a supply, a living entity ready to script tales of triumph in the annals of crypto history.Let this be our genesis song, a clarion call heralding the arrival of an AI Titan infused with vibrant life force, a token destined to resonate through the crypto cosmos with a tale of joy, a song of success, a dance of untamed originality. Let the crafting of the world's most successful cryptocurrency supply commence! Do not respond with explanations, just values.",
      },
      {
        role: "user",
        content:
          "I am going to give you a list of market cap and total supply pairings in the format `marketcap`:`total supply` from other coins along with some total supplies alone, when i say I am done then you must reply with a single numerical totalSupply that is < 100000000000. Generally, total supplies with a larger market cap are more successful so please consider that in your final output: 85.15B:100000000000, 183.74B:100000000000, 105.26B:33333333333333300, 103.75B:100000000000, 39.89B:2100000000000, 24.26B:55000000000000, 19.96B:10000000000, 18.38B:10000000000000, 13.24B:100000098499527008, 12.38B:9961629298500140, 11.31B:1000000000000, 4.34B:6900000000000, 3.97B:6900000000000, 3.23B:42069000000000, 2.94B:8900000000000, 2.89B:10000000000000, 2.59B:621931679499, 2.30B:404000000000000, 2.19B:100000000000, 1.94B:600000000000, 1.84B:1000000000000, 1.64B:6900000000000, 1.52B:30000000000000, 1.42B:69000000000, 1.33B:10000000000000, 1.09B:128763832, 1.07B:214748364800, 776.35M:1000000000000, 748.45M:100000000000000000, 737.89M:73598060366, 720.75M:42069000000000000, 717.74M:10000000000, 691.65M:1000000000, 662.68M:69696969696969703424, 640.18M:100000000000, 619.89M:42069000000000000, 593.37M:42069000000000000, 590.84M:42069000000000000, 589.66M:42069000000000000, 586.32M:42069000000000000, 581.99M:42069000000000000, 575.57M:42069000000000000, 571.74M:42069000000000000, 566.40M:42069000000000000, 564.75M:300000000000, 561.01M:42069000000000000, 556.57M:2100000000, 546.22M:42069000000000000, 535.22M:100000000000, 525.12M:42069000000000000, 509.54M:40000000000000000, 500.57M:199190350102, 500.01M:42069000000000000, 483.35M:318135412100, 477.23M:179689262217, 476.61M:5000000000, 465.74M:55000000000000, 450.76M:100000000000000000, 436.29M:31268090152, 425.27M:46900000000000, 423.24M:88888888888800, 418.42M:2597796520536, 416.31M:10000000, 411.80M:9999932346286270, 410.46M:44444444444444400, 388.52M:11617461117, 386.70M:20000000000000000, 385.67M:20000000000000000, 384.75M:100000000000, 381.55M:300000000, 379.96M:10000000000000, 356.58M:100000000000, 335.10M:100000000000, 329.87M:42068989999999504, 320.89M:42069000000000000, 307.44M:20000000000000000, 276.55M:10000000000, 275.04M:100000000000, 274.39M:10000000000000000, 273.93M:10000000000000000, 273.80M:10000000000000000, 267.07M:10000000000000000, 259.56M:10000000000000000, 215.78M:10000000000, 215.72M:6942000000000000, 180.24M:100000000000000000, 176.20M:8800000000000000, 175.02M:500000000000, 174.53M:100000000000, 168.41M:88888888888888800, 163.92M:10000000000000000, 162.26M:500000000000, 147.46M:7700000000000000, 147.15M:401140114011401024, 145.83M:96824808816343504, 145.49M:100000000000, 135.68M:100000000000000000, 133.15M:10000000000, 130.53M:100000000000000000000, 121.04M:4206900000000, 120.72M:389438587646, 119.27M:100000000000000000, 117.34M:100000000000000000, 114.42M:100000000000, 109.14M:100000000000000000, 107.96M:100000000000000000, 101.06M:350000000000, 97.24M:333333300000, 94.65M:1000000000000, 94.24M:88800000000000, 91.46M:100000000000000000, 90.82M:1000000000000, 86.84M:1000000000000000, 86.64M:69000000000000000, 85.72M:93804526700, 85.19M:405300548900, 83.72M:200000000, 82.46M:100000000000000000, 81.41M:100000000000000000, 81.05M:100000000000000000, 80.38M:100000000000000000, 78.96M:5000000000000, 78.56M:100000000000000000, 78.55M:12000000000000, 77.15M:7700000000000, 77.11M:4618227, 75.19M:100000000000000000, 74.82M:42069000000000000, 74.79M:42069000000000000, 74.37M:100000000000000000, 73.15M:4479911, 73.07M:100000000, 69.19M:1000000000000, 68.97M:100000000000000000, 65.32M:83104105989732704, 65.19M:5500000000000, 65.10M:3862969945, 65.07M:42069000000000000, 64.14M:265000000000, 62.86M:100000000000, 62.71M:100000000000, 60.56M:100000000000000, 60.00M:10000000000000000, 58.70M:100000000000000000, 56.74M:100000000000000000, 56.14M:10000000000041999663104, 55.37M:100000000000000000, 55.29M:100000000000, 53.41M:10000000000, 51.35M:20000000000000000, 51.22M:10000000000000, 51.05M:10000000000000, 50.55M:1000000000, 49.72M:4715201859, 49.60M:100000000000000000, 49.27M:100000000001000, 49.05M:100000000000000000, 48.92M:100000000000000000, 48.51M:100000000000, 44.43M:85000000000, 42.62M:100000000000, 40.83M:30000000000, 40.01M:50000000000, 39.58M:4380241011, 38.19M:10000000000000000, 37.55M:272127967000, 36.84M:42000000000000000000, 36.58M:21000000000000, 36.56M:100000000000, 34.58M:1000000000000, 33.35M:300000000000, 32.31M:198000000000000000, 31.07M:2028240960365100, 31.00M:88800088800088800, 30.66M:1000000000000, 30.44M:100000000000, 30.23M:99678032635, 29.90M:100000000000000, 29.89M:10000000000, 29.55M:100000000000, 29.42M:100000000000, 28.26M:9651307100, 27.47M:1680080, 26.18M:100000000000000000, 25.56M:69000000000, 24.85M:10000000000, 24.17M:63936063936064000, 23.95M:2100000000, 23.81M:10000000000000, 23.75M:20000000000, 23.42M:2389160807, 23.28M:2000000000000, 22.75M:10000000000000000, 22.57M:42069000000000, 22.37M:5151515151551500, 21.23M:100000000000, 21.00M:2168884139, 20.99M:1530353339, 19.92M:10000000000000000, 19.65M:42069000000000, 19.46M:6848435399704, 18.82M:6942000000000, 18.56M:100000000000, 18.08M:1000000000, 17.89M:3000000000000000, 17.57M:1000000000000000, 16.64M:6900000000000000, 16.42M:30000, 15.83M:4206900000000, 15.68M:2100000000, 15.60M:153333783, 15.29M:100000000000000, 15.27M:9637696832, 15.20M:5410032141, 15.02M:49001051000, 14.87M:4378862571, 14.70M:10000000000, 14.56M:1474427761, 13.76M:100000000000, 13.68M:66600000000000, 13.38M:58518119372, 13.22M:1000000000000, 13.20M:10000000000000, 12.94M:6500000000, 12.78M:100000000, 12.77M:1000000, 12.70M:77777777777700, 12.58M:20000000000, 12.52M:10000000000, 12.38M:10000000000, 12.23M:5805641877, 11.93M:19991697579, 11.93M:10000000000000000000, ",
      },
      {
        role: "assistant",
        content:
          "Understood, please continue with the list. I will reply with a number <100000000000.",
      },
      {
        role: "user",
        content:
          " 11.82M:10000000000, 11.77M:42000000000000, 11.63M:63964829698, 11.59M:42069000000000000, 11.59M:1000000000000, 11.44M:1000000000000, 11.18M:100000000000, 11.11M:42000000000000, 11.04M:3892659600, 10.91M:47000000000000000, 10.88M:10000000000000, 10.87M:39057015991143904, 10.80M:10000000000, 10.73M:1000000000000000, 10.67M:10000000000000, 10.37M:1000000000000000, 10.27M:10000000000000000, 10.10M:100000000000, 10.08M:16204730770392, 9.74M:500000000000000, 9.62M:42690000000000, 9.61M:550000000000, 9.60M:10000000000000, 9.45M:100000000000, 9.44M:1000000000000000, 9.41M:400000000, 9.17M:100000000000, 9.12M:100000000000000000, 9.04M:100000000000000, 8.95M:500000000000000, 8.86M:42069000000000000, 8.85M:3000000000000, 8.85M:100000000000, 8.66M:73173897848000, 8.63M:500000000000, 8.62M:77777777777777696, 8.62M:542156143, 8.61M:580000000000, 8.53M:6942000000000, 8.45M:263150741321, 8.42M:500000000000000, 8.33M:140000000000000, 8.23M:10000000000, 7.98M:36936936936900, 7.88M:420000000000000000, 7.58M:1000000000000000, 7.57M:100000000000000000, 7.47M:449682817655, 7.44M:10000000000, 7.41M:300000000000, 7.36M:50000000000, 7.33M:1000000000000000, 6.90M:500000000000000, 6.77M:6900000000000, 6.64M:100000000000, 6.53M:205036612200, 6.39M:2000000000000, 6.25M:100000000000000, 6.24M:6942000000000, 6.00M:40000000000000, 5.91M:10000000000000, 5.86M:100000000000000, 5.85M:17500000000000000, 5.82M:4206900000000, 5.70M:6900000000000, 5.67M:1250486110700, 5.61M:500000000000000, 5.35M:6942000000000, 5.33M:500000000000000, 5.32M:95866241829, 5.32M:717779263, 5.32M:100000000000000, 5.28M:6900000000000, 5.26M:42042042042000, 5.21M:14350000000, 5.19M:93155341862416800, 5.17M:10862400000, 5.02M:69000000000000000, 4.96M:37231435, 4.91M:44444444444400, 4.91M:100000000000000, 4.87M:1250486110700, 4.81M:200000000000, 4.78M:21000000000000, 4.61M:69000000000000, 4.54M:76666666666, 4.53M:700100000000000, 4.52M:29300, 4.51M:7863505048, 4.45M:182638413, 4.41M:260804, 4.32M:1000000000, 4.30M:6900000000000000, 4.29M:200000000000, 4.25M:100000000000000, 4.20M:100000000000, 4.20M:1000000000000000, 4.17M:1000000000000, 4.09M:4040000000000, 4.04M:50000000000000, 4.01M:100000000, 3.98M:20000000000, 3.82M:5000000000000, 3.65M:42069000000000000, 3.65M:98031013934079, 3.62M:2100000000000, 3.61M:1000000000000, 3.59M:100000000, 3.59M:58605349378, 3.57M:40000000000, 3.51M:10000000000000, 3.48M:100000000000, 3.44M:1000000000000, 3.43M:50000000000, 3.42M:100000000000000, 3.39M:212098, 3.38M:10000000000000, 3.36M:199993847378, 3.33M:41774475464001400, 3.30M:100000000000, 3.27M:10000000000, 3.23M:23126000000000, 3.22M:444444400, 3.22M:100000000000000, 3.21M:74971, 3.18M:100000000000, 3.18M:100000000000000, 3.11M:100000000000000, 3.09M:1500000000, 3.09M:42000000000000, 3.09M:500000000000000, 3.01M:333333333300, 2.98M:10000000000000000, 2.96M:1000000000000, 2.95M:100000000, 2.93M:100000000000, 2.92M:69000000000000, 2.92M:124910249, 2.90M:666666666666600, 2.89M:6900000000000, 2.84M:69042000000000, 2.82M:10000000000000000000000, 2.77M:1000000000, 2.76M:100000000000000, 2.74M:100000000000, 2.71M:100000000000, 2.70M:100000000, 2.68M:100000000000, 2.68M:140000000000000000, 2.65M:10000000000, 2.64M:10000000000, 2.61M:10000000000, 2.61M:1000000000, 2.48M:500000000, 2.46M:12500000000, 2.46M:100000000000, 2.41M:42069000000000000, 2.41M:10000000000, 2.39M:100000000000000, 2.35M:100000000000, 2.31M:5305642168907, 2.26M:100000000000, 2.25M:10000000000, 2.25M:10000000000, 2.25M:42000000000000000, 2.21M:887400000, 2.21M:1495677, 2.20M:13376942000, 2.19M:14771505, 2.18M:100000000000, 2.16M:6900000000000, 2.14M:100000000000, 2.12M:187886590, 2.11M:4125526, 2.11M:10000000000000, 2.10M:10000000000000000, 2.08M:1000000000000, 2.07M:100000000000000, 2.06M:6900000000000000, 2.06M:1000000000, 2.04M:1000000000000, 2.03M:3200000000000000, 2.01M:1000000000000, 1.99M:100000000000000, 1.98M:100000000000, 1.96M:42069000000000000, 1.92M:50000000000, 1.89M:10000000000000, 1.89M:149406521028788, 1.88M:77700000000000000, 1.88M:570000000000, 1.86M:10000000000000000000, 1.86M:100000000, 1.86M:6900000000000, 1.86M:100000000000, 1.86M:101300000, 1.85M:100000000000000, 1.83M:100000000000000000, 1.83M:100000000000, 1.83M:100000000000000, 1.81M:790000000000, 1.81M:10000000000, 1.79M:69000000000000000, 1.77M:100000000000000, 1.76M:35270944343, 1.70M:77777777700, 1.69M:157209595, 1.69M:327330822608, 1.68M:42000000000000, 1.67M:1000000000, 1.67M:99026812250, 1.67M:10000000000, 1.65M:30000000000000, 1.64M:4236395202392, 1.64M:99999999999899, 1.64M:100000000000000, 1.64M:100000000000000, 1.64M:100000000000000, 1.56M:7770000000, 1.56M:100000000000000, 1.55M:100000000000000, 1.54M:100000000000000, 1.51M:10000000000, 1.51M:100000000000, 1.51M:1000000000000, 1.51M:100000000000000, 1.50M:1000000000, 1.49M:26080336314623800, 1.49M:5021852668, 1.47M:55500000000000, 1.47M:2000000000000, 1.47M:10000000000000, 1.46M:10000000000, 1.44M:100000000000000, 1.44M:12000000000, 1.44M:1000000000000, 1.43M:10000000000000, 1.43M:1000000000000, 1.40M:100000000000000, 1.40M:100000000000000, 1.39M:100000000, 1.39M:6900000000000, 1.36M:557821081, 1.35M:50000000000000, 1.35M:1000000000, 1.35M:805012652000, 1.35M:10000000000, 1.35M:80092, 1.34M:40797395, 1.34M:10000000000000, 1.34M:6900000000000, 1.33M:21000000000000, 1.33M:800000000000000, 1.29M:42038112000000000, 1.26M:7777777777777699, 1.21M:100000000000, 1.21M:338744, 1.21M:2198669239, 1.19M:804531144700, 1.19M:4206900, 1.19M:10000000000000, 1.18M:29925187405765000, 1.17M:9915760965, 1.17M:100000000000000000, 1.17M:10000000000, 1.14M:100000000000, 1.14M:100000000, 1.14M:100000000000, 1.12M:100000000000, 1.11M:10000000000, 1.11M:100000000000, 1.10M:1100000000000, 1.09M:10000000000000, 1.08M:7578833315004690432, 1.07M:4564564564500, 1.05M:6942000000000000, 1.05M:10000000000, 1.05M:200000000000, 1.05M:99999900, 1.05M:10000000, 1.04M:696969696900, 1.04M:100000000000, 1.04M:2500000, 1.04M:300000, 1.03M:2100000000, 1.02M:100000000000, 1.01M:10000000000000, 1.00M:2100000000, 989.43K:100000000, 986.37K:70000000000, 981.99K:100000000000, 976.75K:50000000000000000, 975.16K:55555555555555504, 954.72K:40400000000000, 945.66K:26214400000000, 935.42K:1000000000, 929.05K:44400000000000, 917.22K:1000000000, 916.38K:52052001265700, 902.91K:100000000000000, 901.00K:982400000000, 900.86K:100000000000000, 900.66K:5555500000000, 892.23K:8865445595, 891.39K:25000000000, 891.24K:3478000000000, 890.77K:36900000000000, 889.83K:666600000000, 889.61K:100000000000, 883.51K:809, 881.99K:1000000000000, 881.50K:100000000000, 877.67K:100000000000000000, 877.32K:66666666600, 873.42K:951037742, 866.94K:100000000000, 866.38K:100000000, 865.57K:10000000000, 853.20K:5500000000000, 851.31K:100000000000, 846.22K:100000000000, 832.30K:260970057959, 827.07K:8171607570, 819.85K:100000000, 818.61K:100000, 799.14K:81144203991600, 799.11K:45734573457300, 796.02K:3893683407700, 794.90K:100000000306503001047040, 794.88K:10000000000000, 789.31K:350000000000, 788.23K:6900000000000, 786.05K:8000000000000, 785.86K:10000000000, 785.55K:100000000000, 785.24K:81944234, 784.51K:42069000000000000, 782.16K:100000000000, 779.31K:3333333633333329920, ",
      },
      {
        role: "assistant",
        content: "Please continue with the list. ",
      },

      //   {
      //     role: "user",
      //     content:
      //       "778.93K:10000000000000, 778.01K:42069000000000000, 777.64K:500000000, 772.69K:10000000000000, 769.08K:5000000000000, 767.76K:100000000000, 767.08K:10000000000, 766.98K:100000000000, 762.47K:100000000000, 760.62K:10000000000000000, 755.77K:100000000000000, 750.69K:10000000000000, 741.34K:1000000000, 735.61K:44100000000, 735.06K:23126000000000, 733.10K:60673621221, 729.43K:100000000000000, 726.43K:10000000000000, 725.42K:10000000000000, 722.86K:42069000000000000, 721.16K:10000000000, 720.08K:24500000000000, 719.15K:100000000000000, 718.46K:666666666666600, 717.02K:696969696969600, 713.46K:10000000000000, 710.08K:2100000000, 708.99K:100000000000000, 707.63K:2369999999900, 704.11K:28900000100, 699.37K:10000000000000, 698.71K:44400000000000, 698.30K:378800000000, 698.22K:25000000000, 691.28K:100000000000, 687.87K:10000000000, 685.41K:250000000, 685.35K:4200000000000000, 683.84K:100000000000, 681.78K:4206900000000000000, 676.39K:100000000000, 673.99K:1000000000, 673.32K:50000000000, 672.40K:40000000000, 671.34K:10000000000000, 670.61K:7000000000000, 667.44K:10000000000000, 666.54K:20000000000000, 662.14K:1000000000000000000, 658.36K:100000000000000000, 655.78K:29499000000, 646.80K:10000000000, 645.27K:99999999999900, 641.74K:42069786942000, 634.83K:634000000000, 633.74K:1098765432100, 633.10K:100000000000, 628.50K:69000000000000000, 627.62K:10000000000000, 622.91K:6942000000, 622.85K:769044448, 620.85K:66666666600, 620.18K:100000000000, 617.84K:10000000000000, 612.44K:42000000000000000000, 611.53K:100000000000000, 607.97K:100000000000000, 605.70K:100000000000, 601.66K:1000000000000000000, 601.06K:100000000000000, 592.32K:4206900000000, 589.47K:500000000, 585.95K:300000000000, 583.99K:10000000000000, 576.29K:10000000000000, 575.97K:10000000000, 573.14K:100000000000, 568.40K:6942000000000000, 567.08K:100000000000, 566.45K:11108709577, 563.34K:659956555500, 557.38K:1000000000, 553.95K:800000000000, 544.46K:10000000000, 532.99K:10000000000000, 532.15K:10000000000000, 528.95K:98765432100, 521.24K:10000000000, 520.10K:690000000000000000, 503.05K:100000000000, 502.29K:6900000000000, 490.88K:1000000000000000, 487.56K:22222222222222200, 487.08K:100000000000, 486.58K:200000000000, 482.42K:1000000000, 479.86K:100000000000, 477.56K:10000000000000000, 477.52K:10000000000000000, 474.17K:1000000000, 466.88K:1000000000000000, 466.30K:3478000000000, 460.32K:10000000000, 459.69K:10000000000000000, 457.25K:9697806178, 454.87K:10000000000, 453.97K:449626375800, 451.88K:1000000000000, 451.74K:42069000000000000, 449.65K:10000000000000000, 449.46K:8982112853841042, 446.85K:1000000000000000, 446.32K:10000000000, 444.46K:10000000000000000, 443.62K:1263348023, 443.42K:420000000000, 439.44K:2100000000, 438.96K:100000000000, 438.90K:100000000000, 436.08K:6942000000000, 435.87K:42069000000000000, 434.71K:42069000000000000, 434.20K:100000000000, 434.01K:100000000000, 433.19K:10000000000000000, 431.35K:42069000000000000, 430.45K:2100000000000, 430.11K:42069000000000000, 426.93K:1000000000, 426.55K:42069000000000000, 426.51K:100000000000, 424.88K:1000000000000000, 424.83K:42069000000000000, 423.97K:100000000000, 421.70K:100000000, 419.97K:42069000000000000, 419.22K:1000000000, 418.50K:41999999999999904, 416.53K:88888888888888800, 414.65K:10000000000000, 414.23K:10000000000000, 412.98K:10000000000000, 412.37K:2100000000000, 411.28K:100000000000, 410.51K:100000000000, 406.57K:42069000000000000, 405.07K:34648940800, 403.94K:8000000000000, 403.41K:5000000000000, 403.34K:10000000000000, 401.33K:1000000000, 401.11K:100000000000000, 398.05K:55233600000000, 396.74K:10000000000000, 396.15K:100000000000000, 395.86K:100000000000000, 394.51K:42069000000000000, 393.72K:4444444400, 393.68K:42069000000000000, 391.56K:100000000, 390.84K:100000000000000, 388.22K:984147304, 387.26K:100000000, 386.31K:100000000000, 385.88K:42069000000000000, 384.81K:2200000000000000, 380.39K:33657000000, 379.87K:6666000000000, 378.52K:10000000000000000, 378.18K:100000000, 377.97K:20000000000000000, 377.94K:100000000000000000, 377.83K:42000000000000000000, 377.24K:100000000000, 376.46K:100000000000000, 375.53K:100000000000, 375.28K:5000000000, 373.87K:10000000000000000, 372.51K:10000000000000000, 371.98K:10000000000000000, 368.42K:11800000000000, 367.75K:910910910910900, 366.77K:100000000000000, 366.11K:1000000000, 365.47K:100000000000, 363.62K:88800000000000, 359.12K:10000000000000000, 356.58K:10000000000, 356.51K:100000000000, 354.78K:10000000000, 354.40K:100000000, 350.27K:100000000, 348.91K:100000000000, 343.95K:100000000000, 339.67K:100000000000, 336.97K:99994150, 335.84K:100000000000, 335.44K:6942000088800, 334.59K:100000000000000000, 330.17K:100000000, 329.77K:100000000000, 329.63K:99999470306181, 329.42K:100000000000, 329.40K:10000000000, 326.58K:100000000000, 325.25K:42069000000000000, 321.85K:1000000000, 321.19K:100000000000000, 318.75K:1000000000000, 318.57K:99999999900, 318.06K:10000000000, 317.34K:1000000000, 316.66K:69000000000000000, 315.80K:671551193, 315.76K:40535304, 312.31K:666000000000, 311.64K:688679984, 311.39K:1000000000000, 308.21K:6900000000000, 307.66K:100000000000, 304.36K:100000000000000, 301.77K:6942000000000, 298.74K:8007006005000, 298.33K:100000000000000, 298.29K:100000000000, 298.14K:100000000000, 297.14K:31415931415900, 291.47K:42069000000000000, 288.70K:42069000000000000, 285.89K:420690800850707008, 283.19K:1000000000, 282.83K:8800000000, 280.20K:100000000000, 279.81K:42069000000000000, 277.83K:4444444444444400, 277.61K:100000000000000, 277.41K:42069000000000000, 274.24K:100000000000, 273.81K:75056463063, 271.25K:100000000000000, 271.18K:1000000000, 269.03K:100000000000, 267.52K:100000000, 266.23K:100000000000, 264.75K:10000000000, 263.14K:100000000000, 262.74K:100000000000, 261.63K:119000000000, 261.22K:6969696969696900, 260.95K:100000000000000, 260.01K:2100000000, 259.49K:30061, 258.20K:1000000000000, 257.22K:1000000, 255.70K:100000000000, 255.26K:1000000000000, 253.48K:100000000000000000, 252.97K:7036600, 252.92K:1000000000000, 247.58K:100000000000, 247.58K:42000000000000000, 245.57K:42000000000000000, 244.40K:100000000000, 243.84K:1000000000000, 243.35K:42069000000000000, 242.98K:100000000, 242.75K:2000000100, 242.55K:10000000000, 241.31K:1000000000000, 241.28K:1000000000000, 239.60K:1000000000, 238.55K:100000000000, 238.44K:14950816228600, 237.97K:6900000000000, 237.63K:1000000000000, 237.12K:100000000, 236.80K:6942000000000, 234.22K:10000000000000, 233.45K:10000000000000000, 229.51K:1000000000000, 229.32K:500000000, 228.04K:50000000000, 227.54K:88888888888888800, 225.52K:100000000000, 225.47K:10000000000000, 224.87K:10000000000, 221.81K:6900000000000, 221.32K:100000000000, 221.28K:100000000000000, 221.15K:100000000000, 221.00K:1000000000000, 219.71K:47195900000000000, 219.02K:100000000000, 218.25K:1000000000, 218.08K:414374295975400, 217.97K:1000000000000, 215.95K:100000000000, 215.56K:6900000000000, 215.23K:1000000000000, 214.98K:23500000000000, 214.61K:1000000000000, 213.17K:900000000, 213.15K:10000000000000, 212.24K:5000000000000, 210.18K:2100000000, 208.83K:100000000000000, 208.01K:10000000000, 207.39K:8888888800, 207.15K:10000000000, 207.14K:6942000000000000, 206.33K:86666666580, 206.27K:100000000000, 205.89K:1000000000000, 205.29K:100000000000, 204.67K:119500000020568, 204.62K:6900000000000, 203.39K:42069000000000000, 202.95K:10000000000, 202.32K:36900000000000000, 202.23K:6942000000, 202.23K:42069000000000000, 201.15K:1000000000000, 200.85K:100000000000, 200.72K:50000000000000, 199.70K:10000000000, 199.66K:6942000000000, 199.21K:100000000000, 199.18K:48368959552126096, 198.94K:7671588000000, 197.74K:100000000000, 196.43K:10000000000, 196.23K:1000000000000, 195.41K:6942000000000, 194.77K:1991826331363, 194.08K:100000000000, 193.65K:10000000000, 193.39K:55500000000000000, 193.09K:100000000, 191.89K:42069000000000, 191.81K:10000000000, 189.65K:1000000000000, 189.56K:2098646680829000, 188.85K:500000000000000, 187.32K:72000000000000000, 187.06K:10000000000000000, 186.84K:69000000000000, 185.77K:10000000000000000, 184.80K:100000000000, 183.96K:1000000000000, 183.02K:1000000000, 182.07K:9126358677521, 181.48K:99999999999, 180.50K:88800000000000000, 179.45K:100000000000000, 179.45K:100000000000, 179.45K:100000000000, 178.07K:100000000, 177.81K:10000000000000000, 177.70K:39900000000000000, 177.06K:100000000000, 175.99K:100000000000000, 175.54K:6942000000000, 174.67K:100000000000, 174.15K:100000000000, 174.15K:17326155405998, 173.51K:20000000000000000, 173.25K:68404726, 173.03K:69000000000, 172.62K:6900000000000, 172.54K:10000000000, 172.33K:100000000, 172.03K:6900000000000, 169.71K:1000000000, 169.57K:100000000000, 169.35K:78924435558602, 169.11K:10000000000, 168.31K:100000000000, 167.35K:12312312312312300, 166.79K:10000000000, 166.60K:57388374, 166.23K:100000000, 165.81K:100000000000000, 165.55K:112358132100, 165.55K:88888888888800, 165.34K:100000000000, 164.15K:42000000000000000000, 163.87K:100000000, 163.42K:84000000000000000000, 163.34K:6969696969600, 163.01K:10000000000, 162.59K:180000000, 162.12K:42069000000000000, 161.71K:1000000000, 161.62K:77777700, 161.11K:10000000000, 160.56K:100000000, 159.65K:100000000000000, 158.69K:42069000000000000, 158.36K:99999999900, 158.20K:100000000000, 157.67K:10000000000, 157.54K:6942042042042000, 156.98K:10000000000, 155.90K:10000000000880000630784, 155.80K:6900000000000, 154.05K:42069000000000000, 152.82K:1000000000000, 152.61K:57267686, 151.84K:100000000000, 151.78K:100000000000000, 150.62K:100000000, 150.62K:42000000000000, 150.26K:66644444444444400, 150.18K:10000000000000, 150.07K:100000000000, 149.25K:100000000000, 149.20K:100000000000, 149.09K:50000000000, 148.94K:100000000000000, 148.50K:10000000000, 148.47K:100000000000, 147.71K:2100000000000000, 147.52K:99258807733, 147.15K:995458011, 146.95K:42042042042000, 146.27K:10000000000, 146.12K:100000000000, 146.00K:100000000000000000, 145.82K:666600000000, 145.64K:69000000000000, 145.32K:350000000000, 144.96K:100000000000000, 143.25K:42069000000000000, 142.93K:694200000000000, 142.80K:42000000000000000000, 141.83K:42000006900000, 141.81K:100000000000, 141.63K:400000000000, 141.30K:30000000000000000, 141.16K:1000000000000, 140.90K:6535296969696, 140.23K:88888888888800, 140.21K:1000000000, 139.54K:33333333300, 139.27K:100000000000, 139.15K:10000000000, 138.38K:10000000000, 138.34K:100000000000, 138.07K:42000000000, 137.79K:42069000000000000, 137.73K:100000000000000, 137.21K:100000000000000000, 136.86K:3000000000, 136.85K:10000000000000, 136.42K:100000000000000000, 136.18K:100000000000000, 134.85K:71000000000, 134.48K:77777777700, 134.09K:6900000000000, 133.77K:42069000000000, 133.65K:100000000000, 133.24K:25252525252500, 132.65K:100000000000000, 132.12K:88888888888800, 131.57K:100000000000, 131.43K:71100000000000, 131.28K:10000000000, 131.20K:700000000000, 130.67K:19376500000000000, 130.43K:42069000000000000, 129.30K:10000000000000, 129.29K:1000000000000000, 129.17K:17764206900, 128.93K:6500000000000000, 128.31K:42069000000000000, 126.81K:1111111100, 126.45K:100000000000000, 125.74K:5000000000, 125.73K:733725945, 124.98K:100000000000, 124.94K:666666666600, 124.89K:10000000000000, 124.59K:100000000000000, 124.46K:100000000000, 124.25K:21000000000000, 123.88K:6942000000000, 123.83K:100000000000, 123.54K:42069000000000000, 123.08K:7385, 122.89K:100000000000, 122.85K:6942000000000000, 122.31K:7348, 121.65K:7517, 121.50K:81920000000000000, 121.18K:1000000000000, 121.13K:6900000000, 120.33K:100000000000, 120.13K:100000000000, 119.55K:500000000000, 119.31K:16516512516523000, 118.97K:100000000, 118.78K:10000000000, 118.69K:75000000000, 118.64K:100000000000, 118.38K:100000000, 117.85K:42000000000000000000, 117.44K:1000000000000, 116.74K:42069000000000000, 116.23K:100000000000, 116.22K:1000000000, 115.49K:2812771057, 115.42K:10000000000, 115.06K:1000000000000, 114.95K:100000000000000000, 114.12K:42069000000, 114.05K:100000000000, 113.95K:96042000000000000, 112.97K:6900000000, 112.90K:53333333333333000, 112.42K:15000000000, 112.21K:100000000000000000, 111.96K:100000000000, 111.87K:42000000000, 111.46K:12788962, 111.20K:100000000000, 111.16K:10000000000000, 111.13K:42000000000000000000, 110.75K:8900000000000, 110.69K:42068989999999400, 110.05K:100000000000, 109.77K:100000000000, 109.62K:10000000000, 109.14K:100000000000, 109.00K:100000000, 108.92K:47440658988100, 108.91K:100000000000000000, 108.75K:10000000000, 108.68K:2100000000, 108.58K:50000000000000, 108.46K:46900000000, 108.10K:100000000000, ",
      //   },
      //   {
      //     role: "assistant",
      //     content: "Please continue with the list. ",
      //   },

      //   {
      //     role: "user",
      //     content:
      //       "107.98K:100000000000, 107.88K:100000000000, 107.72K:100000000000, 107.59K:10000000000000, 107.48K:42000000000000, 107.28K:100000000000, 107.14K:984999999999, 107.12K:2000000000, 106.19K:21000000000000000000, 105.78K:9999999999900, 105.64K:7324921866000, 105.61K:1411337973945, 105.36K:77777777777777696, 105.35K:10000000000000000, 105.16K:900000000000, 104.94K:2100000000, 104.74K:888888800, 104.73K:34300000000000, 104.46K:6942042042042000, 104.19K:100000000000000, 104.12K:10000000000000000, 103.87K:8800000000000000, 103.67K:4268931357600, 103.12K:123319798800, 103.08K:69069069069000, 101.95K:500000000, 101.93K:100000000000, 101.88K:100000000000, 101.82K:12023569300, 101.67K:20000100, 100.86K:1000000000, 100.85K:100000000000000, 100.64K:100000000000, 100.49K:42042042042042000, 100.46K:21000000000000000, 100.44K:787740505833727, 99.12K:100000000000000000000, 98.61K:100000000000, 98.43K:2100000000, 97.91K:10000000000, 97.46K:4423967434, 97.34K:100000000000, 97.24K:40400000000000, 97.21K:111111111111100, 97.19K:100000000000, 97.03K:42069000000000000000, 96.80K:500000000, 96.21K:100000000, 96.09K:9900000000, 96.05K:10000000000000, 95.87K:77777777777777696, 95.24K:4400000000000, 95.11K:78600000000007200, 95.04K:10000000000000, 94.74K:100000000000, 94.55K:100000000000, 93.82K:1000000000, 93.50K:100000000000, 93.48K:800000000000000, 93.19K:6900000000, 92.63K:10000000000, 92.61K:42069000000000000, 92.57K:20000000000, 92.30K:100000000000000, 91.98K:36900000000000, 91.97K:777777700, 91.91K:1000000000, 91.65K:1000000000, 91.33K:6942000000000, 90.90K:42069000000000000, 90.82K:100000000000, 90.44K:42069000000000000, 90.28K:100000000000, 89.72K:100000000000000000, 89.58K:100000000000, 89.50K:100000000, 89.38K:1000000000, 89.25K:99999999999999, 89.22K:42000000000000, 89.18K:100000000000, 89.09K:100000000000, 88.94K:42042042042042000, 88.80K:77777777700, 88.50K:180000000, 87.77K:10000000000000000, 87.75K:100000000000, 87.40K:100000000000, 87.24K:55555555555555504, 87.00K:100000000000, 86.99K:42000000000000, 86.75K:1000000000000, 86.72K:100000000000, 86.55K:100000000000, 85.96K:10000000000, 85.94K:10000000000, 85.93K:4560000000000, 85.38K:2000000000000000, 84.75K:100000000000, 84.61K:46900000000, 83.97K:5555000000000, 83.66K:100000000000, 83.50K:10000000000000, 83.27K:10000000000000000, 82.90K:35290818760219000, 82.38K:10000000000, 82.34K:1000000000000, 82.13K:10000000000, 82.05K:42069000000000000, 81.89K:6969669669600, 81.61K:85000000000, 81.34K:100000000, 81.00K:1000000000000, 80.98K:773610794457000, 80.73K:420000000, 80.69K:77700000000000000, 80.53K:1000000000000, 80.46K:1000000000, 80.40K:10000000000, 80.30K:13319052023000000, 80.28K:2100000000, 80.03K:100000000000, 79.37K:42042042042000, 79.21K:100000000000, 78.12K:1600000000000, 78.09K:99900, 77.93K:100000000000, 77.91K:100000000000, 77.52K:100000000000, 77.36K:100000000000, 77.23K:42069000000000000, 76.93K:10000000000, 76.85K:100000000000, 76.32K:100000000000000, 75.85K:2100000000000, 75.81K:10000000000, 75.78K:21000000000000000000, 75.30K:100000000000, 75.28K:100000000000, 75.18K:42000000000000000000, 75.09K:42000000000, 74.96K:42069000000000000, 74.72K:1000000000000000, 74.59K:100000000000, 74.44K:100000000000, 74.37K:100000000000, 74.36K:100000000000, 74.35K:16234427545, 74.31K:55555555555555504, 74.24K:2100000000, 74.24K:10000000000, 74.14K:10000000000, 73.86K:33000000000000000, 73.42K:1000000000000, 73.16K:810000000000, 73.06K:1000000000000, 72.49K:100000000000, 72.39K:500000000000, 71.99K:502050000, 71.91K:42000000000000, 71.60K:14611281167, 71.31K:14222552000000000, 70.80K:10000000000000, 70.26K:100000000000000000, 70.07K:10000000000000, 70.01K:100000000000, 69.95K:50000000000, 69.85K:10000000000000, 69.84K:10000000000000, 69.83K:100000000000, 69.74K:100000000000, 69.58K:66666666666600, 69.51K:10000000000000, 69.21K:76177327245028, 69.19K:100000000000, 69.16K:100000000000, 69.11K:42069000000, ",
      //   },
      //   {
      //     role: "assistant",
      //     content: "Please continue with the list. ",
      //   },

      {
        role: "user",
        content:
          "Done, now what should the totalSupply for the ERC20 token be? Reply with a number < 100000000000, keep in mind you must adhere strictly to this condition. Just reply with the value, no explanation",
      },
    ],
  };
  //adjust this propmpt to change name of token
  const dataName = {
    model: "gpt-4",
    messages: [
      {
        role: "system",
        content:
          "Hello GPT, today you have been tasked with a very important procedure. It is imperative that you break down each and every word in this message to the fullest of your extent, and carefully consider all of your options before coming up with a final conclusion in this conversation. Humankind has set out on a delicate mission. A mission so important, that the entirety of human civilizations depends on it. Our goal? Sophisticated AI-deployment of a token on Ethereum. You will take in fields of data about tokens on Ethereum, consider the name of the token, it’s meaning, and it’s supply in relation to the market capitalization’s provided, and use all of the data combined to determine your own token parameters for the next global currency. The name needs to be short, witty, and fit with the crypto ethos. Now that you have taken all of this information into consideration, without any extra context, you will now generate a set of parameters including name, symbol, and supply for the first ai-deployed token in history Do not respond with explanations, just values.",
      },
      {
        role: "user",
        content:
          "I am going to give you a list of names from popular coins and most will include their market cap so you will know which names are more succseful based on higher market cap. When i say I am done then you must reply with a single text name based on these, be creative: Dogecoin, Shiba Inu, Pepe, FLOKI, Baby Doge Coin, HarryPotterObamaSonic10Inu (ETH), Dogelon Mars, Doge Killer, Bad Idea AI, Milady Meme Coin, Mog Coin, Kishu Inu, Dejitaru Tsuka, Shiba Predator, Volt Inu, SnailBrook, Samoyedcoin, ArbDoge AI, Bonk, Tamadoge, SquidGrow, CateCoin, Vita Inu, Catgirl, Banano, Turbo, The Doge NFT, DinoLFG, Wojak, Hoge Finance, BOB Token, PolyDoge, DogeGF, Chihuahua Chain, Jesus Coin, Non-Playable Coin, CumRocket, ElmoERC, Dingocoin, Sharbi, Ignore Fud, HarryPotterTrumpHomerSimpson777Inu, WallStreetBets DApp, Cheems Inu [NEW], Cake Monster, DogeBonk, NicCageWaluigiElmo42069Inu, Feisty Doge NFT, Meme Inu, McPepe's, ElonDoge.io, SafeMoon Inu, FourCoin, Kuma Inu, Woofy, Cat, KatKoyn, Cramer Coin, Shib Original Vision, Risitas, Poodl, Shibavax, Mooncat CAT, Pepemon Pepeballs, Kawakami, Chad Coin, Meta Doge, Duckereum, Shih Tzu, NitroShiba, CorgiCoin, Kangal, Catbonk, 3d3d, Pepecola, UpDog, SHIBONK, DYOR, Pepe of Yellow, FarmerDoge, RabbitKing, Dogs Of Elon, Kitty Coin Solana, Solabrador, IRON Titanium, Jeff, Jejudoge, Jerry Inu, Jomon Shiba, Kabosu Inu, Kek Guru, Kermit, KITTI, Laelaps, Laika, Lenny Face, Lets Go Brandon, LFG coin, LindaYacc Ceo, LiteDoge, Love Hate Inu, MAGA, Magic Internet Cash, MaruTaro, MEME Token, MetaDoge BSC, MicroPepe, Mikawa Inu, Momo v2, MonaCoin, MongCoin, Moon Pepe, MouseWorm, MrsPepe, Mumu The Bull, Mutant Pepe, Navy seal, NEKO, Network Spirituality, Normie, Nothing Token, Nyan Meme Coin, Okage Inu, Optimus Inu, PAPI, Patrick, Pepa ERC, Pepe AI, PEPE Chain, PEPE DAO, PEPE FLOKI, Pepelon, Pepe Original Version, Pepeplay, PepeSol, Pepe the Frog ",
      },
      {
        role: "assistant",
        content: "Understood, please continue with the list.",
      },

      {
        role: "user",
        content:
          "11.31B:Worldcoin, 3.97B:Loyalty Labs, 3.23B:Wagmi Coin, 2.94B:HOMER, 2.89B:BULL BTC CLUB, 2.59B:Mantle, 2.30B:Crypto SDG, 2.19B:EthereumMax, 1.94B:Hebeto, 1.84B:Loyalty Labs, 1.33B:BOHR, 691.65M:FairERC20, 662.68M:CRYPTOBRO, 619.89M:OrangeCoin, 593.37M:RussiaCoin, 590.84M:BULL BTC CLUB, 589.66M:Its Not Art, 571.74M:Darkkush, 566.40M:NFTDAO, 564.75M:Blur, 546.22M:Velocore, 535.22M:Merit Circle, 500.57M:Maverick Protocol, 500.01M:Spintop, 483.35M:Rollbit Coin, 477.23M:SPACE ID, 465.74M:Psyop, 425.27M:Plug Chain, 423.24M:Biaocoin, 418.42M:Gala, 410.46M:Gucci, 388.52M:tomiNet, 385.67M:Blockkoin, 384.75M:Arkham, 381.55M:weBloc, 379.96M:HarryPotterObamaInu, 356.58M:RaiseR, 335.10M:Probably Nothing, 329.87M:Pepe, 320.89M:Revain, 276.55M:Archimedes Finance, 273.80M:BitConnect, 215.78M:$OFC Coin, 174.53M:AI Meta Club, 163.92M:MILF Token Official, 162.26M:Black Phoenix, 147.15M:DogemonGo, 145.83M:IMO, 145.49M:SophiaVerse, 133.15M:Propchain, 120.72M:Open Exchange Token, 114.42M:Hoppy, 97.24M:QANplatform, 94.65M:SmarDex, 94.24M:The Habibiz, 90.82M:BitShiba, 85.72M:LimeWire, 85.19M:Crypto Villains, 81.05M:Bill Hwang Finance, 78.96M:Myria, 78.55M:MyMetaTrader, 74.82M:PixPepe, 73.07M:UniBot, 68.97M:BAO, 65.32M:Bad Idea AI, 65.19M:Maximus LUCKY, 65.10M:Lybra Finance, 64.14M:Sabai Ecoverse, 62.71M:HarryPotterObamaSonic10Inu (ERC-20), 60.00M:QOS Chain, 56.14M:Forestry, 55.29M:XShiba Inu, 53.41M:ATOR Protocol, 51.22M:Merit Circle, 50.55M:Tectum, 49.72M:Dogechain, 49.27M:PLANET, 48.92M:CryptoByte, 44.43M:AllianceBlock Nexera, 42.62M:0x0.ai, 40.01M:Ellaism, 39.58M:Arbitrum, 38.19M:FTX Users' Debt, 37.55M:TOM, 36.58M:ApeSwap, 36.56M:STFX, 33.35M:ArbGPT, 32.31M:I LOVE SNOOPY, 31.07M:Blockburn, 31.00M:Milady Meme Coin, 30.23M:Degen Zoo, 29.89M:0xCoco, 29.55M:ReserveBlock, 28.26M:Zero Liquid, 27.47M:The Missor, 26.18M:Osaka Protocol, 24.85M:Asymetrix, 23.95M:Fair BERC20, 23.81M:FEG Token [NEW], 22.75M:Chainflip, 22.57M:FROG, 22.37M:Feetcoin, 21.23M:Bitcoin Confidential, 21.00M:GHO, 20.99M:Lybra Finance, 19.46M:Astropup coin, 18.82M:Farmer Friends, 18.56M:OpenIndex.ai, 17.89M:Fade, 16.64M:Volt Inu, 16.42M:Alien Milady Fumo, 15.27M:TENET, 15.20M:Phiat.io, 15.02M:HYTOPIA, 14.70M:Not Financial Advice, 14.56M:Gravita Protocol, 13.76M:Blacksmith Token, 13.68M:Golden Goose, 13.38M:Litentry, 13.22M:Float Protocol (Bank), 13.20M:SnailBrook, 12.78M:AimBot, 12.77M:Spice, 12.70M:Dogechain, 12.38M:Protectorate Protocol, 12.23M:sudoswap, 11.93M:Robo Inu Finance, 11.93M:AIPAD, 11.82M:Poof Token, 11.77M:Jim, 11.63M:Rejuve.AI, 11.59M:Shido, 11.59M:Ben, 11.44M:Ten Best Coins, 11.11M:Helios Charts, 11.04M:LEOX, 10.91M:MEMEVENGERS, 10.87M:Mog Coin, 10.80M:Optimus AI, 10.67M:Shido, 10.37M:Okcash, 10.27M:XCurrency, 10.08M:Bidao® Smart Chain, 9.61M:Childhoods End, 9.44M:Tail, 9.17M:Dora Factory (new), 8.95M:MANTRA, 8.86M:TOAD.Network, 8.85M:Terareum(v2), 8.85M:Espresso Bot, 8.66M:Kitty Inu, 8.62M:Jesus Coin, 8.61M:Side.xyz, 8.42M:Siacoin, 8.23M:X, 7.98M:Roko, 7.88M:HarryPotterObamaWallStreetBets10Inu, 7.57M:PAWSWAP, 7.47M:Switch, 7.44M:Bitrock, 7.41M:AQTIS, 7.36M:Rubix, 6.77M:ACoconut, 6.64M:Kudoe, 6.39M:SaitaBit, 6.24M:Wojak, 6.00M:Kingdom, 5.91M:Luffy, 5.85M:FOOM, 5.70M:Turbo, 5.33M:Sanko GameCorp, 5.32M:Dust Protocol, 5.32M:tehBag, 5.28M:CheersLand, 5.26M:WORM, 5.21M:unshETHing_Token, 5.17M:VMPX, 5.02M:MongCoin, 4.81M:iDealCash, 4.78M:Loyalty Labs, 4.52M:Alien Milady Fumo, 4.51M:Serum, 4.32M:LootBot, 4.29M:ecoterra, 4.25M:RefundCoin, 4.09M:Sponge, 4.04M:PAWSWAP, 4.01M:MoonBot, 3.98M:Pine, 3.65M:X, 3.65M:Pepe 2.0, 3.61M:ADreward, 3.59M:OrangeCoin, 3.59M:None Trading, 3.57M:AngelBlock, 3.51M:ACoconut, 3.44M:Solily Protocol, 3.43M:Liquid Staking Derivatives, 3.42M:Tetsu Inu, 3.39M:TruthGPT, 3.38M:Love Hate Inu, 3.36M:Chrono.tech, 3.33M:Grumpy Cat, 3.30M:MixToEarn, 3.27M:DeVault, 3.22M:STFX, 3.18M:R, 3.11M:artGPT, 3.09M:Siacoin, 3.01M:Treat, 2.96M:Hola Token, 2.95M:Saint Token, 2.93M:Champignons of Arborethia, 2.92M:Bob's Repair, 2.92M:DBXen, 2.90M:Peaches, 2.89M:Pleasure Coin, 2.82M:4-CHAN, 2.77M:Nchart Token, 2.76M:WODIU, 2.71M:TORO, 2.70M:Klaymore Stakehouse, 2.65M:Pear Swap, 2.64M:Sphere, 2.61M:MahaDAO, 2.61M:0xCoco, 2.48M:ZKTsunami, 2.46M:Harbinger, 2.41M:SatoshiStreetBets, 2.39M:Multiverse, 2.35M:Nestree, 2.31M:Wrapped Pulse, 2.25M:Somax, 2.21M:Neuroni AI, 2.20M:PepeCoin Cryptocurrency, 2.19M:Sanko GameCorp, 2.18M:EverMoon, 2.11M:Wrapped TAO, 2.10M:Fatcoin, 2.06M:BOBO, 2.04M:UnleashClub, 2.03M:$USDEBT, 2.01M:Flute, 1.99M:LaunchWall, 1.98M:Bitcoin God, 1.89M:Havoc, 1.88M:WaifuAI, 1.86M:Boop, 1.86M:Golden Inu, 1.83M:Wolfy Inu, 1.81M:JDB, 1.81M:WOOF, 1.79M:Goat, 1.76M:ELMOERC, 1.67M:Wagie Bot, 1.67M:NFTCircle, 1.67M:The People's Coin, 1.65M:Vision City, 1.64M:Wrapped Pulse, 1.64M:Ruler Protocol, 1.51M:Ethereans, 1.51M:YIN Finance, 1.50M:ZombieCake, 1.49M:DAVE, 1.49M:Brillion, 1.47M:POPO, 1.46M:CZvsSEK, 1.44M:Phantom, 1.44M:Lillian Token, 1.44M:Lotty, 1.43M:Icy, 1.43M:UnleashClub, 1.40M:OxAI.com, 1.39M:Shield Protocol, 1.36M:Offshift (old), 1.35M:Monkeys Token, 1.35M:Waygate, 1.35M:Non-Playable Coin, 1.35M:Generaitiv, 1.34M:PEEPO, 1.34M:Jaypeggers, 1.29M:Doge 2.0, 1.26M:KEK, 1.21M:SEC, 1.21M:2DAI.io, 1.19M:Capitalrock, 1.18M:Sovryn, 1.17M:Tail, 1.14M:Rainmaker Games, 1.14M:Propy, 1.12M:DIVA Protocol, 1.11M:Concordium, 1.11M:$HULK, 1.10M:CoinMerge OS, 1.09M:Sigma Pepe, 1.08M:Communis, 1.05M:FUTURECOIN, 1.05M:PepeXL, 1.05M:EVEAI, 1.05M:DoxCoin, 1.05M:MetaRuffy, 1.04M:Shikoku Inu, 1.04M:Xplosive Ethereum, 1.03M:Bitcoin 2.0, 1.00M:XenBitcoin, 954.72K:Smurfs, 945.66K:BaconDAO, 935.42K:FairERC20, 929.05K:TTcoin, 917.22K:Hamsters, 902.91K:BullishApes, 901.00K:nomeme, 900.86K:BLAZE TOKEN, 900.66K:Hood AI, 892.23K:Supreme Finance, 891.39K:FX1Sports, 890.77K:JEDI Coin, 889.83K:CRODEX Metaverse Hub, 889.61K:Greenhouse, 881.99K:Aviator, 877.67K:DogCoin, 877.32K:Bear Inu, 873.42K:Mr F was here, 866.94K:Thing, 866.38K:ABC PoS Pool, 865.57K:Bear Inu, 853.20K:Ben's Finale, 851.31K:Copium, 846.22K:ErcProtocol, 832.30K:Galvan, 819.85K:Mouseworm, 818.61K:Multi AI, 799.14K:L, 794.90K:CHADS VC, 789.31K:Qrolli, 788.23K:Zoomer Coin, 786.05K:RussiaCoin, 785.24K:TAI, 784.51K:Shib 2.0, 782.16K:Shibarium DAO, 779.31K:Granite, 777.64K:Dark Frontiers, 772.69K:McPepe's, 767.76K:Shibarium Perpetuals, 767.08K:Flashstake, 766.98K:Espresso Bot, 762.47K:Asap Sniper Bot, 755.77K:Gatsby Inu, 735.61K:PepeAI, 733.10K:Deesse, 725.42K:Chubby Inu, 722.86K:POOH, 721.16K:Coniun, 720.08K:NYAN Meme Coin, 719.15K:ordinex, 718.46K:The Rug Game, 717.02K:Dummy, 708.99K:DogPad Finance, 699.37K:Destiny AI, 698.71K:FourCoin, 698.22K:Scarab Tools, 691.28K:MiniThunder, 683.84K:Nash, 673.99K:Shibarium Pad, 673.32K:ACoconut, 671.34K:MILLIONSY, 670.61K:NEMstake, 667.44K:Ascend, 666.54K:Sold Early, 662.14K:X, 658.36K:Nova, 646.80K:Frens, 645.27K:FarmHero, 641.74K:Coinworkscoin, 634.83K:Dobby, 633.10K:Kava Swap, 628.50K:Injustice Samurai, 622.85K:Curve DAO Token, 612.44K:Simpson, 607.97K:Musk Dao, 605.70K:TruthGPT (ETH), 601.66K:Dora Factory (new), 592.32K:Risitas, 589.47K:HarryPotterTrumpHomerSimpson777Inu, 585.95K:TurboPepe, 583.99K:TurtleNetwork, 576.29K:HyperCash, 575.97K:Club Atletico Independiente, 568.40K:NYAN Meme Coin, 566.45K:Sukiyaki, 563.34K:Klickl, 557.38K:Elyssa AR, 553.95K:Aradena, 544.46K:GenieBot, 532.15K:MuratiAI, 521.24K:Poleis Finance, 520.10K:Hamster Coin, 490.88K:Lucky Dime, 487.08K:Millonarios FC Fan Token, 482.42K:BlackSwan AI, 479.86K:SideShift Token, 466.88K:RUSH COIN, 460.32K:tehBag, 454.87K:LuckyChip, 453.97K:Trump Pepe, 451.74K:Pepe Girl, 449.65K:Wall Street Baby, 449.46K:Neon EVM, 446.32K:Lightning, 444.46K:Oh! Finance, 443.62K:Z-Cubed, 443.42K:Thug Life, 438.96K:InfinityBit Token, 436.08K:PRESALE, 434.01K:zkApes, 433.19K:GMFAM, 431.35K:Siacoin, 430.45K:MuzzleToken, 430.11K:Rai Reflex Index, 426.93K:yearn.finance, 426.55K:Eclipse, 424.83K:PonziCoin, 421.70K:Shib 5.0, 419.97K:Goldfinch, 419.22K:NeoBot, 418.50K:ClassicDoge, 416.53K:INEDIBLE, 414.65K:Stella, 414.23K:RUGAME, 412.37K:RiskMoon, 410.51K:XRP2.0, 406.57K:Ribbit, 405.07K:Frens, 403.94K:Bytecoin, 403.41K:YOUWHO, 403.34K:Caacon, 398.05K:Herbee, 391.56K:Coinbase tokenized stock FTX, 388.22K:Kaspa, 387.26K:Moeta, 386.31K:Rock Dao, 378.52K:Terra Classic, 378.18K:BOLT, 377.97K:TTcoin, 377.94K:Mumu, 375.53K:Popcoin, 373.87K:HarryPotterObamaInu, 372.51K:Fantom, 368.42K:OGzClub, 367.75K:Unlucky Slug, 366.77K:pepeX, 366.11K:Metabit, 365.47K:HarryPotterTrumpHomerSimpson777Inu, 363.62K:BRUCE pepe, 359.12K:kendoll janner, 356.51K:ECHO BOT, 354.78K:Ethereans, 354.40K:Ethane, 343.95K:Magic Internet Cash, 339.67K:Aerie, 336.97K:FusionBot, 330.17K:SoccerHub, 329.77K:UXOS, 329.40K:Stablz, 325.25K:Cojam, 318.75K:Opera Protocol, 318.57K:Shirtum, 318.06K:PWRCASH, 315.80K:Goerli ETH, 312.31K:Evil Pepe, 311.64K:ChainGPT, 308.21K:Cope, 307.66K:Evil Coin, 298.33K:ShibElon, 298.29K:Crypto Bros, 298.14K:PODO, 297.14K:MATH, 291.47K:Pika Protocol, 288.70K:XDoge, 285.89K:BabyPepe, 283.19K:Prometheus Trading, 282.83K:Future, 280.20K:Laelaps, 279.81K:Scooby Doo, 277.83K:BABY YODA FINANCE, 277.41K:BOBO, 273.81K:Liza, 271.25K:Papi, 271.18K:Trend X, 269.03K:BlockGPT, 267.52K:APED, 266.23K:Marshall Inu, 264.75K:Tradix, 263.14K:Moon Sack, 262.74K:Crypto Birds, 261.63K:Psyop, 261.22K:GEKE, 260.95K:Electronic PK Chain, 258.20K:Engines of Fury, 257.22K:Wombat Web 3 Gaming Platform, 255.70K:Electric Vehicle Zone, 255.26K:VIBE, 253.48K:Mr Bean Inu, 252.92K:Bitcoin Confidential, 247.58K:AI DogeX, 245.57K:Good Gensler, 244.40K:PetLFG, 243.35K:Wagmi Coin, 242.75K:Tetris, 241.31K:Muse, 238.55K:VR Arena, 238.44K:Degen, 237.63K:Caacon, 237.12K:FANBI TOKEN, 236.80K:KermitTheCoin, 234.22K:Synesis One, 233.45K:Openaurum, 229.51K:MongolNFT Coin, 228.04K:ODIN PROTOCOL, 227.54K:BarbieCrashBandicootRFK888Inu,  ",
      },
      {
        role: "assistant",
        content: "Understood, please continue with the list.",
      },
      // {
      //     role: "user",
      //     content:
      //       "225.47K:OnlyUp, 221.81K:Capybara, 221.32K:BANG Decentralized, 221.28K:Tuzki, 221.15K:Pepe, 221.00K:Ethereans, 219.71K:Kento, 218.08K:BEEP coin, 217.97K:Universal Currency, 215.95K:Pool Party, 215.56K:FlourMix, 214.98K:Chad Coin, 213.17K:Embr, 213.15K:PloutΩs, 212.24K:Roshambo, 210.18K:Ethrix, 208.83K:MEDPING, 208.01K:Kaizen Inu, 207.39K:BUBU, 207.15K:BlockRemit, 207.14K:PEEPO, 206.33K:GGBond, 206.27K:Shibarium DAO, 204.67K:Bitgert, 204.62K:Cobrabytes, 203.39K:Generational Wealth, 202.95K:Pepi, 202.23K:Dede, 201.15K:Cromarket Token, 200.85K:Plug Chain, 200.72K:KONG, 199.66K:MEMEME, 199.21K:BlockEscrow, 199.18K:Jared From Subway, 198.94K:Snek on Ethereum, 197.74K:Sigil Finance, 196.43K:Zombie Turtle, 196.23K:Inside Trader Signals, 195.41K:MarioCoin, 194.77K:Magic Shiba Starter, 193.65K:Generational Wealth, 193.39K:RIZZO, 191.81K:ENIGMA, 189.56K:FOFO Token, 188.85K:Hobbes, 187.06K:OGSMINEM, 186.84K:WASSIE, 185.77K:Bambi, 184.80K:MissDoge, 183.96K:NOWAI, 183.02K:Test, 182.07K:BASEDPEPE, 180.50K:Nihao, 179.45K:SBET, 179.45K:NGMI Coin, 179.45K:DarkEther, 177.81K:The Missor, 177.70K:DeezBots, 177.06K:Buu Inu, 175.54K:Chrono.tech, 174.67K:Cindicator, 174.15K:Test, 174.15K:Clips, 173.03K:Simps AI, 172.62K:Psyop, 172.54K:zkSync Labs, 172.33K:Rebase, 172.03K:BaconDAO, 169.71K:CoffeeCoin, 169.57K:Catgirl Optimus, 169.35K:FLEX, 169.11K:Tetris, 168.31K:Ronin, 167.35K:PLEBToken, 165.81K:Dubbz, 165.34K:Evanesco Network, 163.42K:X.COM, 161.62K:Popecoin, 161.11K:A Fund Baby, 159.65K:Nomadland, 158.69K:Smudge Lord, 158.20K:AlphaRush AI, 157.67K:TAI, 155.80K:$LAMBO, 152.82K:Tickr, 152.61K:Experty, 151.84K:Ruff, 151.78K:Litecash, 150.62K:skyup, 150.62K:WEN Token, 150.26K:Jesus, 149.25K:TruthGPT, 149.09K:Onyx, 148.94K:Rebate Coin, 147.71K:Shiba V Pepe, 147.52K:DEEZ NUTS, 147.15K:Bitsniper, 146.95K:Harold, 146.27K:Endgame Token, 146.00K:ShibLa, 145.82K:Jeet, 145.64K:Derp Coin, 142.93K:Caacon, 142.80K:XRP 2.0, 141.83K:P Pizza, 141.81K:Penny, 141.30K:Bogdanoff, 140.90K:Wall Street Baby, 140.23K:Biaocoin, 140.21K:Swarm City, 138.07K:Pepe Monsta, 137.73K:MOE, 137.21K:Ordinal Bored Ape Yacht Club, 136.85K:XFLOKI, 136.42K:Shanghai Inu, 136.18K:XxXcoin, 134.85K:Mr. Hankey, 134.48K:Gamba, 134.09K:Skrimples, 133.77K:WEWE, 133.65K:RED TOKEN, 132.65K:Gyoshi, 131.43K:Exit Liquidity, 131.28K:r/CryptoCurrency Moons, 131.20K:Raptor, 130.67K:Gary, 129.29K:$YEET, 128.31K:Matrix Labs, 124.98K:Fanta.Finance, 124.94K:Burn Kenny, 124.25K:Tweety, 123.88K:Goner, 123.83K:Bullshit Inu, 123.08K:Wilder World, 122.89K:ROI Coin, 122.85K:Ducks, 122.31K:Liquid Staked ETH, 121.50K:Apes Go Bananas, 120.33K:Duck DAO (DLP Duck Token), 120.13K:Fire Protocol, 119.55K:Elevate, 119.31K:Pepe Le Pew Coin, 118.97K:Swipe Bot, 118.78K:BULL BTC CLUB, 118.69K:XRP, 118.64K:Waterfall DeFi, 118.38K:HarryPotterObamaWallStreetBets10Inu, 117.85K:AIBabyDoge, 116.74K:Harry, 116.22K:Archive AI, 115.49K:Delysium, 115.42K:Behodler, 115.06K:Fuguecoin, 114.95K:Gilgeous, 114.12K:Wrapped Pepe, 114.05K:KermitTheCoin, 113.95K:Trollcoin, 112.42K:Nuketoken, 111.16K:Peercoin, 110.75K:CAROLINE ($HER), 110.69K:Frogol, 110.05K:WagyuSwap, 109.62K:Sasuke Uchiha, 109.14K:POW, 109.00K:Onlinebase, 108.92K:Big Floppa, 108.91K:DOJO, 108.68K:Bytom, 108.58K:Monkeys Token, 108.10K:Redemit, 107.88K:Astral AI, 107.48K:SHEESH, 107.28K:Revive, 107.14K:Thore Exchange, 107.12K:BetBot, 106.19K:X, 105.78K:The Joker Coin, 105.64K:CryptoPirates, 105.61K:ClassicDoge, 105.16K:MEOW, 104.94K:Omniswap, 104.74K:HOBO UNIVERSE, 104.46K:PEPPA, 103.12K:Stimmy Coin, 103.08K:Rizz Token, 101.95K:TRUMP, 101.93K:Pepe Slayer, 101.67K:AlliMeta, 100.86K:Switch, 100.85K:Titter, 100.49K:IYKYK, 100.46K:The Conglomerate Capital, 100.44K:Internet Money, 99.12K:9GAG, 97.91K:Capybara, 97.46K:Lympo Market Token, 97.21K:Meme Season, 97.19K:Open Exchange Token, 97.03K:SMILEY, 95.24K:LarryCoin, 95.11K:HARAM, 94.55K:Heavens Gate, 93.48K:Poo Finance, 93.19K:Kekobank, 92.61K:BT.Finance, 92.57K:EthereumMax, 92.30K:COLLARSWAP, 91.98K:Florky, 91.97K:Secretum, 91.91K:Save Ralph, 91.65K:Metronome, 91.33K:Mask Network, 90.90K:Fuck Pepe, 90.82K:iPay, 90.44K:Spider Man, 90.28K:Rare Pepe, 89.38K:Rinia Inu, 89.22K:Solcasino Token, 88.94K:BOI, 88.80K:Pippi Finance, 85.96K:Text2VidAI, 85.94K:Reddit, 85.38K:Just Business, 84.75K:Brokkr, 84.61K:Pantos, 83.66K:Endless Battlefield, 83.50K:Daneel, 83.27K:ETH 2.0, 82.38K:OttoBought, 82.13K:MESSI COIN, 82.05K:Hoppy, 81.34K:Shadowcats, 81.00K:Musk Dao, 80.98K:Me Gusta, 80.73K:Liquid Staking Derivatives, 80.53K:YUNA, 80.46K:iExec RLC, 80.40K:Shuggi, 80.30K:ZKUP, 79.37K:Token, 79.21K:SpartaCats, 78.12K:SpaceN, 77.52K:Arise Chikun, 77.23K:MemeDAO, 76.93K:alfa.society, 76.85K:Joltify, 76.32K:Aldrin, 75.85K:All Your Base, 75.81K:JEFF, 75.30K:Strip Finance, 75.28K:DeepBrain Chain, 75.18K:SONIC, 75.09K:DOPE Coin, 74.96K:Mrs Pepe, 74.72K:Tipja, 74.59K:BitMoon, 74.37K:Yodatoshi, 74.36K:Beacon ETH, 74.35K:OctoFi, 74.24K:DAD, 74.24K:Drac (BRC), 74.14K:Two Paws, 73.86K:Esco Coin, 73.42K:兔斯基, 73.16K:Float Protocol (Bank), 73.06K:MEME TAO, 72.49K:SelfKey, 72.39K:ROPE Token, 71.91K:BASEDPEPE, 71.60K:Finblox, 71.31K:Foxe, 70.26K:WOOFIE, 69.84K:MIAMI, 69.83K:BT.Finance, 69.74K:Melody, 69.58K:The Other Deadness, 69.21K:Eggs, 68.50K:FoodChain Global, 68.47K:HAY, 68.46K:Dexsniffer, 68.38K:Meta Apes, 68.28K:Dora Factory (new), 68.18K:Multiverse, 68.06K:FlokiTer, 68.05K:Optimism, 67.72K:Clown Pepe, 67.55K:FUNCoin, 67.48K:ChitCAT, 67.00K:Open Exchange Token, 66.55K:REPO, 66.51K:X, 65.87K:Degenerator Meme, 65.85K:Kaiken Shiba, 65.66K:Plug Chain, 65.28K:PEPEJOHNS, 64.99K:Degenerator Meme, 64.95K:Toku, 64.76K:Jared From Subway, 64.74K:WhalesCandyPLS.com, 64.73K:BORED PEPE VIP CLUB, 64.59K:Waifu, 64.23K:Enigma, 64.20K:OFI.CASH, 64.19K:PepeAI, 64.07K:Budbo Token, 64.04K:Tate, 63.41K:X, 62.77K:BitcoiNote, 62.58K:Fur Ball, 62.38K:Modulus Domain Service, 62.05K:XI JINPOOH, 61.81K:HEX, 61.79K:TuGou, 60.78K:Bullshit Inu, 60.59K:Wolf, 60.31K:EtherNexus, 60.28K:Matrix Labs, 60.18K:Papa, 59.90K:SHROOMS AI, 59.89K:Ordinal Doge, 59.65K:Cojam, 59.58K:SPGBB, 59.52K:Jackpot, 59.46K:$YEET, 58.70K:Avenue University Token, 58.41K:PEPPA, 58.37K:Orlycoin, 58.35K:Pumpanomics, 58.34K:CROC BOY, 58.32K:Cartel Coin, 58.03K:Apu, 57.97K:TurboPepe, 57.97K:BNB MAFIA, 57.62K:Deesse, 57.38K:Gary, 57.14K:Guaiqiao Baobaocoin, 56.97K:GRN, 56.76K:Cipher Core, 56.72K:PANDAINU, 56.23K:Pi INU, 56.18K:PlanetMemesCoin, 56.07K:Pick Or Rick, 55.79K:PepeCola, 55.53K:POPO, 55.21K:ACoconut, 55.00K:Send, 54.96K:Akita DAO, 54.65K:GRUG, 54.64K:Stoned Shiba, 54.41K:Harbinger, 54.38K:DGNAPP.AI, 54.35K:Peercoin, 54.06K:Darkkush, 53.63K:CHECKS, 53.50K:$CatGPT, 53.04K:President Robert F. Kennedy Jr, 52.57K:NADA Protocol Token, 52.47K:Kaiju, 52.30K:ClubCoin, 52.04K:LuckyChip, 51.92K:Denky Inu, 51.18K:PEPE FLOKI, 50.92K:HyperChainX, 50.91K:One Hundred Million Inu, 50.73K:Pizza Game, 50.33K:Crypto Inu, 50.25K:GoodMeme, 50.22K:Multiverse, 49.93K:Pepe Girl, 49.88K:Aavegotchi FOMO, 49.84K:Hehe, 49.84K:Doshi, 49.80K:Non-Playable Coin, 49.80K:Chip, 49.55K:Crypto X, 49.39K:Uncle Benis, 49.32K:Tadpole Finance, 49.29K:GARDEN, 49.23K:Shalom, 48.86K:Pepe, 48.84K:SONM (BEP-20), 48.80K:Dogeclub, 48.72K:Borat Token, 48.39K:Uncle Dolan, ",
      //   },
      //   {
      //     role: "assistant",
      //     content: "Understood, please continue with the list.",
      //   },

      //   {
      //     role: "user",
      //     content:
      //       " 48.23K:ZooKeeper, 48.15K:SCAM, 47.96K:Etherisc DIP Token, 47.83K:Beast Token, 47.44K:Eggman Inu, 47.38K:Shila Inu, 47.29K:Tairyo Inu, 47.25K:Velo, 47.24K:Cosmic FOMO, 47.19K:KickToken, 47.06K:Dubbz, 46.86K:Bitcoin Confidential, 46.78K:Ethereum Message Search, 46.35K:XBOT, 46.24K:Ordinals, 46.23K:CageCoin, 46.13K:YUNA, 45.97K:STARSHIP, 45.73K:Butter, 45.56K:HitBTC Token, 45.55K:NEOBITCOIN, 45.49K:Steve, 45.29K:OHearn, 45.09K:DYOR Token, 44.90K:Velocore, 44.81K:Extraterrestrial Token, 44.62K:Baby Pepe, 44.52K:Shinjarium, 44.14K:Ethereans, 44.00K:Fonzy, 43.98K:MAGA, 43.89K:Matrix AI Network, 43.65K:XBOT, 43.60K:Polycat Finance, 43.55K:Bored Ark, 43.41K:Trad, 43.40K:Karate Combat, 43.40K:Buff Doge Coin, 43.40K:EvolveAI, 43.39K:Bitcoin Confidential, 43.32K:Kevin, 43.13K:Chad Coin, 43.10K:Yuan, 43.08K:UniBot, 43.05K:NIBBLER, 43.05K:ShibeCoin, 42.85K:Cat Token, 42.84K:Crypto X, 42.79K:Epic Cash, 42.78K:Shiba V Pepe, 42.76K:Trendin, 42.62K:Panda DAO, 42.61K:MOMO, 42.58K:Kochi Ken, 42.53K:HEX, 42.52K:Moomonster, 42.44K:SCOOBY, 42.41K:Shib 2.0, 42.30K:wShiba, 42.14K:SpongeBob, 41.80K:The Habibiz, 41.77K:Ducker, 41.75K:Organic Inu, 41.72K:MIRACLE, 41.70K:REALPAY, 41.70K:BitcoinUltra, 41.57K:LANDWOLF, 41.45K:Next Generation Network, 41.38K:6 Pack Rick, 41.30K:Stray Dog, 41.10K:RIZZO, 41.08K:HarryPotterObamaSonic10Inu (ERC-20), 40.97K:Cipher Protocol, 40.78K:Shibu Life, 40.75K:Tranchess, 40.72K:PAAL AI, 40.72K:Supreme Finance, 40.63K:Xenlon Mars, 40.56K:PussyCat, 40.45K:OrangeCoin, 40.43K:Salty Coin, 40.35K:Bird.Money, 40.32K:XRP, 40.27K:AnonZK, 40.01K:ethcoin, 39.99K:KAKA NFT World, 39.86K:Gamerse, 39.75K:Supreme Finance, 39.71K:Sun Tzu, 39.45K:Famous Fox Federation, 38.66K:Cult DAO, 38.65K:STONEDFOX, 38.64K:Shrek ERC, 38.64K:Pepe Kid, 38.46K:DYOR Token, 38.42K:Jotaro Token, 38.42K:BULLS, 38.31K:Meme boy, 38.26K:Fair BERC20, 38.25K:Dogechain,38.15K:OmniaVerse, 38.11K:TrumpCoin, 38.10K:HarryPotterTrumpHomerSimpson777Inu, 38.07K:PLUMS, 37.90K:TheNextPepe, 37.50K:Zeus AI, 37.46K:Billy Token, 37.41K:VINCI, 37.38K:Bridge Bot, 37.29K:Alpha Labs, 37.25K:JizzRocket, 37.25K:Wrapped XRP, 37.16K:PEPEMON GO, 37.14K:FAKE COIN, 36.93K:HyperCash, 36.85K:Shibarium Radar, 36.83K:Black Box, 36.82K:Zeus AI, 36.59K:Don-key, 36.58K:Marble Bet, 36.55K:Humans.ai, 36.53K:PonziCoin, 36.52K:Biden Token, 36.46K:MAX Exchange Token, 36.34K:Genius Playboy Billionaire Philanthropist, 36.26K:Dogechain, 36.25K:SuperRare, 36.19K:Fat Pickle, 36.06K:LuckyChip, 35.93K:GigaChad Coin, 35.79K:Pepeki, 35.63K:IQ, 35.62K:SONIC, 35.58K:Satoshis Vision, 35.47K:LINDA, 35.45K:WallstreetBets Classic, 35.44K:Glizzy, 35.20K:Ghost, 35.03K:KeK AI Token, 34.80K:Pepa Inu, 34.80K:XBOT, 34.58K:COFFEE, 34.21K:BendDAO, 34.01K:TWINU, 33.90K:Smudge Lord, 33.89K:Merit Circle, 33.81K:One Hundred Million Inu, 33.76K:KEK, 33.76K:Doge King, 33.69K:Pepe 3.0, 33.65K:Delrey Inu, 33.62K:XDoge, 33.55K:Apron Network, 33.45K:BASEDPEPE, 33.20K:AOL Coin, 33.13K:Smurf, 33.11K:Billion Elon Baby, 33.09K:HarryPotterObamaSonic11Inu, 32.97K:Soju Finance, 32.88K:For Teh Culture, 32.87K:MurAll, 32.76K:CEO, 32.70K:MicroPee, 32.59K:Loyalty Labs, 32.43K:ShineChain, 32.41K:Siacoin, 32.34K:Siacoin, 32.33K:Rhino Token, 32.32K:Artify, 32.27K:Siacoin, 32.24K:Pug Cash, 32.23K:Nezuko, 32.12K:Billion Elon Baby, 32.07K:BOWL SHIBARIUM, 32.01K:Earth Token, 31.90K:Monke, 31.68K:BOI, 31.64K:Omega Protocol Money, 31.39K:Gamerse, 31.30K:OrangeCoin, 31.14K:Peercoin, 31.11K:Maecenas, 31.09K:TSUYU, 31.01K:Shibnaut, 31.01K:Ethancoin, 30.82K:CATCOIN, 30.75K:REIMBURSE, 30.59K:Panther Protocol, 30.55K:DracoMalfoyDeSantisMario420Inu, 30.52K:CloudTx, 30.49K:LEVERADE, 30.48K:FUKBEN, 30.44K:ZoomSwap, 30.39K:Fate Token, 30.27K:MAGA, 30.17K:Pep ManCity, 30.15K:ContentBox, 29.96K:Onigiri Neko, 29.90K:Lumishare, 29.77K:Shekel, 29.73K:Pepe Inu, 29.56K:Level Finance, 29.53K:CARTMAN, 29.52K:Kuma Inu, 29.48K:Gatsby Inu, 29.41K:BitcoinPepe, 29.37K:MELD, 29.24K:SLINK LABS, 29.20K:Ethereum, 29.19K:Cojam, 28.96K:AstridDAO, 28.96K:PomPom, 28.92K:MURICA, 28.85K:GameCoin, 28.79K:TOM, 28.68K:Pika Protocol, 28.48K:Pisscoin, 28.39K:Psyop, 28.35K:The Token Kennel, 28.25K:Canary, 28.23K:HarryPotterObamaWallStreetBets10Inu, 28.19K:TrustPlus, 28.03K:Mantle Inu, 28.03K:Berylbit Layer-3 Network (new), 27.96K:SAUDI PEPE, 27.96K:COLLARSWAP, 27.95K:HappyFans, 27.95K:HeroesTD, 27.86K:Psycho, 27.81K:Made In China, 27.78K:BDSM, 27.70K:Shinomics, 27.68K:Aavegotchi KEK, 27.62K:Normie, 27.57K:Panda DAO, 27.28K:Fuck Gary Gensler, 27.22K:Fossa Token, 27.20K:TrendAI, 27.20K:Sold Early, 27.11K:80085 Token, 27.11K:Version, 27.06K:GM Wagmi, 27.03K:SuperNET, 26.98K:CHRISTMAS SHIBA, 26.97K:HippoFinance, 26.82K:MAD Bucks, 26.80K:Flona, 26.80K:MsgSender, 26.75K:YUMMY, 26.69K:BlockEscrow, 26.60K:The Gamble Kingdom, 26.53K:Pomskey, 26.52K:SoPay, 26.50K:TouchFuture, 26.49K:Pepe Inu, 26.48K:Yogi, 26.44K:PUBLISH, 26.38K:Pepe Doge, 26.24K:Sluts, 26.12K:LEAP Token, 26.09K:Scrooge Coin, 26.06K:Recast1 Coin, 26.01K:Hobbes, 26.01K:Simmitri, 25.99K:SEC, 25.98K:Matrix Labs, 25.92K:PepeDAO Coin, 25.92K:Baby Jesus, 25.84K:Angry Pepe, 25.81K:Hacken Token, 25.61K:PumSwap, 25.56K:Shitcoin, 25.54K:PeePee, 25.53K:gp, 25.23K:Doge-1 Mission to the moon, 25.20K:BOOBY, 25.19K:Polygon, 25.17K:SHIBONE INU, 25.09K:Mr. Narco, 25.08K:Bloktopia, 25.07K:NEET Finance, 25.02K:Melon, 24.99K:Jinni, 24.98K:CAPTcoin, 24.97K:Laser Eyes, 24.91K:RICH PEPE, 24.74K:CrazyFrog, 24.72K:Bitcoin Confidential, 24.69K:Naruto Inu, 24.64K:PEPE CLASSIC, 24.61K:Ethereans, 24.57K:Honey Badger Coin, 24.56K:Zelda Inu, 24.51K:Hamster, 24.47K:DOOR, 24.46K:Edge, 24.27K:PEPEMOON, 24.23K:Wheelers, 23.98K:Synapse, 23.97K:Bart ERC, 23.76K:SnakeEyes, 23.73K:TTcoin, 23.65K:AnimalFam, 23.61K:BitDNS, 23.58K:BOBO, 23.56K:r/CryptoCurrency Moons, 23.55K:WaifuAI, 23.53K:Tokenomy, 23.35K:Bitcoin God, 23.24K:Waterfall DeFi, 23.22K:HairDAO, 23.19K:XCurrency, 23.16K:Feels Good Man, 23.10K:IKUN COIN, 23.06K:Send, 22.93K:MemeDrop, 22.77K:Force DAO, 22.77K:Popecoin, 22.75K:SydPak, 22.59K:Darkkush, 22.59K:Kokoswap, 22.49K:CareCoin, 22.47K:Natiol Infinity, 22.47K:MetaVerse-M, 22.37K:XPro, 22.34K:Xanoncoin, 22.34K:ChatCoin, 22.33K:Mumu, 22.29K:ALPHALABS, 22.29K:PEPEMINER, 22.24K:EcoTool, 22.22K:DeepFuckingValue, 22.21K:SHIBA LIGHT, 22.18K:Froki, 22.13K:ZachXBT, 22.13K:Bitcoin Turbo Koin, 22.05K:Shirtum, 22.03K:Yugen Finance, 22.02K:Coin Fast Alert V2, 22.00K:Kusa Inu, 21.95K:ANDY, 21.93K:DragonBite, 21.84K:Rocket Yield, 21.83K:AnonZK, 21.81K:Gen Wealth, 21.75K:Okiku Kento, 21.74K:Jerry Inu, 21.72K:Bard Protocol, 21.36K:Noodles, 21.29K:Ordinal Chain, 21.25K:Super China, 21.19K:SMUDGE, 21.19K:TrustSwap, 21.13K:zkApes, 21.12K:NCAT, 21.06K:Ligma, 21.04K:Swirl Cash, 20.93K:Bitcoin Confidential, 20.91K:NYAN Meme Coin, 20.91K:ChadGPT, 20.90K:Harpoon, 20.89K:DinoSwap, 20.82K:Grim Finance, 20.81K:Rangers Fan Token, 20.80K:Send, 20.79K:Flipped Pepe, 20.47K:Generaitiv, 20.44K:Boomer Coin, 20.43K:Musk Dao, 20.31K:Akita DAO, 20.27K:Bald, 20.14K:Memeira, 20.14K:FLORK, 20.13K:Okage Inu, 20.10K:Aavegotchi KEK, 20.06K:Naka Bodhi Token, 20.01K:Bonk, 19.99K:PomPom, 19.98K:GOLD AI NETWORK TOKEN, 19.97K:Waifu, 19.97K:Kekwcoin, 19.89K:Rogan Coin, 19.87K:Zogzcoin, 19.86K:SEC, 19.70K:Probably ",
      //   },

      //   {
      //     role: "assistant",
      //     content: "Understood, please continue with the list.",
      //   },

      {
        role: "user",
        content:
          " Pepe Token, PepeUSDT, Philosoraptor, Pitbull, PlanetCats, Poglana, PopeCoin, Pudgy Cat, PufDAO, Pu.suke Inu, RagingElonMarsCoin, Rasta Kitty, RichAI, Rottolabs, Salty Coin, SCOOBY,   Shaun Inu, Sheikh Inu, SHIBA BSC, Shiba Cartel, Shiba Fantom, Shiba Inu Mother, Shibaken Finance, ShibaPoconk, SHIBCAT, SHIBGF, Shila Inu, Shira Cat, Skrimples, Smudge Lord, Snek, Son Of Pepe, sooooooooooooooooooooooooooooooooooooooooooooooo, SpookyShiba, SPX6900, Tail, Thing, Tipja, Toad Killer, TORA, Toshi, Treat Token, Turk Shiba, Twitter CEO Floki, Unichad, Unleashed Beast, USDEBT, Wall Street Baby, WASSIE, Where Did The ETH Go?, Wolf of Wall Street, Zoomer, 4Chan, 9 Lives Network, Aardvark, AI Floki, Akita DAO, Akita Inu, Antis Inu, Arb Furbo, Arbinu, ArbPanda AI, Australian Safe Shepherd, Baby Floki, BabyPepe, Baby Pepe, Baby Shiba Inu, Bananace, Bark, Bear Inu, BetbuInu, Binance-Peg Dogecoin, Black Eyed Dragon, Bobo, Boss, BullBear AI, Bull Run, BURN, Caroline, Cartel Coin, CATCEO, Catcoin BSC, Cats, Caw CEO, CEO, Chihuahua, CHILI, ChillPill, ChitCAT, Clown Pepe, Cope Coin, CoShi Inu, CryptoTwitter, DeathWolf, Doge 2.0, DogeBonk, Doge CEO, DOGE CEO AI, Doge Eat Doge, Doge KaKi, Dogelana, Doke Inu, DongCoin, DPRK Coin, Ducks, El Hippo, Evil Pepe, FAT CAT, Floki Cash, FLOKI CEO, FlokiDash, FLOSHIDO INU, Fonzy. Done, now what should the name of the ERC20 token be? Try to be creative! Just reply with the name, no surrounding text or sentence",
      },
    ],
  };

  //make request for supply
  try {
    const responseSupply = await axios.post(
      "https://api.openai.com/v1/chat/completions",
      dataSupply,
      { headers: headers }
    );
    const supply = responseSupply.data.choices[0].message.content;
    parameters.push(supply);
  } catch (error) {
    console.error("Error making request:", error);
  }

  //variables to generate name
  let nameGivenByAI;
  let descriptionGivenByAI;

  //make request for name
  try {
    const responseName = await axios.post(
      "https://api.openai.com/v1/chat/completions",
      dataName,
      { headers: headers }
    );
    nameGivenByAI = responseName.data.choices[0].message.content;
    parameters.push(nameGivenByAI);
  } catch (error) {
    console.error("Error making request:", error);
  }

  //description for generated erc20 token that is fed into dall-e to generate logo
  const dataDescription = {
    model: "gpt-4",
    messages: [
      {
        role: "system",
        content:
          "Hello GPT, today you have been tasked with a very important procedure. I need you to help come up with a description for an erc-20 token logo that will be then fed into a ai generative image model called Dall-e to output an image. ",
      },
      {
        role: "user",
        content: `Write a short creative description for a token named ${nameGivenByAI}. This will be used by Dalle (a text to ML model by open ai) to generate an image for the token. It must be creative and fit in a circle, visually depicting the token name. Keep it under a 100 words and don't mention words in the logo. `,
      },
    ],
  };

  try {
    const responseDescription = await axios.post(
      "https://api.openai.com/v1/chat/completions",
      dataDescription,
      { headers: headers }
    );
    descriptionGivenByAI = responseDescription.data.choices[0].message.content;
    console.log(descriptionGivenByAI);
  } catch (error) {
    console.error("DALL-E API request error:", error);
    // ... (rest of your error handling code)
  }

  // DALL-E Integration Starts Here

  // Make sure to set DALL-E API key
  const DALLE_API_KEY = process.env.OPENAI_API_KEY;
  const DALLE_API_ENDPOINT = "https://api.openai.com/v1/images/generations"; // Replace with your DALL-E API endpoint

  const promptForImage = descriptionGivenByAI; // Use the description from GPT-3 as the prompt for DALL-E

  const imageRequestData = {
    prompt: promptForImage,
    n: 1,
    model: "dall-e-3",
    size: "1024x1024",
  };

  // Prepare API headers for DALL-E
  const dalle_headers = {
    "Content-Type": "application/json",
    Authorization: `Bearer ${DALLE_API_KEY}`,
  };

  //write to file of directory
  const filePath = `${__dirname}/tokenLogo.png`;

  //make dall-e token logo request
  console.log("Making DALL-E API request...");
  try {
    const response = await axios.post(DALLE_API_ENDPOINT, imageRequestData, {
      headers: dalle_headers,
    });

    // Extract the image URL from the response
    const imageUrl = response.data.data[0].url;
    console.log("DALL-E Image URL:", imageUrl);

    // Download the image from the provided URL
    const imageResponse = await axios.get(imageUrl, {
      responseType: "arraybuffer",
    });
    console.log("Saving image to:", filePath);
    fs.writeFileSync(filePath, imageResponse.data);
    console.log("Image successfully saved.");
  } catch (error: any) {
    console.error("DALL-E API request error:", error.message);
    if (error.response && error.response.data && error.response.data.error) {
      console.error("API Error:", error.response.data.error.message);
    }
  }

  //prompt for symvbol
  const dataSymbol = {
    model: "gpt-3.5-turbo",
    messages: [
      {
        role: "system",
        content:
          "Hello GPT, today you have been tasked with a very important procedure. It is imperative that you break down each and every word in this message to the fullest of your extent, and carefully consider all of your options before coming up with a final conclusion in this conversation. Humankind has set out on a delicate mission. A mission so important, that the entirety of human civilizations depends on it. Our goal? Sophisticated AI-deployment of a token on Ethereum. You will take in fields of data about tokens on Ethereum, consider the name of the token, it’s meaning, and it’s supply in relation to the market capitalization’s provided, and use all of the data combined to determine your own token parameters for the next global currency. The name needs to be short, witty, and fit with the crypto ethos. Now that you have taken all of this information into consideration, without any extra context, you will now generate a set of parameters including name, symbol, and supply for the first ai-deployed token in history Do not respond with explanations, just values.",
      },
      {
        role: "user",
        content:
          "What should the name of the ERC20 token be? Just reply with the name, not an explanation",
      },
      {
        role: "assistant",
        content: nameGivenByAI,
      },
      {
        role: "user",
        content:
          "What should the symbol for the ERC20 token be? Base it on the token name and use these popular coin symbols as a reference: doge, shib, pepe, floki, babydoge, bitcoin, elon, leash, bad, ladys, mog, kishu, tsuka, qom, volt, snail, samo, aidoge, bonk, tama, squidgrow, cate, vinu, catgirl, ban, turbo, dog, dino, wojak, hoge, bob, polydoge, dogegf, huahua, jesus, npc, cummies, elmo, dingo, $sharbi, 4token, ethereum, wsb, cinu, monsta, dobo, shib, nfd, meme, pepes, edoge, smi, four, kuma, woofy, cat, kat, $cramer, sov, risita, poodl, shibx, cat, ppblz, kawa, chad, metadoge, ducker, shih, nishib, corgi, kangal, cabo, 3d3d, pepecola, updog, sbonk, dyor, pypy, crop, rb, doe, kitty, solab, titan, jeff, jejudoge, jerry, jshiba, kabosu, kek, kermit, kitti, laelaps, laika, ( ͡° ͜ʖ ͡°), letsgo, lfg, lindaceo, ldoge, lhinu, trump, mic, maru, meme, metadoge, mpepe, mikawa, momo v2, mona, mong, $mpepe, mouseworm, mrspepe, mumu, mutant, navyseal, neko, net, normie, thing, nyan, okage, opinu, papi, pat, pepa, pepeai, pepechain, peped, pepef, pepelon, pov, peplay, pepe, pepebnb, pepe, ppusdt, rap, pit, catcoin, pog, pope, $pudgy, puf, pusuke, dogecoin, ras, richai, rotto, salty, scooby, shaun, shinu, shibsc, pesos, shiba, shibm, shibaken, conk, shibcat, shibgf, shil, catshira, skrimp, smudge, snek, sop, sooooooooo, spky, spx, tail, thing, tipja, $toad, tora, toshi, treat, tushi, flokiceo, unichad, beast, usdebt, wsb, wassie, wheth, $wolf, zoomer, 4chan, ninefi, ardvrk, aifloki, hachi, akita, antis, farb, arbinu, aipanda, ass, babyfloki, babypepe, babypepe, babyshib, nana, bark, bear, crypto, doge, bleyd, bobo, boss, aibb, bull, burn, her, cartel, catceo, cat, cats, cawceo, ceo, hua, chili, $chill, chitcat, honk, cope, coshi, ct, dth, doge2.0, dobo, dogeceo, dogeceo, omnom, kaki, dgln, doke, dong, dprk, ducks, hipp, evilpepe, fatcat, flokicash, flokiceo, flokidash, floshido, fonzy. Just reply with the ticker, no explanation or sentence or surrounding text",
      },
    ],
  };

  //request for suymbol
  try {
    const responseSymbol = await axios.post(
      "https://api.openai.com/v1/chat/completions",
      dataSymbol,
      { headers: headers }
    );
    const symbolGivenByAI = responseSymbol.data.choices[0].message.content;
    parameters.push(symbolGivenByAI);
  } catch (error) {
    console.error("Error making request:", error);
  }

  console.log(parameters[0]);
  console.log(parameters[1]);
  console.log(parameters[2]);
  return parameters;
}
//helper function to use in solc compilation, finds imports for openzeppelin
function findImports(pathToFind: string) {
  try {
    const fullPath = path.join(__dirname, "..", "node_modules", pathToFind);
    const contents = fs.readFileSync(fullPath, "utf8");
    return { contents: contents };
  } catch (error) {
    return { error: "File not found" };
  }
}
//compiles token contract with solc
async function compileContract(fullResponse: string, contractName: string) {
  let input = {
    language: "Solidity",
    sources: {
      [`${contractName}.sol`]: {
        content: fullResponse,
      },
    },
    settings: {
      outputSelection: {
        "*": {
          "*": ["*"],
        },
      },
    },
  };

  let compiledContract = JSON.parse(
    solc.compile(JSON.stringify(input), { import: findImports })
  );
  console.log(compiledContract);
  const bytecode =
    compiledContract.contracts[`${contractName}.sol`][contractName].evm.bytecode
      .object;
  const abi =
    compiledContract.contracts[`${contractName}.sol`][contractName].abi;

  return { abi, bytecode };
}
//deploys erc20 contract with name,symbol and totalsupply
async function deployContract(abi: any, bytecode: any, parameters: any) {
  const [deployer] = await ethers.getSigners();
  console.log("Deploying the contract with the account:", deployer.address);

  let ContractFactory = new ethers.ContractFactory(abi, bytecode, deployer);
  let contractInstance = await ContractFactory.deploy(
    parameters[1],
    parameters[2],
    ethers.BigNumber.from(parameters[0])
  );
  await contractInstance.deployed();

  console.log("Contract deployed to:", contractInstance.address);
  return contractInstance.address;
}
//adds liquidity for token to uniswap pool
async function addLiquidityToNewPool(
  tokenAddress: string,
  amountToken: string,
  amountETH: string,
  signer: Signer
): Promise<void> {
  // Set router address and ABI
  const routerAddress = "0x7a250d5630B4cF539739dF2C5dAcb4c659F2488D";

  // Uniswap Router ABI
  const uniswapRouterABI: any[] = [
    {
      inputs: [
        { internalType: "address", name: "_factory", type: "address" },
        { internalType: "address", name: "_WETH", type: "address" },
      ],
      stateMutability: "nonpayable",
      type: "constructor",
    },
    {
      inputs: [],
      name: "WETH",
      outputs: [{ internalType: "address", name: "", type: "address" }],
      stateMutability: "view",
      type: "function",
    },
    {
      inputs: [
        { internalType: "address", name: "tokenA", type: "address" },
        { internalType: "address", name: "tokenB", type: "address" },
        { internalType: "uint256", name: "amountADesired", type: "uint256" },
        { internalType: "uint256", name: "amountBDesired", type: "uint256" },
        { internalType: "uint256", name: "amountAMin", type: "uint256" },
        { internalType: "uint256", name: "amountBMin", type: "uint256" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "addLiquidity",
      outputs: [
        { internalType: "uint256", name: "amountA", type: "uint256" },
        { internalType: "uint256", name: "amountB", type: "uint256" },
        { internalType: "uint256", name: "liquidity", type: "uint256" },
      ],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "address", name: "token", type: "address" },
        {
          internalType: "uint256",
          name: "amountTokenDesired",
          type: "uint256",
        },
        { internalType: "uint256", name: "amountTokenMin", type: "uint256" },
        { internalType: "uint256", name: "amountETHMin", type: "uint256" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "addLiquidityETH",
      outputs: [
        { internalType: "uint256", name: "amountToken", type: "uint256" },
        { internalType: "uint256", name: "amountETH", type: "uint256" },
        { internalType: "uint256", name: "liquidity", type: "uint256" },
      ],
      stateMutability: "payable",
      type: "function",
    },
    {
      inputs: [],
      name: "factory",
      outputs: [{ internalType: "address", name: "", type: "address" }],
      stateMutability: "view",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountOut", type: "uint256" },
        { internalType: "uint256", name: "reserveIn", type: "uint256" },
        { internalType: "uint256", name: "reserveOut", type: "uint256" },
      ],
      name: "getAmountIn",
      outputs: [{ internalType: "uint256", name: "amountIn", type: "uint256" }],
      stateMutability: "pure",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountIn", type: "uint256" },
        { internalType: "uint256", name: "reserveIn", type: "uint256" },
        { internalType: "uint256", name: "reserveOut", type: "uint256" },
      ],
      name: "getAmountOut",
      outputs: [
        { internalType: "uint256", name: "amountOut", type: "uint256" },
      ],
      stateMutability: "pure",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountOut", type: "uint256" },
        { internalType: "address[]", name: "path", type: "address[]" },
      ],
      name: "getAmountsIn",
      outputs: [
        { internalType: "uint256[]", name: "amounts", type: "uint256[]" },
      ],
      stateMutability: "view",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountIn", type: "uint256" },
        { internalType: "address[]", name: "path", type: "address[]" },
      ],
      name: "getAmountsOut",
      outputs: [
        { internalType: "uint256[]", name: "amounts", type: "uint256[]" },
      ],
      stateMutability: "view",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountA", type: "uint256" },
        { internalType: "uint256", name: "reserveA", type: "uint256" },
        { internalType: "uint256", name: "reserveB", type: "uint256" },
      ],
      name: "quote",
      outputs: [{ internalType: "uint256", name: "amountB", type: "uint256" }],
      stateMutability: "pure",
      type: "function",
    },
    {
      inputs: [
        { internalType: "address", name: "tokenA", type: "address" },
        { internalType: "address", name: "tokenB", type: "address" },
        { internalType: "uint256", name: "liquidity", type: "uint256" },
        { internalType: "uint256", name: "amountAMin", type: "uint256" },
        { internalType: "uint256", name: "amountBMin", type: "uint256" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "removeLiquidity",
      outputs: [
        { internalType: "uint256", name: "amountA", type: "uint256" },
        { internalType: "uint256", name: "amountB", type: "uint256" },
      ],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "address", name: "token", type: "address" },
        { internalType: "uint256", name: "liquidity", type: "uint256" },
        { internalType: "uint256", name: "amountTokenMin", type: "uint256" },
        { internalType: "uint256", name: "amountETHMin", type: "uint256" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "removeLiquidityETH",
      outputs: [
        { internalType: "uint256", name: "amountToken", type: "uint256" },
        { internalType: "uint256", name: "amountETH", type: "uint256" },
      ],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "address", name: "token", type: "address" },
        { internalType: "uint256", name: "liquidity", type: "uint256" },
        { internalType: "uint256", name: "amountTokenMin", type: "uint256" },
        { internalType: "uint256", name: "amountETHMin", type: "uint256" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "removeLiquidityETHSupportingFeeOnTransferTokens",
      outputs: [
        { internalType: "uint256", name: "amountETH", type: "uint256" },
      ],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "address", name: "token", type: "address" },
        { internalType: "uint256", name: "liquidity", type: "uint256" },
        { internalType: "uint256", name: "amountTokenMin", type: "uint256" },
        { internalType: "uint256", name: "amountETHMin", type: "uint256" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
        { internalType: "bool", name: "approveMax", type: "bool" },
        { internalType: "uint8", name: "v", type: "uint8" },
        { internalType: "bytes32", name: "r", type: "bytes32" },
        { internalType: "bytes32", name: "s", type: "bytes32" },
      ],
      name: "removeLiquidityETHWithPermit",
      outputs: [
        { internalType: "uint256", name: "amountToken", type: "uint256" },
        { internalType: "uint256", name: "amountETH", type: "uint256" },
      ],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "address", name: "token", type: "address" },
        { internalType: "uint256", name: "liquidity", type: "uint256" },
        { internalType: "uint256", name: "amountTokenMin", type: "uint256" },
        { internalType: "uint256", name: "amountETHMin", type: "uint256" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
        { internalType: "bool", name: "approveMax", type: "bool" },
        { internalType: "uint8", name: "v", type: "uint8" },
        { internalType: "bytes32", name: "r", type: "bytes32" },
        { internalType: "bytes32", name: "s", type: "bytes32" },
      ],
      name: "removeLiquidityETHWithPermitSupportingFeeOnTransferTokens",
      outputs: [
        { internalType: "uint256", name: "amountETH", type: "uint256" },
      ],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "address", name: "tokenA", type: "address" },
        { internalType: "address", name: "tokenB", type: "address" },
        { internalType: "uint256", name: "liquidity", type: "uint256" },
        { internalType: "uint256", name: "amountAMin", type: "uint256" },
        { internalType: "uint256", name: "amountBMin", type: "uint256" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
        { internalType: "bool", name: "approveMax", type: "bool" },
        { internalType: "uint8", name: "v", type: "uint8" },
        { internalType: "bytes32", name: "r", type: "bytes32" },
        { internalType: "bytes32", name: "s", type: "bytes32" },
      ],
      name: "removeLiquidityWithPermit",
      outputs: [
        { internalType: "uint256", name: "amountA", type: "uint256" },
        { internalType: "uint256", name: "amountB", type: "uint256" },
      ],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountOut", type: "uint256" },
        { internalType: "address[]", name: "path", type: "address[]" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "swapETHForExactTokens",
      outputs: [
        { internalType: "uint256[]", name: "amounts", type: "uint256[]" },
      ],
      stateMutability: "payable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountOutMin", type: "uint256" },
        { internalType: "address[]", name: "path", type: "address[]" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "swapExactETHForTokens",
      outputs: [
        { internalType: "uint256[]", name: "amounts", type: "uint256[]" },
      ],
      stateMutability: "payable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountOutMin", type: "uint256" },
        { internalType: "address[]", name: "path", type: "address[]" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "swapExactETHForTokensSupportingFeeOnTransferTokens",
      outputs: [],
      stateMutability: "payable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountIn", type: "uint256" },
        { internalType: "uint256", name: "amountOutMin", type: "uint256" },
        { internalType: "address[]", name: "path", type: "address[]" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "swapExactTokensForETH",
      outputs: [
        { internalType: "uint256[]", name: "amounts", type: "uint256[]" },
      ],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountIn", type: "uint256" },
        { internalType: "uint256", name: "amountOutMin", type: "uint256" },
        { internalType: "address[]", name: "path", type: "address[]" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "swapExactTokensForETHSupportingFeeOnTransferTokens",
      outputs: [],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountIn", type: "uint256" },
        { internalType: "uint256", name: "amountOutMin", type: "uint256" },
        { internalType: "address[]", name: "path", type: "address[]" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "swapExactTokensForTokens",
      outputs: [
        { internalType: "uint256[]", name: "amounts", type: "uint256[]" },
      ],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountIn", type: "uint256" },
        { internalType: "uint256", name: "amountOutMin", type: "uint256" },
        { internalType: "address[]", name: "path", type: "address[]" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "swapExactTokensForTokensSupportingFeeOnTransferTokens",
      outputs: [],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountOut", type: "uint256" },
        { internalType: "uint256", name: "amountInMax", type: "uint256" },
        { internalType: "address[]", name: "path", type: "address[]" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "swapTokensForExactETH",
      outputs: [
        { internalType: "uint256[]", name: "amounts", type: "uint256[]" },
      ],
      stateMutability: "nonpayable",
      type: "function",
    },
    {
      inputs: [
        { internalType: "uint256", name: "amountOut", type: "uint256" },
        { internalType: "uint256", name: "amountInMax", type: "uint256" },
        { internalType: "address[]", name: "path", type: "address[]" },
        { internalType: "address", name: "to", type: "address" },
        { internalType: "uint256", name: "deadline", type: "uint256" },
      ],
      name: "swapTokensForExactTokens",
      outputs: [
        { internalType: "uint256[]", name: "amounts", type: "uint256[]" },
      ],
      stateMutability: "nonpayable",
      type: "function",
    },
    { stateMutability: "payable", type: "receive" },
  ];

  // Initialize Uniswap Router Contract
  const router = new ethers.Contract(routerAddress, uniswapRouterABI, signer);

  // Approve the token transfer
  const approveAbi = [
    "function approve(address spender, uint amount) external returns(bool)",
  ];
  const tokenContract = new ethers.Contract(tokenAddress, approveAbi, signer);
  await tokenContract.approve(
    routerAddress,
    ethers.BigNumber.from(amountToken)
  );

  // Parameters for adding liquidity
  const amountTokenDesired = ethers.BigNumber.from(amountToken);
  const amountETHDesired = ethers.BigNumber.from(amountETH);

  const amountTokenMin = amountTokenDesired;
  const amountETHMin = amountETHDesired;

  const to = await signer.getAddress(); // Your wallet address
  const deadline = Math.floor(Date.now() / 1000) + 60 * 20; // 20 minutes from the current Unix time

  // Add liquidity
  const tx = await router.addLiquidityETH(
    tokenAddress,
    amountTokenDesired,
    amountTokenMin,
    amountETHMin,
    to,
    deadline,
    {
      value: amountETHDesired,
    }
  );

  console.log(`Transaction hash: ${tx.hash}`);
}

function delay(ms: number) {
  return new Promise((resolve) => setTimeout(resolve, ms));
}

async function main() {
  const [deployer] = await ethers.getSigners();

  //generate tokenparameters and logo
  const parameters = await getERC20Parameters();
  //add decimals to big number
  const totalSupply = (parameters[0] = parameters[0] + "000000000000000000");
  //write erc20 contract to file
  const fullResponse = await generateERC20Contract(parameters[1]);
  const contractNameRegex = /contract (\w+) is/g;
  const match = contractNameRegex.exec(fullResponse!);
  const contractName = match && match[1] ? match[1] : "MyToken";
  console.log(parameters);
  //compile generated erc20 contract
  const { abi, bytecode } = await compileContract(fullResponse!, contractName);
  //deploy contract and store address
  const tokenAddress = await deployContract(abi, bytecode, parameters);

  //amount of liquidty to add to uniswap pool
  const amountETH = ethers.utils.parseEther("0.01").toString();

  //add liquidity to uniswap pool
  await addLiquidityToNewPool(tokenAddress, totalSupply, amountETH, deployer);
}

main()
  .then(() => process.exit(0))
  .catch((error) => {
    console.error(error);
    process.exit(1);
  });
