/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2017-2024 Yegor Bugayenko
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.jpeek;

import com.beust.jcommander.ParameterException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import org.cactoos.Scalar;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.llorllale.cactoos.matchers.Assertion;
import org.llorllale.cactoos.matchers.IsTrue;
import org.llorllale.cactoos.matchers.Throws;

/**
 * Test case for {@link Main}.
 * @since 0.1
 * @checkstyle JavadocMethodCheck (500 lines)
 */
@SuppressWarnings("PMD.AvoidDuplicateLiterals")
final class MainTest {

    @Test
    void printsHelp() throws IOException {
        Main.main("--help");
    }

    @Test
    void createsXmlReports(@TempDir final Path temp) throws IOException {
        final Path output = temp.resolve("x3");
        final Path input = Paths.get(".");
        Main.main("--sources", input.toString(), "--target", output.toString());
        new Assertion<>(
            "Must create LCOM5 report",
            Files.exists(output.resolve("LCOM5.xml")),
            new IsTrue()
        ).affirm();
    }

    @Test
    void crashesIfInvalidInput() {
        new Assertion<>(
            "Must throw an exception if parameter is invalid",
            () -> {
                Main.main("hello");
                return "";
            }, new Throws<>(ParameterException.class)
        ).affirm();
    }

    @Test
    void crashesIfNoOverwriteAndTargetExists(@TempDir final Path target) {
        new Assertion<>(
            "Must throw an exception if target exists and no overwrite",
            () -> {
                Main.main(
                    "--sources", Paths.get(".").toString(),
                    "--target", target.toString()
                );
                return "";
            }, new Throws<>(IllegalStateException.class)
        ).affirm();
    }

    @Test
    @SuppressWarnings({"unchecked", "rawtypes"})
    void crashesIfOverwriteAndSourceEqualsToTarget(@TempDir final Path source) {
        new Assertion(
            "Must throw an exception",
            (Scalar<Boolean>) () -> {
                Main.main(
                    "--sources", source.toString(),
                    "--target", source.toString(),
                    "--overwrite"
                );
                return true;
            },
            new Throws(
                "Invalid paths - can't be equal if overwrite option is set.",
                IllegalArgumentException.class
            )
        ).affirm();
    }

    @Test
    @SuppressWarnings({"unchecked", "rawtypes"})
    void crashesIfMetricsHaveInvalidNames(@TempDir final Path target) {
        new Assertion(
            "Must throw an exception",
            (Scalar<Boolean>) () -> {
                Main.main(
                    "--sources", Paths.get(".").toString(),
                    "--target", target.toString(),
                    "--metrics", "#%$!"
                );
                return true;
            },
            new Throws(
                "Invalid metric name: '#%$!'",
                IllegalArgumentException.class
            )
        ).affirm();
    }

    @Test
    void createsXmlReportsIfOverwriteAndTargetExists(@TempDir final Path target)
        throws IOException {
        Main.main(
            "--sources", Paths.get(".").toString(),
            "--target", target.toString(),
            "--overwrite"
        );
        new Assertion<>(
            "Must exists LCOM5.xml",
            Files.exists(target.resolve("LCOM5.xml")),
            new IsTrue()
        ).affirm();
    }
}
