from conan import ConanFile
from conan.errors import ConanInvalidConfiguration
from conan.tools.build import check_min_cppstd
from conan.tools.files import copy, get, rmdir
from conan.tools.cmake import CMake, CMakeToolchain, cmake_layout
from conan.tools.scm import Version
import os

required_conan_version = ">=1.50.0"


class MagicEnumConan(ConanFile):
    name = "magic_enum"
    description = (
        "Header-only C++17 library provides static reflection for enums, work "
        "with any enum type without any macro or boilerplate code."
    )
    license = "MIT"
    url = "https://github.com/conan-io/conan-center-index"
    homepage = "https://github.com/Neargye/magic_enum"
    topics = ("cplusplus", "enum-to-string", "string-to-enum", "serialization",
              "reflection", "header-only", "compile-time")
    package_type = "header-library"
    settings = "os", "arch", "compiler", "build_type"
    no_copy_source = True

    @property
    def _min_cppstd(self):
        return "17"

    @property
    def _compilers_minimum_version(self):
        return {
            "gcc": "9",
            "Visual Studio": "15",
            "msvc": "191",
            "clang": "5",
            "apple-clang": "10",
        }

    def layout(self):
        cmake_layout(self, src_folder="src")

    def package_id(self):
        self.info.clear()

    def validate(self):
        if self.settings.compiler.get_safe("cppstd"):
            check_min_cppstd(self, self._min_cppstd)
        minimum_version = self._compilers_minimum_version.get(str(self.settings.compiler), False)
        if minimum_version and Version(self.settings.compiler.version) < minimum_version:
            raise ConanInvalidConfiguration(
                f"{self.ref} requires C++{self._min_cppstd}, which your compiler does not support.",
            )

    def source(self):
        get(self, **self.conan_data["sources"][self.version], strip_root=True)

    def generate(self):
        tc = CMakeToolchain(self)
        tc.cache_variables["MAGIC_ENUM_OPT_BUILD_EXAMPLES"] = False
        tc.cache_variables["MAGIC_ENUM_OPT_BUILD_TESTS"] = False
        tc.generate()

    def build(self):
        cmake = CMake(self)
        cmake.configure()
        cmake.build()

    def package(self):
        cmake = CMake(self)
        cmake.install()
        if "0.9.4" <= Version(self.version) <= "0.9.6":
            copy(self, "*", os.path.join(self.package_folder, "include"), os.path.join(self.package_folder, "include", "magic_enum"))
        copy(self, "LICENSE", self.source_folder, os.path.join(self.package_folder, "licenses"))
        rmdir(self, os.path.join(self.package_folder, "lib"))
        rmdir(self, os.path.join(self.package_folder, "share"))

    def package_info(self):
        self.cpp_info.set_property("cmake_file_name", "magic_enum")
        self.cpp_info.set_property("cmake_target_name", "magic_enum::magic_enum")
        self.cpp_info.set_property("pkg_config_name", "magic_enum")
        self.cpp_info.bindirs = []
        self.cpp_info.libdirs = []
