from conan import ConanFile
from conan.tools.cmake import CMake, CMakeToolchain, cmake_layout
from conan.tools.files import export_conandata_patches, apply_conandata_patches, copy, get, rmdir
from conan.tools.scm import Version
import os

required_conan_version = ">=1.53.0"

class LibmaxminddbConan(ConanFile):
    name = "libmaxminddb"
    description = "C library for the MaxMind DB file format"
    license = "Apache-2.0"
    url = "https://github.com/conan-io/conan-center-index"
    homepage = "http://maxmind.github.io/libmaxminddb/"
    topics = ("maxmind", "geoip")
    settings = "os", "arch", "compiler", "build_type"
    package_type = "library"
    options = {
        "shared": [True, False],
        "fPIC": [True, False],
        "with_binaries": [True, False],
    }
    default_options = {
        "shared": False,
        "fPIC": True,
        "with_binaries": True,
    }

    def export_sources(self):
        export_conandata_patches(self)

    def config_options(self):
        if self.settings.os == "Windows":
            del self.options.fPIC
        if Version(self.version) < "1.10.0":
            del self.options.with_binaries

    def configure(self):
        if self.options.shared:
            self.options.rm_safe("fPIC")
        self.settings.rm_safe("compiler.libcxx")
        self.settings.rm_safe("compiler.cppstd")

    def layout(self):
        cmake_layout(self, src_folder="src")

    def source(self):
        get(self, **self.conan_data["sources"][self.version],
            destination=self.source_folder, strip_root=True)

    def generate(self):
        tc = CMakeToolchain(self)
        tc.variables["BUILD_TESTING"] = False
        if "with_binaries" in self.options:
            tc.variables["MAXMINDDB_BUILD_BINARIES"] = self.options.with_binaries
        # Honor BUILD_SHARED_LIBS from conan_toolchain (see https://github.com/conan-io/conan/issues/11840)
        tc.cache_variables["CMAKE_POLICY_DEFAULT_CMP0077"] = "NEW"
        tc.generate()

    def build(self):
        apply_conandata_patches(self)
        cmake = CMake(self)
        cmake.configure()
        cmake.build()

    def package(self):
        copy(self, "LICENSE", src=self.source_folder, dst=os.path.join(self.package_folder, "licenses"))
        cmake = CMake(self)
        cmake.install()
        rmdir(self, os.path.join(self.package_folder, "lib", "cmake"))
        rmdir(self, os.path.join(self.package_folder, "lib", "pkgconfig"))

    def package_info(self):
        self.cpp_info.set_property("cmake_file_name", "maxminddb")
        self.cpp_info.set_property("cmake_target_name", "maxminddb::maxminddb")
        self.cpp_info.set_property("pkg_config_name", "libmaxminddb")
        self.cpp_info.libs = ["maxminddb"]
        if self.settings.os == "Windows":
            self.cpp_info.system_libs = ["ws2_32"]

        if self.settings.os != "Windows" and self.options.get_safe("with_binaries", True):
            bin_path = os.path.join(self.package_folder, "bin")
            self.env_info.PATH.append(bin_path)

        # TODO: to remove in conan v2 once cmake_find_package_* generators removed
        self.cpp_info.names["cmake_find_package"] = "maxminddb"
        self.cpp_info.names["cmake_find_package_multi"] = "maxminddb"
