from conan import ConanFile
from conan.errors import ConanInvalidConfiguration
from conan.tools.apple import fix_apple_shared_install_name, XCRun
from conan.tools.build import cross_building
from conan.tools.env import VirtualRunEnv
from conan.tools.files import copy, get, rm
from conan.tools.gnu import Autotools, AutotoolsDeps, AutotoolsToolchain
from conan.tools.layout import basic_layout
import os

required_conan_version = ">=1.54.0"


class LdnsConan(ConanFile):
    name = "ldns"
    license = "BSD-3-Clause"
    url = "https://github.com/conan-io/conan-center-index"
    homepage = "https://www.nlnetlabs.nl/projects/ldns"
    description = "LDNS is a DNS library that facilitates DNS tool programming"
    topics = ("dns")
    package_type = "library"
    settings = "os", "compiler", "build_type", "arch"
    options = {
        "shared": [True, False],
        "fPIC": [True, False],
    }
    default_options = {
        "shared": False,
        "fPIC": True,
    }

    def config_options(self):
        if self.settings.os == "Windows":
            self.options.rm_safe("fPIC")

    def configure(self):
        if self.options.shared:
            self.options.rm_safe("fPIC")
        self.settings.rm_safe("compiler.libcxx")
        self.settings.rm_safe("compiler.cppstd")

    def layout(self):
        basic_layout(self, src_folder="src")

    def requirements(self):
        self.requires("openssl/[>=1.1 <4]")

    def validate(self):
        if self.settings.os == "Windows":
            raise ConanInvalidConfiguration("Windows is not supported by the ldns recipe. Contributions are welcome.")

    def source(self):
        get(self, **self.conan_data["sources"][self.version], strip_root=True)

    def generate(self):
        if not cross_building(self):
            env = VirtualRunEnv(self)
            env.generate(scope="build")

        tc = AutotoolsDeps(self)
        try:
            # This fixes an issue of linking against ldns in combination of openssl:shared=False, ldns:shared=True, and an older GCC:
            # > hidden symbol `pthread_atfork' in /usr/lib/x86_64-linux-gnu/libpthread_nonshared.a(pthread_atfork.oS) is referenced by DSO
            # OpenSSL adds -lpthread to link with POSIX thread library. Instead, it should use -pthread compiler flag which additionally
            # sets required macros at compile time.
            tc.environment.remove("LIBS", "-lpthread")
            tc.environment.append("CFLAGS", "-pthread")
        except ValueError:
            pass
        tc.generate()

        tc = AutotoolsToolchain(self)
        tc.configure_args.extend([
            "--disable-rpath",
            f"--with-ssl={self.dependencies['openssl'].package_folder}",
            # DNSSEC algorithm support
            "--enable-ecdsa",
            "--enable-ed25519",
            "--enable-ed448",
            "--disable-dsa",
            "--disable-gost",
            "--enable-full-dane",
            # tooling
            "--disable-ldns-config",
            "--without-drill",
            "--without-examples",
            # library bindings
            "--without-pyldns",
            "--without-p5-dns-ldns",
        ])
        if self.settings.compiler == "apple-clang":
            tc.configure_args.append(f"--with-xcode-sdk={XCRun(self).sdk_version}")
        tc.generate()

    def build(self):
        autotools = Autotools(self)
        autotools.configure()
        autotools.make()

    def package(self):
        autotools = Autotools(self)
        for target in ["install-h", "install-lib"]:
            autotools.install(target=target)
        rm(self, "*.la", os.path.join(self.package_folder, "lib"))
        copy(self, pattern="LICENSE", src=self.source_folder, dst=os.path.join(self.package_folder, "licenses"))
        fix_apple_shared_install_name(self)

    def package_info(self):
        self.cpp_info.libs = ["ldns"]
