from conans import ConanFile, tools, Meson, VisualStudioBuildEnvironment
from conans.errors import ConanInvalidConfiguration
from conan.tools.microsoft import msvc_runtime_flag
import glob
import os
import shutil


class GStPluginsBadConan(ConanFile):
    name = "gst-plugins-bad"
    description = "GStreamer is a development framework for creating applications like media players, video editors, " \
                  "streaming media broadcasters and so on"
    topics = ("gstreamer", "multimedia", "video", "audio", "broadcasting", "framework", "media")
    url = "https://github.com/conan-io/conan-center-index"
    homepage = "https://gstreamer.freedesktop.org/"
    license = "GPL-2.0-only"
    settings = "os", "arch", "compiler", "build_type"
    options = {
        "shared": [True, False],
        "fPIC": [True, False],
        "with_introspection": [True, False],
        }
    default_options = {
        "shared": False,
        "fPIC": True,
        "with_introspection": False,
        }
    _source_subfolder = "source_subfolder"
    _build_subfolder = "build_subfolder"
    exports_sources = ["patches/*.patch"]

    generators = "pkg_config"

    @property
    def _is_msvc(self):
        return self.settings.compiler == "Visual Studio"

    def validate(self):
        if self.options.shared != self.options["gstreamer"].shared or \
            self.options.shared != self.options["glib"].shared or \
            self.options.shared != self.options["gst-plugins-base"].shared:
                # https://gitlab.freedesktop.org/gstreamer/gst-build/-/issues/133
                raise ConanInvalidConfiguration("GLib, GStreamer and GstPlugins must be either all shared, or all static")
        if tools.Version(self.version) >= "1.18.2" and\
           self.settings.compiler == "gcc" and\
           tools.Version(self.settings.compiler.version) < "5":
            raise ConanInvalidConfiguration(
                "gst-plugins-bad %s does not support gcc older than 5" % self.version
            )
        if self.options.shared and str(msvc_runtime_flag(self)).startswith("MT"):
            raise ConanInvalidConfiguration('shared build with static runtime is not supported due to the FlsAlloc limit')

    def configure(self):
        if self.options.shared:
            del self.options.fPIC
        del self.settings.compiler.libcxx
        del self.settings.compiler.cppstd
        self.options['gstreamer'].shared = self.options.shared
        self.options['gst-plugins-base'].shared = self.options.shared

    def config_options(self):
        if self.settings.os == 'Windows':
            del self.options.fPIC

    def requirements(self):
        self.requires("glib/2.70.1")
        self.requires("gstreamer/1.19.1")
        self.requires("gst-plugins-base/1.19.1")

    def build_requirements(self):
        self.build_requires("meson/0.54.2")
        if not tools.which("pkg-config"):
            self.build_requires("pkgconf/1.7.4")
        if self.settings.os == 'Windows':
            self.build_requires("winflexbison/2.5.24")
        else:
            self.build_requires("bison/3.7.6")
            self.build_requires("flex/2.6.4")
        if self.options.with_introspection:
            self.build_requires("gobject-introspection/1.68.0")

    def source(self):
        tools.get(**self.conan_data["sources"][self.version],
                  destination=self._source_subfolder, strip_root=True)

    def _configure_meson(self):
        defs = dict()

        def add_flag(name, value):
            if name in defs:
                defs[name] += " " + value
            else:
                defs[name] = value

        def add_compiler_flag(value):
            add_flag("c_args", value)
            add_flag("cpp_args", value)

        def add_linker_flag(value):
            add_flag("c_link_args", value)
            add_flag("cpp_link_args", value)

        meson = Meson(self)
        if self.settings.compiler == "Visual Studio":
            add_linker_flag("-lws2_32")
            add_compiler_flag("-%s" % self.settings.compiler.runtime)
            if int(str(self.settings.compiler.version)) < 14:
                add_compiler_flag("-Dsnprintf=_snprintf")
        if self.settings.get_safe("compiler.runtime"):
            defs["b_vscrt"] = str(self.settings.compiler.runtime).lower()
        defs["tools"] = "disabled"
        defs["examples"] = "disabled"
        defs["benchmarks"] = "disabled"
        defs["tests"] = "disabled"
        defs["wrap_mode"] = "nofallback"
        defs["introspection"] = "enabled" if self.options.with_introspection else "disabled"
        meson.configure(build_folder=self._build_subfolder,
                        source_folder=self._source_subfolder,
                        defs=defs)
        return meson

    def build(self):
        for patch in self.conan_data.get("patches", {}).get(self.version, []):
            tools.patch(**patch)

        with tools.environment_append(VisualStudioBuildEnvironment(self).vars) if self._is_msvc else tools.no_op():
            meson = self._configure_meson()
            meson.build()

    def _fix_library_names(self, path):
        # regression in 1.16
        if self.settings.compiler == "Visual Studio":
            with tools.chdir(path):
                for filename_old in glob.glob("*.a"):
                    filename_new = filename_old[3:-2] + ".lib"
                    self.output.info("rename %s into %s" % (filename_old, filename_new))
                    shutil.move(filename_old, filename_new)

    def package(self):
        self.copy(pattern="COPYING", dst="licenses", src=self._source_subfolder)
        with tools.environment_append(VisualStudioBuildEnvironment(self).vars) if self._is_msvc else tools.no_op():
            meson = self._configure_meson()
            meson.install()

        self._fix_library_names(os.path.join(self.package_folder, "lib"))
        self._fix_library_names(os.path.join(self.package_folder, "lib", "gstreamer-1.0"))
        tools.rmdir(os.path.join(self.package_folder, "share"))
        tools.rmdir(os.path.join(self.package_folder, "lib", "pkgconfig"))
        tools.rmdir(os.path.join(self.package_folder, "lib", "gstreamer-1.0", "pkgconfig"))
        tools.remove_files_by_mask(self.package_folder, "*.pdb")

    def package_info(self):

        plugins = ["accurip",
                   "adpcmdec",
                   "adpcmenc",
                   "aiff",
                   "asfmux",
                   "audiobuffersplit",
                   "audiofxbad",
                   "audiolatency",
                   "audiomixmatrix",
                   "audiovisualizers",
                   "autoconvert",
                   "bayer",
                   "camerabin",
                   "codecalpha",
                   "coloreffects",
                   "debugutilsbad",
                   "dvbsubenc",
                   "dvbsuboverlay",
                   "dvdspu",
                   "faceoverlay",
                   "festival",
                   "fieldanalysis",
                   "freeverb",
                   "frei0r",
                   "gaudieffects",
                   "gdp",
                   "geometrictransform",
                   "id3tag",
                   "inter",
                   "interlace",
                   "ivfparse",
                   "ivtc",
                   "jp2kdecimator",
                   "jpegformat",
                   "rfbsrc",
                   "midi",
                   "mpegpsdemux",
                   "mpegpsmux",
                   "mpegtsdemux",
                   "mpegtsmux",
                   "mxf",
                   "netsim",
                   "rtponvif",
                   "pcapparse",
                   "pnm",
                   "proxy",
                   "legacyrawparse",
                   "removesilence",
                   "rist",
                   "rtmp2",
                   "rtpmanagerbad",
                   "sdpelem",
                   "segmentclip",
                   "siren",
                   "smooth",
                   "speed",
                   "subenc",
                   "switchbin",
                   "timecode",
                   "transcode",
                   "videofiltersbad",
                   "videoframe_audiolevel",
                   "videoparsersbad",
                   "videosignal",
                   "vmnc",
                   "y4mdec"]

        gst_plugin_path = os.path.join(self.package_folder, "lib", "gstreamer-1.0")
        if self.options.shared:
            self.output.info("Appending GST_PLUGIN_PATH env var : %s" % gst_plugin_path)
            self.cpp_info.bindirs.append(gst_plugin_path)
            self.runenv_info.prepend_path("GST_PLUGIN_PATH", gst_plugin_path)
            self.env_info.GST_PLUGIN_PATH.append(gst_plugin_path)
        else:
            self.cpp_info.defines.append("GST_PLUGINS_BAD_STATIC")
            self.cpp_info.libdirs.append(gst_plugin_path)
            self.cpp_info.libs.extend(["gst%s" % plugin for plugin in plugins])

        self.cpp_info.includedirs = ["include", os.path.join("include", "gstreamer-1.0")]
