from conan import ConanFile
from conan.errors import ConanInvalidConfiguration
from conan.tools.build import check_min_cppstd
from conan.tools.files import get, copy, rmdir
from conan.tools.cmake import CMake, CMakeDeps, CMakeToolchain, cmake_layout
from conan.tools.scm import Version

import os

required_conan_version = ">=1.52.0"

class DawJsonLinkConan(ConanFile):
    name = "daw_json_link"
    description = "Static JSON parsing in C++"
    license = "BSL-1.0"
    url = "https://github.com/conan-io/conan-center-index"
    homepage = "https://github.com/beached/daw_json_link"
    topics = ("json", "parse", "json-parser", "serialization", "constexpr", "header-only")
    package_type = "header-library"
    settings = "os", "arch", "compiler", "build_type"
    no_copy_source = True
    short_paths = True

    @property
    def _minimum_cpp_standard(self):
        return 17

    @property
    def _compilers_minimum_version(self):
        return {
            "Visual Studio": "16",
            "msvc": "192",
            "gcc": "8",
            "clang": "7",
            "apple-clang": "12",
        }

    def layout(self):
        cmake_layout(self, src_folder="src")

    def requirements(self):
        corresponding_daw_header_version = self.conan_data["daw_headers_mapping"][self.version]
        self.requires(f"daw_header_libraries/{corresponding_daw_header_version}")
        corresponding_daw_utf_version = self.conan_data["daw_utf_mapping"][self.version]
        self.requires(f"daw_utf_range/{corresponding_daw_utf_version}")

    def package_id(self):
        self.info.clear()

    def validate(self):
        if self.settings.compiler.get_safe("cppstd"):
            check_min_cppstd(self, self._minimum_cpp_standard)
        minimum_version = self._compilers_minimum_version.get(str(self.settings.compiler), False)
        if minimum_version and Version(self.settings.compiler.version) < minimum_version:
            raise ConanInvalidConfiguration(
                f"{self.ref} requires C++{self._minimum_cpp_standard}, which your compiler does not support."
            )

    def source(self):
        get(self, **self.conan_data["sources"][self.version], strip_root=True)

    def generate(self):
        tc = CMakeToolchain(self)
        tc.variables["DAW_USE_PACKAGE_MANAGEMENT"] = True
        tc.generate()

        deps = CMakeDeps(self)
        deps.generate()

    def build(self):
        cmake = CMake(self)
        cmake.configure()

    def package(self):
        copy(self, pattern="LICENSE", dst=os.path.join(self.package_folder, "licenses"), src=self.source_folder)
        cmake = CMake(self)
        cmake.install()
        rmdir(self, os.path.join(self.package_folder, "lib", "cmake"))
        rmdir(self, os.path.join(self.package_folder, "share"))

    def package_info(self):
        self.cpp_info.bindirs = []
        self.cpp_info.libdirs = []

        self.cpp_info.set_property("cmake_file_name", "daw-json-link")
        self.cpp_info.set_property("cmake_target_name", "daw::daw-json-link")
        self.cpp_info.components["daw"].set_property("cmake_target_name", "daw::daw-json-link")
        self.cpp_info.components["daw"].requires = ["daw_header_libraries::daw", "daw_utf_range::daw"]

        # TODO: to remove in conan v2 once cmake_find_package_* generators removed
        self.cpp_info.filenames["cmake_find_package"] = "daw-json-link"
        self.cpp_info.filenames["cmake_find_package_multi"] = "daw-json-link"
        self.cpp_info.names["cmake_find_package"] = "daw"
        self.cpp_info.names["cmake_find_package_multi"] = "daw"
        self.cpp_info.components["daw"].names["cmake_find_package"] = "daw-json-link"
        self.cpp_info.components["daw"].names["cmake_find_package_multi"] = "daw-json-link"
