import os

from conan import ConanFile
from conan.tools.build import check_min_cppstd
from conan.tools.cmake import CMake, cmake_layout, CMakeToolchain
from conan.tools.env import VirtualBuildEnv
from conan.tools.files import apply_conandata_patches, copy, export_conandata_patches, get, rmdir
from conan.tools.microsoft import is_msvc
from conan.tools.scm import Version
from conan.errors import ConanInvalidConfiguration

required_conan_version = ">=1.52.0"


class UTConan(ConanFile):
    name = "boost-ext-ut"
    description = ("C++20 single header/single module, "
                   "macro-free micro Unit Testing Framework")
    topics = ("ut", "header-only", "unit-test", "test", "tdd", "bdd")
    url = "https://github.com/conan-io/conan-center-index"
    homepage = "https://boost-ext.github.io/ut/"
    license = "BSL-1.0"
    settings = "os", "compiler", "arch", "build_type"
    no_copy_source = True
    options = { "disable_module": [True, False], }
    default_options = { "disable_module": False, }

    @property
    def _minimum_cpp_standard(self):
        return 17 if self.settings.compiler in ["clang", "gcc"] and Version(self.version) <= "1.1.8" else 20

    @property
    def _minimum_compilers_version(self):
        return {
            "apple-clang": "12" if Version(self.version) < "2.0.0" else "14",
            "clang": "9" if Version(self.version) < "2.0.0" else "10",
            "gcc": "9" if Version(self.version) < "2.0.0" else "10",
            "msvc": "192",
            "Visual Studio": "14",
        }

    def export_sources(self):
        export_conandata_patches(self)

    def config_options(self):
        if Version(self.version) <= "1.1.8":
            del self.options.disable_module
        elif is_msvc(self):
            self.options.disable_module = True

    def layout(self):
        cmake_layout(self, src_folder="src")

    def validate(self):
        if (Version(self.version) == "2.1.0" and self.settings.os == "Linux" and self.settings.compiler == "clang" and
                12 <= Version(self.settings.compiler.version) <= 16 and "libstdc++" in self.settings.compiler.libcxx):
            # https://github.com/boost-ext/ut/issues/637
            raise ConanInvalidConfiguration(f"{self.ref} does support Clang + libstdc++. Use -s compiler.libcxx=libc++ or Clang >16.")

        if self.settings.compiler.get_safe("cppstd"):
            check_min_cppstd(self, self._minimum_cpp_standard)
        if Version(self.version) <= "1.1.8" and is_msvc(self):
            raise ConanInvalidConfiguration(f"{self.ref} may not be built with MSVC. "
                                            "Please use at least version 1.1.9 with MSVC.")
        minimum_version = self._minimum_compilers_version.get(str(self.settings.compiler), False)
        if minimum_version and Version(self.settings.compiler.version) < minimum_version:
            raise ConanInvalidConfiguration(
                f"{self.ref} requires at least version {minimum_version} of the {self.settings.compiler} compiler."
            )

        if is_msvc(self):
            if not self.options.get_safe("disable_module", True):
                self.output.warn("The 'disable_module' option must be enabled when using MSVC.")
        if not is_msvc(self):
            min_version = self._minimum_compilers_version.get(
                str(self.settings.compiler))
            if not min_version:
                self.output.warn(f"{self.ref} recipe lacks information about the {self.settings.compiler} "
                                 "compiler support.")
            else:
                if Version(self.settings.compiler.version) < min_version:
                    raise ConanInvalidConfiguration(
                        f"{self.ref} requires C++{self._minimum_cpp_standard} support. "
                        f"The current compiler {self.settings.compiler} {self.settings.compiler.version} does not support it.")

    def build_requirements(self):
        if Version(self.version) >= "2.0.0":
            self.tool_requires("cmake/[>=3.21 <4]")

    def source(self):
        get(self, **self.conan_data["sources"][self.version], strip_root=True)

    def generate(self):
        tc = CMakeToolchain(self)
        tc.cache_variables["BOOST_UT_BUILD_BENCHMARKS"] = False
        tc.cache_variables["BOOST_UT_BUILD_EXAMPLES"] = False
        tc.cache_variables["BOOST_UT_BUILD_TESTS"] = not self.conf.get("tools.build:skip_test", default=True, check_type=bool)
        tc.cache_variables["PROJECT_DISABLE_VERSION_SUFFIX"] = True
        disable_module = self.options.get_safe("disable_module")
        if disable_module:
            tc.cache_variables["BOOST_UT_DISABLE_MODULE"] = disable_module
        tc.generate()
        virtual_build_env = VirtualBuildEnv(self)
        virtual_build_env.generate()

    def build(self):
        apply_conandata_patches(self)
        cmake = CMake(self)
        cmake.configure()
        cmake.build()

    def package(self):
        copy(self, "LICENSE*", self.source_folder, os.path.join(self.package_folder, "licenses"))
        cmake = CMake(self)
        cmake.install()
        rmdir(self, os.path.join(self.package_folder, "lib", "cmake"))

    def package_id(self):
        self.info.clear()

    def package_info(self):
        newer_than_1_1_8 = Version(self.version) > "1.1.8"
        namespace = "Boost" if newer_than_1_1_8 else "boost"
        self.cpp_info.set_property("cmake_file_name", "ut")
        self.cpp_info.set_property("cmake_target_name", f"{namespace}::ut")

        self.cpp_info.names["cmake_find_package"] = namespace
        self.cpp_info.names["cmake_find_package_multi"] = namespace
        self.cpp_info.filenames["cmake_find_package"] = "ut"
        self.cpp_info.filenames["cmake_find_package_multi"] = "ut"
        self.cpp_info.components["ut"].names["cmake_find_package"] = "ut"
        self.cpp_info.components["ut"].names["cmake_find_package_multi"] = "ut"

        if newer_than_1_1_8:
            self.cpp_info.components["ut"].includedirs = [os.path.join("include", "ut-" + self.version, "include")]

        if self.options.get_safe("disable_module"):
            self.cpp_info.components["ut"].defines = ["BOOST_UT_DISABLE_MODULE=1"]
