// Copyright (c) 2021, Tom Ritter
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice,
//       this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.

import $ from 'jquery';
import * as monaco from 'monaco-editor';
import _ from 'underscore';
import {MonacoPane} from './pane.js';
import * as monacoConfig from '../monaco-config.js';
import {Container} from 'golden-layout';
import {MonacoPaneState} from './pane.interfaces.js';
import {Hub} from '../hub.js';
import {ToolInputViewState} from './tool-input-view.interfaces.js';
import {CompilationResult} from '../compilation/compilation.interfaces.js';
import {CompilerInfo} from '../compiler.interfaces.js';
import {SiteSettings} from '../settings.js';
import {ToolState} from '../components.interfaces.js';

export class ToolInputView extends MonacoPane<monaco.editor.IStandaloneCodeEditor, ToolInputViewState> {
    _toolId: string;
    _toolName: string;
    shouldSetSelectionInitially: boolean;
    debouncedEmitChange: (() => void) & _.Cancelable;
    cursorSelectionThrottledFunction: ((e: any) => void) & _.Cancelable;
    lastChangeEmitted: string | undefined;
    constructor(hub: Hub, container: Container, state: ToolInputViewState & MonacoPaneState) {
        if ((state as any).compilerId) state.id = (state as any).compilerId;
        super(hub, container, state);

        this._toolId = state.toolId;
        this._toolName = state.toolName;
        // TODO according to TS typing this should always be true
        this.shouldSetSelectionInitially = !!this.selection;

        this.updateTitle();
        this.onSettingsChange(this.settings);
    }

    override getInitialHTML() {
        return $('#tool-input').html();
    }

    override createEditor(editorRoot: HTMLElement) {
        this.editor = monaco.editor.create(
            editorRoot,
            monacoConfig.extendConfig({
                value: '',
                language: 'plaintext',
                readOnly: false,
                glyphMargin: true,
            }),
        );
    }

    override getPrintName() {
        return 'Tool Input';
    }

    override registerCallbacks() {
        this.eventHub.on('toolClosed', this.onToolClose, this);
        this.eventHub.on('toolInputViewCloseRequest', this.onToolInputViewCloseRequest, this);
        this.eventHub.on('setToolInput', this.onSetToolInput, this);

        this.container.layoutManager.on('initialised', () => {
            // Once initialized, let everyone know what text we have.
            this.maybeEmitChange(false);
        });
        this.eventHub.on('initialised', () => this.maybeEmitChange(false), this);

        this.editor.getModel()?.onDidChangeContent(() => {
            this.debouncedEmitChange();
            this.updateState();
        });

        this.cursorSelectionThrottledFunction = _.throttle(this.onDidChangeCursorSelection.bind(this), 500);
        this.editor.onDidChangeCursorSelection(e => {
            this.cursorSelectionThrottledFunction(e);
        });
    }

    override getDefaultPaneName() {
        return '';
    }

    override getPaneName() {
        return `Tool Input ${this._toolName} (Compiler #${this.compilerInfo.compilerId})`;
    }

    override getCurrentState() {
        const parent = super.getCurrentState();
        return {
            ...parent,
            toolId: this._toolId,
            toolName: this._toolName,
            compilerId: this.compilerInfo.compilerId,
            selection: this.selection,
        };
    }

    override onCompiler(
        compilerId: number,
        compiler: CompilerInfo | null,
        options: string,
        editorId: number,
        treeId: number,
    ) {}

    override onCompileResult(compilerId: number, compiler: CompilerInfo, result: CompilationResult) {}

    override close() {
        this.eventHub.unsubscribe();
        this.eventHub.emit('toolInputViewClosed', this.compilerInfo.compilerId, this._toolId, this.getInput());
        this.editor.dispose();
    }

    onToolClose(compilerId: number, toolSettings: ToolState) {
        if (this.compilerInfo.compilerId === compilerId && this._toolId === toolSettings.toolId) {
            // We can't immediately close as an outer loop somewhere in GoldenLayout is iterating over
            // the hierarchy. We can't modify while it's being iterated over.
            this.close();
            _.defer(function (self: ToolInputView) {
                self.container.close();
            }, this);
        }
    }

    onToolInputViewCloseRequest(compilerId: number, toolId: string) {
        if (this.compilerInfo.compilerId === compilerId && this._toolId === toolId) {
            this.close();
            _.defer(function (self: ToolInputView) {
                self.container.close();
            }, this);
        }
    }

    override onCompilerClose(id: number) {
        if (id === this.compilerInfo.compilerId) {
            // We can't immediately close as an outer loop somewhere in GoldenLayout is iterating over
            // the hierarchy. We can't modify while it's being iterated over.
            this.close();
            _.defer(function (self: ToolInputView) {
                self.container.close();
            }, this);
        }
    }

    override onSettingsChange(newSettings: SiteSettings) {
        super.onSettingsChange(newSettings);
        this.debouncedEmitChange = _.debounce(() => {
            this.maybeEmitChange(false);
        }, newSettings.delayAfterChange);
    }

    override onDidChangeCursorSelection(e: monaco.editor.ICursorSelectionChangedEvent) {
        // On initialization this callback fires with the default selection
        // overwriting any selection from state. If we are awaiting initial
        // selection setting then don't update our selection.
        if (!this.shouldSetSelectionInitially) {
            this.selection = e.selection;
            this.updateState();
        }
    }

    onSetToolInput(compilerId: number, toolId: string, value: string) {
        if (this.compilerInfo.compilerId === compilerId && this._toolId === toolId) {
            const ret = this.editor.getModel()?.setValue(value);
            if (this.shouldSetSelectionInitially && this.selection) {
                this.editor.setSelection(this.selection);
                this.editor.revealLinesInCenter(this.selection.startLineNumber, this.selection.endLineNumber);
                this.shouldSetSelectionInitially = false;
            }
            return ret;
        }
    }

    getInput() {
        if (!this.editor.getModel()) {
            return '';
        }
        return this.editor.getModel()?.getValue() ?? '';
    }

    maybeEmitChange(force: boolean) {
        const input = this.getInput();
        if (!force && input === this.lastChangeEmitted) return;

        this.lastChangeEmitted = input;
        this.eventHub.emit('toolInputChange', this.compilerInfo.compilerId, this._toolId, this.lastChangeEmitted);
    }
}
