// Copyright (c) 2024, Compiler Explorer Authors
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice,
//       this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.

import * as monaco from 'monaco-editor';

function definition(): monaco.languages.IMonarchLanguage {
    return {
        defaultToken: '',

        brackets: [{open: '(', close: ')', token: 'delimiter.parenthesis'}],

        typeKeywords: ['i32', 'i64', 'f32', 'f64', 'v128', 'funcref', 'externref'],

        numbers:
            /^(?:[+-]?\d+(?:(?:N|(?:[eE][+-]?\d+))|(?:\.?\d*(?:M|(?:[eE][+-]?\d+))?)|\/\d+|[xX][0-9a-fA-F]+|r[0-9a-zA-Z]+)?(?=[\\[\]\s"#'(),;@^`{}~]|$))/,

        characters:
            /^(?:\\(?:backspace|formfeed|newline|return|space|tab|o[0-7]{3}|u[0-9A-Fa-f]{4}|x[0-9A-Fa-f]{4}|.)?(?=[\\[\]\s"(),;@^`{}~]|$))/,

        escapes: /^\\(?:["'\\bfnrt]|x[0-9A-Fa-f]{1,4}|u[0-9A-Fa-f]{4}|U[0-9A-Fa-f]{8})/,

        qualifiedSymbols: /^(?:[^\\/[\]\d\s"#'(),;@^`{}~])*(?=[\\[\]\s"(),;@^`{}~]|$)/,

        coreSymbols: [
            'module',
            'func',
            'if',
            'else',
            'then',
            'export',
            'param',
            'result',
            'call',
            'call_indirect',
            'return_call',
            'return_call_indirect',
            'call_ref',
            'return_call_ref',
            'drop',
            'select',
            'select',
            'local.get',
            'local.set',
            'local.tee',
            'global.get',
            'global.set',
            'table.get',
            'table.set',
            'i32.const',
            'i64.const',
            'f32.const',
            'f64.const',
            'ref.null',
            'ref.is_null',
            'ref.func',
            'ref.as_non_null',
            'ref.eq',
            'i32.load',
            'i64.load',
            'f32.load',
            'f64.load',
            'i32.load8_s',
            'i32.load8_u',
            'i32.load16_s',
            'i32.load16_u',
            'i64.load8_s',
            'i64.load8_u',
            'i64.load16_s',
            'i64.load16_u',
            'i64.load32_s',
            'i64.load32_u',
            'i32.store',
            'i64.store',
            'f32.store',
            'f64.store',
            'i32.store8',
            'i32.store16',
            'i64.store8',
            'i64.store16',
            'i64.store32',
            'memory.size',
            'memory.grow',
            'i32.add',
            'i32.sub',
            'i32.mul',
            'i64.add',
            'i64.sub',
            'i64.mul',
            'i32.eqz',
            'i32.eq',
            'i32.ne',
            'i32.lt_s',
            'i32.lt_u',
            'i32.gt_s',
            'i32.gt_u',
            'i32.le_s',
            'i32.le_u',
            'i32.ge_s',
            'i32.ge_u',
            'i64.eqz',
            'i64.eq',
            'i64.ne',
            'i64.lt_s',
            'i64.lt_u',
            'i64.gt_s',
            'i64.gt_u',
            'i64.le_s',
            'i64.le_u',
            'i64.ge_s',
            'i64.ge_u',
            'f32.eq',
            'f32.ne',
            'f32.lt',
            'f32.gt',
            'f32.le',
            'f32.ge',
            'f64.eq',
            'f64.ne',
            'f64.lt',
            'f64.gt',
            'f64.le',
            'f64.ge',
            'i32.clz',
            'i32.ctz',
            'i32.popcnt',
            'i32.div_s',
            'i32.div_u',
            'i32.rem_s',
            'i32.rem_u',
            'i32.and',
            'i32.or',
            'i32.xor',
            'i32.shl',
            'i32.shr_s',
            'i32.shr_u',
            'i32.rotl',
            'i32.rotr',
            'i64.clz',
            'i64.ctz',
            'i64.popcnt',
            'i64.div_s',
            'i64.div_u',
            'i64.rem_s',
            'i64.rem_u',
            'i64.and',
            'i64.or',
            'i64.xor',
            'i64.shl',
            'i64.shr_s',
            'i64.shr_u',
            'i64.rotl',
            'i64.rotr',
            'f32.abs',
            'f32.neg',
            'f32.ceil',
            'f32.floor',
            'f32.trunc',
            'f32.nearest',
            'f32.sqrt',
            'f32.add',
            'f32.sub',
            'f32.mul',
            'f32.div',
            'f32.min',
            'f32.max',
            'f32.copysign',
            'f64.abs',
            'f64.neg',
            'f64.ceil',
            'f64.floor',
            'f64.trunc',
            'f64.nearest',
            'f64.sqrt',
            'f64.add',
            'f64.sub',
            'f64.mul',
            'f64.div',
            'f64.min',
            'f64.max',
            'f64.copysign',
            'i32.wrap_i64',
            'i32.trunc_f32_s',
            'i32.trunc_f32_u',
            'i32.trunc_f64_s',
            'i32.trunc_f64_u',
            'i64.extend_i32_s',
            'i64.extend_i32_u',
            'i64.trunc_f32_s',
            'i64.trunc_f32_u',
            'i64.trunc_f64_s',
            'i64.trunc_f64_u',
            'f32.convert_i32_s',
            'f32.convert_i32_u',
            'f32.convert_i64_s',
            'f32.convert_i64_u',
            'f32.demote_f64',
            'f64.convert_i32_s',
            'f64.convert_i32_u',
            'f64.convert_i64_s',
            'f64.convert_i64_u',
            'f64.promote_f32',
            'i32.reinterpret_f32',
            'i64.reinterpret_f64',
            'f32.reinterpret_i32',
            'f64.reinterpret_i64',
            'i32.extend8_s',
            'i32.extend16_s',
            'i64.extend8_s',
            'i64.extend16_s',
            'i64.extend32_s',
            'f64.acos',
            'f64.asin',
            'f64.atan',
            'f64.cos',
            'f64.sin',
            'f64.tan',
            'f64.exp',
            'f64.log',
            'f64.atan2',
            'f64.pow',
            'f64.mod',
            'v128.load',
            'v128.load8x8_s',
            'v128.load8x8_u',
            'v128.load16x4_s',
            'v128.load16x4_u',
            'v128.load32x2_s',
            'v128.load32x2_u',
            'v128.load8_splat',
            'v128.load16_splat',
            'v128.load32_splat',
            'v128.load64_splat',
            'v128.store',
            'v128.load32_zero',
            'v128.load64_zero',
            'v128.load8_lane',
            'v128.load16_lane',
            'v128.load32_lane',
            'v128.load64_lane',
            'v128.store8_lane',
            'v128.store16_lane',
            'v128.store32_lane',
            'v128.store64_lane',
            'v128.const',
            'i8x16.shuffle',
            'i8x16.swizzle',
            'i8x16.splat',
            'i16x8.splat',
            'i32x4.splat',
            'i64x2.splat',
            'f32x4.splat',
            'f64x2.splat',
            'i8x16.eq',
            'i8x16.ne',
            'i8x16.lt_s',
            'i8x16.lt_u',
            'i8x16.gt_s',
            'i8x16.gt_u',
            'i8x16.le_s',
            'i8x16.le_u',
            'i8x16.ge_s',
            'i8x16.ge_u',
            'i16x8.eq',
            'i16x8.ne',
            'i16x8.lt_s',
            'i16x8.lt_u',
            'i16x8.gt_s',
            'i16x8.gt_u',
            'i16x8.le_s',
            'i16x8.le_u',
            'i16x8.ge_s',
            'i16x8.ge_u',
            'i32x4.eq',
            'i32x4.ne',
            'i32x4.lt_s',
            'i32x4.lt_u',
            'i32x4.gt_s',
            'i32x4.gt_u',
            'i32x4.le_s',
            'i32x4.le_u',
            'i32x4.ge_s',
            'i32x4.ge_u',
            'f32x4.eq',
            'f32x4.ne',
            'f32x4.lt',
            'f32x4.gt',
            'f32x4.le',
            'f32x4.ge',
            'f64x2.eq',
            'f64x2.ne',
            'f64x2.lt',
            'f64x2.gt',
            'f64x2.le',
            'f64x2.ge',
            'v128.not',
            'v128.and',
            'v128.andnot',
            'v128.or',
            'v128.xor',
            'v128.bitselect',
            'v128.any_true',
            'f32x4.demote_f64x2_zero',
            'f64x2.promote_low_f32x4',
            'i8x16.abs',
            'i8x16.neg',
            'i8x16.popcnt',
            'i8x16.all_true',
            'i8x16.bitmask',
            'i8x16.narrow_i16x8_s',
            'i8x16.narrow_i16x8_u',
            'f32x4.ceil',
            'f32x4.floor',
            'f32x4.trunc',
            'f32x4.nearest',
            'i8x16.shl',
            'i8x16.shr_s',
            'i8x16.shr_u',
            'i8x16.add',
            'i8x16.add_sat_s',
            'i8x16.add_sat_u',
            'i8x16.sub',
            'i8x16.sub_sat_s',
            'i8x16.sub_sat_u',
            'f64x2.ceil',
            'f64x2.floor',
            'i8x16.min_s',
            'i8x16.min_u',
            'i8x16.max_s',
            'i8x16.max_u',
            'f64x2.trunc',
            'i8x16.avgr_u',
            'i16x8.extadd_pairwise_i8x16_s',
            'i16x8.extadd_pairwise_i8x16_u',
            'i32x4.extadd_pairwise_i16x8_s',
            'i32x4.extadd_pairwise_i16x8_u',
            'i16x8.abs',
            'i16x8.neg',
            'i16x8.q15mulr_sat_s',
            'i16x8.all_true',
            'i16x8.bitmask',
            'i16x8.narrow_i32x4_s',
            'i16x8.narrow_i32x4_u',
            'i16x8.extend_low_i8x16_s',
            'i16x8.extend_high_i8x16_s',
            'i16x8.extend_low_i8x16_u',
            'i16x8.extend_high_i8x16_u',
            'i16x8.shl',
            'i16x8.shr_s',
            'i16x8.shr_u',
            'i16x8.add',
            'i16x8.add_sat_s',
            'i16x8.add_sat_u',
            'i16x8.sub',
            'i16x8.sub_sat_s',
            'i16x8.sub_sat_u',
            'f64x2.nearest',
            'i16x8.mul',
            'i16x8.min_s',
            'i16x8.min_u',
            'i16x8.max_s',
            'i16x8.max_u',
            'i16x8.avgr_u',
            'i16x8.extmul_low_i8x16_s',
            'i16x8.extmul_high_i8x16_s',
            'i16x8.extmul_low_i8x16_u',
            'i16x8.extmul_high_i8x16_u',
            'i32x4.abs',
            'i32x4.neg',
            'i32x4.all_true',
            'i32x4.bitmask',
            'i32x4.extend_low_i16x8_s',
            'i32x4.extend_high_i16x8_s',
            'i32x4.extend_low_i16x8_u',
            'i32x4.extend_high_i16x8_u',
            'i32x4.shl',
            'i32x4.shr_s',
            'i32x4.shr_u',
            'i32x4.add',
            'i32x4.sub',
            'i32x4.mul',
            'i32x4.min_s',
            'i32x4.min_u',
            'i32x4.max_s',
            'i32x4.max_u',
            'i32x4.dot_i16x8_s',
            'i32x4.extmul_low_i16x8_s',
            'i32x4.extmul_high_i16x8_s',
            'i32x4.extmul_low_i16x8_u',
            'i32x4.extmul_high_i16x8_u',
            'i64x2.abs',
            'i64x2.neg',
            'i64x2.all_true',
            'i64x2.bitmask',
            'i64x2.extend_low_i32x4_s',
            'i64x2.extend_high_i32x4_s',
            'i64x2.extend_low_i32x4_u',
            'i64x2.extend_high_i32x4_u',
            'i64x2.shl',
            'i64x2.shr_s',
            'i64x2.shr_u',
            'i64x2.add',
            'i64x2.sub',
            'i64x2.mul',
            'i64x2.eq',
            'i64x2.ne',
            'i64x2.lt_s',
            'i64x2.gt_s',
            'i64x2.le_s',
            'i64x2.ge_s',
            'i64x2.extmul_low_i32x4_s',
            'i64x2.extmul_high_i32x4_s',
            'i64x2.extmul_low_i32x4_u',
            'i64x2.extmul_high_i32x4_u',
            'f32x4.abs',
            'f32x4.neg',
            'f32x4.sqrt',
            'f32x4.add',
            'f32x4.sub',
            'f32x4.mul',
            'f32x4.div',
            'f32x4.min',
            'f32x4.max',
            'f32x4.pmin',
            'f32x4.pmax',
            'f64x2.abs',
            'f64x2.neg',
            'f64x2.sqrt',
            'f64x2.add',
            'f64x2.sub',
            'f64x2.mul',
            'f64x2.div',
            'f64x2.min',
            'f64x2.max',
            'f64x2.pmin',
            'f64x2.pmax',
            'i32x4.trunc_sat_f32x4_s',
            'i32x4.trunc_sat_f32x4_u',
            'f32x4.convert_i32x4_s',
            'f32x4.convert_i32x4_u',
            'i32x4.trunc_sat_f64x2_s_zero',
            'i32x4.trunc_sat_f64x2_u_zero',
            'f64x2.convert_low_i32x4_s',
            'f64x2.convert_low_i32x4_u',
            'i8x16.relaxed_swizzle',
            'i32x4.relaxed_trunc_f32x4_s',
            'i32x4.relaxed_trunc_f32x4_u',
            'i32x4.relaxed_trunc_f64x2_s_zero',
            'i32x4.relaxed_trunc_f64x2_u_zero',
            'f32x4.qfma',
            'f32x4.qfms',
            'f64x2.qfma',
            'f64x2.qfms',
            'i8x16.relaxed_laneselect',
            'i16x8.relaxed_laneselect',
            'i32x4.relaxed_laneselect',
            'i64x2.relaxed_laneselect',
            'f32x4.relaxed_min',
            'f32x4.relaxed_max',
            'f64x2.relaxed_min',
            'f64x2.relaxed_max',
            'i16x8.relaxed_q15mulr_s',
            'i16x8.dot_i8x16_i7x16_s',
            'i32x4.dot_i8x16_i7x16_add_s',
            'i8x16.extract_lane_s',
            'i8x16.extract_lane_u',
            'i16x8.extract_lane_s',
            'i16x8.extract_lane_u',
            'i32x4.extract_lane',
            'i64x2.extract_lane',
            'f32x4.extract_lane',
            'f64x2.extract_lane',
            'i8x16.replace_lane',
            'i16x8.replace_lane',
            'i32x4.replace_lane',
            'i64x2.replace_lane',
            'f32x4.replace_lane',
            'f64x2.replace_lane',
            'i32.trunc_sat_f32_s',
            'i32.trunc_sat_f32_u',
            'i32.trunc_sat_f64_s',
            'i32.trunc_sat_f64_u',
            'i64.trunc_sat_f32_s',
            'i64.trunc_sat_f32_u',
            'i64.trunc_sat_f64_s',
            'i64.trunc_sat_f64_u',
            'data.drop',
            'table.init',
            'elem.drop',
            'table.copy',
            'table.size',
            'memory.init',
            'memory.copy',
            'memory.fill',
            'table.grow',
            'table.fill',
            'memory.atomic.notify',
            'memory.atomic.wait32',
            'memory.atomic.wait64',
            'i32.atomic.load',
            'i64.atomic.load',
            'i32.atomic.load8_u',
            'i32.atomic.load16_u',
            'i64.atomic.load8_u',
            'i64.atomic.load16_u',
            'i64.atomic.load32_u',
            'i32.atomic.store',
            'i64.atomic.store',
            'i32.atomic.store8',
            'i32.atomic.store16',
            'i64.atomic.store8',
            'i64.atomic.store16',
            'i64.atomic.store32',
            'i32.atomic.rmw.add',
            'i64.atomic.rmw.add',
            'i32.atomic.rmw8.add_u',
            'i32.atomic.rmw16.add_u',
            'i64.atomic.rmw8.add_u',
            'i64.atomic.rmw16.add_u',
            'i64.atomic.rmw32.add_u',
            'i32.atomic.rmw.sub',
            'i64.atomic.rmw.sub',
            'i32.atomic.rmw8.sub_u',
            'i32.atomic.rmw16.sub_u',
            'i64.atomic.rmw8.sub_u',
            'i64.atomic.rmw16.sub_u',
            'i64.atomic.rmw32.sub_u',
            'i32.atomic.rmw.and',
            'i64.atomic.rmw.and',
            'i32.atomic.rmw8.and_u',
            'i32.atomic.rmw16.and_u',
            'i64.atomic.rmw8.and_u',
            'i64.atomic.rmw16.and_u',
            'i64.atomic.rmw32.and_u',
            'i32.atomic.rmw.or',
            'i64.atomic.rmw.or',
            'i32.atomic.rmw8.or_u',
            'i32.atomic.rmw16.or_u',
            'i64.atomic.rmw8.or_u',
            'i64.atomic.rmw16.or_u',
            'i64.atomic.rmw32.or_u',
            'i32.atomic.rmw.xor',
            'i64.atomic.rmw.xor',
            'i32.atomic.rmw8.xor_u',
            'i32.atomic.rmw16.xor_u',
            'i64.atomic.rmw8.xor_u',
            'i64.atomic.rmw16.xor_u',
            'i64.atomic.rmw32.xor_u',
            'i32.atomic.rmw.xchg',
            'i64.atomic.rmw.xchg',
            'i32.atomic.rmw8.xchg_u',
            'i32.atomic.rmw16.xchg_u',
            'i64.atomic.rmw8.xchg_u',
            'i64.atomic.rmw16.xchg_u',
            'i64.atomic.rmw32.xchg_u',
            'i32.atomic.rmw.cmpxchg',
            'i64.atomic.rmw.cmpxchg',
            'i32.atomic.rmw8.cmpxchg_u',
            'i32.atomic.rmw16.cmpxchg_u',
            'i64.atomic.rmw8.cmpxchg_u',
            'i64.atomic.rmw16.cmpxchg_u',
            'i64.atomic.rmw32.cmpxchg_u',
            'atomic.fence',
            'struct.new',
            'struct.new_default',
            'struct.get',
            'struct.get_s',
            'struct.get_u',
            'struct.set',
            'array.new',
            'array.new_default',
            'array.new_fixed',
            'array.new_data',
            'array.new_elem',
            'array.get',
            'array.get_s',
            'array.get_u',
            'array.set',
            'array.len',
            'array.fill',
            'array.copy',
            'array.init_data',
            'array.init_elem',
            'ref.test',
            'ref.test null',
            'ref.cast',
            'ref.cast null',
            'br_on_cast',
            'br_on_cast_fail',
            'any.convert_extern',
            'extern.convert_any',
            'ref.i31',
            'i31.get_s',
            'i31.get_u',
            'ref.cast_nop',
            'string.new_utf8',
            'string.new_wtf16',
            'string.const',
            'string.measure_utf8',
            'string.measure_wtf8',
            'string.measure_wtf16',
            'string.encode_utf8',
            'string.encode_wtf16',
            'string.concat',
            'string.eq',
            'string.is_usv_sequence',
            'string.new_lossy_utf8',
            'string.new_wtf8',
            'string.encode_lossy_utf8',
            'string.encode_wtf8',
            'string.new_utf8_try',
            'string.as_wtf8',
            'stringview_wtf8.advance',
            'stringview_wtf8.encode_utf8',
            'stringview_wtf8.slice',
            'stringview_wtf8.encode_lossy_utf8',
            'stringview_wtf8.encode_wtf8',
            'string.as_wtf16',
            'stringview_wtf16.length',
            'stringview_wtf16.get_codeunit',
            'stringview_wtf16.encode',
            'stringview_wtf16.slice',
            'string.as_iter',
            'stringview_iter.next',
            'stringview_iter.advance',
            'stringview_iter.rewind',
            'stringview_iter.slice',
            'string.compare',
            'string.from_code_point',
            'string.hash',
            'string.new_utf8_array',
            'string.new_wtf16_array',
            'string.encode_utf8_array',
            'string.encode_wtf16_array',
            'string.new_lossy_utf8_array',
            'string.new_wtf8_array',
            'string.encode_lossy_utf8_array',
            'string.encode_wtf8_array',
            'string.new_utf8_array_try',
        ],

        tokenizer: {
            root: [
                // whitespaces and comments
                {include: '@whitespace'},

                // numbers
                [/@numbers/, 'number'],

                // characters
                [/@characters/, 'string'],

                // strings
                {include: '@string'},

                // brackets
                [/[()]/, '@brackets'],

                // symbols
                [
                    /@qualifiedSymbols/,
                    {
                        cases: {
                            '@coreSymbols': 'keyword',
                            '@typeKeywords': 'type',
                            '@default': 'identifier',
                        },
                    },
                ],
            ],

            whitespace: [
                [/[\s,]+/, 'white'],
                [/;.*$/, 'comment'],
            ],

            comment: [[/[^()]/, 'comment']],

            string: [[/"/, 'string', '@multiLineString']],

            multiLineString: [
                [/"/, 'string', '@popall'],
                [/@escapes/, 'string.escape'],
                [/./, 'string'],
            ],
        },
    };
}

const def = definition();
monaco.languages.register({id: 'wat'});
monaco.languages.setMonarchTokensProvider('wat', def);

export = def;
