import {AssemblyInstructionInfo} from '../base.js';

export function getAsmOpcode(opcode: string | undefined): AssemblyInstructionInfo | undefined {
    if (!opcode) return;
    switch (opcode) {
        case "ABS":
            return {
                "tooltip": "Absolute value computes the absolute value of the signed integer value in the source register, and writes the result to the destination register.",
                "html": "<p>Absolute value computes the absolute value of the signed integer value in the source register, and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADC":
            return {
                "tooltip": "Add with Carry adds two register values and the Carry flag value, and writes the result to the destination register.",
                "html": "<p>Add with Carry adds two register values and the Carry flag value, and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADCLB":
            return {
                "tooltip": "Add the even-numbered elements of the first source vector and the 1-bit carry from the least-significant bit of the odd-numbered elements of the second source vector to the even-numbered elements of the destination and accumulator vector. The 1-bit carry output is placed in the corresponding odd-numbered element of the destination vector.",
                "html": "<p>Add the even-numbered elements of the first source vector and the 1-bit carry from the least-significant bit of the odd-numbered elements of the second source vector to the even-numbered elements of the destination and accumulator vector. The 1-bit carry output is placed in the corresponding odd-numbered element of the destination vector.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADCLT":
            return {
                "tooltip": "Add the odd-numbered elements of the first source vector and the 1-bit carry from the least-significant bit of the odd-numbered elements of the second source vector to the even-numbered elements of the destination and accumulator vector. The 1-bit carry output is placed in the corresponding odd-numbered element of the destination vector.",
                "html": "<p>Add the odd-numbered elements of the first source vector and the 1-bit carry from the least-significant bit of the odd-numbered elements of the second source vector to the even-numbered elements of the destination and accumulator vector. The 1-bit carry output is placed in the corresponding odd-numbered element of the destination vector.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADCS":
            return {
                "tooltip": "Add with Carry, setting flags, adds two register values and the Carry flag value, and writes the result to the destination register. It updates the condition flags based on the result.",
                "html": "<p>Add with Carry, setting flags, adds two register values and the Carry flag value, and writes the result to the destination register. It updates the condition flags based on the result.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADD":
            return {
                "tooltip": "Add (extended register) adds a register value and a sign or zero-extended register value, followed by an optional left shift amount, and writes the result to the destination register. The argument that is extended from the <Rm> register can be a byte, halfword, word, or doubleword.",
                "html": "<p>Add (extended register) adds a register value and a sign or zero-extended register value, followed by an optional left shift amount, and writes the result to the destination register. The argument that is extended from the <syntax>&lt;Rm&gt;</syntax> register can be a byte, halfword, word, or doubleword.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDG":
            return {
                "tooltip": "Add with Tag adds an immediate value scaled by the Tag granule to the address in the source register, modifies the Logical Address Tag of the address using an immediate value, and writes the result to the destination register. Tags specified in GCR_EL1.Exclude are excluded from the possible outputs when modifying the Logical Address Tag.",
                "html": "<p>Add with Tag adds an immediate value scaled by the Tag granule to the address in the source register, modifies the Logical Address Tag of the address using an immediate value, and writes the result to the destination register. Tags specified in GCR_EL1.Exclude are excluded from the possible outputs when modifying the Logical Address Tag.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDHA":
            return {
                "tooltip": "Add each element of the source vector to the corresponding active element of each horizontal slice of a ZA tile. The tile elements are predicated by a pair of governing predicates. An element of a horizontal slice is considered active if its corresponding element in the second governing predicate is TRUE and the element corresponding to its horizontal slice number in the first governing predicate is TRUE. Inactive elements in the destination tile remain unmodified.",
                "html": "<p>Add each element of the source vector to the corresponding active element of each horizontal slice of a ZA tile. The tile elements are predicated by a pair of governing predicates. An element of a horizontal slice is considered active if its corresponding element in the second governing predicate is TRUE and the element corresponding to its horizontal slice number in the first governing predicate is TRUE. Inactive elements in the destination tile remain unmodified.</p><p>ID_AA64SMFR0_EL1.I16I64 indicates whether the 64-bit integer variant is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDHN":
        case "ADDHN2":
            return {
                "tooltip": "Add returning High Narrow. This instruction adds each vector element in the first source SIMD&FP register to the corresponding vector element in the second source SIMD&FP register, places the most significant half of the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&FP register.",
                "html": "<p>Add returning High Narrow. This instruction adds each vector element in the first source SIMD&amp;FP register to the corresponding vector element in the second source SIMD&amp;FP register, places the most significant half of the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&amp;FP register.</p><p>The results are truncated. For rounded results, see <xref linkend=\"A64.instructions.RADDHN_advsimd\">RADDHN</xref>.</p><p>The <instruction>ADDHN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>ADDHN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDHNB":
            return {
                "tooltip": "Add each vector element of the first source vector to the corresponding vector element of the second source vector, and place the most significant half of the result in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. This instruction is unpredicated.",
                "html": "<p>Add each vector element of the first source vector to the corresponding vector element of the second source vector, and place the most significant half of the result in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDHNT":
            return {
                "tooltip": "Add each vector element of the first source vector to the corresponding vector element of the second source vector, and place the most significant half of the result in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. This instruction is unpredicated.",
                "html": "<p>Add each vector element of the first source vector to the corresponding vector element of the second source vector, and place the most significant half of the result in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDP":
            return {
                "tooltip": "Add Pair of elements (scalar). This instruction adds two vector elements in the source SIMD&FP register and writes the scalar result into the destination SIMD&FP register.",
                "html": "<p>Add Pair of elements (scalar). This instruction adds two vector elements in the source SIMD&amp;FP register and writes the scalar result into the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDPL":
            return {
                "tooltip": "Add the current predicate register size in bytes multiplied by an immediate in the range -32 to 31 to the 64-bit source general-purpose register or current stack pointer and place the result in the 64-bit destination general-purpose register or current stack pointer.",
                "html": "<p>Add the current predicate register size in bytes multiplied by an immediate in the range -32 to 31 to the 64-bit source general-purpose register or current stack pointer and place the result in the 64-bit destination general-purpose register or current stack pointer.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDQV":
            return {
                "tooltip": "Unsigned addition of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&FP destination register. Inactive elements in the source vector are treated as zero.",
                "html": "<p>Unsigned addition of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&amp;FP destination register. Inactive elements in the source vector are treated as zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDS":
            return {
                "tooltip": "Add (extended register), setting flags, adds a register value and a sign or zero-extended register value, followed by an optional left shift amount, and writes the result to the destination register. The argument that is extended from the <Rm> register can be a byte, halfword, word, or doubleword. It updates the condition flags based on the result.",
                "html": "<p>Add (extended register), setting flags, adds a register value and a sign or zero-extended register value, followed by an optional left shift amount, and writes the result to the destination register. The argument that is extended from the <syntax>&lt;Rm&gt;</syntax> register can be a byte, halfword, word, or doubleword. It updates the condition flags based on the result.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDSPL":
            return {
                "tooltip": "Add the Streaming SVE predicate register size in bytes multiplied by an immediate in the range -32 to 31 to the 64-bit source general-purpose register or current stack pointer and place the result in the 64-bit destination general-purpose register or current stack pointer.",
                "html": "<p>Add the Streaming SVE predicate register size in bytes multiplied by an immediate in the range -32 to 31 to the 64-bit source general-purpose register or current stack pointer and place the result in the 64-bit destination general-purpose register or current stack pointer.</p><p>This instruction does not require the PE to be in Streaming SVE mode.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDSVL":
            return {
                "tooltip": "Add the Streaming SVE vector register size in bytes multiplied by an immediate in the range -32 to 31 to the 64-bit source general-purpose register or current stack pointer, and place the result in the 64-bit destination general-purpose register or current stack pointer.",
                "html": "<p>Add the Streaming SVE vector register size in bytes multiplied by an immediate in the range -32 to 31 to the 64-bit source general-purpose register or current stack pointer, and place the result in the 64-bit destination general-purpose register or current stack pointer.</p><p>This instruction does not require the PE to be in Streaming SVE mode.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDV":
            return {
                "tooltip": "Add across Vector. This instruction adds every vector element in the source SIMD&FP register together, and writes the scalar result to the destination SIMD&FP register.",
                "html": "<p>Add across Vector. This instruction adds every vector element in the source SIMD&amp;FP register together, and writes the scalar result to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDVA":
            return {
                "tooltip": "Add each element of the source vector to the corresponding active element of each vertical slice of a ZA tile. The tile elements are predicated by a pair of governing predicates. An element of a vertical slice is considered active if its corresponding element in the first governing predicate is TRUE and the element corresponding to its vertical slice number in the second governing predicate is TRUE. Inactive elements in the destination tile remain unmodified.",
                "html": "<p>Add each element of the source vector to the corresponding active element of each vertical slice of a ZA tile. The tile elements are predicated by a pair of governing predicates. An element of a vertical slice is considered active if its corresponding element in the first governing predicate is TRUE and the element corresponding to its vertical slice number in the second governing predicate is TRUE. Inactive elements in the destination tile remain unmodified.</p><p>ID_AA64SMFR0_EL1.I16I64 indicates whether the 64-bit integer variant is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADDVL":
            return {
                "tooltip": "Add the current vector register size in bytes multiplied by an immediate in the range -32 to 31 to the 64-bit source general-purpose register or current stack pointer, and place the result in the 64-bit destination general-purpose register or current stack pointer.",
                "html": "<p>Add the current vector register size in bytes multiplied by an immediate in the range -32 to 31 to the 64-bit source general-purpose register or current stack pointer, and place the result in the 64-bit destination general-purpose register or current stack pointer.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADR":
            return {
                "tooltip": "Form PC-relative address adds an immediate value to the PC value to form a PC-relative address, and writes the result to the destination register.",
                "html": "<p>Form PC-relative address adds an immediate value to the PC value to form a PC-relative address, and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ADRP":
            return {
                "tooltip": "Form PC-relative address to 4KB page adds an immediate value that is shifted left by 12 bits, to the PC value to form a PC-relative address, with the bottom 12 bits masked out, and writes the result to the destination register.",
                "html": "<p>Form PC-relative address to 4KB page adds an immediate value that is shifted left by 12 bits, to the PC value to form a PC-relative address, with the bottom 12 bits masked out, and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "AESD":
            return {
                "tooltip": "AES single round decryption.",
                "html": "<p>AES single round decryption.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "AESE":
            return {
                "tooltip": "AES single round encryption.",
                "html": "<p>AES single round encryption.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "AESIMC":
            return {
                "tooltip": "AES inverse mix columns.",
                "html": "<p>AES inverse mix columns.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "AESMC":
            return {
                "tooltip": "AES mix columns.",
                "html": "<p>AES mix columns.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "AND":
            return {
                "tooltip": "Bitwise AND (vector). This instruction performs a bitwise AND between the two source SIMD&FP registers, and writes the result to the destination SIMD&FP register.",
                "html": "<p>Bitwise AND (vector). This instruction performs a bitwise AND between the two source SIMD&amp;FP registers, and writes the result to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ANDQV":
            return {
                "tooltip": "Bitwise AND of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&FP destination register. Inactive elements in the source vector are treated as all ones.",
                "html": "<p>Bitwise AND of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&amp;FP destination register. Inactive elements in the source vector are treated as all ones.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ANDS":
            return {
                "tooltip": "Bitwise AND (immediate), setting flags, performs a bitwise AND of a register value and an immediate value, and writes the result to the destination register. It updates the condition flags based on the result.",
                "html": "<p>Bitwise AND (immediate), setting flags, performs a bitwise AND of a register value and an immediate value, and writes the result to the destination register. It updates the condition flags based on the result.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ANDV":
            return {
                "tooltip": "Bitwise AND horizontally across all lanes of a vector, and place the result in the SIMD&FP scalar destination register. Inactive elements in the source vector are treated as all ones.",
                "html": "<p>Bitwise AND horizontally across all lanes of a vector, and place the result in the SIMD&amp;FP scalar destination register. Inactive elements in the source vector are treated as all ones.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ASR":
            return {
                "tooltip": "Arithmetic Shift Right (register) shifts a register value right by a variable number of bits, shifting in copies of its sign bit, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is right-shifted.",
                "html": "<p>Arithmetic Shift Right (register) shifts a register value right by a variable number of bits, shifting in copies of its sign bit, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is right-shifted.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ASRD":
            return {
                "tooltip": "Shift right by immediate, preserving the sign bit, each active element of the source vector, and destructively place the results in the corresponding elements of the source vector. The result rounds toward zero as in a signed division. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Shift right by immediate, preserving the sign bit, each active element of the source vector, and destructively place the results in the corresponding elements of the source vector. The result rounds toward zero as in a signed division. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ASRR":
            return {
                "tooltip": "Reversed shift right, preserving the sign bit, active elements of the second source vector by corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. The shift amount operand is a vector of unsigned elements in which all bits are significant, and not used modulo the element size. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Reversed shift right, preserving the sign bit, active elements of the second source vector by corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. The shift amount operand is a vector of unsigned elements in which all bits are significant, and not used modulo the element size. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ASRV":
            return {
                "tooltip": "Arithmetic Shift Right Variable shifts a register value right by a variable number of bits, shifting in copies of its sign bit, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is right-shifted.",
                "html": "<p>Arithmetic Shift Right Variable shifts a register value right by a variable number of bits, shifting in copies of its sign bit, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is right-shifted.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "AT":
            return {
                "tooltip": "Address Translate. For more information, see op0==0b01, cache maintenance, TLB maintenance, and address translation instructions.",
                "html": "<p>Address Translate. For more information, see <xref linkend=\"BABEJJJE\">op0==0b01, cache maintenance, TLB maintenance, and address translation instructions</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "AUTDA":
        case "AUTDZA":
            return {
                "tooltip": "Authenticate Data address, using key A. This instruction authenticates a data address, using a modifier and key A.",
                "html": "<p>Authenticate Data address, using key A. This instruction authenticates a data address, using a modifier and key A.</p><p>The address is in the general-purpose register that is specified by <syntax>&lt;Xd&gt;</syntax>.</p><p>The modifier is:</p><p>If the authentication passes, the upper bits of the address are restored to enable subsequent use of the address. For information on behavior if the authentication fails, see <xref>Faulting on pointer authentication</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "AUTDB":
        case "AUTDZB":
            return {
                "tooltip": "Authenticate Data address, using key B. This instruction authenticates a data address, using a modifier and key B.",
                "html": "<p>Authenticate Data address, using key B. This instruction authenticates a data address, using a modifier and key B.</p><p>The address is in the general-purpose register that is specified by <syntax>&lt;Xd&gt;</syntax>.</p><p>The modifier is:</p><p>If the authentication passes, the upper bits of the address are restored to enable subsequent use of the address. For information on behavior if the authentication fails, see <xref>Faulting on pointer authentication</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "AUTIA":
        case "AUTIA1716":
        case "AUTIASP":
        case "AUTIAZ":
        case "AUTIZA":
            return {
                "tooltip": "Authenticate Instruction address, using key A. This instruction authenticates an instruction address, using a modifier and key A.",
                "html": "<p>Authenticate Instruction address, using key A. This instruction authenticates an instruction address, using a modifier and key A.</p><p>The address is:</p><p>The modifier is:</p><p>If the authentication passes, the upper bits of the address are restored to enable subsequent use of the address. For information on behavior if the authentication fails, see <xref>Faulting on pointer authentication</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "AUTIB":
        case "AUTIB1716":
        case "AUTIBSP":
        case "AUTIBZ":
        case "AUTIZB":
            return {
                "tooltip": "Authenticate Instruction address, using key B. This instruction authenticates an instruction address, using a modifier and key B.",
                "html": "<p>Authenticate Instruction address, using key B. This instruction authenticates an instruction address, using a modifier and key B.</p><p>The address is:</p><p>The modifier is:</p><p>If the authentication passes, the upper bits of the address are restored to enable subsequent use of the address. For information on behavior if the authentication fails, see <xref>Faulting on pointer authentication</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "AXFLAG":
            return {
                "tooltip": "Convert floating-point condition flags from Arm to external format. This instruction converts the state of the PSTATE.{N,Z,C,V} flags from a form representing the result of an Arm floating-point scalar compare instruction to an alternative representation required by some software.",
                "html": "<p>Convert floating-point condition flags from Arm to external format. This instruction converts the state of the PSTATE.{N,Z,C,V} flags from a form representing the result of an Arm floating-point scalar compare instruction to an alternative representation required by some software.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "B.AL":
        case "B.CC":
        case "B.CS":
        case "B.EQ":
        case "B.GE":
        case "B.GT":
        case "B.HI":
        case "B.LE":
        case "B.LS":
        case "B.LT":
        case "B.MI":
        case "B.NE":
        case "B.PL":
        case "B.VC":
        case "B.VS":
        case "B.cond":
            return {
                "tooltip": "Branch conditionally to a label at a PC-relative offset, with a hint that this is not a subroutine call or return.",
                "html": "<p>Branch conditionally to a label at a PC-relative offset, with a hint that this is not a subroutine call or return.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "B":
            return {
                "tooltip": "Branch causes an unconditional branch to a label at a PC-relative offset, with a hint that this is not a subroutine call or return.",
                "html": "<p>Branch causes an unconditional branch to a label at a PC-relative offset, with a hint that this is not a subroutine call or return.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BC.AL":
        case "BC.CC":
        case "BC.CS":
        case "BC.EQ":
        case "BC.GE":
        case "BC.GT":
        case "BC.HI":
        case "BC.LE":
        case "BC.LS":
        case "BC.LT":
        case "BC.MI":
        case "BC.NE":
        case "BC.PL":
        case "BC.VC":
        case "BC.VS":
        case "BC.cond":
            return {
                "tooltip": "Branch Consistent conditionally to a label at a PC-relative offset, with a hint that this branch will behave very consistently and is very unlikely to change direction.",
                "html": "<p>Branch Consistent conditionally to a label at a PC-relative offset, with a hint that this branch will behave very consistently and is very unlikely to change direction.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BCAX":
            return {
                "tooltip": "Bit Clear and exclusive-OR performs a bitwise AND of the 128-bit vector in a source SIMD&FP register and the complement of the vector in another source SIMD&FP register, then performs a bitwise exclusive-OR of the resulting vector and the vector in a third source SIMD&FP register, and writes the result to the destination SIMD&FP register.",
                "html": "<p>Bit Clear and exclusive-OR performs a bitwise AND of the 128-bit vector in a source SIMD&amp;FP register and the complement of the vector in another source SIMD&amp;FP register, then performs a bitwise exclusive-OR of the resulting vector and the vector in a third source SIMD&amp;FP register, and writes the result to the destination SIMD&amp;FP register.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SHA3\">FEAT_SHA3</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BDEP":
            return {
                "tooltip": "This instruction scatters the lowest-numbered contiguous bits within each element of the first source vector to the bit positions indicated by non-zero bits in the corresponding mask element of the second source vector, preserving their order, and set the bits corresponding to a zero mask bit to zero. This instruction is unpredicated.",
                "html": "<p>This instruction scatters the lowest-numbered contiguous bits within each element of the first source vector to the bit positions indicated by non-zero bits in the corresponding mask element of the second source vector, preserving their order, and set the bits corresponding to a zero mask bit to zero. This instruction is unpredicated.</p><p>ID_AA64ZFR0_EL1.BitPerm indicates whether this instruction is implemented.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BEXT":
            return {
                "tooltip": "This instruction gathers bits in each element of the first source vector from the bit positions indicated by non-zero bits in the corresponding mask element of the second source vector to the lowest-numbered contiguous bits of the corresponding destination element, preserving their order, and sets the remaining higher-numbered bits to zero. This instruction is unpredicated.",
                "html": "<p>This instruction gathers bits in each element of the first source vector from the bit positions indicated by non-zero bits in the corresponding mask element of the second source vector to the lowest-numbered contiguous bits of the corresponding destination element, preserving their order, and sets the remaining higher-numbered bits to zero. This instruction is unpredicated.</p><p>ID_AA64ZFR0_EL1.BitPerm indicates whether this instruction is implemented.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFADD":
            return {
                "tooltip": "Add active BFloat16 elements of the second source vector to corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Add active BFloat16 elements of the second source vector to corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.</p><p>This instruction follows SVE2.1 non-widening BFloat16 numerical behaviors.</p><p>ID_AA64ZFR0_EL1.B16B16 indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFC":
            return {
                "tooltip": "Bitfield Clear sets a bitfield of <width> bits at bit position <lsb> of the destination register to zero, leaving the other destination bits unchanged.",
                "html": "<p>Bitfield Clear sets a bitfield of <syntax>&lt;width&gt;</syntax> bits at bit position <syntax>&lt;lsb&gt;</syntax> of the destination register to zero, leaving the other destination bits unchanged.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFCLAMP":
            return {
                "tooltip": "Clamp each BFloat16 element in the two or four destination vectors to between the BFloat16 minimum value in the corresponding element of the first source vector and the BFloat16 maximum value in the corresponding element of the second source vector and destructively place the clamped results in the corresponding elements of the two or four destination vectors.",
                "html": "<p>Clamp each BFloat16 element in the two or four destination vectors to between the BFloat16 minimum value in the corresponding element of the first source vector and the BFloat16 maximum value in the corresponding element of the second source vector and destructively place the clamped results in the corresponding elements of the two or four destination vectors.</p><p>Regardless of the value of FPCR.AH, the behavior is as follows for each minimum number and maximum number operation:</p><p>This instruction follows SME2.1 non-widening BFloat16 numerical behaviors corresponding to instructions that place their results in two or four SVE Z vectors.</p><p>This instruction is unpredicated.</p><p>ID_AA64SMFR0_EL1.B16B16 indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFCVT":
            return {
                "tooltip": "Floating-point convert from single-precision to BFloat16 format (scalar) converts the single-precision floating-point value in the 32-bit SIMD&FP source register to BFloat16 format and writes the result in the 16-bit SIMD&FP destination register.",
                "html": "<p>Floating-point convert from single-precision to BFloat16 format (scalar) converts the single-precision floating-point value in the 32-bit SIMD&amp;FP source register to BFloat16 format and writes the result in the 16-bit SIMD&amp;FP destination register.</p><p><xref linkend=\"AArch64.id_aa64isar1_el1\">ID_AA64ISAR1_EL1</xref>.BF16 indicates whether this instruction is supported.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFCVTN":
        case "BFCVTN2":
            return {
                "tooltip": "Floating-point convert from single-precision to BFloat16 format (vector) reads each single-precision element in the SIMD&FP source vector, converts each value to BFloat16 format, and writes the results in the lower or upper half of the SIMD&FP destination vector. The result elements are half the width of the source elements.",
                "html": "<p>Floating-point convert from single-precision to BFloat16 format (vector) reads each single-precision element in the SIMD&amp;FP source vector, converts each value to BFloat16 format, and writes the results in the lower or upper half of the SIMD&amp;FP destination vector. The result elements are half the width of the source elements.</p><p>The BFCVTN instruction writes the half-width results to the lower half of the destination vector and clears the upper half to zero, while the BFCVTN2 instruction writes the results to the upper half of the destination vector without affecting the other bits in the register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFCVTNT":
            return {
                "tooltip": "Convert to BFloat16 from single-precision in each active floating-point element of the source vector, and place the results in the odd-numbered 16-bit elements of the destination vector, leaving the even-numbered elements unchanged. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Convert to BFloat16 from single-precision in each active floating-point element of the source vector, and place the results in the odd-numbered 16-bit elements of the destination vector, leaving the even-numbered elements unchanged. Inactive elements in the destination vector register remain unmodified.</p><p>ID_AA64ZFR0_EL1.BF16 indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFDOT":
            return {
                "tooltip": "BFloat16 floating-point dot product (vector, by element). This instruction delimits the source vectors into pairs of BFloat16 elements. The BFloat16 pair within the second source vector is specified using an immediate index. The index range is from 0 to 3 inclusive.",
                "html": "<p>BFloat16 floating-point dot product (vector, by element). This instruction delimits the source vectors into pairs of BFloat16 elements. The BFloat16 pair within the second source vector is specified using an immediate index. The index range is from 0 to 3 inclusive.</p><p>If FEAT_EBF16 is not implemented or <xref linkend=\"AArch64.fpcr\">FPCR</xref>.EBF is 0, this instruction:</p><p>If FEAT_EBF16 is implemented and <xref linkend=\"AArch64.fpcr\">FPCR</xref>.EBF is 1, then this instruction:</p><p>Irrespective of FEAT_EBF16 and <xref linkend=\"AArch64.fpcr\">FPCR</xref>.EBF, this instruction:</p><p><xref linkend=\"AArch64.id_aa64isar1_el1\">ID_AA64ISAR1_EL1</xref>.BF16 indicates whether this instruction is supported.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFI":
            return {
                "tooltip": "Bitfield Insert copies a bitfield of <width> bits from the least significant bits of the source register to bit position <lsb> of the destination register, leaving the other destination bits unchanged.",
                "html": "<p>Bitfield Insert copies a bitfield of <syntax>&lt;width&gt;</syntax> bits from the least significant bits of the source register to bit position <syntax>&lt;lsb&gt;</syntax> of the destination register, leaving the other destination bits unchanged.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFM":
            return {
                "tooltip": "Bitfield Move is usually accessed via one of its aliases, which are always preferred for disassembly.",
                "html": "<p>Bitfield Move is usually accessed via one of its aliases, which are always preferred for disassembly.</p><p>If <syntax>&lt;imms&gt;</syntax> is greater than or equal to <syntax>&lt;immr&gt;</syntax>, this copies a bitfield of (<syntax>&lt;imms&gt;</syntax>-<syntax>&lt;immr&gt;</syntax>+1) bits starting from bit position <syntax>&lt;immr&gt;</syntax> in the source register to the least significant bits of the destination register.</p><p>If <syntax>&lt;imms&gt;</syntax> is less than <syntax>&lt;immr&gt;</syntax>, this copies a bitfield of (<syntax>&lt;imms&gt;</syntax>+1) bits from the least significant bits of the source register to bit position (regsize-<syntax>&lt;immr&gt;</syntax>) of the destination register, where regsize is the destination register size of 32 or 64 bits.</p><p>In both cases the other bits of the destination register remain unchanged.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMAX":
            return {
                "tooltip": "Determine the maximum of BFloat16 elements of the second source vector and the corresponding BFloat16 elements of the two or four first source vectors and destructively place the results in the corresponding elements of the two or four first source vectors.",
                "html": "<p>Determine the maximum of BFloat16 elements of the second source vector and the corresponding BFloat16 elements of the two or four first source vectors and destructively place the results in the corresponding elements of the two or four first source vectors.</p><p>When FPCR.AH is 0, the behavior is as follows:</p><p>When FPCR.AH is 1, the behavior is as follows:</p><p>This instruction follows SME2.1 non-widening BFloat16 numerical behaviors corresponding to instructions that place their results in two or four SVE Z vectors.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMAXNM":
            return {
                "tooltip": "Determine the maximum number value of BFloat16 elements of the second source vector and the corresponding BFloat16 elements of the two or four first source vectors and destructively place the results in the corresponding elements of the two or four first source vectors.",
                "html": "<p>Determine the maximum number value of BFloat16 elements of the second source vector and the corresponding BFloat16 elements of the two or four first source vectors and destructively place the results in the corresponding elements of the two or four first source vectors.</p><p>Regardless of the value of FPCR.AH, the behavior is as follows:</p><p>This instruction follows SME2.1 non-widening BFloat16 numerical behaviors corresponding to instructions that place their results in two or four SVE Z vectors.</p><p>This instruction is unpredicated.</p><p>ID_AA64SMFR0_EL1.B16B16 indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMIN":
            return {
                "tooltip": "Determine the mininum of BFloat16 elements of the second source vector and the corresponding BFloat16 elements of the two or four first source vectors and destructively place the results in the corresponding elements of the two or four first source vectors.",
                "html": "<p>Determine the mininum of BFloat16 elements of the second source vector and the corresponding BFloat16 elements of the two or four first source vectors and destructively place the results in the corresponding elements of the two or four first source vectors.</p><p>When FPCR.AH is 0, the behavior is as follows:</p><p>When FPCR.AH is 1, the behavior is as follows:</p><p>This instruction follows SME2.1 non-widening BFloat16 numerical behaviors corresponding to instructions that place their results in two or four SVE Z vectors.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMINNM":
            return {
                "tooltip": "Determine the minimum number value of BFloat16 elements of the second source vector and the corresponding BFloat16 elements of the two or four first source vectors and destructively place the results in the corresponding elements of the two or four first source vectors.",
                "html": "<p>Determine the minimum number value of BFloat16 elements of the second source vector and the corresponding BFloat16 elements of the two or four first source vectors and destructively place the results in the corresponding elements of the two or four first source vectors.</p><p>Regardless of the value of FPCR.AH, the behavior is as follows:</p><p>This instruction follows SME2.1 non-widening BFloat16 numerical behaviors corresponding to instructions that place their results in two or four SVE Z vectors.</p><p>This instruction is unpredicated.</p><p>ID_AA64SMFR0_EL1.B16B16 indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMLA":
            return {
                "tooltip": "Multiply the corresponding active BFloat16 elements of the first and second source vectors and add to elements of the third source (addend) vector without intermediate rounding. Destructively place the results in the destination and third source (addend) vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply the corresponding active BFloat16 elements of the first and second source vectors and add to elements of the third source (addend) vector without intermediate rounding. Destructively place the results in the destination and third source (addend) vector. Inactive elements in the destination vector register remain unmodified.</p><p>This instruction follows SVE2.1 non-widening BFloat16 numerical behaviors.</p><p>ID_AA64ZFR0_EL1.B16B16 indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMLALB":
        case "BFMLALT":
            return {
                "tooltip": "BFloat16 floating-point widening multiply-add long (by element) widens the even-numbered (bottom) or odd-numbered (top) 16-bit elements in the first source vector, and the indexed element in the second source vector from Bfloat16 to single-precision format. The instruction then multiplies and adds these values without intermediate rounding to single-precision elements of the destination vector that overlap with the corresponding BFloat16 elements in the first source vector.",
                "html": "<p>BFloat16 floating-point widening multiply-add long (by element) widens the even-numbered (bottom) or odd-numbered (top) 16-bit elements in the first source vector, and the indexed element in the second source vector from Bfloat16 to single-precision format. The instruction then multiplies and adds these values without intermediate rounding to single-precision elements of the destination vector that overlap with the corresponding BFloat16 elements in the first source vector.</p><p><xref linkend=\"AArch64.id_aa64isar1_el1\">ID_AA64ISAR1_EL1</xref>.BF16 indicates whether this instruction is supported.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMLAL":
            return {
                "tooltip": "This BFloat16 floating-point multiply-add long instruction widens all 16-bit BFloat16 elements in the one, two, or four first source vectors and the indexed element of the second source vector to single-precision format, then multiplies the corresponding elements and destructively adds these values without intermediate rounding to the overlapping 32-bit single-precision elements of the ZA double-vector groups.",
                "html": "<p>This BFloat16 floating-point multiply-add long instruction widens all 16-bit BFloat16 elements in the one, two, or four first source vectors and the indexed element of the second source vector to single-precision format, then multiplies the corresponding elements and destructively adds these values without intermediate rounding to the overlapping 32-bit single-precision elements of the ZA double-vector groups.</p><p>The BF16 elements within the second source vector are specified using a 3-bit immediate index which selects the same element position within each 128-bit vector segment.</p><p>The lowest of the two consecutive vector numbers forming the double-vector group within all of, each half of, or each quarter of the ZA array are selected by the sum of the vector select register and immediate offset, modulo all, half, or quarter the number of ZA array vectors.</p><p>The <arm-defined-word>vector group</arm-defined-word> symbol, <syntax>VGx2</syntax> or <syntax>VGx4</syntax>, indicates that the ZA operand consists of two or four ZA double-vector groups respectively. The <arm-defined-word>vector group</arm-defined-word> symbol is preferred for disassembly, but optional in assembler source code.</p><p>This instruction follows SME ZA-targeting floating-point behaviors.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMLS":
            return {
                "tooltip": "Multiply the corresponding active BFloat16 elements of the first and second source vectors and subtract from elements of the third source (addend) vector without intermediate rounding. Destructively place the results in the destination and third source (addend) vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply the corresponding active BFloat16 elements of the first and second source vectors and subtract from elements of the third source (addend) vector without intermediate rounding. Destructively place the results in the destination and third source (addend) vector. Inactive elements in the destination vector register remain unmodified.</p><p>This instruction follows SVE2.1 non-widening BFloat16 numerical behaviors.</p><p>ID_AA64ZFR0_EL1.B16B16 indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMLSL":
            return {
                "tooltip": "This BFloat16 floating-point multiply-subtract long instruction widens all 16-bit BFloat16 elements in the one, two, or four first source vectors and the indexed element of the second source vector to single-precision format, then multiplies the corresponding elements and destructively subtracts these values without intermediate rounding from the overlapping 32-bit single-precision elements of the ZA double-vector groups.",
                "html": "<p>This BFloat16 floating-point multiply-subtract long instruction widens all 16-bit BFloat16 elements in the one, two, or four first source vectors and the indexed element of the second source vector to single-precision format, then multiplies the corresponding elements and destructively subtracts these values without intermediate rounding from the overlapping 32-bit single-precision elements of the ZA double-vector groups.</p><p>The BF16 elements within the second source vector are specified using a 3-bit immediate index which selects the same element position within each 128-bit vector segment.</p><p>The lowest of the two consecutive vector numbers forming the double-vector group within all of, each half of, or each quarter of the ZA array are selected by the sum of the vector select register and immediate offset, modulo all, half, or quarter the number of ZA array vectors.</p><p>The <arm-defined-word>vector group</arm-defined-word> symbol, <syntax>VGx2</syntax> or <syntax>VGx4</syntax>, indicates that the ZA operand consists of two or four ZA double-vector groups respectively. The <arm-defined-word>vector group</arm-defined-word> symbol is preferred for disassembly, but optional in assembler source code.</p><p>This instruction follows SME ZA-targeting floating-point behaviors.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMLSLB":
            return {
                "tooltip": "This BFloat16 floating-point multiply-subtract long instruction widens the even-numbered BFloat16 elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and subtracts these values without intermediate rounding from the single-precision elements of the destination vector that overlap with the corresponding BFloat16 elements in the source vectors. This instruction is unpredicated.",
                "html": "<p>This BFloat16 floating-point multiply-subtract long instruction widens the even-numbered BFloat16 elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and subtracts these values without intermediate rounding from the single-precision elements of the destination vector that overlap with the corresponding BFloat16 elements in the source vectors. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMLSLT":
            return {
                "tooltip": "This BFloat16 floating-point multiply-subtract long instruction widens the odd-numbered BFloat16 elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and subtracts these values without intermediate rounding from the single-precision elements of the destination vector that overlap with the corresponding BFloat16 elements in the source vectors. This instruction is unpredicated.",
                "html": "<p>This BFloat16 floating-point multiply-subtract long instruction widens the odd-numbered BFloat16 elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and subtracts these values without intermediate rounding from the single-precision elements of the destination vector that overlap with the corresponding BFloat16 elements in the source vectors. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMMLA":
            return {
                "tooltip": "BFloat16 floating-point matrix multiply-accumulate into 2x2 matrix.",
                "html": "<p>BFloat16 floating-point matrix multiply-accumulate into 2x2 matrix.</p><p>If FEAT_EBF16 is not implemented or <xref linkend=\"AArch64.fpcr\">FPCR</xref>.EBF is 0, this instruction:</p><p>If FEAT_EBF16 is implemented and <xref linkend=\"AArch64.fpcr\">FPCR</xref>.EBF is 1, then this instruction:</p><p>Irrespective of FEAT_EBF16 and <xref linkend=\"AArch64.fpcr\">FPCR</xref>.EBF, this instruction:</p><p><xref linkend=\"AArch64.id_aa64isar1_el1\">ID_AA64ISAR1_EL1</xref>.BF16 indicates whether this instruction is supported.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMOPA":
            return {
                "tooltip": "The BFloat16 floating-point sum of outer products and accumulate instruction works with a 32-bit element ZA tile.",
                "html": "<p>The BFloat16 floating-point sum of outer products and accumulate instruction works with a 32-bit element ZA tile.</p><p>This instruction multiplies the SVL<sub>S</sub>\u00d72 sub-matrix of BFloat16 values held in the first source vector by the 2\u00d7SVL<sub>S</sub> sub-matrix of BFloat16 values in the second source vector.</p><p>Each source vector is independently predicated by a corresponding governing predicate. When a 16-bit source element is Inactive it is treated as having the value +0.0, but if both pairs of source vector elements that correspond to a 32-bit destination element contain Inactive elements, then the destination element remains unmodified.</p><p>The resulting SVL<sub>S</sub>\u00d7SVL<sub>S</sub> single-precision floating-point sum of outer products is then destructively added to the single-precision floating-point destination tile. This is equivalent to performing a 2-way dot product and accumulate to each of the destination tile elements.</p><p>Each 32-bit container of the first source vector holds 2 consecutive column elements of each row of a SVL<sub>S</sub>\u00d72 sub-matrix. Similarly, each 32-bit container of the second source vector holds 2 consecutive row elements of each column of a 2\u00d7SVL<sub>S</sub> sub-matrix.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMOPS":
            return {
                "tooltip": "The BFloat16 floating-point sum of outer products and subtract instruction works with a 32-bit element ZA tile.",
                "html": "<p>The BFloat16 floating-point sum of outer products and subtract instruction works with a 32-bit element ZA tile.</p><p>This instruction multiplies the SVL<sub>S</sub>\u00d72 sub-matrix of BFloat16 values held in the first source vector by the 2\u00d7SVL<sub>S</sub> sub-matrix of BFloat16 values in the second source vector.</p><p>Each source vector is independently predicated by a corresponding governing predicate. When a 16-bit source element is Inactive it is treated as having the value +0.0, but if both pairs of source vector elements that correspond to a 32-bit destination element contain Inactive elements, then the destination element remains unmodified.</p><p>The resulting SVL<sub>S</sub>\u00d7SVL<sub>S</sub> single-precision floating-point sum of outer products is then destructively subtracted from the single-precision floating-point destination tile. This is equivalent to performing a 2-way dot product and subtract from each of the destination tile elements.</p><p>Each 32-bit container of the first source vector holds 2 consecutive column elements of each row of a SVL<sub>S</sub>\u00d72 sub-matrix. Similarly, each 32-bit container of the second source vector holds 2 consecutive row elements of each column of a 2\u00d7SVL<sub>S</sub> sub-matrix.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFMUL":
            return {
                "tooltip": "Multiply active BFloat16 elements of the second source vector to corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply active BFloat16 elements of the second source vector to corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.</p><p>This instruction follows SVE2.1 non-widening BFloat16 numerical behaviors.</p><p>ID_AA64ZFR0_EL1.B16B16 indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFSUB":
            return {
                "tooltip": "Subtract active BFloat16 elements of the second source vector from corresponding BFloat16 elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Subtract active BFloat16 elements of the second source vector from corresponding BFloat16 elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.</p><p>This instruction follows SVE2.1 non-widening BFloat16 numerical behaviors.</p><p>ID_AA64ZFR0_EL1.B16B16 indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFVDOT":
            return {
                "tooltip": "The instruction computes the sum-of-products of each vertical pair of BFloat16 values in the corresponding elements of the two first source vectors with the pair of BFloat16 values in the indexed 32-bit group of the corresponding 128-bit segment of the second source vector. The single-precision sum-of-products results are destructively added to the corresponding single-precision elements of the two ZA single-vector groups.",
                "html": "<p>The instruction computes the sum-of-products of each vertical pair of BFloat16 values in the corresponding elements of the two first source vectors with the pair of BFloat16 values in the indexed 32-bit group of the corresponding 128-bit segment of the second source vector. The single-precision sum-of-products results are destructively added to the corresponding single-precision elements of the two ZA single-vector groups.</p><p>The BF16 pairs within the second source vector are specified using an immediate index which selects the same BF16 pair position within each 128-bit vector segment. The element index range is from 0 to 3.</p><p>The vector numbers forming the single-vector group within each half of the ZA array are selected by the sum of the vector select register and immediate offset, modulo half the number of ZA array vectors.</p><p>The <arm-defined-word>vector group</arm-defined-word> symbol <syntax>VGx2</syntax> indicates that the ZA operand consists of two ZA single-vector groups. The <arm-defined-word>vector group</arm-defined-word> symbol is preferred for disassembly, but optional in assembler source code.</p><p>This instruction follows SME2 ZA-targeting BFloat16 numerical behaviors.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BFXIL":
            return {
                "tooltip": "Bitfield Extract and Insert Low copies a bitfield of <width> bits starting from bit position <lsb> in the source register to the least significant bits of the destination register, leaving the other destination bits unchanged.",
                "html": "<p>Bitfield Extract and Insert Low copies a bitfield of <syntax>&lt;width&gt;</syntax> bits starting from bit position <syntax>&lt;lsb&gt;</syntax> in the source register to the least significant bits of the destination register, leaving the other destination bits unchanged.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BGRP":
            return {
                "tooltip": "This instruction separates bits in each element of the first source vector by gathering from the bit positions indicated by non-zero bits in the corresponding mask element of the second source vector to the lowest-numbered contiguous bits of the corresponding destination element, and from positions indicated by zero bits to the highest-numbered bits of the destination element, preserving the bit order within each group. This instruction is unpredicated.",
                "html": "<p>This instruction separates bits in each element of the first source vector by gathering from the bit positions indicated by non-zero bits in the corresponding mask element of the second source vector to the lowest-numbered contiguous bits of the corresponding destination element, and from positions indicated by zero bits to the highest-numbered bits of the destination element, preserving the bit order within each group. This instruction is unpredicated.</p><p>ID_AA64ZFR0_EL1.BitPerm indicates whether this instruction is implemented.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BIC":
            return {
                "tooltip": "Bitwise bit Clear (vector, immediate). This instruction reads each vector element from the destination SIMD&FP register, performs a bitwise AND between each result and the complement of an immediate constant, places the result into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Bitwise bit Clear (vector, immediate). This instruction reads each vector element from the destination SIMD&amp;FP register, performs a bitwise AND between each result and the complement of an immediate constant, places the result into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BICS":
            return {
                "tooltip": "Bitwise Bit Clear (shifted register), setting flags, performs a bitwise AND of a register value and the complement of an optionally-shifted register value, and writes the result to the destination register. It updates the condition flags based on the result.",
                "html": "<p>Bitwise Bit Clear (shifted register), setting flags, performs a bitwise AND of a register value and the complement of an optionally-shifted register value, and writes the result to the destination register. It updates the condition flags based on the result.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BIF":
            return {
                "tooltip": "Bitwise Insert if False. This instruction inserts each bit from the first source SIMD&FP register into the destination SIMD&FP register if the corresponding bit of the second source SIMD&FP register is 0, otherwise leaves the bit in the destination register unchanged.",
                "html": "<p>Bitwise Insert if False. This instruction inserts each bit from the first source SIMD&amp;FP register into the destination SIMD&amp;FP register if the corresponding bit of the second source SIMD&amp;FP register is 0, otherwise leaves the bit in the destination register unchanged.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BIT":
            return {
                "tooltip": "Bitwise Insert if True. This instruction inserts each bit from the first source SIMD&FP register into the SIMD&FP destination register if the corresponding bit of the second source SIMD&FP register is 1, otherwise leaves the bit in the destination register unchanged.",
                "html": "<p>Bitwise Insert if True. This instruction inserts each bit from the first source SIMD&amp;FP register into the SIMD&amp;FP destination register if the corresponding bit of the second source SIMD&amp;FP register is 1, otherwise leaves the bit in the destination register unchanged.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BL":
            return {
                "tooltip": "Branch with Link branches to a PC-relative offset, setting the register X30 to PC+4. It provides a hint that this is a subroutine call.",
                "html": "<p>Branch with Link branches to a PC-relative offset, setting the register X30 to PC+4. It provides a hint that this is a subroutine call.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BLR":
            return {
                "tooltip": "Branch with Link to Register calls a subroutine at an address in a register, setting register X30 to PC+4.",
                "html": "<p>Branch with Link to Register calls a subroutine at an address in a register, setting register X30 to PC+4.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BLRAA":
        case "BLRAAZ":
        case "BLRAB":
        case "BLRABZ":
            return {
                "tooltip": "Branch with Link to Register, with pointer authentication. This instruction authenticates the address in the general-purpose register that is specified by <Xn>, using a modifier and the specified key, and calls a subroutine at the authenticated address, setting register X30 to PC+4.",
                "html": "<p>Branch with Link to Register, with pointer authentication. This instruction authenticates the address in the general-purpose register that is specified by <syntax>&lt;Xn&gt;</syntax>, using a modifier and the specified key, and calls a subroutine at the authenticated address, setting register X30 to PC+4.</p><p>The modifier is:</p><p>Key A is used for <instruction>BLRAA</instruction> and <instruction>BLRAAZ</instruction>. Key B is used for <instruction>BLRAB</instruction> and <instruction>BLRABZ</instruction>.</p><p>If the authentication passes, the PE continues execution at the target of the branch. For information on behavior if the authentication fails, see <xref>Faulting on pointer authentication</xref>.</p><p>The authenticated address is not written back to the general-purpose register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BMOPA":
            return {
                "tooltip": "This instruction works with 32-bit element ZA tile. This instruction generates an outer product of the first source SVLS\u00d71 vector and the second source 1\u00d7SVLS vector. Each outer product element is obtained as population count of the bitwise XNOR result of the corresponding 32-bit elements of the first source vector and the second source vector. Each source vector is independently predicated by a corresponding governing predicate. When either source vector element is inactive the corresponding destination tile element remains unmodified. The resulting SVLS\u00d7SVLS product is then destructively added to the destination tile.",
                "html": "<p>This instruction works with 32-bit element ZA tile. This instruction generates an outer product of the first source SVL<sub>S</sub>\u00d71 vector and the second source 1\u00d7SVL<sub>S</sub> vector. Each outer product element is obtained as population count of the bitwise XNOR result of the corresponding 32-bit elements of the first source vector and the second source vector. Each source vector is independently predicated by a corresponding governing predicate. When either source vector element is inactive the corresponding destination tile element remains unmodified. The resulting SVL<sub>S</sub>\u00d7SVL<sub>S</sub> product is then destructively added to the destination tile.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BMOPS":
            return {
                "tooltip": "This instruction works with 32-bit element ZA tile. This instruction generates an outer product of the first source SVLS\u00d71 vector and the second source 1\u00d7SVLS vector. Each outer product element is obtained as population count of the bitwise XNOR result of the corresponding 32-bit elements of the first source vector and the second source vector. Each source vector is independently predicated by a corresponding governing predicate. When either source vector element is inactive the corresponding destination tile element remains unmodified. The resulting SVLS\u00d7SVLS product is then destructively subtracted from the destination tile.",
                "html": "<p>This instruction works with 32-bit element ZA tile. This instruction generates an outer product of the first source SVL<sub>S</sub>\u00d71 vector and the second source 1\u00d7SVL<sub>S</sub> vector. Each outer product element is obtained as population count of the bitwise XNOR result of the corresponding 32-bit elements of the first source vector and the second source vector. Each source vector is independently predicated by a corresponding governing predicate. When either source vector element is inactive the corresponding destination tile element remains unmodified. The resulting SVL<sub>S</sub>\u00d7SVL<sub>S</sub> product is then destructively subtracted from the destination tile.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BR":
            return {
                "tooltip": "Branch to Register branches unconditionally to an address in a register, with a hint that this is not a subroutine return.",
                "html": "<p>Branch to Register branches unconditionally to an address in a register, with a hint that this is not a subroutine return.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRAA":
        case "BRAAZ":
        case "BRAB":
        case "BRABZ":
            return {
                "tooltip": "Branch to Register, with pointer authentication. This instruction authenticates the address in the general-purpose register that is specified by <Xn>, using a modifier and the specified key, and branches to the authenticated address.",
                "html": "<p>Branch to Register, with pointer authentication. This instruction authenticates the address in the general-purpose register that is specified by <syntax>&lt;Xn&gt;</syntax>, using a modifier and the specified key, and branches to the authenticated address.</p><p>The modifier is:</p><p>Key A is used for <instruction>BRAA</instruction> and <instruction>BRAAZ</instruction>. Key B is used for <instruction>BRAB</instruction> and <instruction>BRABZ</instruction>.</p><p>If the authentication passes, the PE continues execution at the target of the branch. For information on behavior if the authentication fails, see <xref>Faulting on pointer authentication</xref>.</p><p>The authenticated address is not written back to the general-purpose register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRB":
            return {
                "tooltip": "Branch Record Buffer. For more information, see op0==0b01, cache maintenance, TLB maintenance, and address translation instructions.",
                "html": "<p>Branch Record Buffer. For more information, see <xref linkend=\"BABEJJJE\">op0==0b01, cache maintenance, TLB maintenance, and address translation instructions</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRK":
            return {
                "tooltip": "Breakpoint instruction. A BRK instruction generates a Breakpoint Instruction exception. The PE records the exception in ESR_ELx, using the EC value 0x3c, and captures the value of the immediate argument in ESR_ELx.ISS.",
                "html": "<p>Breakpoint instruction. A <instruction>BRK</instruction> instruction generates a Breakpoint Instruction exception. The PE records the exception in <xref linkend=\"ESR_ELx\">ESR_ELx</xref>, using the EC value <hexnumber>0x3c</hexnumber>, and captures the value of the immediate argument in <xref linkend=\"ESR_ELx\">ESR_ELx</xref>.ISS.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRKA":
            return {
                "tooltip": "Sets destination predicate elements up to and including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register remain unmodified or are set to zero, depending on whether merging or zeroing predication is selected. Does not set the condition flags.",
                "html": "<p>Sets destination predicate elements up to and including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register remain unmodified or are set to zero, depending on whether merging or zeroing predication is selected. Does not set the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRKAS":
            return {
                "tooltip": "Sets destination predicate elements up to and including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Sets destination predicate elements up to and including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRKB":
            return {
                "tooltip": "Sets destination predicate elements up to but not including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register remain unmodified or are set to zero, depending on whether merging or zeroing predication is selected. Does not set the condition flags.",
                "html": "<p>Sets destination predicate elements up to but not including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register remain unmodified or are set to zero, depending on whether merging or zeroing predication is selected. Does not set the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRKBS":
            return {
                "tooltip": "Sets destination predicate elements up to but not including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Sets destination predicate elements up to but not including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRKN":
            return {
                "tooltip": "If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise leaves the destination and second source predicate unchanged. Does not set the condition flags.",
                "html": "<p>If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise leaves the destination and second source predicate unchanged. Does not set the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRKNS":
            return {
                "tooltip": "If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise leaves the destination and second source predicate unchanged. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise leaves the destination and second source predicate unchanged. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRKPA":
            return {
                "tooltip": "If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise sets destination predicate elements up to and including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.",
                "html": "<p>If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise sets destination predicate elements up to and including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRKPAS":
            return {
                "tooltip": "If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise sets destination predicate elements up to and including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise sets destination predicate elements up to and including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRKPB":
            return {
                "tooltip": "If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise sets destination predicate elements up to but not including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.",
                "html": "<p>If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise sets destination predicate elements up to but not including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BRKPBS":
            return {
                "tooltip": "If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise sets destination predicate elements up to but not including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>If the last active element of the first source predicate is false then set the destination predicate to all-false. Otherwise sets destination predicate elements up to but not including the first active and true source element to true, then sets subsequent elements to false. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BSL1N":
            return {
                "tooltip": "Selects bits from the inverted first source vector where the corresponding bit in the third source vector is '1', and from the second source vector where the corresponding bit in the third source vector is '0'. The result is placed destructively in the destination and first source vector. This instruction is unpredicated.",
                "html": "<p>Selects bits from the inverted first source vector where the corresponding bit in the third source vector is '1', and from the second source vector where the corresponding bit in the third source vector is '0'. The result is placed destructively in the destination and first source vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BSL2N":
            return {
                "tooltip": "Selects bits from the first source vector where the corresponding bit in the third source vector is '1', and from the inverted second source vector where the corresponding bit in the third source vector is '0'. The result is placed destructively in the destination and first source vector. This instruction is unpredicated.",
                "html": "<p>Selects bits from the first source vector where the corresponding bit in the third source vector is '1', and from the inverted second source vector where the corresponding bit in the third source vector is '0'. The result is placed destructively in the destination and first source vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BSL":
            return {
                "tooltip": "Bitwise Select. This instruction sets each bit in the destination SIMD&FP register to the corresponding bit from the first source SIMD&FP register when the original destination bit was 1, otherwise from the second source SIMD&FP register.",
                "html": "<p>Bitwise Select. This instruction sets each bit in the destination SIMD&amp;FP register to the corresponding bit from the first source SIMD&amp;FP register when the original destination bit was 1, otherwise from the second source SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "BTI":
            return {
                "tooltip": "Branch Target Identification. A BTI instruction is used to guard against the execution of instructions which are not the intended target of a branch.",
                "html": "<p>Branch Target Identification. A <instruction>BTI</instruction> instruction is used to guard against the execution of instructions which are not the intended target of a branch.</p><p>Outside of a guarded memory region, a <instruction>BTI</instruction> instruction executes as a <instruction>NOP</instruction>. Within a guarded memory region while <xref linkend=\"BEIDIGBH\">PSTATE</xref>.BTYPE != <binarynumber>0b00</binarynumber>, a <instruction>BTI</instruction> instruction compatible with the current value of PSTATE.BTYPE will not generate a Branch Target Exception and will allow execution of subsequent instructions within the memory region.</p><p>The operand <syntax>&lt;targets&gt;</syntax> passed to a <instruction>BTI</instruction> instruction determines the values of <xref linkend=\"BEIDIGBH\">PSTATE</xref>.BTYPE which the <instruction>BTI</instruction> instruction is compatible with.</p><p>Within a guarded memory region, when <xref linkend=\"BEIDIGBH\">PSTATE</xref>.BTYPE != <binarynumber>0b00</binarynumber>, all instructions will generate a Branch Target Exception, other than <instruction>BRK</instruction>, <instruction>BTI</instruction>, <instruction>HLT</instruction>, <instruction>PACIASP</instruction>, and <instruction>PACIBSP</instruction>, which might not. See the individual instructions for more information.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CADD":
            return {
                "tooltip": "Add the real and imaginary components of the integral complex numbers from the first source vector to the complex numbers from the second source vector which have first been rotated by 90 or 270 degrees in the direction from the positive real axis towards the positive imaginary axis, when considered in polar representation, equivalent to multiplying the complex numbers in the second source vector by \u00b1j beforehand. Destructively place the results in the corresponding elements of the first source vector. This instruction is unpredicated.",
                "html": "<p>Add the real and imaginary components of the integral complex numbers from the first source vector to the complex numbers from the second source vector which have first been rotated by 90 or 270 degrees in the direction from the positive real axis towards the positive imaginary axis, when considered in polar representation, equivalent to multiplying the complex numbers in the second source vector by \u00b1<arm-defined-word>j</arm-defined-word> beforehand. Destructively place the results in the corresponding elements of the first source vector. This instruction is unpredicated.</p><p>Each complex number is represented in a vector register as an even/odd pair of elements with the real part in the even-numbered element and the imaginary part in the odd-numbered element.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CAS":
        case "CASA":
        case "CASAL":
        case "CASL":
            return {
                "tooltip": "Compare and Swap word or doubleword in memory reads a 32-bit word or 64-bit doubleword from memory, and compares it against the value held in a first register. If the comparison is equal, the value in a second register is written to memory. If the write is performed, the read and write occur atomically such that no other modification of the memory location can take place between the read and write.",
                "html": "<p>Compare and Swap word or doubleword in memory reads a 32-bit word or 64-bit doubleword from memory, and compares it against the value held in a first register. If the comparison is equal, the value in a second register is written to memory. If the write is performed, the read and write occur atomically such that no other modification of the memory location can take place between the read and write.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>The architecture permits that the data read clears any exclusive monitors associated with that location, even if the compare subsequently fails.</p><p>If the instruction generates a synchronous Data Abort, the register which is compared and loaded, that is <syntax>&lt;Ws&gt;</syntax>, or <syntax>&lt;Xs&gt;</syntax>, is restored to the value held in the register before the instruction was executed.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CASAB":
        case "CASALB":
        case "CASB":
        case "CASLB":
            return {
                "tooltip": "Compare and Swap byte in memory reads an 8-bit byte from memory, and compares it against the value held in a first register. If the comparison is equal, the value in a second register is written to memory. If the write is performed, the read and write occur atomically such that no other modification of the memory location can take place between the read and write.",
                "html": "<p>Compare and Swap byte in memory reads an 8-bit byte from memory, and compares it against the value held in a first register. If the comparison is equal, the value in a second register is written to memory. If the write is performed, the read and write occur atomically such that no other modification of the memory location can take place between the read and write.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>The architecture permits that the data read clears any exclusive monitors associated with that location, even if the compare subsequently fails.</p><p>If the instruction generates a synchronous Data Abort, the register which is compared and loaded, that is <syntax>&lt;Ws&gt;</syntax>, is restored to the values held in the register before the instruction was executed.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CASAH":
        case "CASALH":
        case "CASH":
        case "CASLH":
            return {
                "tooltip": "Compare and Swap halfword in memory reads a 16-bit halfword from memory, and compares it against the value held in a first register. If the comparison is equal, the value in a second register is written to memory. If the write is performed, the read and write occur atomically such that no other modification of the memory location can take place between the read and write.",
                "html": "<p>Compare and Swap halfword in memory reads a 16-bit halfword from memory, and compares it against the value held in a first register. If the comparison is equal, the value in a second register is written to memory. If the write is performed, the read and write occur atomically such that no other modification of the memory location can take place between the read and write.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>The architecture permits that the data read clears any exclusive monitors associated with that location, even if the compare subsequently fails.</p><p>If the instruction generates a synchronous Data Abort, the register which is compared and loaded, that is <syntax>&lt;Ws&gt;</syntax>, is restored to the values held in the register before the instruction was executed.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CASP":
        case "CASPA":
        case "CASPAL":
        case "CASPL":
            return {
                "tooltip": "Compare and Swap Pair of words or doublewords in memory reads a pair of 32-bit words or 64-bit doublewords from memory, and compares them against the values held in the first pair of registers. If the comparison is equal, the values in the second pair of registers are written to memory. If the writes are performed, the reads and writes occur atomically such that no other modification of the memory location can take place between the reads and writes.",
                "html": "<p>Compare and Swap Pair of words or doublewords in memory reads a pair of 32-bit words or 64-bit doublewords from memory, and compares them against the values held in the first pair of registers. If the comparison is equal, the values in the second pair of registers are written to memory. If the writes are performed, the reads and writes occur atomically such that no other modification of the memory location can take place between the reads and writes.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>The architecture permits that the data read clears any exclusive monitors associated with that location, even if the compare subsequently fails.</p><p>If the instruction generates a synchronous Data Abort, the registers which are compared and loaded, that is <syntax>&lt;Ws&gt;</syntax> and <syntax>&lt;W(s+1)&gt;</syntax>, or <syntax>&lt;Xs&gt;</syntax> and <syntax>&lt;X(s+1)&gt;</syntax>, are restored to the values held in the registers before the instruction was executed.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CBNZ":
            return {
                "tooltip": "Compare and Branch on Nonzero compares the value in a register with zero, and conditionally branches to a label at a PC-relative offset if the comparison is not equal. It provides a hint that this is not a subroutine call or return. This instruction does not affect the condition flags.",
                "html": "<p>Compare and Branch on Nonzero compares the value in a register with zero, and conditionally branches to a label at a PC-relative offset if the comparison is not equal. It provides a hint that this is not a subroutine call or return. This instruction does not affect the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CBZ":
            return {
                "tooltip": "Compare and Branch on Zero compares the value in a register with zero, and conditionally branches to a label at a PC-relative offset if the comparison is equal. It provides a hint that this is not a subroutine call or return. This instruction does not affect condition flags.",
                "html": "<p>Compare and Branch on Zero compares the value in a register with zero, and conditionally branches to a label at a PC-relative offset if the comparison is equal. It provides a hint that this is not a subroutine call or return. This instruction does not affect condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CCMN":
            return {
                "tooltip": "Conditional Compare Negative (immediate) sets the value of the condition flags to the result of the comparison of a register value and a negated immediate value if the condition is TRUE, and an immediate value otherwise.",
                "html": "<p>Conditional Compare Negative (immediate) sets the value of the condition flags to the result of the comparison of a register value and a negated immediate value if the condition is TRUE, and an immediate value otherwise.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CCMP":
            return {
                "tooltip": "Conditional Compare (immediate) sets the value of the condition flags to the result of the comparison of a register value and an immediate value if the condition is TRUE, and an immediate value otherwise.",
                "html": "<p>Conditional Compare (immediate) sets the value of the condition flags to the result of the comparison of a register value and an immediate value if the condition is TRUE, and an immediate value otherwise.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CDOT":
            return {
                "tooltip": "The complex integer dot product instructions delimit the source vectors into pairs of 8-bit or 16-bit signed integer complex numbers. Within each pair, the complex numbers in the first source vector are multiplied by the corresponding complex numbers in the second source vector and the resulting wide real or wide imaginary part of the product is accumulated into a 32-bit or 64-bit destination vector element which overlaps all four of the elements that comprise a pair of complex number values in the first source vector.",
                "html": "<p>The complex integer dot product instructions delimit the source vectors into pairs of 8-bit or 16-bit signed integer complex numbers. Within each pair, the complex numbers in the first source vector are multiplied by the corresponding complex numbers in the second source vector and the resulting wide real or wide imaginary part of the product is accumulated into a 32-bit or 64-bit destination vector element which overlaps all four of the elements that comprise a pair of complex number values in the first source vector.</p><p>As a result each instruction implicitly deinterleaves the real and imaginary components of their complex number inputs, so that the destination vector accumulates 4\u00d7wide real sums or 4\u00d7wide imaginary sums.</p><p>The complex numbers in the second source vector are rotated by 0, 90, 180 or 270 degrees in the direction from the positive real axis towards the positive imaginary axis, when considered in polar representation, by performing the following transformations prior to the dot product operations:</p><p></p><p>Each complex number is represented in a vector register as an even/odd pair of elements with the real part in the even-numbered element and the imaginary part in the odd-numbered element.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CFINV":
            return {
                "tooltip": "Invert Carry Flag. This instruction inverts the value of the PSTATE.C flag.",
                "html": "<p>Invert Carry Flag. This instruction inverts the value of the PSTATE.C flag.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CFP":
            return {
                "tooltip": "Control Flow Prediction Restriction by Context prevents control flow predictions that predict execution addresses based on information gathered from earlier execution within a particular execution context. Control flow predictions determined by the actions of code in the target execution context or contexts appearing in program order before the instruction cannot be used to exploitatively control speculative execution occurring after the instruction is complete and synchronized.",
                "html": "<p>Control Flow Prediction Restriction by Context prevents control flow predictions that predict execution addresses based on information gathered from earlier execution within a particular execution context. Control flow predictions determined by the actions of code in the target execution context or contexts appearing in program order before the instruction cannot be used to exploitatively control speculative execution occurring after the instruction is complete and synchronized.</p><p>For more information, see <xref linkend=\"AArch64.cfp_rctx\">CFP RCTX, Control Flow Prediction Restriction by Context</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CHKFEAT":
            return {
                "tooltip": "Check feature status. This instruction indicates the status of features.",
                "html": "<p>Check feature status. This instruction indicates the status of features.</p><p>If FEAT_CHK is not implemented, this instruction executes as a <instruction>NOP</instruction>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CINC":
            return {
                "tooltip": "Conditional Increment returns, in the destination register, the value of the source register incremented by 1 if the condition is TRUE, and otherwise returns the value of the source register.",
                "html": "<p>Conditional Increment returns, in the destination register, the value of the source register incremented by 1 if the condition is TRUE, and otherwise returns the value of the source register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CINV":
            return {
                "tooltip": "Conditional Invert returns, in the destination register, the bitwise inversion of the value of the source register if the condition is TRUE, and otherwise returns the value of the source register.",
                "html": "<p>Conditional Invert returns, in the destination register, the bitwise inversion of the value of the source register if the condition is TRUE, and otherwise returns the value of the source register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CLASTA":
            return {
                "tooltip": "From the source vector register extract the element after the last active element, or if the last active element is the final element extract element zero, and then zero-extend that element to destructively place in the destination and first source general-purpose register. If there are no active elements then destructively zero-extend the least significant element-size bits of the destination and first source general-purpose register.",
                "html": "<p>From the source vector register extract the element after the last active element, or if the last active element is the final element extract element zero, and then zero-extend that element to destructively place in the destination and first source general-purpose register. If there are no active elements then destructively zero-extend the least significant element-size bits of the destination and first source general-purpose register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CLASTB":
            return {
                "tooltip": "From the source vector register extract the last active element, and then zero-extend that element to destructively place in the destination and first source general-purpose register. If there are no active elements then destructively zero-extend the least significant element-size bits of the destination and first source general-purpose register.",
                "html": "<p>From the source vector register extract the last active element, and then zero-extend that element to destructively place in the destination and first source general-purpose register. If there are no active elements then destructively zero-extend the least significant element-size bits of the destination and first source general-purpose register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CLRBHB":
            return {
                "tooltip": "Clear Branch History clears the branch history for the current context to the extent that branch history information created before the CLRBHB instruction cannot be used by code before the CLRBHB instruction to exploitatively control the execution of any indirect branches in code in the current context that appear in program order after the instruction.",
                "html": "<p>Clear Branch History clears the branch history for the current context to the extent that branch history information created before the <instruction>CLRBHB</instruction> instruction cannot be used by code before the <instruction>CLRBHB</instruction> instruction to exploitatively control the execution of any indirect branches in code in the current context that appear in program order after the instruction.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CLREX":
            return {
                "tooltip": "Clear Exclusive clears the local monitor of the executing PE.",
                "html": "<p>Clear Exclusive clears the local monitor of the executing PE.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CLS":
            return {
                "tooltip": "Count Leading Sign bits (vector). This instruction counts the number of consecutive bits following the most significant bit that are the same as the most significant bit in each vector element in the source SIMD&FP register, places the result into a vector, and writes the vector to the destination SIMD&FP register. The count does not include the most significant bit itself.",
                "html": "<p>Count Leading Sign bits (vector). This instruction counts the number of consecutive bits following the most significant bit that are the same as the most significant bit in each vector element in the source SIMD&amp;FP register, places the result into a vector, and writes the vector to the destination SIMD&amp;FP register. The count does not include the most significant bit itself.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CLZ":
            return {
                "tooltip": "Count Leading Zero bits (vector). This instruction counts the number of consecutive zeros, starting from the most significant bit, in each vector element in the source SIMD&FP register, places the result into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Count Leading Zero bits (vector). This instruction counts the number of consecutive zeros, starting from the most significant bit, in each vector element in the source SIMD&amp;FP register, places the result into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMEQ":
            return {
                "tooltip": "Compare bitwise Equal (vector). This instruction compares each vector element from the first source SIMD&FP register with the corresponding vector element from the second source SIMD&FP register, and if the comparison is equal sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Compare bitwise Equal (vector). This instruction compares each vector element from the first source SIMD&amp;FP register with the corresponding vector element from the second source SIMD&amp;FP register, and if the comparison is equal sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMGE":
            return {
                "tooltip": "Compare signed Greater than or Equal (vector). This instruction compares each vector element in the first source SIMD&FP register with the corresponding vector element in the second source SIMD&FP register and if the first signed integer value is greater than or equal to the second signed integer value sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Compare signed Greater than or Equal (vector). This instruction compares each vector element in the first source SIMD&amp;FP register with the corresponding vector element in the second source SIMD&amp;FP register and if the first signed integer value is greater than or equal to the second signed integer value sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMGT":
            return {
                "tooltip": "Compare signed Greater than (vector). This instruction compares each vector element in the first source SIMD&FP register with the corresponding vector element in the second source SIMD&FP register and if the first signed integer value is greater than the second signed integer value sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Compare signed Greater than (vector). This instruction compares each vector element in the first source SIMD&amp;FP register with the corresponding vector element in the second source SIMD&amp;FP register and if the first signed integer value is greater than the second signed integer value sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMHI":
            return {
                "tooltip": "Compare unsigned Higher (vector). This instruction compares each vector element in the first source SIMD&FP register with the corresponding vector element in the second source SIMD&FP register and if the first unsigned integer value is greater than the second unsigned integer value sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Compare unsigned Higher (vector). This instruction compares each vector element in the first source SIMD&amp;FP register with the corresponding vector element in the second source SIMD&amp;FP register and if the first unsigned integer value is greater than the second unsigned integer value sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMHS":
            return {
                "tooltip": "Compare unsigned Higher or Same (vector). This instruction compares each vector element in the first source SIMD&FP register with the corresponding vector element in the second source SIMD&FP register and if the first unsigned integer value is greater than or equal to the second unsigned integer value sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Compare unsigned Higher or Same (vector). This instruction compares each vector element in the first source SIMD&amp;FP register with the corresponding vector element in the second source SIMD&amp;FP register and if the first unsigned integer value is greater than or equal to the second unsigned integer value sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMLA":
            return {
                "tooltip": "Multiply the duplicated real components for rotations 0 and 180, or imaginary components for rotations 90 and 270, of the integral numbers in the first source vector by the corresponding complex number in the second source vector rotated by 0, 90, 180 or 270 degrees in the direction from the positive real axis towards the positive imaginary axis, when considered in polar representation.",
                "html": "<p>Multiply the duplicated real components for rotations 0 and 180, or imaginary components for rotations 90 and 270, of the integral numbers in the first source vector by the corresponding complex number in the second source vector rotated by 0, 90, 180 or 270 degrees in the direction from the positive real axis towards the positive imaginary axis, when considered in polar representation.</p><p>Then add the products to the corresponding components of the complex numbers in the addend vector. Destructively place the results in the corresponding elements of the addend vector. This instruction is unpredicated.</p><p>These transformations permit the creation of a variety of multiply-add and multiply-subtract operations on complex numbers by combining two of these instructions with the same vector operands but with rotations that are 90 degrees apart.</p><p>Each complex number is represented in a vector register as an even/odd pair of elements with the real part in the even-numbered element and the imaginary part in the odd-numbered element.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMLE":
            return {
                "tooltip": "Compare signed Less than or Equal to zero (vector). This instruction reads each vector element in the source SIMD&FP register and if the signed integer value is less than or equal to zero sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Compare signed Less than or Equal to zero (vector). This instruction reads each vector element in the source SIMD&amp;FP register and if the signed integer value is less than or equal to zero sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMLT":
            return {
                "tooltip": "Compare signed Less than zero (vector). This instruction reads each vector element in the source SIMD&FP register and if the signed integer value is less than zero sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Compare signed Less than zero (vector). This instruction reads each vector element in the source SIMD&amp;FP register and if the signed integer value is less than zero sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMN":
            return {
                "tooltip": "Compare Negative (extended register) adds a register value and a sign or zero-extended register value, followed by an optional left shift amount. The argument that is extended from the <Rm> register can be a byte, halfword, word, or doubleword. It updates the condition flags based on the result, and discards the result.",
                "html": "<p>Compare Negative (extended register) adds a register value and a sign or zero-extended register value, followed by an optional left shift amount. The argument that is extended from the <syntax>&lt;Rm&gt;</syntax> register can be a byte, halfword, word, or doubleword. It updates the condition flags based on the result, and discards the result.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMP":
            return {
                "tooltip": "Compare (extended register) subtracts a sign or zero-extended register value, followed by an optional left shift amount, from a register value. The argument that is extended from the <Rm> register can be a byte, halfword, word, or doubleword. It updates the condition flags based on the result, and discards the result.",
                "html": "<p>Compare (extended register) subtracts a sign or zero-extended register value, followed by an optional left shift amount, from a register value. The argument that is extended from the <syntax>&lt;Rm&gt;</syntax> register can be a byte, halfword, word, or doubleword. It updates the condition flags based on the result, and discards the result.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMP<cc>":
        case "CMPAL":
        case "CMPCC":
        case "CMPCS":
        case "CMPEQ":
        case "CMPGE":
        case "CMPGT":
        case "CMPHI":
        case "CMPLE":
        case "CMPLS":
        case "CMPLT":
        case "CMPMI":
        case "CMPNE":
        case "CMPPL":
        case "CMPVC":
        case "CMPVS":
            return {
                "tooltip": "Compare active integer elements in the source vector with an immediate, and place the boolean results of the specified comparison in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Compare active integer elements in the source vector with an immediate, and place the boolean results of the specified comparison in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p><p></p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMPLO":
            return {
                "tooltip": "Compare active unsigned integer elements in the first source vector being lower than corresponding unsigned elements in the second source vector, and place the boolean results of the comparison in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Compare active unsigned integer elements in the first source vector being lower than corresponding unsigned elements in the second source vector, and place the boolean results of the comparison in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMPP":
            return {
                "tooltip": "Compare with Tag subtracts the 56-bit address held in the second source register from the 56-bit address held in the first source register, updates the condition flags based on the result of the subtraction, and discards the result.",
                "html": "<p>Compare with Tag subtracts the 56-bit address held in the second source register from the 56-bit address held in the first source register, updates the condition flags based on the result of the subtraction, and discards the result.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CMTST":
            return {
                "tooltip": "Compare bitwise Test bits nonzero (vector). This instruction reads each vector element in the first source SIMD&FP register, performs an AND with the corresponding vector element in the second source SIMD&FP register, and if the result is not zero, sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Compare bitwise Test bits nonzero (vector). This instruction reads each vector element in the first source SIMD&amp;FP register, performs an AND with the corresponding vector element in the second source SIMD&amp;FP register, and if the result is not zero, sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CNEG":
            return {
                "tooltip": "Conditional Negate returns, in the destination register, the negated value of the source register if the condition is TRUE, and otherwise returns the value of the source register.",
                "html": "<p>Conditional Negate returns, in the destination register, the negated value of the source register if the condition is TRUE, and otherwise returns the value of the source register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CNOT":
            return {
                "tooltip": "Logically invert the boolean value in each active element of the source vector, and place the results in the corresponding elements of the destination vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Logically invert the boolean value in each active element of the source vector, and place the results in the corresponding elements of the destination vector. Inactive elements in the destination vector register remain unmodified.</p><p>Boolean TRUE is any non-zero value in a source, and one in a result element. Boolean FALSE is always zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CNT":
            return {
                "tooltip": "Count bits counts the number of binary one bits in the value of the source register, and writes the result to the destination register.",
                "html": "<p>Count bits counts the number of binary one bits in the value of the source register, and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CNTB":
        case "CNTD":
        case "CNTH":
        case "CNTW":
            return {
                "tooltip": "Determines the number of active elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then places the result in the scalar destination.",
                "html": "<p>Determines the number of active elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then places the result in the scalar destination.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CNTP":
            return {
                "tooltip": "Counts the number of active and true elements in the source predicate and places the scalar result in the destination general-purpose register. Inactive predicate elements are not counted.",
                "html": "<p>Counts the number of active and true elements in the source predicate and places the scalar result in the destination general-purpose register. Inactive predicate elements are not counted.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "COMPACT":
            return {
                "tooltip": "Read the active elements from the source vector and pack them into the lowest-numbered elements of the destination vector. Then set any remaining elements of the destination vector to zero.",
                "html": "<p>Read the active elements from the source vector and pack them into the lowest-numbered elements of the destination vector. Then set any remaining elements of the destination vector to zero.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "COSP":
            return {
                "tooltip": "Clear Other Speculative Predictions by Context prevents predictions, other than Cache prefetch, Control flow, and Data Value predictions, that predict execution addresses based on information gathered from earlier execution within a particular execution context. Predictions, other than Cache prefetch, Control flow, and Data Value predictions, determined by the actions of code in the target execution context or contexts appearing in program order before the instruction cannot exploitatively control any speculative access occurring after the instruction is complete and synchronized.",
                "html": "<p>Clear Other Speculative Predictions by Context prevents predictions, other than Cache prefetch, Control flow, and Data Value predictions, that predict execution addresses based on information gathered from earlier execution within a particular execution context. Predictions, other than Cache prefetch, Control flow, and Data Value predictions, determined by the actions of code in the target execution context or contexts appearing in program order before the instruction cannot exploitatively control any speculative access occurring after the instruction is complete and synchronized.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPP":
            return {
                "tooltip": "Cache Prefetch Prediction Restriction by Context prevents cache allocation predictions that predict execution addresses based on information gathered from earlier execution within a particular execution context. The actions of code in the target execution context or contexts appearing in program order before the instruction cannot exploitatively control cache prefetch predictions occurring after the instruction is complete and synchronized.",
                "html": "<p>Cache Prefetch Prediction Restriction by Context prevents cache allocation predictions that predict execution addresses based on information gathered from earlier execution within a particular execution context. The actions of code in the target execution context or contexts appearing in program order before the instruction cannot exploitatively control cache prefetch predictions occurring after the instruction is complete and synchronized.</p><p>For more information, see <xref linkend=\"AArch64.cpp_rctx\">CPP RCTX, Cache Prefetch Prediction Restriction by Context</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPY":
            return {
                "tooltip": "Copy a signed integer immediate to each active element in the destination vector. Inactive elements in the destination vector register are set to zero.",
                "html": "<p>Copy a signed integer immediate to each active element in the destination vector. Inactive elements in the destination vector register are set to zero.</p><p>The immediate operand is a signed value in the range -128 to +127, and for element widths of 16 bits or higher it may also be a signed multiple of 256 in the range -32768 to +32512 (excluding 0).</p><p>The immediate is encoded in 8 bits with an optional left shift by 8. The preferred disassembly when the shift option is specified is \"<syntax>#&lt;simm8&gt;, LSL #8</syntax>\". However an assembler and disassembler may also allow use of the shifted 16-bit value unless the immediate is 0 and the shift amount is 8, which must be unambiguously described as \"<syntax>#0, LSL #8</syntax>\".</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFE":
        case "CPYFM":
        case "CPYFP":
            return {
                "tooltip": "Memory Copy Forward-only. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFP, then CPYFM, and then CPYFE.",
                "html": "<p>Memory Copy Forward-only. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFP, then CPYFM, and then CPYFE.</p><p>CPYFP performs some preconditioning of the arguments suitable for using the CPYFM instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFM performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFE performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFEN":
        case "CPYFMN":
        case "CPYFPN":
            return {
                "tooltip": "Memory Copy Forward-only, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPN, then CPYFMN, and then CPYFEN.",
                "html": "<p>Memory Copy Forward-only, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPN, then CPYFMN, and then CPYFEN.</p><p>CPYFPN performs some preconditioning of the arguments suitable for using the CPYFMN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFEN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFERN":
        case "CPYFMRN":
        case "CPYFPRN":
            return {
                "tooltip": "Memory Copy Forward-only, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPRN, then CPYFMRN, and then CPYFERN.",
                "html": "<p>Memory Copy Forward-only, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPRN, then CPYFMRN, and then CPYFERN.</p><p>CPYFPRN performs some preconditioning of the arguments suitable for using the CPYFMRN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMRN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFERN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFERT":
        case "CPYFMRT":
        case "CPYFPRT":
            return {
                "tooltip": "Memory Copy Forward-only, reads unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPRT, then CPYFMRT, and then CPYFERT.",
                "html": "<p>Memory Copy Forward-only, reads unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPRT, then CPYFMRT, and then CPYFERT.</p><p>CPYFPRT performs some preconditioning of the arguments suitable for using the CPYFMRT instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMRT performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFERT performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFERTN":
        case "CPYFMRTN":
        case "CPYFPRTN":
            return {
                "tooltip": "Memory Copy Forward-only, reads unprivileged, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPRTN, then CPYFMRTN, and then CPYFERTN.",
                "html": "<p>Memory Copy Forward-only, reads unprivileged, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPRTN, then CPYFMRTN, and then CPYFERTN.</p><p>CPYFPRTN performs some preconditioning of the arguments suitable for using the CPYFMRTN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMRTN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFERTN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFERTRN":
        case "CPYFMRTRN":
        case "CPYFPRTRN":
            return {
                "tooltip": "Memory Copy Forward-only, reads unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPRTRN, then CPYFMRTRN, and then CPYFERTRN.",
                "html": "<p>Memory Copy Forward-only, reads unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPRTRN, then CPYFMRTRN, and then CPYFERTRN.</p><p>CPYFPRTRN performs some preconditioning of the arguments suitable for using the CPYFMRTRN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMRTRN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFERTRN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFERTWN":
        case "CPYFMRTWN":
        case "CPYFPRTWN":
            return {
                "tooltip": "Memory Copy Forward-only, reads unprivileged, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPRTWN, then CPYFMRTWN, and then CPYFERTWN.",
                "html": "<p>Memory Copy Forward-only, reads unprivileged, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPRTWN, then CPYFMRTWN, and then CPYFERTWN.</p><p>CPYFPRTWN performs some preconditioning of the arguments suitable for using the CPYFMRTWN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMRTWN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFERTWN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFET":
        case "CPYFMT":
        case "CPYFPT":
            return {
                "tooltip": "Memory Copy Forward-only, reads and writes unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPT, then CPYFMT, and then CPYFET.",
                "html": "<p>Memory Copy Forward-only, reads and writes unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPT, then CPYFMT, and then CPYFET.</p><p>CPYFPT performs some preconditioning of the arguments suitable for using the CPYFMT instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMT performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFET performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFETN":
        case "CPYFMTN":
        case "CPYFPTN":
            return {
                "tooltip": "Memory Copy Forward-only, reads and writes unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPTN, then CPYFMTN, and then CPYFETN.",
                "html": "<p>Memory Copy Forward-only, reads and writes unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPTN, then CPYFMTN, and then CPYFETN.</p><p>CPYFPTN performs some preconditioning of the arguments suitable for using the CPYFMTN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMTN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFETN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFETRN":
        case "CPYFMTRN":
        case "CPYFPTRN":
            return {
                "tooltip": "Memory Copy Forward-only, reads and writes unprivileged, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPTRN, then CPYFMTRN, and then CPYFETRN.",
                "html": "<p>Memory Copy Forward-only, reads and writes unprivileged, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPTRN, then CPYFMTRN, and then CPYFETRN.</p><p>CPYFPTRN performs some preconditioning of the arguments suitable for using the CPYFMTRN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMTRN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFETRN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFETWN":
        case "CPYFMTWN":
        case "CPYFPTWN":
            return {
                "tooltip": "Memory Copy Forward-only, reads and writes unprivileged, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPTWN, then CPYFMTWN, and then CPYFETWN.",
                "html": "<p>Memory Copy Forward-only, reads and writes unprivileged, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPTWN, then CPYFMTWN, and then CPYFETWN.</p><p>CPYFPTWN performs some preconditioning of the arguments suitable for using the CPYFMTWN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMTWN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFETWN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFEWN":
        case "CPYFMWN":
        case "CPYFPWN":
            return {
                "tooltip": "Memory Copy Forward-only, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPWN, then CPYFMWN, and then CPYFEWN.",
                "html": "<p>Memory Copy Forward-only, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPWN, then CPYFMWN, and then CPYFEWN.</p><p>CPYFPWN performs some preconditioning of the arguments suitable for using the CPYFMWN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMWN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFEWN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFEWT":
        case "CPYFMWT":
        case "CPYFPWT":
            return {
                "tooltip": "Memory Copy Forward-only, writes unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPWT, then CPYFMWT, and then CPYFEWT.",
                "html": "<p>Memory Copy Forward-only, writes unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPWT, then CPYFMWT, and then CPYFEWT.</p><p>CPYFPWT performs some preconditioning of the arguments suitable for using the CPYFMWT instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMWT performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFEWT performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFEWTN":
        case "CPYFMWTN":
        case "CPYFPWTN":
            return {
                "tooltip": "Memory Copy Forward-only, writes unprivileged, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPWTN, then CPYFMWTN, and then CPYFEWTN.",
                "html": "<p>Memory Copy Forward-only, writes unprivileged, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPWTN, then CPYFMWTN, and then CPYFEWTN.</p><p>CPYFPWTN performs some preconditioning of the arguments suitable for using the CPYFMWTN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMWTN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFEWTN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFEWTRN":
        case "CPYFMWTRN":
        case "CPYFPWTRN":
            return {
                "tooltip": "Memory Copy Forward-only, writes unprivileged, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPWTRN, then CPYFMWTRN, and then CPYFEWTRN.",
                "html": "<p>Memory Copy Forward-only, writes unprivileged, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPWTRN, then CPYFMWTRN, and then CPYFEWTRN.</p><p>CPYFPWTRN performs some preconditioning of the arguments suitable for using the CPYFMWTRN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMWTRN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFEWTRN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYFEWTWN":
        case "CPYFMWTWN":
        case "CPYFPWTWN":
            return {
                "tooltip": "Memory Copy Forward-only, writes unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPWTWN, then CPYFMWTWN, and then CPYFEWTWN.",
                "html": "<p>Memory Copy Forward-only, writes unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYFPWTWN, then CPYFMWTWN, and then CPYFEWTWN.</p><p>CPYFPWTWN performs some preconditioning of the arguments suitable for using the CPYFMWTWN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFMWTWN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYFEWTWN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>The memory copy performed by these instructions is in the forward direction only, so the instructions are suitable for a memory copy only where there is no overlap between the source and destination locations, or where the source address is greater than the destination address.</p><p>The architecture supports two algorithms for the memory copy: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYE":
        case "CPYM":
        case "CPYP":
            return {
                "tooltip": "Memory Copy. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYP, then CPYM, and then CPYE.",
                "html": "<p>Memory Copy. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYP, then CPYM, and then CPYE.</p><p>CPYP performs some preconditioning of the arguments suitable for using the CPYM instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYM performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYE performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYP, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYEN":
        case "CPYMN":
        case "CPYPN":
            return {
                "tooltip": "Memory Copy, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPN, then CPYMN, and then CPYEN.",
                "html": "<p>Memory Copy, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPN, then CPYMN, and then CPYEN.</p><p>CPYPN performs some preconditioning of the arguments suitable for using the CPYMN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYEN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYERN":
        case "CPYMRN":
        case "CPYPRN":
            return {
                "tooltip": "Memory Copy, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPRN, then CPYMRN, and then CPYERN.",
                "html": "<p>Memory Copy, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPRN, then CPYMRN, and then CPYERN.</p><p>CPYPRN performs some preconditioning of the arguments suitable for using the CPYMRN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMRN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYERN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPRN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYERT":
        case "CPYMRT":
        case "CPYPRT":
            return {
                "tooltip": "Memory Copy, reads unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPRT, then CPYMRT, and then CPYERT.",
                "html": "<p>Memory Copy, reads unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPRT, then CPYMRT, and then CPYERT.</p><p>CPYPRT performs some preconditioning of the arguments suitable for using the CPYMRT instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMRT performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYERT performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPRT, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYERTN":
        case "CPYMRTN":
        case "CPYPRTN":
            return {
                "tooltip": "Memory Copy, reads unprivileged, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPRTN, then CPYMRTN, and then CPYERTN.",
                "html": "<p>Memory Copy, reads unprivileged, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPRTN, then CPYMRTN, and then CPYERTN.</p><p>CPYPRTN performs some preconditioning of the arguments suitable for using the CPYMRTN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMRTN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYERTN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPRTN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYERTRN":
        case "CPYMRTRN":
        case "CPYPRTRN":
            return {
                "tooltip": "Memory Copy, reads unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPRTRN, then CPYMRTRN, and then CPYERTRN.",
                "html": "<p>Memory Copy, reads unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPRTRN, then CPYMRTRN, and then CPYERTRN.</p><p>CPYPRTRN performs some preconditioning of the arguments suitable for using the CPYMRTRN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMRTRN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYERTRN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPRTRN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYERTWN":
        case "CPYMRTWN":
        case "CPYPRTWN":
            return {
                "tooltip": "Memory Copy, reads unprivileged, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPRTWN, then CPYMRTWN, and then CPYERTWN.",
                "html": "<p>Memory Copy, reads unprivileged, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPRTWN, then CPYMRTWN, and then CPYERTWN.</p><p>CPYPRTWN performs some preconditioning of the arguments suitable for using the CPYMRTWN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMRTWN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYERTWN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPRTWN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYET":
        case "CPYMT":
        case "CPYPT":
            return {
                "tooltip": "Memory Copy, reads and writes unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPT, then CPYMT, and then CPYET.",
                "html": "<p>Memory Copy, reads and writes unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPT, then CPYMT, and then CPYET.</p><p>CPYPT performs some preconditioning of the arguments suitable for using the CPYMT instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMT performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYET performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPT, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYETN":
        case "CPYMTN":
        case "CPYPTN":
            return {
                "tooltip": "Memory Copy, reads and writes unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPTN, then CPYMTN, and then CPYETN.",
                "html": "<p>Memory Copy, reads and writes unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPTN, then CPYMTN, and then CPYETN.</p><p>CPYPTN performs some preconditioning of the arguments suitable for using the CPYMTN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMTN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYETN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPTN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYETRN":
        case "CPYMTRN":
        case "CPYPTRN":
            return {
                "tooltip": "Memory Copy, reads and writes unprivileged, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPTRN, then CPYMTRN, and then CPYETRN.",
                "html": "<p>Memory Copy, reads and writes unprivileged, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPTRN, then CPYMTRN, and then CPYETRN.</p><p>CPYPTRN performs some preconditioning of the arguments suitable for using the CPYMTRN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMTRN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYETRN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPTRN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYETWN":
        case "CPYMTWN":
        case "CPYPTWN":
            return {
                "tooltip": "Memory Copy, reads and writes unprivileged, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPTWN, then CPYMTWN, and then CPYETWN.",
                "html": "<p>Memory Copy, reads and writes unprivileged, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPTWN, then CPYMTWN, and then CPYETWN.</p><p>CPYPTWN performs some preconditioning of the arguments suitable for using the CPYMTWN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMTWN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYETWN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPTWN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYEWN":
        case "CPYMWN":
        case "CPYPWN":
            return {
                "tooltip": "Memory Copy, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPWN, then CPYMWN, and then CPYEWN.",
                "html": "<p>Memory Copy, writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPWN, then CPYMWN, and then CPYEWN.</p><p>CPYPWN performs some preconditioning of the arguments suitable for using the CPYMWN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMWN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYEWN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPWN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYEWT":
        case "CPYMWT":
        case "CPYPWT":
            return {
                "tooltip": "Memory Copy, writes unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPWT, then CPYMWT, and then CPYEWT.",
                "html": "<p>Memory Copy, writes unprivileged. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPWT, then CPYMWT, and then CPYEWT.</p><p>CPYPWT performs some preconditioning of the arguments suitable for using the CPYMWT instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMWT performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYEWT performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPWT, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYEWTN":
        case "CPYMWTN":
        case "CPYPWTN":
            return {
                "tooltip": "Memory Copy, writes unprivileged, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPWTN, then CPYMWTN, and then CPYEWTN.",
                "html": "<p>Memory Copy, writes unprivileged, reads and writes non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPWTN, then CPYMWTN, and then CPYEWTN.</p><p>CPYPWTN performs some preconditioning of the arguments suitable for using the CPYMWTN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMWTN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYEWTN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPWTN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYEWTRN":
        case "CPYMWTRN":
        case "CPYPWTRN":
            return {
                "tooltip": "Memory Copy, writes unprivileged, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPWTRN, then CPYMWTRN, and then CPYEWTRN.",
                "html": "<p>Memory Copy, writes unprivileged, reads non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPWTRN, then CPYMWTRN, and then CPYEWTRN.</p><p>CPYPWTRN performs some preconditioning of the arguments suitable for using the CPYMWTRN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMWTRN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYEWTRN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPWTRN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CPYEWTWN":
        case "CPYMWTWN":
        case "CPYPWTWN":
            return {
                "tooltip": "Memory Copy, writes unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPWTWN, then CPYMWTWN, and then CPYEWTWN.",
                "html": "<p>Memory Copy, writes unprivileged and non-temporal. These instructions perform a memory copy. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: CPYPWTWN, then CPYMWTWN, and then CPYEWTWN.</p><p>CPYPWTWN performs some preconditioning of the arguments suitable for using the CPYMWTWN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYMWTWN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory copy. CPYEWTWN performs the last part of the memory copy.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory copy allows some optimization of the size that can be performed.</p><p>For CPYPWTWN, the following saturation logic is applied:</p><p>If Xn&lt;63:55&gt; != 000000000, the copy size Xn is saturated to <hexnumber>0x007FFFFFFFFFFFFF</hexnumber>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CRC32B":
        case "CRC32H":
        case "CRC32W":
        case "CRC32X":
            return {
                "tooltip": "CRC32 checksum performs a cyclic redundancy check (CRC) calculation on a value held in a general-purpose register. It takes an input CRC value in the first source operand, performs a CRC on the input value in the second source operand, and returns the output CRC value. The second source operand can be 8, 16, 32, or 64 bits. To align with common usage, the bit order of the values is reversed as part of the operation, and the polynomial 0x04C11DB7 is used for the CRC calculation.",
                "html": "<p><instruction>CRC32</instruction> checksum performs a cyclic redundancy check (CRC) calculation on a value held in a general-purpose register. It takes an input CRC value in the first source operand, performs a CRC on the input value in the second source operand, and returns the output CRC value. The second source operand can be 8, 16, 32, or 64 bits. To align with common usage, the bit order of the values is reversed as part of the operation, and the polynomial <hexnumber>0x04C11DB7</hexnumber> is used for the CRC calculation.</p><p>In an Armv8.0 implementation, this is an <arm-defined-word>optional</arm-defined-word> instruction. From Armv8.1, it is mandatory for all implementations to implement this instruction.</p><p><xref linkend=\"AArch64.id_aa64isar0_el1\">ID_AA64ISAR0_EL1</xref>.CRC32 indicates whether this instruction is supported.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CRC32CB":
        case "CRC32CH":
        case "CRC32CW":
        case "CRC32CX":
            return {
                "tooltip": "CRC32 checksum performs a cyclic redundancy check (CRC) calculation on a value held in a general-purpose register. It takes an input CRC value in the first source operand, performs a CRC on the input value in the second source operand, and returns the output CRC value. The second source operand can be 8, 16, 32, or 64 bits. To align with common usage, the bit order of the values is reversed as part of the operation, and the polynomial 0x1EDC6F41 is used for the CRC calculation.",
                "html": "<p><instruction>CRC32</instruction> checksum performs a cyclic redundancy check (CRC) calculation on a value held in a general-purpose register. It takes an input CRC value in the first source operand, performs a CRC on the input value in the second source operand, and returns the output CRC value. The second source operand can be 8, 16, 32, or 64 bits. To align with common usage, the bit order of the values is reversed as part of the operation, and the polynomial <hexnumber>0x1EDC6F41</hexnumber> is used for the CRC calculation.</p><p>In an Armv8.0 implementation, this is an <arm-defined-word>optional</arm-defined-word> instruction. From Armv8.1, it is mandatory for all implementations to implement this instruction.</p><p><xref linkend=\"AArch64.id_aa64isar0_el1\">ID_AA64ISAR0_EL1</xref>.CRC32 indicates whether this instruction is supported.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CSDB":
            return {
                "tooltip": "Consumption of Speculative Data Barrier is a memory barrier that controls speculative execution and data value prediction.",
                "html": "<p>Consumption of Speculative Data Barrier is a memory barrier that controls speculative execution and data value prediction.</p><p>No instruction other than branch instructions appearing in program order after the CSDB can be speculatively executed using the results of any:</p><p>For purposes of the definition of CSDB, PSTATE.{N,Z,C,V} is not considered a data value. This definition permits:</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CSEL":
            return {
                "tooltip": "If the condition is true, Conditional Select writes the value of the first source register to the destination register. If the condition is false, it writes the value of the second source register to the destination register.",
                "html": "<p>If the condition is true, Conditional Select writes the value of the first source register to the destination register. If the condition is false, it writes the value of the second source register to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CSET":
            return {
                "tooltip": "Conditional Set sets the destination register to 1 if the condition is TRUE, and otherwise sets it to 0.",
                "html": "<p>Conditional Set sets the destination register to 1 if the condition is TRUE, and otherwise sets it to 0.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CSETM":
            return {
                "tooltip": "Conditional Set Mask sets all bits of the destination register to 1 if the condition is TRUE, and otherwise sets all bits to 0.",
                "html": "<p>Conditional Set Mask sets all bits of the destination register to 1 if the condition is TRUE, and otherwise sets all bits to 0.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CSINC":
            return {
                "tooltip": "Conditional Select Increment returns, in the destination register, the value of the first source register if the condition is TRUE, and otherwise returns the value of the second source register incremented by 1.",
                "html": "<p>Conditional Select Increment returns, in the destination register, the value of the first source register if the condition is TRUE, and otherwise returns the value of the second source register incremented by 1.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CSINV":
            return {
                "tooltip": "Conditional Select Invert returns, in the destination register, the value of the first source register if the condition is TRUE, and otherwise returns the bitwise inversion value of the second source register.",
                "html": "<p>Conditional Select Invert returns, in the destination register, the value of the first source register if the condition is TRUE, and otherwise returns the bitwise inversion value of the second source register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CSNEG":
            return {
                "tooltip": "Conditional Select Negation returns, in the destination register, the value of the first source register if the condition is TRUE, and otherwise returns the negated value of the second source register.",
                "html": "<p>Conditional Select Negation returns, in the destination register, the value of the first source register if the condition is TRUE, and otherwise returns the negated value of the second source register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CTERMEQ":
        case "CTERMNE":
            return {
                "tooltip": "Detect termination conditions in serialized vector loops. Tests whether the comparison between the scalar source operands holds true and if not tests the state of the !Last condition flag (C) which indicates whether the previous flag-setting predicate instruction selected the last element of the vector partition.",
                "html": "<p>Detect termination conditions in serialized vector loops. Tests whether the comparison between the scalar source operands holds true and if not tests the state of the <arm-defined-word>!Last</arm-defined-word> condition flag (C) which indicates whether the previous flag-setting predicate instruction selected the last element of the vector partition.</p><p>The Z and C condition flags are preserved by this instruction. The N and V condition flags are set as a pair to generate one of the following conditions for a subsequent conditional instruction:</p><p></p><p>The scalar source operands are 32-bit or 64-bit general-purpose registers of the same size.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "CTZ":
            return {
                "tooltip": "Count Trailing Zeros counts the number of consecutive binary zero bits, starting from the least significant bit in the source register, and places the count in the destination register.",
                "html": "<p>Count Trailing Zeros counts the number of consecutive binary zero bits, starting from the least significant bit in the source register, and places the count in the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DC":
            return {
                "tooltip": "Data Cache operation. For more information, see op0==0b01, cache maintenance, TLB maintenance, and address translation instructions.",
                "html": "<p>Data Cache operation. For more information, see <xref linkend=\"BABEJJJE\">op0==0b01, cache maintenance, TLB maintenance, and address translation instructions</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DCPS1":
            return {
                "tooltip": "Debug Change PE State to EL1, when executed in Debug state",
                "html": "<p>Debug Change PE State to EL1, when executed in Debug state:</p><p>The target exception level of a DCPS1 instruction is:</p><p>When the target Exception level of a DCPS1 instruction is ELx, on executing this instruction:</p><p>This instruction is <arm-defined-word>undefined</arm-defined-word> at EL0 in Non-secure state if EL2 is implemented and <xref linkend=\"AArch64.hcr_el2\">HCR_EL2</xref>.TGE == 1.</p><p>This instruction is always <arm-defined-word>undefined</arm-defined-word> in Non-debug state.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DCPS2":
            return {
                "tooltip": "Debug Change PE State to EL2, when executed in Debug state",
                "html": "<p>Debug Change PE State to EL2, when executed in Debug state:</p><p>The target exception level of a DCPS2 instruction is:</p><p>When the target Exception level of a DCPS2 instruction is ELx, on executing this instruction:</p><p>This instruction is <arm-defined-word>undefined</arm-defined-word> at the following exception levels:</p><p>This instruction is always <arm-defined-word>undefined</arm-defined-word> in Non-debug state.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DCPS3":
            return {
                "tooltip": "Debug Change PE State to EL3, when executed in Debug state",
                "html": "<p>Debug Change PE State to EL3, when executed in Debug state:</p><p>The target exception level of a DCPS3 instruction is EL3.</p><p>On executing a DCPS3 instruction:</p><p>This instruction is <arm-defined-word>undefined</arm-defined-word> at all exception levels if either:</p><p>This instruction is always <arm-defined-word>undefined</arm-defined-word> in Non-debug state.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DECB":
        case "DECD":
        case "DECH":
        case "DECW":
            return {
                "tooltip": "Determines the number of active elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to decrement the scalar destination.",
                "html": "<p>Determines the number of active elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to decrement the scalar destination.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DECP":
            return {
                "tooltip": "Counts the number of true elements in the source predicate and then uses the result to decrement the scalar destination.",
                "html": "<p>Counts the number of true elements in the source predicate and then uses the result to decrement the scalar destination.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DGH":
            return {
                "tooltip": "Data Gathering Hint is a hint instruction that indicates that it is not expected to be performance optimal to merge memory accesses with Normal Non-cacheable or Device-GRE attributes appearing in program order before the hint instruction with any memory accesses appearing after the hint instruction into a single memory transaction on an interconnect.",
                "html": "<p>Data Gathering Hint is a hint instruction that indicates that it is not expected to be performance optimal to merge memory accesses with Normal Non-cacheable or Device-GRE attributes appearing in program order before the hint instruction with any memory accesses appearing after the hint instruction into a single memory transaction on an interconnect.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DMB":
            return {
                "tooltip": "Data Memory Barrier is a memory barrier that ensures the ordering of observations of memory accesses, see Data Memory Barrier.",
                "html": "<p>Data Memory Barrier is a memory barrier that ensures the ordering of observations of memory accesses, see <xref linkend=\"BEIIECBH\">Data Memory Barrier</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DSB":
            return {
                "tooltip": "Data Synchronization Barrier is a memory barrier that ensures the completion of memory accesses, see Data Synchronization Barrier.",
                "html": "<p>Data Synchronization Barrier is a memory barrier that ensures the completion of memory accesses, see <xref linkend=\"BEICEFJH\">Data Synchronization Barrier</xref>.</p><p>A DSB instruction with the nXS qualifier is complete when the subset of these memory accesses with the XS attribute set to 0 are complete. It does not require that memory accesses with the XS attribute set to 1 are complete.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DUP":
            return {
                "tooltip": "Duplicate vector element to vector or scalar. This instruction duplicates the vector element at the specified element index in the source SIMD&FP register into a scalar or each element in a vector, and writes the result to the destination SIMD&FP register.",
                "html": "<p>Duplicate vector element to vector or scalar. This instruction duplicates the vector element at the specified element index in the source SIMD&amp;FP register into a scalar or each element in a vector, and writes the result to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DUPM":
            return {
                "tooltip": "Unconditionally broadcast the logical bitmask immediate into each element of the destination vector. This instruction is unpredicated. The immediate is a 64-bit value consisting of a single run of ones or zeros repeating every 2, 4, 8, 16, 32 or 64 bits.",
                "html": "<p>Unconditionally broadcast the logical bitmask immediate into each element of the destination vector. This instruction is unpredicated. The immediate is a 64-bit value consisting of a single run of ones or zeros repeating every 2, 4, 8, 16, 32 or 64 bits.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DUPQ":
            return {
                "tooltip": "Unconditionally broadcast the indexed element within each 128-bit source vector segment to all elements of the corresponding destination vector segment. This instruction is unpredicated.",
                "html": "<p>Unconditionally broadcast the indexed element within each 128-bit source vector segment to all elements of the corresponding destination vector segment. This instruction is unpredicated.</p><p>The immediate element index is in the range of 0 to 15 (bytes), 7 (halfwords), 3 (words) or 1 (doublewords).</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "DVP":
            return {
                "tooltip": "Data Value Prediction Restriction by Context prevents data value predictions that predict execution addresses based on information gathered from earlier execution within a particular execution context. Data value predictions determined by the actions of code in the target execution context or contexts appearing in program order before the instruction cannot be used to exploitatively control speculative execution occurring after the instruction is complete and synchronized.",
                "html": "<p>Data Value Prediction Restriction by Context prevents data value predictions that predict execution addresses based on information gathered from earlier execution within a particular execution context. Data value predictions determined by the actions of code in the target execution context or contexts appearing in program order before the instruction cannot be used to exploitatively control speculative execution occurring after the instruction is complete and synchronized.</p><p>For more information, see <xref linkend=\"AArch64.dvp_rctx\">DVP RCTX, Data Value Prediction Restriction by Context</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "EON":
            return {
                "tooltip": "Bitwise Exclusive-OR NOT (shifted register) performs a bitwise exclusive-OR NOT of a register value and an optionally-shifted register value, and writes the result to the destination register.",
                "html": "<p>Bitwise Exclusive-OR NOT (shifted register) performs a bitwise exclusive-OR NOT of a register value and an optionally-shifted register value, and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "EOR3":
            return {
                "tooltip": "Three-way Exclusive-OR performs a three-way exclusive-OR of the values in the three source SIMD&FP registers, and writes the result to the destination SIMD&FP register.",
                "html": "<p>Three-way Exclusive-OR performs a three-way exclusive-OR of the values in the three source SIMD&amp;FP registers, and writes the result to the destination SIMD&amp;FP register.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SHA3\">FEAT_SHA3</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "EOR":
            return {
                "tooltip": "Bitwise Exclusive-OR (vector). This instruction performs a bitwise exclusive-OR operation between the two source SIMD&FP registers, and places the result in the destination SIMD&FP register.",
                "html": "<p>Bitwise Exclusive-OR (vector). This instruction performs a bitwise exclusive-OR operation between the two source SIMD&amp;FP registers, and places the result in the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "EORBT":
            return {
                "tooltip": "Interleaving exclusive OR between the even-numbered elements of the first source vector register and the odd-numbered elements of the second source vector register, placing the result in the even-numbered elements of the destination vector, leaving the odd-numbered elements unchanged. This instruction is unpredicated.",
                "html": "<p>Interleaving exclusive OR between the even-numbered elements of the first source vector register and the odd-numbered elements of the second source vector register, placing the result in the even-numbered elements of the destination vector, leaving the odd-numbered elements unchanged. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "EORQV":
            return {
                "tooltip": "Bitwise exclusive OR of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&FP destination register. Inactive elements in the source vector are treated as all zeros.",
                "html": "<p>Bitwise exclusive OR of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&amp;FP destination register. Inactive elements in the source vector are treated as all zeros.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "EORS":
            return {
                "tooltip": "Bitwise exclusive OR active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Bitwise exclusive OR active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "EORTB":
            return {
                "tooltip": "Interleaving exclusive OR between the odd-numbered elements of the first source vector register and the even-numbered elements of the second source vector register, placing the result in the odd-numbered elements of the destination vector, leaving the even-numbered elements unchanged. This instruction is unpredicated.",
                "html": "<p>Interleaving exclusive OR between the odd-numbered elements of the first source vector register and the even-numbered elements of the second source vector register, placing the result in the odd-numbered elements of the destination vector, leaving the even-numbered elements unchanged. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "EORV":
            return {
                "tooltip": "Bitwise exclusive OR horizontally across all lanes of a vector, and place the result in the SIMD&FP scalar destination register. Inactive elements in the source vector are treated as zero.",
                "html": "<p>Bitwise exclusive OR horizontally across all lanes of a vector, and place the result in the SIMD&amp;FP scalar destination register. Inactive elements in the source vector are treated as zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ERET":
            return {
                "tooltip": "Exception Return using the ELR and SPSR for the current Exception level. When executed, the PE restores PSTATE from the SPSR, and branches to the address held in the ELR.",
                "html": "<p>Exception Return using the ELR and SPSR for the current Exception level. When executed, the PE restores <xref linkend=\"BEIDIGBH\">PSTATE</xref> from the SPSR, and branches to the address held in the ELR.</p><p>The PE checks the SPSR for the current Exception level for an illegal return event. See <xref linkend=\"BEIEGDFD\">Illegal return events from AArch64 state</xref>.</p><p><instruction>ERET</instruction> is <arm-defined-word>undefined</arm-defined-word> at EL0.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ERETAA":
        case "ERETAB":
            return {
                "tooltip": "Exception Return, with pointer authentication. This instruction authenticates the address in ELR, using SP as the modifier and the specified key, the PE restores PSTATE from the SPSR for the current Exception level, and branches to the authenticated address.",
                "html": "<p>Exception Return, with pointer authentication. This instruction authenticates the address in ELR, using SP as the modifier and the specified key, the PE restores <xref linkend=\"BEIDIGBH\">PSTATE</xref> from the SPSR for the current Exception level, and branches to the authenticated address.</p><p>Key A is used for <instruction>ERETAA</instruction>. Key B is used for <instruction>ERETAB</instruction>.</p><p>If the authentication passes, the PE continues execution at the target of the branch. For information on behavior if the authentication fails, see <xref>Faulting on pointer authentication</xref>.</p><p>The authenticated address is not written back to ELR.</p><p>The PE checks the SPSR for the current Exception level for an illegal return event. See <xref linkend=\"BEIEGDFD\">Illegal return events from AArch64 state</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ESB":
            return {
                "tooltip": "Error Synchronization Barrier is an error synchronization event that might also update DISR_EL1 and VDISR_EL2.",
                "html": "<p>Error Synchronization Barrier is an error synchronization event that might also update DISR_EL1 and VDISR_EL2.</p><p>This instruction can be used at all Exception levels and in Debug state.</p><p>In Debug state, this instruction behaves as if SError interrupts are masked at all Exception levels. See Error Synchronization Barrier in the Arm(R) Reliability, Availability, and Serviceability (RAS) Specification, Armv8, for Armv8-A architecture profile.</p><p>If the RAS Extension is not implemented, this instruction executes as a <instruction>NOP</instruction>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "EXT":
            return {
                "tooltip": "Extract vector from pair of vectors. This instruction extracts the lowest vector elements from the second source SIMD&FP register and the highest vector elements from the first source SIMD&FP register, concatenates the results into a vector, and writes the vector to the destination SIMD&FP register vector. The index value specifies the lowest vector element to extract from the first source register, and consecutive elements are extracted from the first, then second, source registers until the destination vector is filled.",
                "html": "<p>Extract vector from pair of vectors. This instruction extracts the lowest vector elements from the second source SIMD&amp;FP register and the highest vector elements from the first source SIMD&amp;FP register, concatenates the results into a vector, and writes the vector to the destination SIMD&amp;FP register vector. The index value specifies the lowest vector element to extract from the first source register, and consecutive elements are extracted from the first, then second, source registers until the destination vector is filled.</p><p></p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "EXTQ":
            return {
                "tooltip": "For each 128-bit vector segment of the result, copy the indexed byte up to and including the last byte of the corresponding first source vector segment to the bottom of the result segment, then fill the remainder of the result segment starting from the first byte of the corresponding second source vector segment. The result segments are destructively placed in the corresponding first source vector segment. This instruction is unpredicated.",
                "html": "<p>For each 128-bit vector segment of the result, copy the indexed byte up to and including the last byte of the corresponding first source vector segment to the bottom of the result segment, then fill the remainder of the result segment starting from the first byte of the corresponding second source vector segment. The result segments are destructively placed in the corresponding first source vector segment. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "EXTR":
            return {
                "tooltip": "Extract register extracts a register from a pair of registers.",
                "html": "<p>Extract register extracts a register from a pair of registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FABD":
            return {
                "tooltip": "Floating-point Absolute Difference (vector). This instruction subtracts the floating-point values in the elements of the second source SIMD&FP register, from the corresponding floating-point values in the elements of the first source SIMD&FP register, places the absolute value of each result in a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Absolute Difference (vector). This instruction subtracts the floating-point values in the elements of the second source SIMD&amp;FP register, from the corresponding floating-point values in the elements of the first source SIMD&amp;FP register, places the absolute value of each result in a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FABS":
            return {
                "tooltip": "Floating-point Absolute value (vector). This instruction calculates the absolute value of each vector element in the source SIMD&FP register, writes the result to a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Absolute value (vector). This instruction calculates the absolute value of each vector element in the source SIMD&amp;FP register, writes the result to a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FACGE":
            return {
                "tooltip": "Floating-point Absolute Compare Greater than or Equal (vector). This instruction compares the absolute value of each floating-point value in the first source SIMD&FP register with the absolute value of the corresponding floating-point value in the second source SIMD&FP register and if the first value is greater than or equal to the second value sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Floating-point Absolute Compare Greater than or Equal (vector). This instruction compares the absolute value of each floating-point value in the first source SIMD&amp;FP register with the absolute value of the corresponding floating-point value in the second source SIMD&amp;FP register and if the first value is greater than or equal to the second value sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FACGT":
            return {
                "tooltip": "Floating-point Absolute Compare Greater than (vector). This instruction compares the absolute value of each vector element in the first source SIMD&FP register with the absolute value of the corresponding vector element in the second source SIMD&FP register and if the first value is greater than the second value sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Floating-point Absolute Compare Greater than (vector). This instruction compares the absolute value of each vector element in the first source SIMD&amp;FP register with the absolute value of the corresponding vector element in the second source SIMD&amp;FP register and if the first value is greater than the second value sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FACLE":
            return {
                "tooltip": "Compare active absolute values of floating-point elements in the first source vector being less than or equal to corresponding absolute values of elements in the second source vector, and place the boolean results of the comparison in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.",
                "html": "<p>Compare active absolute values of floating-point elements in the first source vector being less than or equal to corresponding absolute values of elements in the second source vector, and place the boolean results of the comparison in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FACLT":
            return {
                "tooltip": "Compare active absolute values of floating-point elements in the first source vector being less than corresponding absolute values of elements in the second source vector, and place the boolean results of the comparison in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.",
                "html": "<p>Compare active absolute values of floating-point elements in the first source vector being less than corresponding absolute values of elements in the second source vector, and place the boolean results of the comparison in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FADD":
            return {
                "tooltip": "Floating-point Add (vector). This instruction adds corresponding vector elements in the two source SIMD&FP registers, writes the result into a vector, and writes the vector to the destination SIMD&FP register. All the values in this instruction are floating-point values.",
                "html": "<p>Floating-point Add (vector). This instruction adds corresponding vector elements in the two source SIMD&amp;FP registers, writes the result into a vector, and writes the vector to the destination SIMD&amp;FP register. All the values in this instruction are floating-point values.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FADDA":
            return {
                "tooltip": "Floating-point add a SIMD&FP scalar source and all active lanes of the vector source and place the result destructively in the SIMD&FP scalar source register. Vector elements are processed strictly in order from low to high, with the scalar source providing the initial value. Inactive elements in the source vector are ignored.",
                "html": "<p>Floating-point add a SIMD&amp;FP scalar source and all active lanes of the vector source and place the result destructively in the SIMD&amp;FP scalar source register. Vector elements are processed strictly in order from low to high, with the scalar source providing the initial value. Inactive elements in the source vector are ignored.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FADDP":
            return {
                "tooltip": "Floating-point Add Pair of elements (scalar). This instruction adds two floating-point vector elements in the source SIMD&FP register and writes the scalar result into the destination SIMD&FP register.",
                "html": "<p>Floating-point Add Pair of elements (scalar). This instruction adds two floating-point vector elements in the source SIMD&amp;FP register and writes the scalar result into the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FADDQV":
            return {
                "tooltip": "Floating-point addition of the same element numbers from each 128-bit source vector segment using a recursive pairwise reduction, placing each result into the corresponding element number of the 128-bit SIMD&FP destination register. Inactive elements in the source vector are treated as +0.0.",
                "html": "<p>Floating-point addition of the same element numbers from each 128-bit source vector segment using a recursive pairwise reduction, placing each result into the corresponding element number of the 128-bit SIMD&amp;FP destination register. Inactive elements in the source vector are treated as +0.0.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FADDV":
            return {
                "tooltip": "Floating-point add horizontally over all lanes of a vector using a recursive pairwise reduction, and place the result in the SIMD&FP scalar destination register. Inactive elements in the source vector are treated as +0.0.",
                "html": "<p>Floating-point add horizontally over all lanes of a vector using a recursive pairwise reduction, and place the result in the SIMD&amp;FP scalar destination register. Inactive elements in the source vector are treated as +0.0.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCADD":
            return {
                "tooltip": "Floating-point Complex Add.",
                "html": "<p>Floating-point Complex Add.</p><p>This instruction operates on complex numbers that are represented in SIMD&amp;FP registers as pairs of elements, with the more significant element holding the imaginary part of the number and the less significant element holding the real part of the number. Each element holds a floating-point value. It performs the following computation on the corresponding complex number element pairs from the two source registers:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCCMP":
            return {
                "tooltip": "Floating-point Conditional quiet Compare (scalar). This instruction compares the two SIMD&FP source register values and writes the result to the PSTATE.{N, Z, C, V} flags. If the condition does not pass then the PSTATE.{N, Z, C, V} flags are set to the flag bit specifier.",
                "html": "<p>Floating-point Conditional quiet Compare (scalar). This instruction compares the two SIMD&amp;FP source register values and writes the result to the <xref linkend=\"BEIDIGBH\">PSTATE</xref>.{N, Z, C, V} flags. If the condition does not pass then the <xref linkend=\"BEIDIGBH\">PSTATE</xref>.{N, Z, C, V} flags are set to the flag bit specifier.</p><p>This instruction raises an Invalid Operation floating-point exception if either or both of the operands is a signaling NaN.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCCMPE":
            return {
                "tooltip": "Floating-point Conditional signaling Compare (scalar). This instruction compares the two SIMD&FP source register values and writes the result to the PSTATE.{N, Z, C, V} flags. If the condition does not pass then the PSTATE.{N, Z, C, V} flags are set to the flag bit specifier.",
                "html": "<p>Floating-point Conditional signaling Compare (scalar). This instruction compares the two SIMD&amp;FP source register values and writes the result to the <xref linkend=\"BEIDIGBH\">PSTATE</xref>.{N, Z, C, V} flags. If the condition does not pass then the <xref linkend=\"BEIDIGBH\">PSTATE</xref>.{N, Z, C, V} flags are set to the flag bit specifier.</p><p>This instruction raises an Invalid Operation floating-point exception if either or both of the operands is any type of NaN.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCLAMP":
            return {
                "tooltip": "Clamp each floating-point element in the two or four destination vectors to between the floating-point minimum value in the corresponding element of the first source vector and the floating-point maximum value in the corresponding element of the second source vector and destructively place the clamped results in the corresponding elements of the two or four destination vectors.",
                "html": "<p>Clamp each floating-point element in the two or four destination vectors to between the floating-point minimum value in the corresponding element of the first source vector and the floating-point maximum value in the corresponding element of the second source vector and destructively place the clamped results in the corresponding elements of the two or four destination vectors.</p><p>Regardless of the value of FPCR.AH, the behavior is as follows for each mininum number and maximum number operation:</p><p>This instruction follows SME2 floating-point numerical behaviors corresponding to instructions that place their results in one or more SVE Z vectors.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCMEQ":
            return {
                "tooltip": "Floating-point Compare Equal (vector). This instruction compares each floating-point value from the first source SIMD&FP register, with the corresponding floating-point value from the second source SIMD&FP register, and if the comparison is equal sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Floating-point Compare Equal (vector). This instruction compares each floating-point value from the first source SIMD&amp;FP register, with the corresponding floating-point value from the second source SIMD&amp;FP register, and if the comparison is equal sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCMGE":
            return {
                "tooltip": "Floating-point Compare Greater than or Equal (vector). This instruction reads each floating-point value in the first source SIMD&FP register and if the value is greater than or equal to the corresponding floating-point value in the second source SIMD&FP register sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Floating-point Compare Greater than or Equal (vector). This instruction reads each floating-point value in the first source SIMD&amp;FP register and if the value is greater than or equal to the corresponding floating-point value in the second source SIMD&amp;FP register sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCMGT":
            return {
                "tooltip": "Floating-point Compare Greater than (vector). This instruction reads each floating-point value in the first source SIMD&FP register and if the value is greater than the corresponding floating-point value in the second source SIMD&FP register sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Floating-point Compare Greater than (vector). This instruction reads each floating-point value in the first source SIMD&amp;FP register and if the value is greater than the corresponding floating-point value in the second source SIMD&amp;FP register sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCMLA":
            return {
                "tooltip": "Floating-point Complex Multiply Accumulate (by element).",
                "html": "<p>Floating-point Complex Multiply Accumulate (by element).</p><p>This instruction operates on complex numbers that are represented in SIMD&amp;FP registers as pairs of elements, with the more significant element holding the imaginary part of the number and the less significant element holding the real part of the number. Each element holds a floating-point value. It performs the following computation on complex numbers from the first source register and the destination register with the specified complex number from the second source register:</p><p>The multiplication and addition operations are performed as a fused multiply-add, without any intermediate rounding.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCMLE":
            return {
                "tooltip": "Floating-point Compare Less than or Equal to zero (vector). This instruction reads each floating-point value in the source SIMD&FP register and if the value is less than or equal to zero sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Floating-point Compare Less than or Equal to zero (vector). This instruction reads each floating-point value in the source SIMD&amp;FP register and if the value is less than or equal to zero sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCMLT":
            return {
                "tooltip": "Floating-point Compare Less than zero (vector). This instruction reads each floating-point value in the source SIMD&FP register and if the value is less than zero sets every bit of the corresponding vector element in the destination SIMD&FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&FP register to zero.",
                "html": "<p>Floating-point Compare Less than zero (vector). This instruction reads each floating-point value in the source SIMD&amp;FP register and if the value is less than zero sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to one, otherwise sets every bit of the corresponding vector element in the destination SIMD&amp;FP register to zero.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCMP":
            return {
                "tooltip": "Floating-point quiet Compare (scalar). This instruction compares the two SIMD&FP source register values, or the first SIMD&FP source register value and zero. It writes the result to the PSTATE.{N, Z, C, V} flags.",
                "html": "<p>Floating-point quiet Compare (scalar). This instruction compares the two SIMD&amp;FP source register values, or the first SIMD&amp;FP source register value and zero. It writes the result to the <xref linkend=\"BEIDIGBH\">PSTATE</xref>.{N, Z, C, V} flags.</p><p>This instruction raises an Invalid Operation floating-point exception if either or both of the operands is a signaling NaN.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCMPE":
            return {
                "tooltip": "Floating-point signaling Compare (scalar). This instruction compares the two SIMD&FP source register values, or the first SIMD&FP source register value and zero. It writes the result to the PSTATE.{N, Z, C, V} flags.",
                "html": "<p>Floating-point signaling Compare (scalar). This instruction compares the two SIMD&amp;FP source register values, or the first SIMD&amp;FP source register value and zero. It writes the result to the <xref linkend=\"BEIDIGBH\">PSTATE</xref>.{N, Z, C, V} flags.</p><p>This instruction raises an Invalid Operation floating-point exception if either or both of the operands is any type of NaN.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCPY":
            return {
                "tooltip": "Copy a floating-point immediate into each active element in the destination vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Copy a floating-point immediate into each active element in the destination vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCSEL":
            return {
                "tooltip": "Floating-point Conditional Select (scalar). This instruction allows the SIMD&FP destination register to take the value from either one or the other of two SIMD&FP source registers. If the condition passes, the first SIMD&FP source register value is taken, otherwise the second SIMD&FP source register value is taken.",
                "html": "<p>Floating-point Conditional Select (scalar). This instruction allows the SIMD&amp;FP destination register to take the value from either one or the other of two SIMD&amp;FP source registers. If the condition passes, the first SIMD&amp;FP source register value is taken, otherwise the second SIMD&amp;FP source register value is taken.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVT":
            return {
                "tooltip": "Floating-point Convert precision (scalar). This instruction converts the floating-point value in the SIMD&FP source register to the precision for the destination register data type using the rounding mode that is determined by the FPCR and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Convert precision (scalar). This instruction converts the floating-point value in the SIMD&amp;FP source register to the precision for the destination register data type using the rounding mode that is determined by the <xref linkend=\"AArch64.fpcr\">FPCR</xref> and writes the result to the SIMD&amp;FP destination register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTAS":
            return {
                "tooltip": "Floating-point Convert to Signed integer, rounding to nearest with ties to Away (vector). This instruction converts each element in a vector from a floating-point value to a signed integer value using the Round to Nearest with Ties to Away rounding mode and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Convert to Signed integer, rounding to nearest with ties to Away (vector). This instruction converts each element in a vector from a floating-point value to a signed integer value using the Round to Nearest with Ties to Away rounding mode and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTAU":
            return {
                "tooltip": "Floating-point Convert to Unsigned integer, rounding to nearest with ties to Away (vector). This instruction converts each element in a vector from a floating-point value to an unsigned integer value using the Round to Nearest with Ties to Away rounding mode and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Convert to Unsigned integer, rounding to nearest with ties to Away (vector). This instruction converts each element in a vector from a floating-point value to an unsigned integer value using the Round to Nearest with Ties to Away rounding mode and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTL":
        case "FCVTL2":
            return {
                "tooltip": "Floating-point Convert to higher precision Long (vector). This instruction reads each element in a vector in the SIMD&FP source register, converts each value to double the precision of the source element using the rounding mode that is determined by the FPCR, and writes each result to the equivalent element of the vector in the SIMD&FP destination register.",
                "html": "<p>Floating-point Convert to higher precision Long (vector). This instruction reads each element in a vector in the SIMD&amp;FP source register, converts each value to double the precision of the source element using the rounding mode that is determined by the <xref linkend=\"AArch64.fpcr\">FPCR</xref>, and writes each result to the equivalent element of the vector in the SIMD&amp;FP destination register.</p><p>Where the operation lengthens a 64-bit vector to a 128-bit vector, the <instruction>FCVTL2</instruction> variant operates on the elements in the top 64 bits of the source register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTLT":
            return {
                "tooltip": "Convert odd-numbered floating-point elements from the source vector to the next higher precision, and place the results in the active overlapping double-width elements of the destination vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Convert odd-numbered floating-point elements from the source vector to the next higher precision, and place the results in the active overlapping double-width elements of the destination vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTMS":
            return {
                "tooltip": "Floating-point Convert to Signed integer, rounding toward Minus infinity (vector). This instruction converts a scalar or each element in a vector from a floating-point value to a signed integer value using the Round towards Minus Infinity rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Convert to Signed integer, rounding toward Minus infinity (vector). This instruction converts a scalar or each element in a vector from a floating-point value to a signed integer value using the Round towards Minus Infinity rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the Security state and Exception level in which the instruction is executed, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTMU":
            return {
                "tooltip": "Floating-point Convert to Unsigned integer, rounding toward Minus infinity (vector). This instruction converts a scalar or each element in a vector from a floating-point value to an unsigned integer value using the Round towards Minus Infinity rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Convert to Unsigned integer, rounding toward Minus infinity (vector). This instruction converts a scalar or each element in a vector from a floating-point value to an unsigned integer value using the Round towards Minus Infinity rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the Security state and Exception level in which the instruction is executed, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTN":
        case "FCVTN2":
            return {
                "tooltip": "Floating-point Convert to lower precision Narrow (vector). This instruction reads each vector element in the SIMD&FP source register, converts each result to half the precision of the source element, writes the final result to a vector, and writes the vector to the lower or upper half of the destination SIMD&FP register. The destination vector elements are half as long as the source vector elements. The rounding mode is determined by the FPCR.",
                "html": "<p>Floating-point Convert to lower precision Narrow (vector). This instruction reads each vector element in the SIMD&amp;FP source register, converts each result to half the precision of the source element, writes the final result to a vector, and writes the vector to the lower or upper half of the destination SIMD&amp;FP register. The destination vector elements are half as long as the source vector elements. The rounding mode is determined by the <xref linkend=\"AArch64.fpcr\">FPCR</xref>.</p><p>The <instruction>FCVTN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>FCVTN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the Security state and Exception level in which the instruction is executed, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTNS":
            return {
                "tooltip": "Floating-point Convert to Signed integer, rounding to nearest with ties to even (vector). This instruction converts a scalar or each element in a vector from a floating-point value to a signed integer value using the Round to Nearest rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Convert to Signed integer, rounding to nearest with ties to even (vector). This instruction converts a scalar or each element in a vector from a floating-point value to a signed integer value using the Round to Nearest rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the Security state and Exception level in which the instruction is executed, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTNT":
            return {
                "tooltip": "Convert active floating-point elements from the source vector to the next lower precision, and place the results in the odd-numbered half-width elements of the destination vector, leaving the even-numbered elements unchanged. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Convert active floating-point elements from the source vector to the next lower precision, and place the results in the odd-numbered half-width elements of the destination vector, leaving the even-numbered elements unchanged. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTNU":
            return {
                "tooltip": "Floating-point Convert to Unsigned integer, rounding to nearest with ties to even (vector). This instruction converts a scalar or each element in a vector from a floating-point value to an unsigned integer value using the Round to Nearest rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Convert to Unsigned integer, rounding to nearest with ties to even (vector). This instruction converts a scalar or each element in a vector from a floating-point value to an unsigned integer value using the Round to Nearest rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the Security state and Exception level in which the instruction is executed, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTPS":
            return {
                "tooltip": "Floating-point Convert to Signed integer, rounding toward Plus infinity (vector). This instruction converts a scalar or each element in a vector from a floating-point value to a signed integer value using the Round towards Plus Infinity rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Convert to Signed integer, rounding toward Plus infinity (vector). This instruction converts a scalar or each element in a vector from a floating-point value to a signed integer value using the Round towards Plus Infinity rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the Security state and Exception level in which the instruction is executed, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTPU":
            return {
                "tooltip": "Floating-point Convert to Unsigned integer, rounding toward Plus infinity (vector). This instruction converts a scalar or each element in a vector from a floating-point value to an unsigned integer value using the Round towards Plus Infinity rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Convert to Unsigned integer, rounding toward Plus infinity (vector). This instruction converts a scalar or each element in a vector from a floating-point value to an unsigned integer value using the Round towards Plus Infinity rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the Security state and Exception level in which the instruction is executed, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTX":
            return {
                "tooltip": "Convert active double-precision floating-point elements from the source vector to single-precision, rounding to Odd, and place the results in the even-numbered 32-bit elements of the destination vector, while setting the odd-numbered elements to zero. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Convert active double-precision floating-point elements from the source vector to single-precision, rounding to Odd, and place the results in the even-numbered 32-bit elements of the destination vector, while setting the odd-numbered elements to zero. Inactive elements in the destination vector register remain unmodified.</p><p>Rounding to Odd (aka Von Neumann rounding) permits a two-step conversion from double-precision to half-precision without incurring intermediate rounding errors.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTXN":
        case "FCVTXN2":
            return {
                "tooltip": "Floating-point Convert to lower precision Narrow, rounding to odd (vector). This instruction reads each vector element in the source SIMD&FP register, narrows each value to half the precision of the source element using the Round to Odd rounding mode, writes the result to a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Convert to lower precision Narrow, rounding to odd (vector). This instruction reads each vector element in the source SIMD&amp;FP register, narrows each value to half the precision of the source element using the Round to Odd rounding mode, writes the result to a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>This instruction uses the Round to Odd rounding mode which is not defined by the IEEE 754-2008 standard. This rounding mode ensures that if the result of the conversion is inexact the least significant bit of the mantissa is forced to 1. This rounding mode enables a floating-point value to be converted to a lower precision format via an intermediate precision format while avoiding double rounding errors. For example, a 64-bit floating-point value can be converted to a correctly rounded 16-bit floating-point value by first using this instruction to produce a 32-bit value and then using another instruction with the wanted rounding mode to convert the 32-bit value to the final 16-bit floating-point value.</p><p>The <instruction>FCVTXN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>FCVTXN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTXNT":
            return {
                "tooltip": "Convert active double-precision floating-point elements from the source vector to single-precision, rounding to Odd, and place the results in the odd-numbered 32-bit elements of the destination vector, leaving the even-numbered elements unchanged. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Convert active double-precision floating-point elements from the source vector to single-precision, rounding to Odd, and place the results in the odd-numbered 32-bit elements of the destination vector, leaving the even-numbered elements unchanged. Inactive elements in the destination vector register remain unmodified.</p><p>Rounding to Odd (aka Von Neumann rounding) permits a two-step conversion from double-precision to half-precision without incurring intermediate rounding errors.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTZS":
            return {
                "tooltip": "Floating-point Convert to Signed fixed-point, rounding toward Zero (vector). This instruction converts a scalar or each element in a vector from floating-point to fixed-point signed integer using the Round towards Zero rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Convert to Signed fixed-point, rounding toward Zero (vector). This instruction converts a scalar or each element in a vector from floating-point to fixed-point signed integer using the Round towards Zero rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the Security state and Exception level in which the instruction is executed, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FCVTZU":
            return {
                "tooltip": "Floating-point Convert to Unsigned fixed-point, rounding toward Zero (vector). This instruction converts a scalar or each element in a vector from floating-point to fixed-point unsigned integer using the Round towards Zero rounding mode, and writes the result to the general-purpose destination register.",
                "html": "<p>Floating-point Convert to Unsigned fixed-point, rounding toward Zero (vector). This instruction converts a scalar or each element in a vector from floating-point to fixed-point unsigned integer using the Round towards Zero rounding mode, and writes the result to the general-purpose destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the Security state and Exception level in which the instruction is executed, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FDIV":
            return {
                "tooltip": "Floating-point Divide (vector). This instruction divides the floating-point values in the elements in the first source SIMD&FP register, by the floating-point values in the corresponding elements in the second source SIMD&FP register, places the results in a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Divide (vector). This instruction divides the floating-point values in the elements in the first source SIMD&amp;FP register, by the floating-point values in the corresponding elements in the second source SIMD&amp;FP register, places the results in a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FDIVR":
            return {
                "tooltip": "Reversed divide active floating-point elements of the second source vector by corresponding floating-point elements of the first source vector and destructively place the quotient in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Reversed divide active floating-point elements of the second source vector by corresponding floating-point elements of the first source vector and destructively place the quotient in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FDOT":
            return {
                "tooltip": "This instruction computes the fused sum-of-products of a pair of half-precision floating-point values held in each 32-bit element of the first source and second source vectors, without intermediate rounding, and then destructively adds the single-precision sum-of-products to the corresponding single-precision element of the destination vector.",
                "html": "<p>This instruction computes the fused sum-of-products of a pair of half-precision floating-point values held in each 32-bit element of the first source and second source vectors, without intermediate rounding, and then destructively adds the single-precision sum-of-products to the corresponding single-precision element of the destination vector.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FDUP":
            return {
                "tooltip": "Unconditionally broadcast the floating-point immediate into each element of the destination vector. This instruction is unpredicated.",
                "html": "<p>Unconditionally broadcast the floating-point immediate into each element of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FEXPA":
            return {
                "tooltip": "The FEXPA instruction accelerates the polynomial series calculation of the exp(x) function.",
                "html": "<p>The <instruction>FEXPA</instruction> instruction accelerates the polynomial series calculation of the <arm-defined-word>exp(x)</arm-defined-word> function.</p><p>The double-precision variant copies the low 52 bits of an entry from a hard-wired table of 64-bit coefficients, indexed by the low 6 bits of each element of the source vector, and prepends to that the next 11 bits of the source element (src&lt;16:6&gt;), setting the sign bit to zero.</p><p>The single-precision variant copies the low 23 bits of an entry from hard-wired table of 32-bit coefficients, indexed by the low 6 bits of each element of the source vector, and prepends to that the next 8 bits of the source element (src&lt;13:6&gt;), setting the sign bit to zero.</p><p>The half-precision variant copies the low 10 bits of an entry from hard-wired table of 16-bit coefficients, indexed by the low 5 bits of each element of the source vector, and prepends to that the next 5 bits of the source element (src&lt;9:5&gt;), setting the sign bit to zero.</p><p>A coefficient table entry with index <arm-defined-word>m</arm-defined-word> holds the floating-point value 2<sup>(m/64)</sup>, or for the half-precision variant 2<sup>(m/32)</sup>. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FJCVTZS":
            return {
                "tooltip": "Floating-point Javascript Convert to Signed fixed-point, rounding toward Zero. This instruction converts the double-precision floating-point value in the SIMD&FP source register to a 32-bit signed integer using the Round towards Zero rounding mode, and writes the result to the general-purpose destination register. If the result is too large to be represented as a signed 32-bit integer, then the result is the integer modulo 232, as held in a 32-bit signed integer.",
                "html": "<p>Floating-point Javascript Convert to Signed fixed-point, rounding toward Zero. This instruction converts the double-precision floating-point value in the SIMD&amp;FP source register to a 32-bit signed integer using the Round towards Zero rounding mode, and writes the result to the general-purpose destination register. If the result is too large to be represented as a signed 32-bit integer, then the result is the integer modulo 2<sup>32</sup>, as held in a 32-bit signed integer.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FLOGB":
            return {
                "tooltip": "This instruction returns the signed integer base 2 logarithm of each floating-point input element |x| after normalization.",
                "html": "<p>This instruction returns the signed integer base 2 logarithm of each floating-point input element |<arm-defined-word>x</arm-defined-word>| after normalization.</p><p>This is the unbiased exponent of <arm-defined-word>x</arm-defined-word> used in the representation of the floating-point value, such that, for positive <arm-defined-word>x</arm-defined-word>, <arm-defined-word>x</arm-defined-word> = significand \u00d7 2<sup>exponent</sup>.</p><p>The integer results are placed in elements of the destination vector which have the same width (<arm-defined-word>esize</arm-defined-word>) as the floating-point input elements:</p><p>Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMAD":
            return {
                "tooltip": "Multiply the corresponding active floating-point elements of the first and second source vectors and add to elements of the third (addend) vector without intermediate rounding. Destructively place the results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply the corresponding active floating-point elements of the first and second source vectors and add to elements of the third (addend) vector without intermediate rounding. Destructively place the results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMADD":
            return {
                "tooltip": "Floating-point fused Multiply-Add (scalar). This instruction multiplies the values of the first two SIMD&FP source registers, adds the product to the value of the third SIMD&FP source register, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point fused Multiply-Add (scalar). This instruction multiplies the values of the first two SIMD&amp;FP source registers, adds the product to the value of the third SIMD&amp;FP source register, and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMAX":
            return {
                "tooltip": "Floating-point Maximum (vector). This instruction compares corresponding vector elements in the two source SIMD&FP registers, places the larger of each of the two floating-point values into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Maximum (vector). This instruction compares corresponding vector elements in the two source SIMD&amp;FP registers, places the larger of each of the two floating-point values into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 0, the behavior is as follows:</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 1, the behavior is as follows:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMAXNM":
            return {
                "tooltip": "Floating-point Maximum Number (vector). This instruction compares corresponding vector elements in the two source SIMD&FP registers, writes the larger of the two floating-point values into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Maximum Number (vector). This instruction compares corresponding vector elements in the two source SIMD&amp;FP registers, writes the larger of the two floating-point values into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Regardless of the value of <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH, the behavior is as follows:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMAXNMP":
            return {
                "tooltip": "Floating-point Maximum Number of Pair of elements (scalar). This instruction compares two vector elements in the source SIMD&FP register and writes the largest of the floating-point values as a scalar to the destination SIMD&FP register.",
                "html": "<p>Floating-point Maximum Number of Pair of elements (scalar). This instruction compares two vector elements in the source SIMD&amp;FP register and writes the largest of the floating-point values as a scalar to the destination SIMD&amp;FP register.</p><p>Regardless of the value of <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH, the behavior is as follows for each pairwise operation:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMAXNMQV":
            return {
                "tooltip": "Floating-point maximum number of the same element numbers from each 128-bit source vector segment using a recursive pairwise reduction, placing each result into the corresponding element number of the 128-bit SIMD&FP destination register. Inactive elements in the source vector are treated as the default NaN.",
                "html": "<p>Floating-point maximum number of the same element numbers from each 128-bit source vector segment using a recursive pairwise reduction, placing each result into the corresponding element number of the 128-bit SIMD&amp;FP destination register. Inactive elements in the source vector are treated as the default NaN.</p><p>Regardless of the value of FPCR.AH, the behavior is as follows:</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMAXNMV":
            return {
                "tooltip": "Floating-point Maximum Number across Vector. This instruction compares all the vector elements in the source SIMD&FP register, and writes the largest of the values as a scalar to the destination SIMD&FP register. All the values in this instruction are floating-point values.",
                "html": "<p>Floating-point Maximum Number across Vector. This instruction compares all the vector elements in the source SIMD&amp;FP register, and writes the largest of the values as a scalar to the destination SIMD&amp;FP register. All the values in this instruction are floating-point values.</p><p>Regardless of the value of <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH, the behavior is as follows:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMAXP":
            return {
                "tooltip": "Floating-point Maximum of Pair of elements (scalar). This instruction compares two vector elements in the source SIMD&FP register and writes the largest of the floating-point values as a scalar to the destination SIMD&FP register.",
                "html": "<p>Floating-point Maximum of Pair of elements (scalar). This instruction compares two vector elements in the source SIMD&amp;FP register and writes the largest of the floating-point values as a scalar to the destination SIMD&amp;FP register.</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 0, the behavior is as follows for each pairwise operation:</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 1, the behavior is as follows for each pairwise operation:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMAXQV":
            return {
                "tooltip": "Floating-point maximum of the same element numbers from each 128-bit source vector segment using a recursive pairwise reduction, placing each result into the corresponding element number of the 128-bit SIMD&FP destination register. Inactive elements in the source vector are treated as -Infinity.",
                "html": "<p>Floating-point maximum of the same element numbers from each 128-bit source vector segment using a recursive pairwise reduction, placing each result into the corresponding element number of the 128-bit SIMD&amp;FP destination register. Inactive elements in the source vector are treated as -Infinity.</p><p>When FPCR.AH is 0, the behavior is as follows:</p><p>When FPCR.AH is 1, the behavior is as follows:</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMAXV":
            return {
                "tooltip": "Floating-point Maximum across Vector. This instruction compares all the vector elements in the source SIMD&FP register, and writes the largest of the values as a scalar to the destination SIMD&FP register. All the values in this instruction are floating-point values.",
                "html": "<p>Floating-point Maximum across Vector. This instruction compares all the vector elements in the source SIMD&amp;FP register, and writes the largest of the values as a scalar to the destination SIMD&amp;FP register. All the values in this instruction are floating-point values.</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 0, the behavior is as follows:</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 1, the behavior is as follows:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMIN":
            return {
                "tooltip": "Floating-point minimum (vector). This instruction compares corresponding elements in the vectors in the two source SIMD&FP registers, places the smaller of each of the two floating-point values into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point minimum (vector). This instruction compares corresponding elements in the vectors in the two source SIMD&amp;FP registers, places the smaller of each of the two floating-point values into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 0, the behavior is as follows:</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 1, the behavior is as follows:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMINNM":
            return {
                "tooltip": "Floating-point Minimum Number (vector). This instruction compares corresponding vector elements in the two source SIMD&FP registers, writes the smaller of the two floating-point values into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Minimum Number (vector). This instruction compares corresponding vector elements in the two source SIMD&amp;FP registers, writes the smaller of the two floating-point values into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Regardless of the value of <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH, the behavior is as follows:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMINNMP":
            return {
                "tooltip": "Floating-point Minimum Number of Pair of elements (scalar). This instruction compares two vector elements in the source SIMD&FP register and writes the smallest of the floating-point values as a scalar to the destination SIMD&FP register.",
                "html": "<p>Floating-point Minimum Number of Pair of elements (scalar). This instruction compares two vector elements in the source SIMD&amp;FP register and writes the smallest of the floating-point values as a scalar to the destination SIMD&amp;FP register.</p><p>Regardless of the value of <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH, the behavior is as follows for each pairwise operation:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMINNMQV":
            return {
                "tooltip": "Floating-point minimum number of the same element numbers from each 128-bit source vector segment using a recursive pairwise reduction, placing each result into the corresponding element number of the 128-bit SIMD&FP destination register. Inactive elements in the source vector are treated as the default NaN.",
                "html": "<p>Floating-point minimum number of the same element numbers from each 128-bit source vector segment using a recursive pairwise reduction, placing each result into the corresponding element number of the 128-bit SIMD&amp;FP destination register. Inactive elements in the source vector are treated as the default NaN.</p><p>Regardless of the value of FPCR.AH, the behavior is as follows:</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMINNMV":
            return {
                "tooltip": "Floating-point Minimum Number across Vector. This instruction compares all the vector elements in the source SIMD&FP register, and writes the smallest of the values as a scalar to the destination SIMD&FP register. All the values in this instruction are floating-point values.",
                "html": "<p>Floating-point Minimum Number across Vector. This instruction compares all the vector elements in the source SIMD&amp;FP register, and writes the smallest of the values as a scalar to the destination SIMD&amp;FP register. All the values in this instruction are floating-point values.</p><p>Regardless of the value of <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH, the behavior is as follows:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMINP":
            return {
                "tooltip": "Floating-point Minimum of Pair of elements (scalar). This instruction compares two vector elements in the source SIMD&FP register and writes the smallest of the floating-point values as a scalar to the destination SIMD&FP register.",
                "html": "<p>Floating-point Minimum of Pair of elements (scalar). This instruction compares two vector elements in the source SIMD&amp;FP register and writes the smallest of the floating-point values as a scalar to the destination SIMD&amp;FP register.</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 0, the behavior is as follows for each pairwise operation:</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 1, the behavior is as follows for each pairwise operation:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMINQV":
            return {
                "tooltip": "Floating-point minimum of the same element numbers from each 128-bit source vector segment using a recursive pairwise reduction, placing each result into the corresponding element number of the 128-bit SIMD&FP destination register. Inactive elements in the source vector are treated as +Infinity.",
                "html": "<p>Floating-point minimum of the same element numbers from each 128-bit source vector segment using a recursive pairwise reduction, placing each result into the corresponding element number of the 128-bit SIMD&amp;FP destination register. Inactive elements in the source vector are treated as +Infinity.</p><p>When FPCR.AH is 0, the behavior is as follows:</p><p>When FPCR.AH is 1, the behavior is as follows:</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMINV":
            return {
                "tooltip": "Floating-point Minimum across Vector. This instruction compares all the vector elements in the source SIMD&FP register, and writes the smallest of the values as a scalar to the destination SIMD&FP register. All the values in this instruction are floating-point values.",
                "html": "<p>Floating-point Minimum across Vector. This instruction compares all the vector elements in the source SIMD&amp;FP register, and writes the smallest of the values as a scalar to the destination SIMD&amp;FP register. All the values in this instruction are floating-point values.</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 0, the behavior is as follows:</p><p>When <xref linkend=\"AArch64.fpcr\">FPCR</xref>.AH is 1, the behavior is as follows:</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMLA":
            return {
                "tooltip": "Floating-point fused Multiply-Add to accumulator (by element). This instruction multiplies the vector elements in the first source SIMD&FP register by the specified value in the second source SIMD&FP register, and accumulates the results in the vector elements of the destination SIMD&FP register. All the values in this instruction are floating-point values.",
                "html": "<p>Floating-point fused Multiply-Add to accumulator (by element). This instruction multiplies the vector elements in the first source SIMD&amp;FP register by the specified value in the second source SIMD&amp;FP register, and accumulates the results in the vector elements of the destination SIMD&amp;FP register. All the values in this instruction are floating-point values.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMLAL":
        case "FMLAL2":
            return {
                "tooltip": "Floating-point fused Multiply-Add Long to accumulator (by element). This instruction multiplies the vector elements in the first source SIMD&FP register by the specified value in the second source SIMD&FP register, and accumulates the product to the corresponding vector element of the destination SIMD&FP register. The instruction does not round the result of the multiply before the accumulation.",
                "html": "<p>Floating-point fused Multiply-Add Long to accumulator (by element). This instruction multiplies the vector elements in the first source SIMD&amp;FP register by the specified value in the second source SIMD&amp;FP register, and accumulates the product to the corresponding vector element of the destination SIMD&amp;FP register. The instruction does not round the result of the multiply before the accumulation.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p><p>In Armv8.2 and Armv8.3, this is an <arm-defined-word>optional</arm-defined-word> instruction. From Armv8.4 it is mandatory for all implementations to support it.</p><p><xref linkend=\"AArch64.id_aa64isar0_el1\">ID_AA64ISAR0_EL1</xref>.FHM indicates whether this instruction is supported.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMLALB":
            return {
                "tooltip": "This half-precision floating-point multiply-add long instruction widens the even-numbered half-precision elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and adds these values without intermediate rounding to the single-precision elements of the destination vector that overlap with the corresponding half-precision elements in the source vectors. This instruction is unpredicated.",
                "html": "<p>This half-precision floating-point multiply-add long instruction widens the even-numbered half-precision elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and adds these values without intermediate rounding to the single-precision elements of the destination vector that overlap with the corresponding half-precision elements in the source vectors. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMLALT":
            return {
                "tooltip": "This half-precision floating-point multiply-add long instruction widens the odd-numbered half-precision elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and adds these values without intermediate rounding to the single-precision elements of the destination vector that overlap with the corresponding half-precision elements in the source vectors. This instruction is unpredicated.",
                "html": "<p>This half-precision floating-point multiply-add long instruction widens the odd-numbered half-precision elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and adds these values without intermediate rounding to the single-precision elements of the destination vector that overlap with the corresponding half-precision elements in the source vectors. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMLS":
            return {
                "tooltip": "Floating-point fused Multiply-Subtract from accumulator (by element). This instruction multiplies the vector elements in the first source SIMD&FP register by the specified value in the second source SIMD&FP register, and subtracts the results from the vector elements of the destination SIMD&FP register. All the values in this instruction are floating-point values.",
                "html": "<p>Floating-point fused Multiply-Subtract from accumulator (by element). This instruction multiplies the vector elements in the first source SIMD&amp;FP register by the specified value in the second source SIMD&amp;FP register, and subtracts the results from the vector elements of the destination SIMD&amp;FP register. All the values in this instruction are floating-point values.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMLSL":
        case "FMLSL2":
            return {
                "tooltip": "Floating-point fused Multiply-Subtract Long from accumulator (by element). This instruction multiplies the negated vector elements in the first source SIMD&FP register by the specified value in the second source SIMD&FP register, and accumulates the product to the corresponding vector element of the destination SIMD&FP register. The instruction does not round the result of the multiply before the accumulation.",
                "html": "<p>Floating-point fused Multiply-Subtract Long from accumulator (by element). This instruction multiplies the negated vector elements in the first source SIMD&amp;FP register by the specified value in the second source SIMD&amp;FP register, and accumulates the product to the corresponding vector element of the destination SIMD&amp;FP register. The instruction does not round the result of the multiply before the accumulation.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p><p>In Armv8.2 and Armv8.3, this is an <arm-defined-word>optional</arm-defined-word> instruction. From Armv8.4 it is mandatory for all implementations to support it.</p><p><xref linkend=\"AArch64.id_aa64isar0_el1\">ID_AA64ISAR0_EL1</xref>.FHM indicates whether this instruction is supported.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMLSLB":
            return {
                "tooltip": "This half-precision floating-point multiply-subtract long instruction widens the even-numbered half-precision elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and subtracts these values without intermediate rounding from the single-precision elements of the destination vector that overlap with the corresponding half-precision elements in the source vectors. This instruction is unpredicated.",
                "html": "<p>This half-precision floating-point multiply-subtract long instruction widens the even-numbered half-precision elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and subtracts these values without intermediate rounding from the single-precision elements of the destination vector that overlap with the corresponding half-precision elements in the source vectors. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMLSLT":
            return {
                "tooltip": "This half-precision floating-point multiply-subtract long instruction widens the odd-numbered half-precision elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and subtracts these values without intermediate rounding from the single-precision elements of the destination vector that overlap with the corresponding half-precision elements in the source vectors. This instruction is unpredicated.",
                "html": "<p>This half-precision floating-point multiply-subtract long instruction widens the odd-numbered half-precision elements in the first source vector and the corresponding elements in the second source vector to single-precision format and then destructively multiplies and subtracts these values without intermediate rounding from the single-precision elements of the destination vector that overlap with the corresponding half-precision elements in the source vectors. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMMLA":
            return {
                "tooltip": "The floating-point matrix multiply-accumulate instruction supports single-precision and double-precision data types in a 2\u00d72 matrix contained in segments of 128 or 256 bits, respectively. It multiplies the 2\u00d72 matrix in each segment of the first source vector by the 2\u00d72 matrix in the corresponding segment of the second source vector. The resulting 2\u00d72 matrix product is then destructively added to the matrix accumulator held in the corresponding segment of the addend and destination vector. This is equivalent to performing a 2-way dot product per destination element. This instruction is unpredicated. The single-precision variant is vector length agnostic. The double-precision variant requires that the current vector length is at least 256 bits, and if the current vector length is not an integer multiple of 256 bits then the trailing bits are set to zero.",
                "html": "<p>The floating-point matrix multiply-accumulate instruction supports single-precision and double-precision data types in a 2\u00d72 matrix contained in segments of 128 or 256 bits, respectively. It multiplies the 2\u00d72 matrix in each segment of the first source vector by the 2\u00d72 matrix in the corresponding segment of the second source vector. The resulting 2\u00d72 matrix product is then destructively added to the matrix accumulator held in the corresponding segment of the addend and destination vector. This is equivalent to performing a 2-way dot product per destination element. This instruction is unpredicated. The single-precision variant is vector length agnostic. The double-precision variant requires that the current vector length is at least 256 bits, and if the current vector length is not an integer multiple of 256 bits then the trailing bits are set to zero.</p><p>ID_AA64ZFR0_EL1.F32MM indicates whether the single-precision variant is implemented.</p><p>ID_AA64ZFR0_EL1.F64MM indicates whether the double-precision variant is implemented.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMOPA":
            return {
                "tooltip": "The half-precision floating-point sum of outer products and accumulate instruction works with a 32-bit element ZA tile.",
                "html": "<p>The half-precision floating-point sum of outer products and accumulate instruction works with a 32-bit element ZA tile.</p><p>This instruction widens the SVL<sub>S</sub>\u00d72 sub-matrix of half-precision floating-point values held in the first source vector to single-precision floating-point values and multiplies it by the widened 2\u00d7SVL<sub>S</sub> sub-matrix of half-precision floating-point values in the second source vector to single-precision floating-point values.</p><p>Each source vector is independently predicated by a corresponding governing predicate. When a 16-bit source element is Inactive it is treated as having the value +0.0, but if both pairs of source vector elements that correspond to a 32-bit destination element contain Inactive elements, then the destination element remains unmodified.</p><p>The resulting SVL<sub>S</sub>\u00d7SVL<sub>S</sub> single-precision floating-point sum of outer products is then destructively added to the single-precision floating-point destination tile. This is equivalent to performing a 2-way dot product and accumulate to each of the destination tile elements.</p><p>Each 32-bit container of the first source vector holds 2 consecutive column elements of each row of a SVL<sub>S</sub>\u00d72 sub-matrix. Similarly, each 32-bit container of the second source vector holds 2 consecutive row elements of each column of a 2\u00d7SVL<sub>S</sub> sub-matrix.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMOPS":
            return {
                "tooltip": "The half-precision floating-point sum of outer products and subtract instruction works with a 32-bit element ZA tile.",
                "html": "<p>The half-precision floating-point sum of outer products and subtract instruction works with a 32-bit element ZA tile.</p><p>This instruction widens the SVL<sub>S</sub>\u00d72 sub-matrix of half-precision floating-point values held in the first source vector to single-precision floating-point values and multiplies it by the widened 2\u00d7SVL<sub>S</sub> sub-matrix of half-precision floating-point values in the second source vector to single-precision floating-point values.</p><p>Each source vector is independently predicated by a corresponding governing predicate. When a 16-bit source element is Inactive it is treated as having the value +0.0, but if both pairs of source vector elements that correspond to a 32-bit destination element contain Inactive elements, then the destination element remains unmodified.</p><p>The resulting SVL<sub>S</sub>\u00d7SVL<sub>S</sub> single-precision floating-point sum of outer products is then destructively subtracted from the single-precision floating-point destination tile. This is equivalent to performing a 2-way dot product and subtract from each of the destination tile elements.</p><p>Each 32-bit container of the first source vector holds 2 consecutive column elements of each row of a SVL<sub>S</sub>\u00d72 sub-matrix. Similarly, each 32-bit container of the second source vector holds 2 consecutive row elements of each column of a 2\u00d7SVL<sub>S</sub> sub-matrix.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMOV":
            return {
                "tooltip": "Floating-point move immediate (vector). This instruction copies an immediate floating-point constant into every element of the SIMD&FP destination register.",
                "html": "<p>Floating-point move immediate (vector). This instruction copies an immediate floating-point constant into every element of the SIMD&amp;FP destination register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMSB":
            return {
                "tooltip": "Multiply the corresponding active floating-point elements of the first and second source vectors and subtract from elements of the third (addend) vector without intermediate rounding. Destructively place the results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply the corresponding active floating-point elements of the first and second source vectors and subtract from elements of the third (addend) vector without intermediate rounding. Destructively place the results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMSUB":
            return {
                "tooltip": "Floating-point Fused Multiply-Subtract (scalar). This instruction multiplies the values of the first two SIMD&FP source registers, negates the product, adds that to the value of the third SIMD&FP source register, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Fused Multiply-Subtract (scalar). This instruction multiplies the values of the first two SIMD&amp;FP source registers, negates the product, adds that to the value of the third SIMD&amp;FP source register, and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMUL":
            return {
                "tooltip": "Floating-point Multiply (by element). This instruction multiplies the vector elements in the first source SIMD&FP register by the specified value in the second source SIMD&FP register, places the results in a vector, and writes the vector to the destination SIMD&FP register. All the values in this instruction are floating-point values.",
                "html": "<p>Floating-point Multiply (by element). This instruction multiplies the vector elements in the first source SIMD&amp;FP register by the specified value in the second source SIMD&amp;FP register, places the results in a vector, and writes the vector to the destination SIMD&amp;FP register. All the values in this instruction are floating-point values.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FMULX":
            return {
                "tooltip": "Floating-point Multiply extended (by element). This instruction multiplies the floating-point values in the vector elements in the first source SIMD&FP register by the specified floating-point value in the second source SIMD&FP register, places the results in a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Multiply extended (by element). This instruction multiplies the floating-point values in the vector elements in the first source SIMD&amp;FP register by the specified floating-point value in the second source SIMD&amp;FP register, places the results in a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>If one value is zero and the other value is infinite, the result is 2.0. In this case, the result is negative if only one of the values is negative, otherwise the result is positive.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FNEG":
            return {
                "tooltip": "Floating-point Negate (vector). This instruction negates the value of each vector element in the source SIMD&FP register, writes the result to a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Negate (vector). This instruction negates the value of each vector element in the source SIMD&amp;FP register, writes the result to a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FNMAD":
            return {
                "tooltip": "Multiply the corresponding active floating-point elements of the first and second source vectors and add to elements of the third (addend) vector without intermediate rounding. Destructively place the negated results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply the corresponding active floating-point elements of the first and second source vectors and add to elements of the third (addend) vector without intermediate rounding. Destructively place the negated results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FNMADD":
            return {
                "tooltip": "Floating-point Negated fused Multiply-Add (scalar). This instruction multiplies the values of the first two SIMD&FP source registers, negates the product, subtracts the value of the third SIMD&FP source register, and writes the result to the destination SIMD&FP register.",
                "html": "<p>Floating-point Negated fused Multiply-Add (scalar). This instruction multiplies the values of the first two SIMD&amp;FP source registers, negates the product, subtracts the value of the third SIMD&amp;FP source register, and writes the result to the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FNMLA":
            return {
                "tooltip": "Multiply the corresponding active floating-point elements of the first and second source vectors and add to elements of the third source (addend) vector without intermediate rounding. Destructively place the negated results in the destination and third source (addend) vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply the corresponding active floating-point elements of the first and second source vectors and add to elements of the third source (addend) vector without intermediate rounding. Destructively place the negated results in the destination and third source (addend) vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FNMLS":
            return {
                "tooltip": "Multiply the corresponding active floating-point elements of the first and second source vectors and subtract from elements of the third source (addend) vector without intermediate rounding. Destructively place the negated results in the destination and third source (addend) vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply the corresponding active floating-point elements of the first and second source vectors and subtract from elements of the third source (addend) vector without intermediate rounding. Destructively place the negated results in the destination and third source (addend) vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FNMSB":
            return {
                "tooltip": "Multiply the corresponding active floating-point elements of the first and second source vectors and subtract from elements of the third (addend) vector without intermediate rounding. Destructively place the negated results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply the corresponding active floating-point elements of the first and second source vectors and subtract from elements of the third (addend) vector without intermediate rounding. Destructively place the negated results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FNMSUB":
            return {
                "tooltip": "Floating-point Negated fused Multiply-Subtract (scalar). This instruction multiplies the values of the first two SIMD&FP source registers, subtracts the value of the third SIMD&FP source register, and writes the result to the destination SIMD&FP register.",
                "html": "<p>Floating-point Negated fused Multiply-Subtract (scalar). This instruction multiplies the values of the first two SIMD&amp;FP source registers, subtracts the value of the third SIMD&amp;FP source register, and writes the result to the destination SIMD&amp;FP register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FNMUL":
            return {
                "tooltip": "Floating-point Multiply-Negate (scalar). This instruction multiplies the floating-point values of the two source SIMD&FP registers, and writes the negation of the result to the destination SIMD&FP register.",
                "html": "<p>Floating-point Multiply-Negate (scalar). This instruction multiplies the floating-point values of the two source SIMD&amp;FP registers, and writes the negation of the result to the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRECPE":
            return {
                "tooltip": "Floating-point Reciprocal Estimate. This instruction finds an approximate reciprocal estimate for each vector element in the source SIMD&FP register, places the result in a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Reciprocal Estimate. This instruction finds an approximate reciprocal estimate for each vector element in the source SIMD&amp;FP register, places the result in a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRECPS":
            return {
                "tooltip": "Floating-point Reciprocal Step. This instruction multiplies the corresponding floating-point values in the vectors of the two source SIMD&FP registers, subtracts each of the products from 2.0, places the resulting floating-point values in a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Reciprocal Step. This instruction multiplies the corresponding floating-point values in the vectors of the two source SIMD&amp;FP registers, subtracts each of the products from 2.0, places the resulting floating-point values in a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRECPX":
            return {
                "tooltip": "Floating-point Reciprocal exponent (scalar). This instruction finds an approximate reciprocal exponent for the source SIMD&FP register and writes the result to the destination SIMD&FP register.",
                "html": "<p>Floating-point Reciprocal exponent (scalar). This instruction finds an approximate reciprocal exponent for the source SIMD&amp;FP register and writes the result to the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRINT32X":
            return {
                "tooltip": "Floating-point Round to 32-bit Integer, using current rounding mode (vector). This instruction rounds a vector of floating-point values in the SIMD&FP source register to integral floating-point values that fit into a 32-bit integer size using the rounding mode that is determined by the FPCR, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Round to 32-bit Integer, using current rounding mode (vector). This instruction rounds a vector of floating-point values in the SIMD&amp;FP source register to integral floating-point values that fit into a 32-bit integer size using the rounding mode that is determined by the <xref linkend=\"AArch64.fpcr\">FPCR</xref>, and writes the result to the SIMD&amp;FP destination register.</p><p>A zero input returns a zero result with the same sign. When one of the result values is not numerically equal to the corresponding input value, an Inexact exception is raised. When an input is infinite, NaN or out-of-range, the instruction returns for the corresponding result value the most negative integer representable in the destination size, and an Invalid Operation floating-point exception is raised.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRINT32Z":
            return {
                "tooltip": "Floating-point Round to 32-bit Integer toward Zero (vector). This instruction rounds a vector of floating-point values in the SIMD&FP source register to integral floating-point values that fit into a 32-bit integer size using the Round towards Zero rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Round to 32-bit Integer toward Zero (vector). This instruction rounds a vector of floating-point values in the SIMD&amp;FP source register to integral floating-point values that fit into a 32-bit integer size using the Round towards Zero rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A zero input returns a zero result with the same sign. When one of the result values is not numerically equal to the corresponding input value, an Inexact exception is raised. When an input is infinite, NaN or out-of-range, the instruction returns for the corresponding result value the most negative integer representable in the destination size, and an Invalid Operation floating-point exception is raised.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRINT64X":
            return {
                "tooltip": "Floating-point Round to 64-bit Integer, using current rounding mode (vector). This instruction rounds a vector of floating-point values in the SIMD&FP source register to integral floating-point values that fit into a 64-bit integer size using the rounding mode that is determined by the FPCR, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Round to 64-bit Integer, using current rounding mode (vector). This instruction rounds a vector of floating-point values in the SIMD&amp;FP source register to integral floating-point values that fit into a 64-bit integer size using the rounding mode that is determined by the <xref linkend=\"AArch64.fpcr\">FPCR</xref>, and writes the result to the SIMD&amp;FP destination register.</p><p>A zero input returns a zero result with the same sign. When one of the result values is not numerically equal to the corresponding input value, an Inexact exception is raised. When an input is infinite, NaN or out-of-range, the instruction returns for the corresponding result value the most negative integer representable in the destination size, and an Invalid Operation floating-point exception is raised.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRINT64Z":
            return {
                "tooltip": "Floating-point Round to 64-bit Integer toward Zero (vector). This instruction rounds a vector of floating-point values in the SIMD&FP source register to integral floating-point values that fit into a 64-bit integer size using the Round towards Zero rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Round to 64-bit Integer toward Zero (vector). This instruction rounds a vector of floating-point values in the SIMD&amp;FP source register to integral floating-point values that fit into a 64-bit integer size using the Round towards Zero rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A zero input returns a zero result with the same sign. When one of the result values is not numerically equal to the corresponding input value, an Inexact exception is raised. When an input is infinite, NaN or out-of-range, the instruction returns for the corresponding result value the most negative integer representable in the destination size, and an Invalid Operation floating-point exception is raised.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRINTA":
            return {
                "tooltip": "Floating-point Round to Integral, to nearest with ties to Away (vector). This instruction rounds a vector of floating-point values in the SIMD&FP source register to integral floating-point values of the same size using the Round to Nearest with Ties to Away rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Round to Integral, to nearest with ties to Away (vector). This instruction rounds a vector of floating-point values in the SIMD&amp;FP source register to integral floating-point values of the same size using the Round to Nearest with Ties to Away rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A zero input gives a zero result with the same sign, an infinite input gives an infinite result with the same sign, and a NaN is propagated as for normal arithmetic.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRINTI":
            return {
                "tooltip": "Floating-point Round to Integral, using current rounding mode (vector). This instruction rounds a vector of floating-point values in the SIMD&FP source register to integral floating-point values of the same size using the rounding mode that is determined by the FPCR, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Round to Integral, using current rounding mode (vector). This instruction rounds a vector of floating-point values in the SIMD&amp;FP source register to integral floating-point values of the same size using the rounding mode that is determined by the <xref linkend=\"AArch64.fpcr\">FPCR</xref>, and writes the result to the SIMD&amp;FP destination register.</p><p>A zero input gives a zero result with the same sign, an infinite input gives an infinite result with the same sign, and a NaN is propagated as for normal arithmetic.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRINTM":
            return {
                "tooltip": "Floating-point Round to Integral, toward Minus infinity (vector). This instruction rounds a vector of floating-point values in the SIMD&FP source register to integral floating-point values of the same size using the Round towards Minus Infinity rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Round to Integral, toward Minus infinity (vector). This instruction rounds a vector of floating-point values in the SIMD&amp;FP source register to integral floating-point values of the same size using the Round towards Minus Infinity rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A zero input gives a zero result with the same sign, an infinite input gives an infinite result with the same sign, and a NaN is propagated as for normal arithmetic.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRINTN":
            return {
                "tooltip": "Floating-point Round to Integral, to nearest with ties to even (vector). This instruction rounds a vector of floating-point values in the SIMD&FP source register to integral floating-point values of the same size using the Round to Nearest rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Round to Integral, to nearest with ties to even (vector). This instruction rounds a vector of floating-point values in the SIMD&amp;FP source register to integral floating-point values of the same size using the Round to Nearest rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A zero input gives a zero result with the same sign, an infinite input gives an infinite result with the same sign, and a NaN is propagated as for normal arithmetic.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRINTP":
            return {
                "tooltip": "Floating-point Round to Integral, toward Plus infinity (vector). This instruction rounds a vector of floating-point values in the SIMD&FP source register to integral floating-point values of the same size using the Round towards Plus Infinity rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Round to Integral, toward Plus infinity (vector). This instruction rounds a vector of floating-point values in the SIMD&amp;FP source register to integral floating-point values of the same size using the Round towards Plus Infinity rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A zero input gives a zero result with the same sign, an infinite input gives an infinite result with the same sign, and a NaN is propagated as for normal arithmetic.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRINTX":
            return {
                "tooltip": "Floating-point Round to Integral exact, using current rounding mode (vector). This instruction rounds a vector of floating-point values in the SIMD&FP source register to integral floating-point values of the same size using the rounding mode that is determined by the FPCR, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Round to Integral exact, using current rounding mode (vector). This instruction rounds a vector of floating-point values in the SIMD&amp;FP source register to integral floating-point values of the same size using the rounding mode that is determined by the <xref linkend=\"AArch64.fpcr\">FPCR</xref>, and writes the result to the SIMD&amp;FP destination register.</p><p>When a result value is not numerically equal to the corresponding input value, an Inexact exception is raised. A zero input gives a zero result with the same sign, an infinite input gives an infinite result with the same sign, and a NaN is propagated as for normal arithmetic.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRINTZ":
            return {
                "tooltip": "Floating-point Round to Integral, toward Zero (vector). This instruction rounds a vector of floating-point values in the SIMD&FP source register to integral floating-point values of the same size using the Round towards Zero rounding mode, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Floating-point Round to Integral, toward Zero (vector). This instruction rounds a vector of floating-point values in the SIMD&amp;FP source register to integral floating-point values of the same size using the Round towards Zero rounding mode, and writes the result to the SIMD&amp;FP destination register.</p><p>A zero input gives a zero result with the same sign, an infinite input gives an infinite result with the same sign, and a NaN is propagated as for normal arithmetic.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRSQRTE":
            return {
                "tooltip": "Floating-point Reciprocal Square Root Estimate. This instruction calculates an approximate square root for each vector element in the source SIMD&FP register, places the result in a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Reciprocal Square Root Estimate. This instruction calculates an approximate square root for each vector element in the source SIMD&amp;FP register, places the result in a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FRSQRTS":
            return {
                "tooltip": "Floating-point Reciprocal Square Root Step. This instruction multiplies corresponding floating-point values in the vectors of the two source SIMD&FP registers, subtracts each of the products from 3.0, divides these results by 2.0, places the results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Reciprocal Square Root Step. This instruction multiplies corresponding floating-point values in the vectors of the two source SIMD&amp;FP registers, subtracts each of the products from 3.0, divides these results by 2.0, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FSCALE":
            return {
                "tooltip": "Multiply the active floating-point elements of the first source vector by 2.0 to the power of the signed integer values in the corresponding elements of the second source vector and destructively place the results in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply the active floating-point elements of the first source vector by 2.0 to the power of the signed integer values in the corresponding elements of the second source vector and destructively place the results in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FSQRT":
            return {
                "tooltip": "Floating-point Square Root (vector). This instruction calculates the square root for each vector element in the source SIMD&FP register, places the result in a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Square Root (vector). This instruction calculates the square root for each vector element in the source SIMD&amp;FP register, places the result in a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref> or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FSUB":
            return {
                "tooltip": "Floating-point Subtract (vector). This instruction subtracts the elements in the vector in the second source SIMD&FP register, from the corresponding elements in the vector in the first source SIMD&FP register, places each result into elements of a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Floating-point Subtract (vector). This instruction subtracts the elements in the vector in the second source SIMD&amp;FP register, from the corresponding elements in the vector in the first source SIMD&amp;FP register, places each result into elements of a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>This instruction can generate a floating-point exception. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FSUBR":
            return {
                "tooltip": "Reversed subtract from an immediate each active floating-point element of the source vector, and destructively place the results in the corresponding elements of the source vector. The immediate may take the value +0.5 or +1.0 only. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Reversed subtract from an immediate each active floating-point element of the source vector, and destructively place the results in the corresponding elements of the source vector. The immediate may take the value +0.5 or +1.0 only. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FTMAD":
            return {
                "tooltip": "The FTMAD instruction calculates the series terms for either sin(x) or cos(x), where the argument x has been adjusted to be in the range -\u03c0/4 < x \u2264 \u03c0/4.",
                "html": "<p>The <instruction>FTMAD</instruction> instruction calculates the series terms for either <arm-defined-word>sin(x)</arm-defined-word> or <arm-defined-word>cos(x)</arm-defined-word>, where the argument <arm-defined-word>x</arm-defined-word> has been adjusted to be in the range -\u03c0/4 &lt; <arm-defined-word>x</arm-defined-word> \u2264 \u03c0/4.</p><p>To calculate the series terms of <arm-defined-word>sin(x)</arm-defined-word> and <arm-defined-word>cos(x)</arm-defined-word> the initial source operands of <instruction>FTMAD</instruction> should be zero in the first source vector and <arm-defined-word>x</arm-defined-word><sup>2</sup> in the second source vector. The <instruction>FTMAD</instruction> instruction is then executed eight times to calculate the sum of eight series terms, which gives a result of sufficient precision.</p><p>The <instruction>FTMAD</instruction> instruction multiplies each element of the first source vector by the absolute value of the corresponding element of the second source vector and performs a fused addition of each product with a value obtained from a table of hard-wired coefficients, and places the results destructively in the first source vector.</p><p>The coefficients are different for <arm-defined-word>sin(x)</arm-defined-word> and <arm-defined-word>cos(x)</arm-defined-word>, and are selected by a combination of the sign bit in the second source element and an immediate index in the range 0 to 7.</p><p></p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FTSMUL":
            return {
                "tooltip": "The FTSMUL instruction calculates the initial value for the FTMAD instruction. The instruction squares each element in the first source vector and then sets the sign bit to a copy of bit 0 of the corresponding element in the second source register, and places the results in the destination vector. This instruction is unpredicated.",
                "html": "<p>The <instruction>FTSMUL</instruction> instruction calculates the initial value for the <instruction>FTMAD</instruction> instruction. The instruction squares each element in the first source vector and then sets the sign bit to a copy of bit 0 of the corresponding element in the second source register, and places the results in the destination vector. This instruction is unpredicated.</p><p>To compute <arm-defined-word>sin(x)</arm-defined-word> or <arm-defined-word>cos(x)</arm-defined-word> the instruction is executed with elements of the first source vector set to <arm-defined-word>x</arm-defined-word>, adjusted to be in the range -\u03c0/4 &lt; <arm-defined-word>x</arm-defined-word> \u2264 \u03c0/4.</p><p>The elements of the second source vector hold the corresponding value of the quadrant <arm-defined-word>q</arm-defined-word> number as an integer not a floating-point value. The value <arm-defined-word>q</arm-defined-word> satisfies the relationship (2q-1) \u00d7 \u03c0/4 &lt; <arm-defined-word>x</arm-defined-word> \u2264 (2q+1) \u00d7 \u03c0/4.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FTSSEL":
            return {
                "tooltip": "The FTSSEL instruction selects the coefficient for the final multiplication in the polynomial series approximation. The instruction places the value 1.0 or a copy of the first source vector element in the destination element, depending on bit 0 of the quadrant number q held in the corresponding element of the second source vector. The sign bit of the destination element is copied from bit 1 of the corresponding value of q. This instruction is unpredicated.",
                "html": "<p>The <instruction>FTSSEL</instruction> instruction selects the coefficient for the final multiplication in the polynomial series approximation. The instruction places the value 1.0 or a copy of the first source vector element in the destination element, depending on bit 0 of the quadrant number <arm-defined-word>q</arm-defined-word> held in the corresponding element of the second source vector. The sign bit of the destination element is copied from bit 1 of the corresponding value of <arm-defined-word>q</arm-defined-word>. This instruction is unpredicated.</p><p>To compute <arm-defined-word>sin(x)</arm-defined-word> or <arm-defined-word>cos(x)</arm-defined-word> the instruction is executed with elements of the first source vector set to <arm-defined-word>x</arm-defined-word>, adjusted to be in the range -\u03c0/4 &lt; <arm-defined-word>x</arm-defined-word> \u2264 \u03c0/4.</p><p>The elements of the second source vector hold the corresponding value of the quadrant <arm-defined-word>q</arm-defined-word> number as an integer not a floating-point value. The value <arm-defined-word>q</arm-defined-word> satisfies the relationship (2q-1) \u00d7 \u03c0/4 &lt; <arm-defined-word>x</arm-defined-word> \u2264 (2q+1) \u00d7 \u03c0/4.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "FVDOT":
            return {
                "tooltip": "The instruction computes the fused sum-of-products of each vertical pair of half-precision floating-point values in the corresponding elements of the two first source vectors with the pair of half-precision floating-point values in the indexed 32-bit group of the corresponding 128-bit segment of the second source vector, without intermediate rounding. The single-precision sum-of-products results are destructively added to the corresponding single-precision elements of the two ZA single-vector groups.",
                "html": "<p>The instruction computes the fused sum-of-products of each vertical pair of half-precision floating-point values in the corresponding elements of the two first source vectors with the pair of half-precision floating-point values in the indexed 32-bit group of the corresponding 128-bit segment of the second source vector, without intermediate rounding. The single-precision sum-of-products results are destructively added to the corresponding single-precision elements of the two ZA single-vector groups.</p><p>The half-precision floating-point pairs within the second source vector are specified using an immediate index which selects the same pair position within each 128-bit vector segment. The element index range is from 0 to 3.</p><p>The vector numbers forming the single-vector group within each half of the ZA array are selected by the sum of the vector select register and immediate offset, modulo half the number of ZA array vectors.</p><p>The <arm-defined-word>vector group</arm-defined-word> symbol <syntax>VGx2</syntax> indicates that the ZA operand consists of two ZA single-vector groups. The <arm-defined-word>vector group</arm-defined-word> symbol is preferred for disassembly, but optional in assembler source code.</p><p>This instruction follows SME ZA-targeting floating-point behaviors.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "GCSBDSYNC":
            return {
                "tooltip": "Guarded Control Stack Barrier. This instruction generates a Guarded control stack data synchronization event.",
                "html": "<p>Guarded Control Stack Barrier. This instruction generates a Guarded control stack data synchronization event.</p><p>If FEAT_GCS is not implemented, this instruction executes as a <instruction>NOP</instruction>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "GCSPOPCX":
            return {
                "tooltip": "Guarded Control Stack Pop and Compare exception return record loads an exception return record from the location indicated by the current Guarded control stack pointer register, compares the loaded values with the current ELR_ELx, SPSR_ELx, and LR, and increments the pointer by the size of a Guarded control stack exception return record.",
                "html": "<p>Guarded Control Stack Pop and Compare exception return record loads an exception return record from the location indicated by the current Guarded control stack pointer register, compares the loaded values with the current ELR_ELx, SPSR_ELx, and LR, and increments the pointer by the size of a Guarded control stack exception return record.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "GCSPOPM":
            return {
                "tooltip": "Guarded Control Stack Pop loads the 64-bit doubleword that is pointed to by the current Guarded control stack pointer, writes it to the destination register, and increments the current Guarded control stack pointer register by the size of a Guarded control stack procedure return record.",
                "html": "<p>Guarded Control Stack Pop loads the 64-bit doubleword that is pointed to by the current Guarded control stack pointer, writes it to the destination register, and increments the current Guarded control stack pointer register by the size of a Guarded control stack procedure return record.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "GCSPOPX":
            return {
                "tooltip": "Guarded Control Stack Pop exception return record loads an exception return record from the location indicated by the current Guarded control stack pointer register, checks that the record is an exception return record, and increments the pointer by the size of a Guarded control stack exception return record.",
                "html": "<p>Guarded Control Stack Pop exception return record loads an exception return record from the location indicated by the current Guarded control stack pointer register, checks that the record is an exception return record, and increments the pointer by the size of a Guarded control stack exception return record.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "GCSPUSHM":
            return {
                "tooltip": "Guarded Control Stack Push decrements the current Guarded control stack pointer register by the size of a Guarded control procedure return record and stores an entry to the Guarded control stack.",
                "html": "<p>Guarded Control Stack Push decrements the current Guarded control stack pointer register by the size of a Guarded control procedure return record and stores an entry to the Guarded control stack.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "GCSPUSHX":
            return {
                "tooltip": "Guarded Control Stack Push exception return record decrements the current Guarded control stack pointer register by the size of a Guarded control stack exception return record and stores an exception return record to the Guarded control stack.",
                "html": "<p>Guarded Control Stack Push exception return record decrements the current Guarded control stack pointer register by the size of a Guarded control stack exception return record and stores an exception return record to the Guarded control stack.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "GCSSS1":
            return {
                "tooltip": "Guarded Control Stack Switch Stack 1 validates that the stack being switched to contains a Valid cap entry, stores an In-progress cap entry to the stack that is being switched to, and sets the current Guarded control stack pointer to the stack that is being switched to.",
                "html": "<p>Guarded Control Stack Switch Stack 1 validates that the stack being switched to contains a Valid cap entry, stores an In-progress cap entry to the stack that is being switched to, and sets the current Guarded control stack pointer to the stack that is being switched to.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "GCSSS2":
            return {
                "tooltip": "Guarded Control Stack Switch Stack 2 validates that the most recent entry of the Guarded control stack being switched to contains an In-progress cap entry, stores a Valid cap entry to the Guarded control stack that is being switched from, and sets Xt to the Guarded control stack pointer that is being switched from.",
                "html": "<p>Guarded Control Stack Switch Stack 2 validates that the most recent entry of the Guarded control stack being switched to contains an In-progress cap entry, stores a Valid cap entry to the Guarded control stack that is being switched from, and sets Xt to the Guarded control stack pointer that is being switched from.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "GCSSTR":
            return {
                "tooltip": "Guarded Control Stack Store stores a doubleword from a register to memory. The address that is used for the store is calculated from a base register.",
                "html": "<p>Guarded Control Stack Store stores a doubleword from a register to memory. The address that is used for the store is calculated from a base register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "GCSSTTR":
            return {
                "tooltip": "Guarded Control Stack unprivileged Store stores a doubleword from a register to memory. The address that is used for the store is calculated from a base register.",
                "html": "<p>Guarded Control Stack unprivileged Store stores a doubleword from a register to memory. The address that is used for the store is calculated from a base register.</p><p>Memory accesses made by the instruction behave as if the instruction was executed at EL0 if the <xref linkend=\"CBAICCHA\">Effective value</xref> of PSTATE.UAO is 0 and either:</p><p>Otherwise, the memory access operates with the restrictions determined by the Exception level at which the instruction is executed.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "GMI":
            return {
                "tooltip": "Tag Mask Insert inserts the tag in the first source register into the excluded set specified in the second source register, writing the new excluded set to the destination register.",
                "html": "<p>Tag Mask Insert inserts the tag in the first source register into the excluded set specified in the second source register, writing the new excluded set to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "HINT":
            return {
                "tooltip": "Hint instruction is for the instruction set space that is reserved for architectural hint instructions.",
                "html": "<p>Hint instruction is for the instruction set space that is reserved for architectural hint instructions.</p><p>Some encodings described here are not allocated in this revision of the architecture, and behave as NOPs. These encodings might be allocated to other hint functionality in future revisions of the architecture and therefore must not be used by software.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "HISTCNT":
            return {
                "tooltip": "This instruction compares each active 32 or 64-bit element of the first source vector with all active elements with an element number less than or equal to its own in the second source vector, and places the count of matching elements in the corresponding element of the destination vector. Inactive elements in the destination vector are set to zero.",
                "html": "<p>This instruction compares each active 32 or 64-bit element of the first source vector with all active elements with an element number less than or equal to its own in the second source vector, and places the count of matching elements in the corresponding element of the destination vector. Inactive elements in the destination vector are set to zero.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "HISTSEG":
            return {
                "tooltip": "This instruction compares each 8-bit byte element of the first source vector with all of the elements in the corresponding 128-bit segment of the second source vector and places the count of matching elements in the corresponding element of the destination vector. This instruction is unpredicated.",
                "html": "<p>This instruction compares each 8-bit byte element of the first source vector with all of the elements in the corresponding 128-bit segment of the second source vector and places the count of matching elements in the corresponding element of the destination vector. This instruction is unpredicated.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "HLT":
            return {
                "tooltip": "Halt instruction. An HLT instruction can generate a Halt Instruction debug event, which causes entry into Debug state.",
                "html": "<p>Halt instruction. An <instruction>HLT</instruction> instruction can generate a Halt Instruction debug event, which causes entry into Debug state.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "HVC":
            return {
                "tooltip": "Hypervisor Call causes an exception to EL2. Software executing at EL1 can use this instruction to call the hypervisor to request a service.",
                "html": "<p>Hypervisor Call causes an exception to EL2. Software executing at EL1 can use this instruction to call the hypervisor to request a service.</p><p>The <instruction>HVC</instruction> instruction is <arm-defined-word>undefined</arm-defined-word>:</p><p>On executing an <instruction>HVC</instruction> instruction, the PE records the exception as a Hypervisor Call exception in <xref linkend=\"ESR_ELx\">ESR_ELx</xref>, using the EC value <hexnumber>0x16</hexnumber>, and the value of the immediate argument.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "IC":
            return {
                "tooltip": "Instruction Cache operation. For more information, see op0==0b01, cache maintenance, TLB maintenance, and address translation instructions.",
                "html": "<p>Instruction Cache operation. For more information, see <xref linkend=\"BABEJJJE\">op0==0b01, cache maintenance, TLB maintenance, and address translation instructions</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "INCB":
        case "INCD":
        case "INCH":
        case "INCW":
            return {
                "tooltip": "Determines the number of active elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to increment the scalar destination.",
                "html": "<p>Determines the number of active elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to increment the scalar destination.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "INCP":
            return {
                "tooltip": "Counts the number of true elements in the source predicate and then uses the result to increment the scalar destination.",
                "html": "<p>Counts the number of true elements in the source predicate and then uses the result to increment the scalar destination.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "INDEX":
            return {
                "tooltip": "Populates the destination vector by setting the first element to the first signed immediate integer operand and monotonically incrementing the value by the second signed immediate integer operand for each subsequent element. This instruction is unpredicated.",
                "html": "<p>Populates the destination vector by setting the first element to the first signed immediate integer operand and monotonically incrementing the value by the second signed immediate integer operand for each subsequent element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "INS":
            return {
                "tooltip": "Insert vector element from another vector element. This instruction copies the vector element of the source SIMD&FP register to the specified vector element of the destination SIMD&FP register.",
                "html": "<p>Insert vector element from another vector element. This instruction copies the vector element of the source SIMD&amp;FP register to the specified vector element of the destination SIMD&amp;FP register.</p><p>This instruction can insert data into individual elements within a SIMD&amp;FP register without clearing the remaining bits to zero.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "INSR":
            return {
                "tooltip": "Shift the destination vector left by one element, and then place a copy of the least-significant bits of the general-purpose register in element 0 of the destination vector. This instruction is unpredicated.",
                "html": "<p>Shift the destination vector left by one element, and then place a copy of the least-significant bits of the general-purpose register in element 0 of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "IRG":
            return {
                "tooltip": "Insert Random Tag inserts a random Logical Address Tag into the address in the first source register, and writes the result to the destination register. Any tags specified in the optional second source register or in GCR_EL1.Exclude are excluded from the selection of the random Logical Address Tag.",
                "html": "<p>Insert Random Tag inserts a random Logical Address Tag into the address in the first source register, and writes the result to the destination register. Any tags specified in the optional second source register or in GCR_EL1.Exclude are excluded from the selection of the random Logical Address Tag.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ISB":
            return {
                "tooltip": "Instruction Synchronization Barrier flushes the pipeline in the PE and is a context synchronization event. For more information, see Instruction Synchronization Barrier (ISB).",
                "html": "<p>Instruction Synchronization Barrier flushes the pipeline in the PE and is a context synchronization event. For more information, see <xref linkend=\"BEIJADJC\">Instruction Synchronization Barrier (ISB)</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LASTA":
            return {
                "tooltip": "If there is an active element then extract the element after the last active element modulo the number of elements from the final source vector register. If there are no active elements, extract element zero. Then zero-extend and place the extracted element in the destination general-purpose register.",
                "html": "<p>If there is an active element then extract the element after the last active element modulo the number of elements from the final source vector register. If there are no active elements, extract element zero. Then zero-extend and place the extracted element in the destination general-purpose register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LASTB":
            return {
                "tooltip": "If there is an active element then extract the last active element from the final source vector register. If there are no active elements, extract the highest-numbered element. Then zero-extend and place the extracted element in the destination general-purpose register.",
                "html": "<p>If there is an active element then extract the last active element from the final source vector register. If there are no active elements, extract the highest-numbered element. Then zero-extend and place the extracted element in the destination general-purpose register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1":
            return {
                "tooltip": "Load multiple single-element structures to one, two, three, or four registers. This instruction loads multiple single-element structures from memory and writes the result to one, two, three, or four SIMD&FP registers.",
                "html": "<p>Load multiple single-element structures to one, two, three, or four registers. This instruction loads multiple single-element structures from memory and writes the result to one, two, three, or four SIMD&amp;FP registers.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1B":
            return {
                "tooltip": "Contiguous load of unsigned bytes to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous load of unsigned bytes to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1D":
            return {
                "tooltip": "Contiguous load of unsigned doublewords to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous load of unsigned doublewords to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1H":
            return {
                "tooltip": "Contiguous load of unsigned halfwords to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous load of unsigned halfwords to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1Q":
            return {
                "tooltip": "Gather load of quadwords to active elements of a vector register from memory addresses generated by a vector base plus a 64-bit unscaled scalar register offset. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load of quadwords to active elements of a vector register from memory addresses generated by a vector base plus a 64-bit unscaled scalar register offset. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1R":
            return {
                "tooltip": "Load one single-element structure and Replicate to all lanes (of one register). This instruction loads a single-element structure from memory and replicates the structure to all the lanes of the SIMD&FP register.",
                "html": "<p>Load one single-element structure and Replicate to all lanes (of one register). This instruction loads a single-element structure from memory and replicates the structure to all the lanes of the SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1RB":
            return {
                "tooltip": "Load a single unsigned byte from a memory address generated by a 64-bit scalar base address plus an immediate offset which is in the range 0 to 63.",
                "html": "<p>Load a single unsigned byte from a memory address generated by a 64-bit scalar base address plus an immediate offset which is in the range 0 to 63.</p><p>Broadcast the loaded data into all active elements of the destination vector, setting the inactive elements to zero. If all elements are inactive then the instruction will not perform a read from Device memory or cause a data abort.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1RD":
            return {
                "tooltip": "Load a single doubleword from a memory address generated by a 64-bit scalar base address plus an immediate offset which is a multiple of 8 in the range 0 to 504.",
                "html": "<p>Load a single doubleword from a memory address generated by a 64-bit scalar base address plus an immediate offset which is a multiple of 8 in the range 0 to 504.</p><p>Broadcast the loaded data into all active elements of the destination vector, setting the inactive elements to zero. If all elements are inactive then the instruction will not perform a read from Device memory or cause a data abort.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1RH":
            return {
                "tooltip": "Load a single unsigned halfword from a memory address generated by a 64-bit scalar base address plus an immediate offset which is a multiple of 2 in the range 0 to 126.",
                "html": "<p>Load a single unsigned halfword from a memory address generated by a 64-bit scalar base address plus an immediate offset which is a multiple of 2 in the range 0 to 126.</p><p>Broadcast the loaded data into all active elements of the destination vector, setting the inactive elements to zero. If all elements are inactive then the instruction will not perform a read from Device memory or cause a data abort.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1ROB":
            return {
                "tooltip": "Load thirty-two contiguous bytes to elements of a 256-bit (octaword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 32 in the range -256 to +224 added to the base address.",
                "html": "<p>Load thirty-two contiguous bytes to elements of a 256-bit (octaword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 32 in the range -256 to +224 added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero.</p><p>The resulting 256-bit vector is then replicated to fill the destination vector. The instruction requires that the current vector length is at least 256 bits, and if the current vector length is not an integer multiple of 256 bits then the trailing bits in the destination vector are set to zero.</p><p>Only the first thirty-two predicate elements are used and higher numbered predicate elements are ignored.</p><p>ID_AA64ZFR0_EL1.F64MM indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1ROD":
            return {
                "tooltip": "Load four contiguous doublewords to elements of a 256-bit (octaword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 32 in the range -256 to +224 added to the base address.",
                "html": "<p>Load four contiguous doublewords to elements of a 256-bit (octaword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 32 in the range -256 to +224 added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero.</p><p>The resulting 256-bit vector is then replicated to fill the destination vector. The instruction requires that the current vector length is at least 256 bits, and if the current vector length is not an integer multiple of 256 bits then the trailing bits in the destination vector are set to zero.</p><p>Only the first four predicate elements are used and higher numbered predicate elements are ignored.</p><p>ID_AA64ZFR0_EL1.F64MM indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1ROH":
            return {
                "tooltip": "Load sixteen contiguous halfwords to elements of a 256-bit (octaword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 32 in the range -256 to +224 added to the base address.",
                "html": "<p>Load sixteen contiguous halfwords to elements of a 256-bit (octaword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 32 in the range -256 to +224 added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero.</p><p>The resulting 256-bit vector is then replicated to fill the destination vector. The instruction requires that the current vector length is at least 256 bits, and if the current vector length is not an integer multiple of 256 bits then the trailing bits in the destination vector are set to zero.</p><p>Only the first sixteen predicate elements are used and higher numbered predicate elements are ignored.</p><p>ID_AA64ZFR0_EL1.F64MM indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1ROW":
            return {
                "tooltip": "Load eight contiguous words to elements of a 256-bit (octaword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 32 in the range -256 to +224 added to the base address.",
                "html": "<p>Load eight contiguous words to elements of a 256-bit (octaword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 32 in the range -256 to +224 added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero.</p><p>The resulting 256-bit vector is then replicated to fill the destination vector. The instruction requires that the current vector length is at least 256 bits, and if the current vector length is not an integer multiple of 256 bits then the trailing bits in the destination vector are set to zero.</p><p>Only the first eight predicate elements are used and higher numbered predicate elements are ignored.</p><p>ID_AA64ZFR0_EL1.F64MM indicates whether this instruction is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1RQB":
            return {
                "tooltip": "Load sixteen contiguous bytes to elements of a short, 128-bit (quadword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 16 in the range -128 to +112 added to the base address.",
                "html": "<p>Load sixteen contiguous bytes to elements of a short, 128-bit (quadword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 16 in the range -128 to +112 added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero. The resulting short vector is then replicated to fill the long destination vector. Only the first sixteen predicate elements are used and higher numbered predicate elements are ignored.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1RQD":
            return {
                "tooltip": "Load two contiguous doublewords to elements of a short, 128-bit (quadword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 16 in the range -128 to +112 added to the base address.",
                "html": "<p>Load two contiguous doublewords to elements of a short, 128-bit (quadword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 16 in the range -128 to +112 added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero. The resulting short vector is then replicated to fill the long destination vector. Only the first two predicate elements are used and higher numbered predicate elements are ignored.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1RQH":
            return {
                "tooltip": "Load eight contiguous halfwords to elements of a short, 128-bit (quadword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 16 in the range -128 to +112 added to the base address.",
                "html": "<p>Load eight contiguous halfwords to elements of a short, 128-bit (quadword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 16 in the range -128 to +112 added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero. The resulting short vector is then replicated to fill the long destination vector. Only the first eight predicate elements are used and higher numbered predicate elements are ignored.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1RQW":
            return {
                "tooltip": "Load four contiguous words to elements of a short, 128-bit (quadword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 16 in the range -128 to +112 added to the base address.",
                "html": "<p>Load four contiguous words to elements of a short, 128-bit (quadword) vector from the memory address generated by a 64-bit scalar base address and immediate index that is a multiple of 16 in the range -128 to +112 added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero. The resulting short vector is then replicated to fill the long destination vector. Only the first four predicate elements are used and higher numbered predicate elements are ignored.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1RSB":
            return {
                "tooltip": "Load a single signed byte from a memory address generated by a 64-bit scalar base address plus an immediate offset which is in the range 0 to 63.",
                "html": "<p>Load a single signed byte from a memory address generated by a 64-bit scalar base address plus an immediate offset which is in the range 0 to 63.</p><p>Broadcast the loaded data into all active elements of the destination vector, setting the inactive elements to zero. If all elements are inactive then the instruction will not perform a read from Device memory or cause a data abort.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1RSH":
            return {
                "tooltip": "Load a single signed halfword from a memory address generated by a 64-bit scalar base address plus an immediate offset which is a multiple of 2 in the range 0 to 126.",
                "html": "<p>Load a single signed halfword from a memory address generated by a 64-bit scalar base address plus an immediate offset which is a multiple of 2 in the range 0 to 126.</p><p>Broadcast the loaded data into all active elements of the destination vector, setting the inactive elements to zero. If all elements are inactive then the instruction will not perform a read from Device memory or cause a data abort.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1RSW":
            return {
                "tooltip": "Load a single signed word from a memory address generated by a 64-bit scalar base address plus an immediate offset which is a multiple of 4 in the range 0 to 252.",
                "html": "<p>Load a single signed word from a memory address generated by a 64-bit scalar base address plus an immediate offset which is a multiple of 4 in the range 0 to 252.</p><p>Broadcast the loaded data into all active elements of the destination vector, setting the inactive elements to zero. If all elements are inactive then the instruction will not perform a read from Device memory or cause a data abort.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1RW":
            return {
                "tooltip": "Load a single unsigned word from a memory address generated by a 64-bit scalar base address plus an immediate offset which is a multiple of 4 in the range 0 to 252.",
                "html": "<p>Load a single unsigned word from a memory address generated by a 64-bit scalar base address plus an immediate offset which is a multiple of 4 in the range 0 to 252.</p><p>Broadcast the loaded data into all active elements of the destination vector, setting the inactive elements to zero. If all elements are inactive then the instruction will not perform a read from Device memory or cause a data abort.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1SB":
            return {
                "tooltip": "Gather load of signed bytes to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is in the range 0 to 31. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load of signed bytes to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is in the range 0 to 31. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1SH":
            return {
                "tooltip": "Gather load of signed halfwords to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 2 in the range 0 to 62. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load of signed halfwords to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 2 in the range 0 to 62. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1SW":
            return {
                "tooltip": "Gather load of signed words to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 4 in the range 0 to 124. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load of signed words to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 4 in the range 0 to 124. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD1W":
            return {
                "tooltip": "Contiguous load of unsigned words to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous load of unsigned words to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD2":
            return {
                "tooltip": "Load multiple 2-element structures to two registers. This instruction loads multiple 2-element structures from memory and writes the result to the two SIMD&FP registers, with de-interleaving.",
                "html": "<p>Load multiple 2-element structures to two registers. This instruction loads multiple 2-element structures from memory and writes the result to the two SIMD&amp;FP registers, with de-interleaving.</p><p>For an example of de-interleaving, see <instruction>LD3 (multiple structures)</instruction>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD2B":
            return {
                "tooltip": "Contiguous load two-byte structures, each to the same element number in two vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load two-byte structures, each to the same element number in two vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the two vector registers, or equivalently to the two consecutive bytes in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the two destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD2D":
            return {
                "tooltip": "Contiguous load two-doubleword structures, each to the same element number in two vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load two-doubleword structures, each to the same element number in two vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the two vector registers, or equivalently to the two consecutive doublewords in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the two destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD2H":
            return {
                "tooltip": "Contiguous load two-halfword structures, each to the same element number in two vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load two-halfword structures, each to the same element number in two vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the two vector registers, or equivalently to the two consecutive halfwords in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the two destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD2Q":
            return {
                "tooltip": "Contiguous load two-quadword structures, each to the same element number in two vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load two-quadword structures, each to the same element number in two vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the two vector registers, or equivalently to the two consecutive quadwords in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the two destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD2R":
            return {
                "tooltip": "Load single 2-element structure and Replicate to all lanes of two registers. This instruction loads a 2-element structure from memory and replicates the structure to all the lanes of the two SIMD&FP registers.",
                "html": "<p>Load single 2-element structure and Replicate to all lanes of two registers. This instruction loads a 2-element structure from memory and replicates the structure to all the lanes of the two SIMD&amp;FP registers.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD2W":
            return {
                "tooltip": "Contiguous load two-word structures, each to the same element number in two vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load two-word structures, each to the same element number in two vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the two vector registers, or equivalently to the two consecutive words in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the two destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD3":
            return {
                "tooltip": "Load multiple 3-element structures to three registers. This instruction loads multiple 3-element structures from memory and writes the result to the three SIMD&FP registers, with de-interleaving.",
                "html": "<p>Load multiple 3-element structures to three registers. This instruction loads multiple 3-element structures from memory and writes the result to the three SIMD&amp;FP registers, with de-interleaving.</p><p></p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD3B":
            return {
                "tooltip": "Contiguous load three-byte structures, each to the same element number in three vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load three-byte structures, each to the same element number in three vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the three vector registers, or equivalently to the three consecutive bytes in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the three destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD3D":
            return {
                "tooltip": "Contiguous load three-doubleword structures, each to the same element number in three vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load three-doubleword structures, each to the same element number in three vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the three vector registers, or equivalently to the three consecutive doublewords in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the three destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD3H":
            return {
                "tooltip": "Contiguous load three-halfword structures, each to the same element number in three vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load three-halfword structures, each to the same element number in three vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the three vector registers, or equivalently to the three consecutive halfwords in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the three destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD3Q":
            return {
                "tooltip": "Contiguous load three-quadword structures, each to the same element number in three vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load three-quadword structures, each to the same element number in three vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the three vector registers, or equivalently to the three consecutive quadwords in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the three destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD3R":
            return {
                "tooltip": "Load single 3-element structure and Replicate to all lanes of three registers. This instruction loads a 3-element structure from memory and replicates the structure to all the lanes of the three SIMD&FP registers.",
                "html": "<p>Load single 3-element structure and Replicate to all lanes of three registers. This instruction loads a 3-element structure from memory and replicates the structure to all the lanes of the three SIMD&amp;FP registers.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD3W":
            return {
                "tooltip": "Contiguous load three-word structures, each to the same element number in three vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load three-word structures, each to the same element number in three vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the three vector registers, or equivalently to the three consecutive words in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the three destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD4":
            return {
                "tooltip": "Load multiple 4-element structures to four registers. This instruction loads multiple 4-element structures from memory and writes the result to the four SIMD&FP registers, with de-interleaving.",
                "html": "<p>Load multiple 4-element structures to four registers. This instruction loads multiple 4-element structures from memory and writes the result to the four SIMD&amp;FP registers, with de-interleaving.</p><p>For an example of de-interleaving, see <instruction>LD3 (multiple structures)</instruction>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD4B":
            return {
                "tooltip": "Contiguous load four-byte structures, each to the same element number in four vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load four-byte structures, each to the same element number in four vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the four vector registers, or equivalently to the four consecutive bytes in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the four destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD4D":
            return {
                "tooltip": "Contiguous load four-doubleword structures, each to the same element number in four vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load four-doubleword structures, each to the same element number in four vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the four vector registers, or equivalently to the four consecutive doublewords in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the four destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD4H":
            return {
                "tooltip": "Contiguous load four-halfword structures, each to the same element number in four vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load four-halfword structures, each to the same element number in four vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the four vector registers, or equivalently to the four consecutive halfwords in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the four destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD4Q":
            return {
                "tooltip": "Contiguous load four-quadword structures, each to the same element number in four vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load four-quadword structures, each to the same element number in four vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the four vector registers, or equivalently to the four consecutive quadwords in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the four destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD4R":
            return {
                "tooltip": "Load single 4-element structure and Replicate to all lanes of four registers. This instruction loads a 4-element structure from memory and replicates the structure to all the lanes of the four SIMD&FP registers.",
                "html": "<p>Load single 4-element structure and Replicate to all lanes of four registers. This instruction loads a 4-element structure from memory and replicates the structure to all the lanes of the four SIMD&amp;FP registers.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD4W":
            return {
                "tooltip": "Contiguous load four-word structures, each to the same element number in four vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous load four-word structures, each to the same element number in four vector registers from the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the four vector registers, or equivalently to the four consecutive words in memory which make up each structure. Inactive elements will not cause a read from Device memory or signal a fault, and the corresponding element is set to zero in each of the four destination vector registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LD64B":
            return {
                "tooltip": "Single-copy Atomic 64-byte Load derives an address from a base register value, loads eight 64-bit doublewords from a memory location, and writes them to consecutive registers, Xt to X(t+7). The data that is loaded is atomic and is required to be 64-byte aligned.",
                "html": "<p>Single-copy Atomic 64-byte Load derives an address from a base register value, loads eight 64-bit doublewords from a memory location, and writes them to consecutive registers, <syntax>Xt</syntax> to <syntax>X(t+7)</syntax>. The data that is loaded is atomic and is required to be 64-byte aligned.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDADD":
        case "LDADDA":
        case "LDADDAL":
        case "LDADDL":
            return {
                "tooltip": "Atomic add on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, adds the value held in a register to it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic add on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, adds the value held in a register to it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDADDAB":
        case "LDADDALB":
        case "LDADDB":
        case "LDADDLB":
            return {
                "tooltip": "Atomic add on byte in memory atomically loads an 8-bit byte from memory, adds the value held in a register to it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic add on byte in memory atomically loads an 8-bit byte from memory, adds the value held in a register to it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDADDAH":
        case "LDADDALH":
        case "LDADDH":
        case "LDADDLH":
            return {
                "tooltip": "Atomic add on halfword in memory atomically loads a 16-bit halfword from memory, adds the value held in a register to it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic add on halfword in memory atomically loads a 16-bit halfword from memory, adds the value held in a register to it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAP1":
            return {
                "tooltip": "Load-Acquire RCpc one single-element structure to one lane of one register. This instruction loads a single-element structure from memory and writes the result to the specified lane of the SIMD&FP register without affecting the other bits of the register.",
                "html": "<p>Load-Acquire RCpc one single-element structure to one lane of one register. This instruction loads a single-element structure from memory and writes the result to the specified lane of the SIMD&amp;FP register without affecting the other bits of the register.</p><p>The instruction has memory ordering semantics, as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, except that:</p><p>This difference in memory ordering is not described in the pseudocode.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAPR":
            return {
                "tooltip": "Load-Acquire RCpc Register derives an address from a base register value, loads a 32-bit word or 64-bit doubleword from the derived address in memory, and writes it to a register.",
                "html": "<p>Load-Acquire RCpc Register derives an address from a base register value, loads a 32-bit word or 64-bit doubleword from the derived address in memory, and writes it to a register.</p><p>The instruction has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, except that:</p><p>This difference in memory ordering is not described in the pseudocode.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAPRB":
            return {
                "tooltip": "Load-Acquire RCpc Register Byte derives an address from a base register value, loads a byte from the derived address in memory, zero-extends it and writes it to a register.",
                "html": "<p>Load-Acquire RCpc Register Byte derives an address from a base register value, loads a byte from the derived address in memory, zero-extends it and writes it to a register.</p><p>The instruction has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, except that:</p><p>This difference in memory ordering is not described in the pseudocode.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAPRH":
            return {
                "tooltip": "Load-Acquire RCpc Register Halfword derives an address from a base register value, loads a halfword from the derived address in memory, zero-extends it and writes it to a register.",
                "html": "<p>Load-Acquire RCpc Register Halfword derives an address from a base register value, loads a halfword from the derived address in memory, zero-extends it and writes it to a register.</p><p>The instruction has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, except that:</p><p>This difference in memory ordering is not described in the pseudocode.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAPUR":
            return {
                "tooltip": "Load-Acquire RCpc SIMD&FP Register (unscaled offset). This instruction loads a SIMD&FP register from memory. The address that is used for the load is calculated from a base register value and an optional immediate offset.",
                "html": "<p>Load-Acquire RCpc SIMD&amp;FP Register (unscaled offset). This instruction loads a SIMD&amp;FP register from memory. The address that is used for the load is calculated from a base register value and an optional immediate offset.</p><p>The instruction has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, except that:</p><p>This difference in memory ordering is not described in the pseudocode.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAPURB":
            return {
                "tooltip": "Load-Acquire RCpc Register Byte (unscaled) calculates an address from a base register and an immediate offset, loads a byte from memory, zero-extends it, and writes it to a register.",
                "html": "<p>Load-Acquire RCpc Register Byte (unscaled) calculates an address from a base register and an immediate offset, loads a byte from memory, zero-extends it, and writes it to a register.</p><p>The instruction has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, except that:</p><p>This difference in memory ordering is not described in the pseudocode.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAPURH":
            return {
                "tooltip": "Load-Acquire RCpc Register Halfword (unscaled) calculates an address from a base register and an immediate offset, loads a halfword from memory, zero-extends it, and writes it to a register.",
                "html": "<p>Load-Acquire RCpc Register Halfword (unscaled) calculates an address from a base register and an immediate offset, loads a halfword from memory, zero-extends it, and writes it to a register.</p><p>The instruction has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, except that:</p><p>This difference in memory ordering is not described in the pseudocode.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAPURSB":
            return {
                "tooltip": "Load-Acquire RCpc Register Signed Byte (unscaled) calculates an address from a base register and an immediate offset, loads a signed byte from memory, sign-extends it, and writes it to a register.",
                "html": "<p>Load-Acquire RCpc Register Signed Byte (unscaled) calculates an address from a base register and an immediate offset, loads a signed byte from memory, sign-extends it, and writes it to a register.</p><p>The instruction has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, except that:</p><p>This difference in memory ordering is not described in the pseudocode.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAPURSH":
            return {
                "tooltip": "Load-Acquire RCpc Register Signed Halfword (unscaled) calculates an address from a base register and an immediate offset, loads a signed halfword from memory, sign-extends it, and writes it to a register.",
                "html": "<p>Load-Acquire RCpc Register Signed Halfword (unscaled) calculates an address from a base register and an immediate offset, loads a signed halfword from memory, sign-extends it, and writes it to a register.</p><p>The instruction has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, except that:</p><p>This difference in memory ordering is not described in the pseudocode.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAPURSW":
            return {
                "tooltip": "Load-Acquire RCpc Register Signed Word (unscaled) calculates an address from a base register and an immediate offset, loads a signed word from memory, sign-extends it, and writes it to a register.",
                "html": "<p>Load-Acquire RCpc Register Signed Word (unscaled) calculates an address from a base register and an immediate offset, loads a signed word from memory, sign-extends it, and writes it to a register.</p><p>The instruction has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, except that:</p><p>This difference in memory ordering is not described in the pseudocode.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAR":
            return {
                "tooltip": "Load-Acquire Register derives an address from a base register value, loads a 32-bit word or 64-bit doubleword from memory, and writes it to a register. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load-Acquire Register derives an address from a base register value, loads a 32-bit word or 64-bit doubleword from memory, and writes it to a register. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>For this instruction, if the destination is WZR/XZR, it is impossible for software to observe the presence of the acquire semantic other than its effect on the arrival at endpoints.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDARB":
            return {
                "tooltip": "Load-Acquire Register Byte derives an address from a base register value, loads a byte from memory, zero-extends it and writes it to a register. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load-Acquire Register Byte derives an address from a base register value, loads a byte from memory, zero-extends it and writes it to a register. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>For this instruction, if the destination is WZR/XZR, it is impossible for software to observe the presence of the acquire semantic other than its effect on the arrival at endpoints.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDARH":
            return {
                "tooltip": "Load-Acquire Register Halfword derives an address from a base register value, loads a halfword from memory, zero-extends it, and writes it to a register. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load-Acquire Register Halfword derives an address from a base register value, loads a halfword from memory, zero-extends it, and writes it to a register. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>For this instruction, if the destination is WZR/XZR, it is impossible for software to observe the presence of the acquire semantic other than its effect on the arrival at endpoints.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAXP":
            return {
                "tooltip": "Load-Acquire Exclusive Pair of Registers derives an address from a base register value, loads two 32-bit words or two 64-bit doublewords from memory, and writes them to two registers. For information on single-copy atomicity and alignment requirements, see Requirements for single-copy atomicity and Alignment of data accesses. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See Synchronization and semaphores. The instruction also has memory ordering semantics, as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load-Acquire Exclusive Pair of Registers derives an address from a base register value, loads two 32-bit words or two 64-bit doublewords from memory, and writes them to two registers. For information on single-copy atomicity and alignment requirements, see <xref linkend=\"CHDDCBCC\">Requirements for single-copy atomicity</xref> and <xref linkend=\"CHDFFEGJ\">Alignment of data accesses</xref>. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. The instruction also has memory ordering semantics, as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAXR":
            return {
                "tooltip": "Load-Acquire Exclusive Register derives an address from a base register value, loads a 32-bit word or 64-bit doubleword from memory, and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See Synchronization and semaphores. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load-Acquire Exclusive Register derives an address from a base register value, loads a 32-bit word or 64-bit doubleword from memory, and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAXRB":
            return {
                "tooltip": "Load-Acquire Exclusive Register Byte derives an address from a base register value, loads a byte from memory, zero-extends it and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See Synchronization and semaphores. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load-Acquire Exclusive Register Byte derives an address from a base register value, loads a byte from memory, zero-extends it and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDAXRH":
            return {
                "tooltip": "Load-Acquire Exclusive Register Halfword derives an address from a base register value, loads a halfword from memory, zero-extends it and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See Synchronization and semaphores. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load-Acquire Exclusive Register Halfword derives an address from a base register value, loads a halfword from memory, zero-extends it and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDCLR":
        case "LDCLRA":
        case "LDCLRAL":
        case "LDCLRL":
            return {
                "tooltip": "Atomic bit clear on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic bit clear on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDCLRAB":
        case "LDCLRALB":
        case "LDCLRB":
        case "LDCLRLB":
            return {
                "tooltip": "Atomic bit clear on byte in memory atomically loads an 8-bit byte from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic bit clear on byte in memory atomically loads an 8-bit byte from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDCLRAH":
        case "LDCLRALH":
        case "LDCLRH":
        case "LDCLRLH":
            return {
                "tooltip": "Atomic bit clear on halfword in memory atomically loads a 16-bit halfword from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic bit clear on halfword in memory atomically loads a 16-bit halfword from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDCLRP":
        case "LDCLRPA":
        case "LDCLRPAL":
        case "LDCLRPL":
            return {
                "tooltip": "Atomic bit clear on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise AND with the complement of the value held in a pair of registers on it, and stores the result back to memory. The value initially loaded from memory is returned in the same pair of registers.",
                "html": "<p>Atomic bit clear on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise AND with the complement of the value held in a pair of registers on it, and stores the result back to memory. The value initially loaded from memory is returned in the same pair of registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDEOR":
        case "LDEORA":
        case "LDEORAL":
        case "LDEORL":
            return {
                "tooltip": "Atomic Exclusive-OR on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic Exclusive-OR on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDEORAB":
        case "LDEORALB":
        case "LDEORB":
        case "LDEORLB":
            return {
                "tooltip": "Atomic Exclusive-OR on byte in memory atomically loads an 8-bit byte from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic Exclusive-OR on byte in memory atomically loads an 8-bit byte from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDEORAH":
        case "LDEORALH":
        case "LDEORH":
        case "LDEORLH":
            return {
                "tooltip": "Atomic Exclusive-OR on halfword in memory atomically loads a 16-bit halfword from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic Exclusive-OR on halfword in memory atomically loads a 16-bit halfword from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDFF1B":
            return {
                "tooltip": "Gather load with first-faulting behavior of unsigned bytes to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is in the range 0 to 31. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load with first-faulting behavior of unsigned bytes to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is in the range 0 to 31. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDFF1D":
            return {
                "tooltip": "Gather load with first-faulting behavior of doublewords to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 8 in the range 0 to 248. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load with first-faulting behavior of doublewords to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 8 in the range 0 to 248. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDFF1H":
            return {
                "tooltip": "Gather load with first-faulting behavior of unsigned halfwords to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 2 in the range 0 to 62. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load with first-faulting behavior of unsigned halfwords to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 2 in the range 0 to 62. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDFF1SB":
            return {
                "tooltip": "Gather load with first-faulting behavior of signed bytes to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is in the range 0 to 31. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load with first-faulting behavior of signed bytes to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is in the range 0 to 31. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDFF1SH":
            return {
                "tooltip": "Gather load with first-faulting behavior of signed halfwords to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 2 in the range 0 to 62. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load with first-faulting behavior of signed halfwords to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 2 in the range 0 to 62. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDFF1SW":
            return {
                "tooltip": "Gather load with first-faulting behavior of signed words to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 4 in the range 0 to 124. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load with first-faulting behavior of signed words to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 4 in the range 0 to 124. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDFF1W":
            return {
                "tooltip": "Gather load with first-faulting behavior of unsigned words to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 4 in the range 0 to 124. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load with first-faulting behavior of unsigned words to active elements of a vector register from memory addresses generated by a vector base plus immediate index. The index is a multiple of 4 in the range 0 to 124. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDG":
            return {
                "tooltip": "Load Allocation Tag loads an Allocation Tag from a memory address, generates a Logical Address Tag from the Allocation Tag and merges it into the destination register. The address used for the load is calculated from the base register and an immediate signed offset scaled by the Tag granule.",
                "html": "<p>Load Allocation Tag loads an Allocation Tag from a memory address, generates a Logical Address Tag from the Allocation Tag and merges it into the destination register. The address used for the load is calculated from the base register and an immediate signed offset scaled by the Tag granule.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDGM":
            return {
                "tooltip": "Load Tag Multiple reads a naturally aligned block of N Allocation Tags, where the size of N is identified in GMID_EL1.BS, and writes the Allocation Tag read from address A to the destination register at 4*A<7:4>+3:4*A<7:4>. Bits of the destination register not written with an Allocation Tag are set to 0.",
                "html": "<p>Load Tag Multiple reads a naturally aligned block of N Allocation Tags, where the size of N is identified in GMID_EL1.BS, and writes the Allocation Tag read from address A to the destination register at 4*A&lt;7:4&gt;+3:4*A&lt;7:4&gt;. Bits of the destination register not written with an Allocation Tag are set to 0.</p><p>This instruction is <arm-defined-word>undefined</arm-defined-word> at EL0.</p><p>This instruction generates an Unchecked access.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDIAPP":
            return {
                "tooltip": "Load-Acquire RCpc ordered Pair of registers calculates an address from a base register value and an optional offset, loads two 32-bit words or two 64-bit doublewords from memory, and writes them to two registers. For information on single-copy atomicity and alignment requirements, see Requirements for single-copy atomicity and Alignment of data accesses. The instruction also has memory ordering semantics, as described in Load-Acquire, Load-AcquirePC, and Store-Release, except that",
                "html": "<p>Load-Acquire RCpc ordered Pair of registers calculates an address from a base register value and an optional offset, loads two 32-bit words or two 64-bit doublewords from memory, and writes them to two registers. For information on single-copy atomicity and alignment requirements, see <xref linkend=\"CHDDCBCC\">Requirements for single-copy atomicity</xref> and <xref linkend=\"CHDFFEGJ\">Alignment of data accesses</xref>. The instruction also has memory ordering semantics, as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, except that:</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDLAR":
            return {
                "tooltip": "Load LOAcquire Register loads a 32-bit word or 64-bit doubleword from memory, and writes it to a register. The instruction also has memory ordering semantics as described in Load LOAcquire, Store LORelease. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load LOAcquire Register loads a 32-bit word or 64-bit doubleword from memory, and writes it to a register. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIJDIJG\">Load LOAcquire, Store LORelease</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>For this instruction, if the destination is WZR/XZR, it is impossible for software to observe the presence of the acquire semantic other than its effect on the arrival at endpoints.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDLARB":
            return {
                "tooltip": "Load LOAcquire Register Byte loads a byte from memory, zero-extends it and writes it to a register. The instruction also has memory ordering semantics as described in Load LOAcquire, Store LORelease. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load LOAcquire Register Byte loads a byte from memory, zero-extends it and writes it to a register. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIJDIJG\">Load LOAcquire, Store LORelease</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>For this instruction, if the destination is WZR/XZR, it is impossible for software to observe the presence of the acquire semantic other than its effect on the arrival at endpoints.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDLARH":
            return {
                "tooltip": "Load LOAcquire Register Halfword loads a halfword from memory, zero-extends it, and writes it to a register. The instruction also has memory ordering semantics as described in Load LOAcquire, Store LORelease. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load LOAcquire Register Halfword loads a halfword from memory, zero-extends it, and writes it to a register. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIJDIJG\">Load LOAcquire, Store LORelease</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>For this instruction, if the destination is WZR/XZR, it is impossible for software to observe the presence of the acquire semantic other than its effect on the arrival at endpoints.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNF1B":
            return {
                "tooltip": "Contiguous load with non-faulting behavior of unsigned bytes to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.",
                "html": "<p>Contiguous load with non-faulting behavior of unsigned bytes to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNF1D":
            return {
                "tooltip": "Contiguous load with non-faulting behavior of doublewords to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.",
                "html": "<p>Contiguous load with non-faulting behavior of doublewords to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNF1H":
            return {
                "tooltip": "Contiguous load with non-faulting behavior of unsigned halfwords to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.",
                "html": "<p>Contiguous load with non-faulting behavior of unsigned halfwords to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNF1SB":
            return {
                "tooltip": "Contiguous load with non-faulting behavior of signed bytes to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.",
                "html": "<p>Contiguous load with non-faulting behavior of signed bytes to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNF1SH":
            return {
                "tooltip": "Contiguous load with non-faulting behavior of signed halfwords to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.",
                "html": "<p>Contiguous load with non-faulting behavior of signed halfwords to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNF1SW":
            return {
                "tooltip": "Contiguous load with non-faulting behavior of signed words to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.",
                "html": "<p>Contiguous load with non-faulting behavior of signed words to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNF1W":
            return {
                "tooltip": "Contiguous load with non-faulting behavior of unsigned words to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.",
                "html": "<p>Contiguous load with non-faulting behavior of unsigned words to elements of a vector register from the memory address generated by a 64-bit scalar base and immediate index in the range -8 to 7 which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address. Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNP":
            return {
                "tooltip": "Load Pair of SIMD&FP registers, with Non-temporal hint. This instruction loads a pair of SIMD&FP registers from memory, issuing a hint to the memory system that the access is non-temporal. The address that is used for the load is calculated from a base register value and an optional immediate offset.",
                "html": "<p>Load Pair of SIMD&amp;FP registers, with Non-temporal hint. This instruction loads a pair of SIMD&amp;FP registers from memory, issuing a hint to the memory system that the access is non-temporal. The address that is used for the load is calculated from a base register value and an optional immediate offset.</p><p>For information about non-temporal pair instructions, see <xref linkend=\"BABJADHH\">Load/Store SIMD and Floating-point Non-temporal pair</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNT1B":
            return {
                "tooltip": "Contiguous load non-temporal of bytes to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous load non-temporal of bytes to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p><p>A non-temporal load is a hint to the system that this data is unlikely to be referenced again soon.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNT1D":
            return {
                "tooltip": "Contiguous load non-temporal of doublewords to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous load non-temporal of doublewords to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p><p>A non-temporal load is a hint to the system that this data is unlikely to be referenced again soon.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNT1H":
            return {
                "tooltip": "Contiguous load non-temporal of halfwords to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous load non-temporal of halfwords to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p><p>A non-temporal load is a hint to the system that this data is unlikely to be referenced again soon.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNT1SB":
            return {
                "tooltip": "Gather load non-temporal of signed bytes to active elements of a vector register from memory addresses generated by a vector base plus a 64-bit unscaled scalar register offset. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load non-temporal of signed bytes to active elements of a vector register from memory addresses generated by a vector base plus a 64-bit unscaled scalar register offset. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>A non-temporal load is a hint to the system that this data is unlikely to be referenced again soon.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNT1SH":
            return {
                "tooltip": "Gather load non-temporal of signed halfwords to active elements of a vector register from memory addresses generated by a vector base plus a 64-bit unscaled scalar register offset. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load non-temporal of signed halfwords to active elements of a vector register from memory addresses generated by a vector base plus a 64-bit unscaled scalar register offset. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>A non-temporal load is a hint to the system that this data is unlikely to be referenced again soon.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNT1SW":
            return {
                "tooltip": "Gather load non-temporal of signed words to active elements of a vector register from memory addresses generated by a vector base plus a 64-bit unscaled scalar register offset. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.",
                "html": "<p>Gather load non-temporal of signed words to active elements of a vector register from memory addresses generated by a vector base plus a 64-bit unscaled scalar register offset. Inactive elements will not cause a read from Device memory or signal faults, and are set to zero in the destination vector.</p><p>A non-temporal load is a hint to the system that this data is unlikely to be referenced again soon.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDNT1W":
            return {
                "tooltip": "Contiguous load non-temporal of words to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous load non-temporal of words to elements of two or four consecutive vector registers from the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements will not cause a read from Device memory or signal a fault, and are set to zero in the destination vector.</p><p>A non-temporal load is a hint to the system that this data is unlikely to be referenced again soon.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDP":
            return {
                "tooltip": "Load Pair of SIMD&FP registers. This instruction loads a pair of SIMD&FP registers from memory. The address that is used for the load is calculated from a base register value and an optional immediate offset.",
                "html": "<p>Load Pair of SIMD&amp;FP registers. This instruction loads a pair of SIMD&amp;FP registers from memory. The address that is used for the load is calculated from a base register value and an optional immediate offset.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDPSW":
            return {
                "tooltip": "Load Pair of Registers Signed Word calculates an address from a base register value and an immediate offset, loads two 32-bit words from memory, sign-extends them, and writes them to two registers. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Pair of Registers Signed Word calculates an address from a base register value and an immediate offset, loads two 32-bit words from memory, sign-extends them, and writes them to two registers. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDR":
            return {
                "tooltip": "Load SIMD&FP Register (immediate offset). This instruction loads an element from memory, and writes the result as a scalar to the SIMD&FP register. The address that is used for the load is calculated from a base register value, a signed immediate offset, and an optional offset that is a multiple of the element size.",
                "html": "<p>Load SIMD&amp;FP Register (immediate offset). This instruction loads an element from memory, and writes the result as a scalar to the SIMD&amp;FP register. The address that is used for the load is calculated from a base register value, a signed immediate offset, and an optional offset that is a multiple of the element size.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDRAA":
        case "LDRAB":
            return {
                "tooltip": "Load Register, with pointer authentication. This instruction authenticates an address from a base register using a modifier of zero and the specified key, adds an immediate offset to the authenticated address, and loads a 64-bit doubleword from memory at this resulting address into a register.",
                "html": "<p>Load Register, with pointer authentication. This instruction authenticates an address from a base register using a modifier of zero and the specified key, adds an immediate offset to the authenticated address, and loads a 64-bit doubleword from memory at this resulting address into a register.</p><p>Key A is used for <instruction>LDRAA</instruction>. Key B is used for <instruction>LDRAB</instruction>.</p><p>If the authentication passes, the PE behaves the same as for an <instruction>LDR</instruction> instruction. For information on behavior if the authentication fails, see <xref>Faulting on pointer authentication</xref>.</p><p>The authenticated address is not written back to the base register, unless the pre-indexed variant of the instruction is used. In this case, the address that is written back to the base register does not include the pointer authentication code.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDRB":
            return {
                "tooltip": "Load Register Byte (immediate) loads a byte from memory, zero-extends it, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Register Byte (immediate) loads a byte from memory, zero-extends it, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDRH":
            return {
                "tooltip": "Load Register Halfword (immediate) loads a halfword from memory, zero-extends it, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Register Halfword (immediate) loads a halfword from memory, zero-extends it, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDRSB":
            return {
                "tooltip": "Load Register Signed Byte (immediate) loads a byte from memory, sign-extends it to either 32 bits or 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Register Signed Byte (immediate) loads a byte from memory, sign-extends it to either 32 bits or 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDRSH":
            return {
                "tooltip": "Load Register Signed Halfword (immediate) loads a halfword from memory, sign-extends it to 32 bits or 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Register Signed Halfword (immediate) loads a halfword from memory, sign-extends it to 32 bits or 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDRSW":
            return {
                "tooltip": "Load Register Signed Word (immediate) loads a word from memory, sign-extends it to 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Register Signed Word (immediate) loads a word from memory, sign-extends it to 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDSET":
        case "LDSETA":
        case "LDSETAL":
        case "LDSETL":
            return {
                "tooltip": "Atomic bit set on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic bit set on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDSETAB":
        case "LDSETALB":
        case "LDSETB":
        case "LDSETLB":
            return {
                "tooltip": "Atomic bit set on byte in memory atomically loads an 8-bit byte from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic bit set on byte in memory atomically loads an 8-bit byte from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDSETAH":
        case "LDSETALH":
        case "LDSETH":
        case "LDSETLH":
            return {
                "tooltip": "Atomic bit set on halfword in memory atomically loads a 16-bit halfword from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic bit set on halfword in memory atomically loads a 16-bit halfword from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDSETP":
        case "LDSETPA":
        case "LDSETPAL":
        case "LDSETPL":
            return {
                "tooltip": "Atomic bit set on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise OR with the value held in a pair of registers on it, and stores the result back to memory. The value initially loaded from memory is returned in the same pair of registers.",
                "html": "<p>Atomic bit set on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise OR with the value held in a pair of registers on it, and stores the result back to memory. The value initially loaded from memory is returned in the same pair of registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDSMAX":
        case "LDSMAXA":
        case "LDSMAXAL":
        case "LDSMAXL":
            return {
                "tooltip": "Atomic signed maximum on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic signed maximum on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDSMAXAB":
        case "LDSMAXALB":
        case "LDSMAXB":
        case "LDSMAXLB":
            return {
                "tooltip": "Atomic signed maximum on byte in memory atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic signed maximum on byte in memory atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDSMAXAH":
        case "LDSMAXALH":
        case "LDSMAXH":
        case "LDSMAXLH":
            return {
                "tooltip": "Atomic signed maximum on halfword in memory atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic signed maximum on halfword in memory atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDSMIN":
        case "LDSMINA":
        case "LDSMINAL":
        case "LDSMINL":
            return {
                "tooltip": "Atomic signed minimum on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic signed minimum on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDSMINAB":
        case "LDSMINALB":
        case "LDSMINB":
        case "LDSMINLB":
            return {
                "tooltip": "Atomic signed minimum on byte in memory atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic signed minimum on byte in memory atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDSMINAH":
        case "LDSMINALH":
        case "LDSMINH":
        case "LDSMINLH":
            return {
                "tooltip": "Atomic signed minimum on halfword in memory atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic signed minimum on halfword in memory atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDTR":
            return {
                "tooltip": "Load Register (unprivileged) loads a word or doubleword from memory, and writes it to a register. The address that is used for the load is calculated from a base register and an immediate offset.",
                "html": "<p>Load Register (unprivileged) loads a word or doubleword from memory, and writes it to a register. The address that is used for the load is calculated from a base register and an immediate offset.</p><p>Memory accesses made by the instruction behave as if the instruction was executed at EL0 if the <xref linkend=\"CBAICCHA\">Effective value</xref> of PSTATE.UAO is 0 and either:</p><p>Otherwise, the memory access operates with the restrictions determined by the Exception level at which the instruction is executed. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDTRB":
            return {
                "tooltip": "Load Register Byte (unprivileged) loads a byte from memory, zero-extends it, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset.",
                "html": "<p>Load Register Byte (unprivileged) loads a byte from memory, zero-extends it, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset.</p><p>Memory accesses made by the instruction behave as if the instruction was executed at EL0 if the <xref linkend=\"CBAICCHA\">Effective value</xref> of PSTATE.UAO is 0 and either:</p><p>Otherwise, the memory access operates with the restrictions determined by the Exception level at which the instruction is executed. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDTRH":
            return {
                "tooltip": "Load Register Halfword (unprivileged) loads a halfword from memory, zero-extends it, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset.",
                "html": "<p>Load Register Halfword (unprivileged) loads a halfword from memory, zero-extends it, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset.</p><p>Memory accesses made by the instruction behave as if the instruction was executed at EL0 if the <xref linkend=\"CBAICCHA\">Effective value</xref> of PSTATE.UAO is 0 and either:</p><p>Otherwise, the memory access operates with the restrictions determined by the Exception level at which the instruction is executed. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDTRSB":
            return {
                "tooltip": "Load Register Signed Byte (unprivileged) loads a byte from memory, sign-extends it to 32 bits or 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset.",
                "html": "<p>Load Register Signed Byte (unprivileged) loads a byte from memory, sign-extends it to 32 bits or 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset.</p><p>Memory accesses made by the instruction behave as if the instruction was executed at EL0 if the <xref linkend=\"CBAICCHA\">Effective value</xref> of PSTATE.UAO is 0 and either:</p><p>Otherwise, the memory access operates with the restrictions determined by the Exception level at which the instruction is executed. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDTRSH":
            return {
                "tooltip": "Load Register Signed Halfword (unprivileged) loads a halfword from memory, sign-extends it to 32 bits or 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset.",
                "html": "<p>Load Register Signed Halfword (unprivileged) loads a halfword from memory, sign-extends it to 32 bits or 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset.</p><p>Memory accesses made by the instruction behave as if the instruction was executed at EL0 if the <xref linkend=\"CBAICCHA\">Effective value</xref> of PSTATE.UAO is 0 and either:</p><p>Otherwise, the memory access operates with the restrictions determined by the Exception level at which the instruction is executed. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDTRSW":
            return {
                "tooltip": "Load Register Signed Word (unprivileged) loads a word from memory, sign-extends it to 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset.",
                "html": "<p>Load Register Signed Word (unprivileged) loads a word from memory, sign-extends it to 64 bits, and writes the result to a register. The address that is used for the load is calculated from a base register and an immediate offset.</p><p>Memory accesses made by the instruction behave as if the instruction was executed at EL0 if the <xref linkend=\"CBAICCHA\">Effective value</xref> of PSTATE.UAO is 0 and either:</p><p>Otherwise, the memory access operates with the restrictions determined by the Exception level at which the instruction is executed. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDUMAX":
        case "LDUMAXA":
        case "LDUMAXAL":
        case "LDUMAXL":
            return {
                "tooltip": "Atomic unsigned maximum on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic unsigned maximum on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDUMAXAB":
        case "LDUMAXALB":
        case "LDUMAXB":
        case "LDUMAXLB":
            return {
                "tooltip": "Atomic unsigned maximum on byte in memory atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic unsigned maximum on byte in memory atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDUMAXAH":
        case "LDUMAXALH":
        case "LDUMAXH":
        case "LDUMAXLH":
            return {
                "tooltip": "Atomic unsigned maximum on halfword in memory atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic unsigned maximum on halfword in memory atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDUMIN":
        case "LDUMINA":
        case "LDUMINAL":
        case "LDUMINL":
            return {
                "tooltip": "Atomic unsigned minimum on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic unsigned minimum on word or doubleword in memory atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDUMINAB":
        case "LDUMINALB":
        case "LDUMINB":
        case "LDUMINLB":
            return {
                "tooltip": "Atomic unsigned minimum on byte in memory atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic unsigned minimum on byte in memory atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDUMINAH":
        case "LDUMINALH":
        case "LDUMINH":
        case "LDUMINLH":
            return {
                "tooltip": "Atomic unsigned minimum on halfword in memory atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.",
                "html": "<p>Atomic unsigned minimum on halfword in memory atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers. The value initially loaded from memory is returned in the destination register.</p><p>For more information about memory ordering semantics, see <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDUR":
            return {
                "tooltip": "Load SIMD&FP Register (unscaled offset). This instruction loads a SIMD&FP register from memory. The address that is used for the load is calculated from a base register value and an optional immediate offset.",
                "html": "<p>Load SIMD&amp;FP Register (unscaled offset). This instruction loads a SIMD&amp;FP register from memory. The address that is used for the load is calculated from a base register value and an optional immediate offset.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDURB":
            return {
                "tooltip": "Load Register Byte (unscaled) calculates an address from a base register and an immediate offset, loads a byte from memory, zero-extends it, and writes it to a register. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Register Byte (unscaled) calculates an address from a base register and an immediate offset, loads a byte from memory, zero-extends it, and writes it to a register. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDURH":
            return {
                "tooltip": "Load Register Halfword (unscaled) calculates an address from a base register and an immediate offset, loads a halfword from memory, zero-extends it, and writes it to a register. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Register Halfword (unscaled) calculates an address from a base register and an immediate offset, loads a halfword from memory, zero-extends it, and writes it to a register. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDURSB":
            return {
                "tooltip": "Load Register Signed Byte (unscaled) calculates an address from a base register and an immediate offset, loads a signed byte from memory, sign-extends it, and writes it to a register. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Register Signed Byte (unscaled) calculates an address from a base register and an immediate offset, loads a signed byte from memory, sign-extends it, and writes it to a register. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDURSH":
            return {
                "tooltip": "Load Register Signed Halfword (unscaled) calculates an address from a base register and an immediate offset, loads a signed halfword from memory, sign-extends it, and writes it to a register. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Register Signed Halfword (unscaled) calculates an address from a base register and an immediate offset, loads a signed halfword from memory, sign-extends it, and writes it to a register. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDURSW":
            return {
                "tooltip": "Load Register Signed Word (unscaled) calculates an address from a base register and an immediate offset, loads a signed word from memory, sign-extends it, and writes it to a register. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Register Signed Word (unscaled) calculates an address from a base register and an immediate offset, loads a signed word from memory, sign-extends it, and writes it to a register. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDXP":
            return {
                "tooltip": "Load Exclusive Pair of Registers derives an address from a base register value, loads two 32-bit words or two 64-bit doublewords from memory, and writes them to two registers. For information on single-copy atomicity and alignment requirements, see Requirements for single-copy atomicity and Alignment of data accesses. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See Synchronization and semaphores. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Exclusive Pair of Registers derives an address from a base register value, loads two 32-bit words or two 64-bit doublewords from memory, and writes them to two registers. For information on single-copy atomicity and alignment requirements, see <xref linkend=\"CHDDCBCC\">Requirements for single-copy atomicity</xref> and <xref linkend=\"CHDFFEGJ\">Alignment of data accesses</xref>. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDXR":
            return {
                "tooltip": "Load Exclusive Register derives an address from a base register value, loads a 32-bit word or a 64-bit doubleword from memory, and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See Synchronization and semaphores. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Exclusive Register derives an address from a base register value, loads a 32-bit word or a 64-bit doubleword from memory, and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDXRB":
            return {
                "tooltip": "Load Exclusive Register Byte derives an address from a base register value, loads a byte from memory, zero-extends it and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See Synchronization and semaphores. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Exclusive Register Byte derives an address from a base register value, loads a byte from memory, zero-extends it and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LDXRH":
            return {
                "tooltip": "Load Exclusive Register Halfword derives an address from a base register value, loads a halfword from memory, zero-extends it and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See Synchronization and semaphores. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Load Exclusive Register Halfword derives an address from a base register value, loads a halfword from memory, zero-extends it and writes it to a register. The memory access is atomic. The PE marks the physical address being accessed as an exclusive access. This exclusive access mark is checked by Store Exclusive instructions. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LSL":
            return {
                "tooltip": "Logical Shift Left (register) shifts a register value left by a variable number of bits, shifting in zeros, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is left-shifted.",
                "html": "<p>Logical Shift Left (register) shifts a register value left by a variable number of bits, shifting in zeros, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is left-shifted.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LSLR":
            return {
                "tooltip": "Reversed shift left active elements of the second source vector by corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. The shift amount operand is a vector of unsigned elements in which all bits are significant, and not used modulo the element size. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Reversed shift left active elements of the second source vector by corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. The shift amount operand is a vector of unsigned elements in which all bits are significant, and not used modulo the element size. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LSLV":
            return {
                "tooltip": "Logical Shift Left Variable shifts a register value left by a variable number of bits, shifting in zeros, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is left-shifted.",
                "html": "<p>Logical Shift Left Variable shifts a register value left by a variable number of bits, shifting in zeros, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is left-shifted.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LSR":
            return {
                "tooltip": "Logical Shift Right (register) shifts a register value right by a variable number of bits, shifting in zeros, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is right-shifted.",
                "html": "<p>Logical Shift Right (register) shifts a register value right by a variable number of bits, shifting in zeros, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is right-shifted.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LSRR":
            return {
                "tooltip": "Reversed shift right, inserting zeroes, active elements of the second source vector by corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. The shift amount operand is a vector of unsigned elements in which all bits are significant, and not used modulo the element size. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Reversed shift right, inserting zeroes, active elements of the second source vector by corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. The shift amount operand is a vector of unsigned elements in which all bits are significant, and not used modulo the element size. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LSRV":
            return {
                "tooltip": "Logical Shift Right Variable shifts a register value right by a variable number of bits, shifting in zeros, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is right-shifted.",
                "html": "<p>Logical Shift Right Variable shifts a register value right by a variable number of bits, shifting in zeros, and writes the result to the destination register. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is right-shifted.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LUTI2":
            return {
                "tooltip": "Copy 8-bit, 16-bit or 32-bit elements from ZT0 to two destination vectors using packed 2-bit indices from a segment of the source vector register. A segment corresponds to a portion of the source vector that is consumed in order to fill the destination vector. The segment is selected by the vector segment index modulo the total number of segments.",
                "html": "<p>Copy 8-bit, 16-bit or 32-bit elements from ZT0 to two destination vectors using packed 2-bit indices from a segment of the source vector register. A segment corresponds to a portion of the source vector that is consumed in order to fill the destination vector. The segment is selected by the vector segment index modulo the total number of segments.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "LUTI4":
            return {
                "tooltip": "Copy 8-bit, 16-bit or 32-bit elements from ZT0 to two destination vectors using packed 4-bit indices from a segment of the source vector register. A segment corresponds to a portion of the source vector that is consumed in order to fill the destination vector. The segment is selected by the vector segment index modulo the total number of segments.",
                "html": "<p>Copy 8-bit, 16-bit or 32-bit elements from ZT0 to two destination vectors using packed 4-bit indices from a segment of the source vector register. A segment corresponds to a portion of the source vector that is consumed in order to fill the destination vector. The segment is selected by the vector segment index modulo the total number of segments.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MAD":
            return {
                "tooltip": "Multiply the corresponding active elements of the first and second source vectors and add to elements of the third (addend) vector. Destructively place the results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply the corresponding active elements of the first and second source vectors and add to elements of the third (addend) vector. Destructively place the results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MADD":
            return {
                "tooltip": "Multiply-Add multiplies two register values, adds a third register value, and writes the result to the destination register.",
                "html": "<p>Multiply-Add multiplies two register values, adds a third register value, and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MATCH":
            return {
                "tooltip": "This instruction compares each active 8-bit or 16-bit character in the first source vector with all of the characters in the corresponding 128-bit segment of the second source vector. Where the first source element detects any matching characters in the second segment it places true in the corresponding element of the destination predicate, otherwise false. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>This instruction compares each active 8-bit or 16-bit character in the first source vector with all of the characters in the corresponding 128-bit segment of the second source vector. Where the first source element detects any matching characters in the second segment it places true in the corresponding element of the destination predicate, otherwise false. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MLA":
            return {
                "tooltip": "Multiply-Add to accumulator (vector, by element). This instruction multiplies the vector elements in the first source SIMD&FP register by the specified value in the second source SIMD&FP register, and accumulates the results with the vector elements of the destination SIMD&FP register. All the values in this instruction are unsigned integer values.",
                "html": "<p>Multiply-Add to accumulator (vector, by element). This instruction multiplies the vector elements in the first source SIMD&amp;FP register by the specified value in the second source SIMD&amp;FP register, and accumulates the results with the vector elements of the destination SIMD&amp;FP register. All the values in this instruction are unsigned integer values.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MLS":
            return {
                "tooltip": "Multiply-Subtract from accumulator (vector, by element). This instruction multiplies the vector elements in the first source SIMD&FP register by the specified value in the second source SIMD&FP register, and subtracts the results from the vector elements of the destination SIMD&FP register. All the values in this instruction are unsigned integer values.",
                "html": "<p>Multiply-Subtract from accumulator (vector, by element). This instruction multiplies the vector elements in the first source SIMD&amp;FP register by the specified value in the second source SIMD&amp;FP register, and subtracts the results from the vector elements of the destination SIMD&amp;FP register. All the values in this instruction are unsigned integer values.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MNEG":
            return {
                "tooltip": "Multiply-Negate multiplies two register values, negates the product, and writes the result to the destination register.",
                "html": "<p>Multiply-Negate multiplies two register values, negates the product, and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MOV":
            return {
                "tooltip": "Read active elements from the source predicate and place in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.",
                "html": "<p>Read active elements from the source predicate and place in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MOVA":
            return {
                "tooltip": "The instruction operates on two consecutive horizontal or vertical slices within a named ZA tile of the specified element size.",
                "html": "<p>The instruction operates on two consecutive horizontal or vertical slices within a named ZA tile of the specified element size.</p><p>The consecutive slice numbers within the tile are selected starting from the sum of the slice index register and immediate offset, modulo the number of such elements in a vector. The immediate offset is a multiple of 2 in the range 0 to the number of elements in a 128-bit vector segment minus 2.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MOVAZ":
            return {
                "tooltip": "The instruction operates on two consecutive horizontal or vertical slices within a named ZA tile of the specified element size. The tile slices are zeroed after moving their contents to the destination vectors.",
                "html": "<p>The instruction operates on two consecutive horizontal or vertical slices within a named ZA tile of the specified element size. The tile slices are zeroed after moving their contents to the destination vectors.</p><p>The consecutive slice numbers within the tile are selected starting from the sum of the slice index register and immediate offset, modulo the number of such elements in a vector. The immediate offset is a multiple of 2 in the range 0 to the number of elements in a 128-bit vector segment minus 2.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MOVI":
            return {
                "tooltip": "Move Immediate (vector). This instruction places an immediate constant into every vector element of the destination SIMD&FP register.",
                "html": "<p>Move Immediate (vector). This instruction places an immediate constant into every vector element of the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MOVK":
            return {
                "tooltip": "Move wide with keep moves an optionally-shifted 16-bit immediate value into a register, keeping other bits unchanged.",
                "html": "<p>Move wide with keep moves an optionally-shifted 16-bit immediate value into a register, keeping other bits unchanged.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MOVN":
            return {
                "tooltip": "Move wide with NOT moves the inverse of an optionally-shifted 16-bit immediate value to a register.",
                "html": "<p>Move wide with NOT moves the inverse of an optionally-shifted 16-bit immediate value to a register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MOVPRFX":
            return {
                "tooltip": "The predicated MOVPRFX instruction is a hint to hardware that the instruction may be combined with the destructive instruction which follows it in program order to create a single constructive operation. Since it is a hint it is also permitted to be implemented as a discrete vector copy, and the result of executing the pair of instructions with or without combining is identical. The choice of combined versus discrete operation may vary dynamically.",
                "html": "<p>The predicated <instruction>MOVPRFX</instruction> instruction is a hint to hardware that the instruction may be combined with the destructive instruction which follows it in program order to create a single constructive operation. Since it is a hint it is also permitted to be implemented as a discrete vector copy, and the result of executing the pair of instructions with or without combining is identical. The choice of combined versus discrete operation may vary dynamically.</p><p>Unless the combination of a constructive operation with merging predication is specifically required, it is strongly recommended that for performance reasons software should prefer to use the zeroing form of predicated <instruction>MOVPRFX</instruction> or the unpredicated <instruction>MOVPRFX</instruction> instruction.</p><p>Although the operation of the instruction is defined as a simple predicated vector copy, it is required that the prefixed instruction at PC+4 must be an SVE destructive binary or ternary instruction encoding, or a unary operation with merging predication, but excluding other <instruction>MOVPRFX</instruction> instructions. The prefixed instruction must specify the same predicate register, and have the same maximum element size (ignoring a fixed 64-bit \"wide vector\" operand), and the same destination vector as the <instruction>MOVPRFX</instruction> instruction. The prefixed instruction must not use the destination register in any other operand position, even if they have different names but refer to the same architectural register state. Any other use is UNPREDICTABLE.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MOVS":
            return {
                "tooltip": "Read active elements from the source predicate and place in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Read active elements from the source predicate and place in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MOVT":
            return {
                "tooltip": "Move 8 bytes to a general-purpose register from the ZT0 register at the byte offset specified by the immediate index. This instruction is UNDEFINED in Non-debug state.",
                "html": "<p>Move 8 bytes to a general-purpose register from the ZT0 register at the byte offset specified by the immediate index. This instruction is UNDEFINED in Non-debug state.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MOVZ":
            return {
                "tooltip": "Move wide with zero moves an optionally-shifted 16-bit immediate value to a register.",
                "html": "<p>Move wide with zero moves an optionally-shifted 16-bit immediate value to a register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MRRS":
            return {
                "tooltip": "Move System Register to two adjacent general-purpose registers allows the PE to read an AArch64 128-bit System register into two adjacent 64-bit general-purpose registers.",
                "html": "<p>Move System Register to two adjacent general-purpose registers allows the PE to read an AArch64 128-bit System register into two adjacent 64-bit general-purpose registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MRS":
            return {
                "tooltip": "Move System Register to general-purpose register allows the PE to read an AArch64 System register into a general-purpose register.",
                "html": "<p>Move System Register to general-purpose register allows the PE to read an AArch64 System register into a general-purpose register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MSB":
            return {
                "tooltip": "Multiply the corresponding active elements of the first and second source vectors and subtract from elements of the third (addend) vector. Destructively place the results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Multiply the corresponding active elements of the first and second source vectors and subtract from elements of the third (addend) vector. Destructively place the results in the destination and first source (multiplicand) vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MSR":
            return {
                "tooltip": "Move immediate value to Special Register moves an immediate value to selected bits of the PSTATE. For more information, see Process state, PSTATE.",
                "html": "<p>Move immediate value to Special Register moves an immediate value to selected bits of the PSTATE. For more information, see <xref linkend=\"BEIDIGBH\">Process state, PSTATE</xref>.</p><p>The bits that can be written by this instruction are:</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MSRR":
            return {
                "tooltip": "Move two adjacent general-purpose registers to System Register allows the PE to write an AArch64 128-bit System register from two adjacent 64-bit general-purpose registers.",
                "html": "<p>Move two adjacent general-purpose registers to System Register allows the PE to write an AArch64 128-bit System register from two adjacent 64-bit general-purpose registers.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MSUB":
            return {
                "tooltip": "Multiply-Subtract multiplies two register values, subtracts the product from a third register value, and writes the result to the destination register.",
                "html": "<p>Multiply-Subtract multiplies two register values, subtracts the product from a third register value, and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MUL":
            return {
                "tooltip": "Multiply (vector, by element). This instruction multiplies the vector elements in the first source SIMD&FP register by the specified value in the second source SIMD&FP register, places the results in a vector, and writes the vector to the destination SIMD&FP register. All the values in this instruction are unsigned integer values.",
                "html": "<p>Multiply (vector, by element). This instruction multiplies the vector elements in the first source SIMD&amp;FP register by the specified value in the second source SIMD&amp;FP register, places the results in a vector, and writes the vector to the destination SIMD&amp;FP register. All the values in this instruction are unsigned integer values.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MVN":
            return {
                "tooltip": "Bitwise NOT (vector). This instruction reads each vector element from the source SIMD&FP register, places the inverse of each value into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Bitwise NOT (vector). This instruction reads each vector element from the source SIMD&amp;FP register, places the inverse of each value into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "MVNI":
            return {
                "tooltip": "Move inverted Immediate (vector). This instruction places the inverse of an immediate constant into every vector element of the destination SIMD&FP register.",
                "html": "<p>Move inverted Immediate (vector). This instruction places the inverse of an immediate constant into every vector element of the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NAND":
            return {
                "tooltip": "Bitwise NAND active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.",
                "html": "<p>Bitwise NAND active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NANDS":
            return {
                "tooltip": "Bitwise NAND active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Bitwise NAND active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NBSL":
            return {
                "tooltip": "Selects bits from the first source vector where the corresponding bit in the third source vector is '1', and from the second source vector where the corresponding bit in the third source vector is '0'. The inverted result is placed destructively in the destination and first source vector. This instruction is unpredicated.",
                "html": "<p>Selects bits from the first source vector where the corresponding bit in the third source vector is '1', and from the second source vector where the corresponding bit in the third source vector is '0'. The inverted result is placed destructively in the destination and first source vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NEG":
            return {
                "tooltip": "Negate (vector). This instruction reads each vector element from the source SIMD&FP register, negates each value, puts the result into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Negate (vector). This instruction reads each vector element from the source SIMD&amp;FP register, negates each value, puts the result into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NEGS":
            return {
                "tooltip": "Negate, setting flags, negates an optionally-shifted register value, and writes the result to the destination register. It updates the condition flags based on the result.",
                "html": "<p>Negate, setting flags, negates an optionally-shifted register value, and writes the result to the destination register. It updates the condition flags based on the result.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NGC":
            return {
                "tooltip": "Negate with Carry negates the sum of a register value and the value of NOT (Carry flag), and writes the result to the destination register.",
                "html": "<p>Negate with Carry negates the sum of a register value and the value of NOT (Carry flag), and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NGCS":
            return {
                "tooltip": "Negate with Carry, setting flags, negates the sum of a register value and the value of NOT (Carry flag), and writes the result to the destination register. It updates the condition flags based on the result.",
                "html": "<p>Negate with Carry, setting flags, negates the sum of a register value and the value of NOT (Carry flag), and writes the result to the destination register. It updates the condition flags based on the result.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NMATCH":
            return {
                "tooltip": "This instruction compares each active 8-bit or 16-bit character in the first source vector with all of the characters in the corresponding 128-bit segment of the second source vector. Where the first source element detects no matching characters in the second segment it places true in the corresponding element of the destination predicate, otherwise false. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>This instruction compares each active 8-bit or 16-bit character in the first source vector with all of the characters in the corresponding 128-bit segment of the second source vector. Where the first source element detects no matching characters in the second segment it places true in the corresponding element of the destination predicate, otherwise false. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NOP":
            return {
                "tooltip": "No Operation does nothing, other than advance the value of the program counter by 4. This instruction can be used for instruction alignment purposes.",
                "html": "<p>No Operation does nothing, other than advance the value of the program counter by 4. This instruction can be used for instruction alignment purposes.</p><p>The timing effects of including a <instruction>NOP</instruction> instruction in a program are not guaranteed. It can increase execution time, leave it unchanged, or even reduce it. Therefore, <instruction>NOP</instruction> instructions are not suitable for timing loops.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NOR":
            return {
                "tooltip": "Bitwise NOR active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.",
                "html": "<p>Bitwise NOR active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Does not set the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NORS":
            return {
                "tooltip": "Bitwise NOR active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Bitwise NOR active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NOT":
            return {
                "tooltip": "Bitwise NOT (vector). This instruction reads each vector element from the source SIMD&FP register, places the inverse of each value into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Bitwise NOT (vector). This instruction reads each vector element from the source SIMD&amp;FP register, places the inverse of each value into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "NOTS":
            return {
                "tooltip": "Bitwise invert each active element of the source predicate, and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Bitwise invert each active element of the source predicate, and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ORN":
            return {
                "tooltip": "Bitwise inclusive OR NOT (vector). This instruction performs a bitwise OR NOT between the two source SIMD&FP registers, and writes the result to the destination SIMD&FP register.",
                "html": "<p>Bitwise inclusive OR NOT (vector). This instruction performs a bitwise OR NOT between the two source SIMD&amp;FP registers, and writes the result to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ORNS":
            return {
                "tooltip": "Bitwise inclusive OR inverted active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Bitwise inclusive OR inverted active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ORQV":
            return {
                "tooltip": "Bitwise inclusive OR of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&FP destination register. Inactive elements in the source vector are treated as all zeros.",
                "html": "<p>Bitwise inclusive OR of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&amp;FP destination register. Inactive elements in the source vector are treated as all zeros.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ORR":
            return {
                "tooltip": "Bitwise inclusive OR (vector, immediate). This instruction reads each vector element from the destination SIMD&FP register, performs a bitwise OR between each result and an immediate constant, places the result into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Bitwise inclusive OR (vector, immediate). This instruction reads each vector element from the destination SIMD&amp;FP register, performs a bitwise OR between each result and an immediate constant, places the result into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ORRS":
            return {
                "tooltip": "Bitwise inclusive OR active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Bitwise inclusive OR active elements of the second source predicate with corresponding elements of the first source predicate and place the results in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ORV":
            return {
                "tooltip": "Bitwise inclusive OR horizontally across all lanes of a vector, and place the result in the SIMD&FP scalar destination register. Inactive elements in the source vector are treated as zero.",
                "html": "<p>Bitwise inclusive OR horizontally across all lanes of a vector, and place the result in the SIMD&amp;FP scalar destination register. Inactive elements in the source vector are treated as zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PACDA":
        case "PACDZA":
            return {
                "tooltip": "Pointer Authentication Code for Data address, using key A. This instruction computes and inserts a pointer authentication code for a data address, using a modifier and key A.",
                "html": "<p>Pointer Authentication Code for Data address, using key A. This instruction computes and inserts a pointer authentication code for a data address, using a modifier and key A.</p><p>The address is in the general-purpose register that is specified by <syntax>&lt;Xd&gt;</syntax>.</p><p>The modifier is:</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PACDB":
        case "PACDZB":
            return {
                "tooltip": "Pointer Authentication Code for Data address, using key B. This instruction computes and inserts a pointer authentication code for a data address, using a modifier and key B.",
                "html": "<p>Pointer Authentication Code for Data address, using key B. This instruction computes and inserts a pointer authentication code for a data address, using a modifier and key B.</p><p>The address is in the general-purpose register that is specified by <syntax>&lt;Xd&gt;</syntax>.</p><p>The modifier is:</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PACGA":
            return {
                "tooltip": "Pointer Authentication Code, using Generic key. This instruction computes the pointer authentication code for a 64-bit value in the first source register, using a modifier in the second source register, and the Generic key. The computed pointer authentication code is written to the most significant 32 bits of the destination register, and the least significant 32 bits of the destination register are set to zero.",
                "html": "<p>Pointer Authentication Code, using Generic key. This instruction computes the pointer authentication code for a 64-bit value in the first source register, using a modifier in the second source register, and the Generic key. The computed pointer authentication code is written to the most significant 32 bits of the destination register, and the least significant 32 bits of the destination register are set to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PACIA":
        case "PACIA1716":
        case "PACIASP":
        case "PACIAZ":
        case "PACIZA":
            return {
                "tooltip": "Pointer Authentication Code for Instruction address, using key A. This instruction computes and inserts a pointer authentication code for an instruction address, using a modifier and key A.",
                "html": "<p>Pointer Authentication Code for Instruction address, using key A. This instruction computes and inserts a pointer authentication code for an instruction address, using a modifier and key A.</p><p>The address is:</p><p>The modifier is:</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PACIB":
        case "PACIB1716":
        case "PACIBSP":
        case "PACIBZ":
        case "PACIZB":
            return {
                "tooltip": "Pointer Authentication Code for Instruction address, using key B. This instruction computes and inserts a pointer authentication code for an instruction address, using a modifier and key B.",
                "html": "<p>Pointer Authentication Code for Instruction address, using key B. This instruction computes and inserts a pointer authentication code for an instruction address, using a modifier and key B.</p><p>The address is:</p><p>The modifier is:</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PEXT":
            return {
                "tooltip": "Expands the source predicate-as-counter into a four-predicate wide mask and copies one quarter of it into the destination predicate register.",
                "html": "<p>Expands the source predicate-as-counter into a four-predicate wide mask and copies one quarter of it into the destination predicate register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PFALSE":
            return {
                "tooltip": "Set all elements in the destination predicate to false.",
                "html": "<p>Set all elements in the destination predicate to false.</p><p>For programmer convenience, an assembler must also accept predicate-as-counter register name for the destination predicate register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PFIRST":
            return {
                "tooltip": "Sets the first active element in the destination predicate to true, otherwise elements from the source predicate are passed through unchanged. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Sets the first active element in the destination predicate to true, otherwise elements from the source predicate are passed through unchanged. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PMOV":
            return {
                "tooltip": "Copy a packed bitmap, where bit value 0b1 represents TRUE and bit value 0b0 represents FALSE, from part of a source vector register to elements of a destination SVE predicate register.",
                "html": "<p>Copy a packed bitmap, where bit value 0b1 represents TRUE and bit value 0b0 represents FALSE, from part of a source vector register to elements of a destination SVE predicate register.</p><p>Because the number of bits in an SVE predicate element scales with the vector element size, the behavior varies according to the specified element size.</p><p>The immediate index is optional, defaulting to 0 if omitted.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PMUL":
            return {
                "tooltip": "Polynomial Multiply. This instruction multiplies corresponding elements in the vectors of the two source SIMD&FP registers, places the results in a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Polynomial Multiply. This instruction multiplies corresponding elements in the vectors of the two source SIMD&amp;FP registers, places the results in a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>For information about multiplying polynomials see <xref linkend=\"BABDGBIC\">Polynomial arithmetic over {0, 1}</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PMULL":
        case "PMULL2":
            return {
                "tooltip": "Polynomial Multiply Long. This instruction multiplies corresponding elements in the lower or upper half of the vectors of the two source SIMD&FP registers, places the results in a vector, and writes the vector to the destination SIMD&FP register. The destination vector elements are twice as long as the elements that are multiplied.",
                "html": "<p>Polynomial Multiply Long. This instruction multiplies corresponding elements in the lower or upper half of the vectors of the two source SIMD&amp;FP registers, places the results in a vector, and writes the vector to the destination SIMD&amp;FP register. The destination vector elements are twice as long as the elements that are multiplied.</p><p>For information about multiplying polynomials, see <xref linkend=\"BABDGBIC\">Polynomial arithmetic over {0, 1}</xref>.</p><p>The <instruction>PMULL</instruction> instruction extracts each source vector from the lower half of each source register. The <instruction>PMULL2</instruction> instruction extracts each source vector from the upper half of each source register.</p><p>The <instruction>PMULL</instruction> and <instruction>PMULL2</instruction> variants that operate on 64-bit source elements are defined only when FEAT_PMULL is implemented.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PMULLB":
            return {
                "tooltip": "Polynomial multiply over [0, 1] the corresponding even-numbered elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Polynomial multiply over [0, 1] the corresponding even-numbered elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p><p>ID_AA64ZFR0_EL1.AES indicates whether the 128-bit element variant is implemented. The 128-bit element variant is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PMULLT":
            return {
                "tooltip": "Polynomial multiply over [0, 1] the corresponding odd-numbered elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Polynomial multiply over [0, 1] the corresponding odd-numbered elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p><p>ID_AA64ZFR0_EL1.AES indicates whether the 128-bit element variant is implemented. The 128-bit element variant is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PNEXT":
            return {
                "tooltip": "An instruction used to construct a loop which iterates over all true elements in the vector select predicate register. If all elements in the first source predicate register are false it determines the first true element in the vector select predicate register, otherwise it determines the next true element in the vector select predicate register that follows the last true element in the first source predicate register. All elements of the destination predicate register are set to false, except the element corresponding to the determined vector select element, if any, which is set to true. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>An instruction used to construct a loop which iterates over all true elements in the vector select predicate register. If all elements in the first source predicate register are false it determines the first true element in the vector select predicate register, otherwise it determines the next true element in the vector select predicate register that follows the last true element in the first source predicate register. All elements of the destination predicate register are set to false, except the element corresponding to the determined vector select element, if any, which is set to true. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PRFB":
            return {
                "tooltip": "Gather prefetch of bytes from the active memory addresses generated by a vector base plus immediate index. The index is in the range 0 to 31. Inactive addresses are not prefetched from memory.",
                "html": "<p>Gather prefetch of bytes from the active memory addresses generated by a vector base plus immediate index. The index is in the range 0 to 31. Inactive addresses are not prefetched from memory.</p><p>The <syntax>&lt;prfop&gt;</syntax> symbol specifies the prefetch hint as a combination of three options: access type <value>PLD</value> for load or <value>PST</value> for store; target cache level <value>L1</value>, <value>L2</value> or <value>L3</value>; temporality (<value>KEEP</value> for temporal or <value>STRM</value> for non-temporal).</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PRFD":
            return {
                "tooltip": "Gather prefetch of doublewords from the active memory addresses generated by a vector base plus immediate index. The index is a multiple of 8 in the range 0 to 248. Inactive addresses are not prefetched from memory.",
                "html": "<p>Gather prefetch of doublewords from the active memory addresses generated by a vector base plus immediate index. The index is a multiple of 8 in the range 0 to 248. Inactive addresses are not prefetched from memory.</p><p>The <syntax>&lt;prfop&gt;</syntax> symbol specifies the prefetch hint as a combination of three options: access type <value>PLD</value> for load or <value>PST</value> for store; target cache level <value>L1</value>, <value>L2</value> or <value>L3</value>; temporality (<value>KEEP</value> for temporal or <value>STRM</value> for non-temporal).</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PRFH":
            return {
                "tooltip": "Gather prefetch of halfwords from the active memory addresses generated by a vector base plus immediate index. The index is a multiple of 2 in the range 0 to 62. Inactive addresses are not prefetched from memory.",
                "html": "<p>Gather prefetch of halfwords from the active memory addresses generated by a vector base plus immediate index. The index is a multiple of 2 in the range 0 to 62. Inactive addresses are not prefetched from memory.</p><p>The <syntax>&lt;prfop&gt;</syntax> symbol specifies the prefetch hint as a combination of three options: access type <value>PLD</value> for load or <value>PST</value> for store; target cache level <value>L1</value>, <value>L2</value> or <value>L3</value>; temporality (<value>KEEP</value> for temporal or <value>STRM</value> for non-temporal).</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PRFM":
            return {
                "tooltip": "Prefetch Memory (immediate) signals the memory system that data memory accesses from a specified address are likely to occur in the near future. The memory system can respond by taking actions that are expected to speed up the memory accesses when they do occur, such as preloading the cache line containing the specified address into one or more caches.",
                "html": "<p>Prefetch Memory (immediate) signals the memory system that data memory accesses from a specified address are likely to occur in the near future. The memory system can respond by taking actions that are expected to speed up the memory accesses when they do occur, such as preloading the cache line containing the specified address into one or more caches.</p><p>The effect of a <instruction>PRFM</instruction> instruction is <arm-defined-word>implementation defined</arm-defined-word>. For more information, see <xref linkend=\"CEGGGIDE\">Prefetch memory</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PRFUM":
            return {
                "tooltip": "Prefetch Memory (unscaled offset) signals the memory system that data memory accesses from a specified address are likely to occur in the near future. The memory system can respond by taking actions that are expected to speed up the memory accesses when they do occur, such as preloading the cache line containing the specified address into one or more caches.",
                "html": "<p>Prefetch Memory (unscaled offset) signals the memory system that data memory accesses from a specified address are likely to occur in the near future. The memory system can respond by taking actions that are expected to speed up the memory accesses when they do occur, such as preloading the cache line containing the specified address into one or more caches.</p><p>The effect of a <instruction>PRFUM</instruction> instruction is <arm-defined-word>implementation defined</arm-defined-word>. For more information, see <xref linkend=\"CEGGGIDE\">Prefetch memory</xref>.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PRFW":
            return {
                "tooltip": "Gather prefetch of words from the active memory addresses generated by a vector base plus immediate index. The index is a multiple of 4 in the range 0 to 124. Inactive addresses are not prefetched from memory.",
                "html": "<p>Gather prefetch of words from the active memory addresses generated by a vector base plus immediate index. The index is a multiple of 4 in the range 0 to 124. Inactive addresses are not prefetched from memory.</p><p>The <syntax>&lt;prfop&gt;</syntax> symbol specifies the prefetch hint as a combination of three options: access type <value>PLD</value> for load or <value>PST</value> for store; target cache level <value>L1</value>, <value>L2</value> or <value>L3</value>; temporality (<value>KEEP</value> for temporal or <value>STRM</value> for non-temporal).</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PSBCSYNC":
            return {
                "tooltip": "Profiling Synchronization Barrier. This instruction is a barrier that ensures that all existing profiling data for the current PE has been formatted, and profiling buffer addresses have been translated such that all writes to the profiling buffer have been initiated.  A following DSB instruction completes when the writes to the profiling buffer have completed.",
                "html": "<p>Profiling Synchronization Barrier. This instruction is a barrier that ensures that all existing profiling data for the current PE has been formatted, and profiling buffer addresses have been translated such that all writes to the profiling buffer have been initiated.  A following <instruction>DSB</instruction> instruction completes when the writes to the profiling buffer have completed.</p><p>If the Statistical Profiling Extension is not implemented, this instruction executes as a <instruction>NOP</instruction>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PSEL":
            return {
                "tooltip": "If the indexed element of the second source predicate is true, place the contents of the first source predicate register into the destination predicate register, otherwise set the destination predicate to all-false. The indexed element is determined by the sum of a general-purpose index register and an immediate, modulo the number of elements. Does not set the condition flags.",
                "html": "<p>If the indexed element of the second source predicate is true, place the contents of the first source predicate register into the destination predicate register, otherwise set the destination predicate to all-false. The indexed element is determined by the sum of a general-purpose index register and an immediate, modulo the number of elements. Does not set the condition flags.</p><p>For programmer convenience, an assembler must also accept predicate-as-counter register names for the destination predicate register and the first source predicate register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PSSBB":
            return {
                "tooltip": "Physical Speculative Store Bypass Barrier is a memory barrier that prevents speculative loads from bypassing earlier stores to the same physical address under certain conditions. For more information and details of the semantics, see Physical Speculative Store Bypass Barrier (PSSBB).",
                "html": "<p>Physical Speculative Store Bypass Barrier is a memory barrier that prevents speculative loads from bypassing earlier stores to the same physical address under certain conditions. For more information and details of the semantics, see <xref linkend=\"CHDECEBA\">Physical Speculative Store Bypass Barrier (PSSBB)</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PTEST":
            return {
                "tooltip": "Sets the First (N), None (Z), !Last (C) condition flags based on the predicate source register, and the V flag to zero.",
                "html": "<p>Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate source register, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PTRUE":
            return {
                "tooltip": "Set elements of the destination predicate to true if the element number satisfies the named predicate constraint, or to false otherwise. If the constraint specifies more elements than are available at the current vector length then all elements of the destination predicate are set to false.",
                "html": "<p>Set elements of the destination predicate to true if the element number satisfies the named predicate constraint, or to false otherwise. If the constraint specifies more elements than are available at the current vector length then all elements of the destination predicate are set to false.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception. Does not set the condition flags.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PTRUES":
            return {
                "tooltip": "Set elements of the destination predicate to true if the element number satisfies the named predicate constraint, or to false otherwise. If the constraint specifies more elements than are available at the current vector length then all elements of the destination predicate are set to false.",
                "html": "<p>Set elements of the destination predicate to true if the element number satisfies the named predicate constraint, or to false otherwise. If the constraint specifies more elements than are available at the current vector length then all elements of the destination predicate are set to false.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "PUNPKHI":
        case "PUNPKLO":
            return {
                "tooltip": "Unpack elements from the lowest or highest half of the source predicate and place in elements of twice their size within the destination predicate. This instruction is unpredicated.",
                "html": "<p>Unpack elements from the lowest or highest half of the source predicate and place in elements of twice their size within the destination predicate. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RADDHN":
        case "RADDHN2":
            return {
                "tooltip": "Rounding Add returning High Narrow. This instruction adds each vector element in the first source SIMD&FP register to the corresponding vector element in the second source SIMD&FP register, places the most significant half of the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&FP register.",
                "html": "<p>Rounding Add returning High Narrow. This instruction adds each vector element in the first source SIMD&amp;FP register to the corresponding vector element in the second source SIMD&amp;FP register, places the most significant half of the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&amp;FP register.</p><p>The results are rounded. For truncated results, see <xref linkend=\"A64.instructions.ADDHN_advsimd\">ADDHN</xref>.</p><p>The <instruction>RADDHN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>RADDHN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RADDHNB":
            return {
                "tooltip": "Add each vector element of the first source vector to the corresponding vector element of the second source vector, and place the most significant rounded half of the result in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. This instruction is unpredicated.",
                "html": "<p>Add each vector element of the first source vector to the corresponding vector element of the second source vector, and place the most significant rounded half of the result in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RADDHNT":
            return {
                "tooltip": "Add each vector element of the first source vector to the corresponding vector element of the second source vector, and place the most significant rounded half of the result in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. This instruction is unpredicated.",
                "html": "<p>Add each vector element of the first source vector to the corresponding vector element of the second source vector, and place the most significant rounded half of the result in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RAX1":
            return {
                "tooltip": "Rotate and Exclusive-OR rotates each 64-bit element of the 128-bit vector in a source SIMD&FP register left by 1, performs a bitwise exclusive-OR of the resulting 128-bit vector and the vector in another source SIMD&FP register, and writes the result to the destination SIMD&FP register.",
                "html": "<p>Rotate and Exclusive-OR rotates each 64-bit element of the 128-bit vector in a source SIMD&amp;FP register left by 1, performs a bitwise exclusive-OR of the resulting 128-bit vector and the vector in another source SIMD&amp;FP register, and writes the result to the destination SIMD&amp;FP register.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SHA3\">FEAT_SHA3</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RBIT":
            return {
                "tooltip": "Reverse Bit order (vector). This instruction reads each vector element from the source SIMD&FP register, reverses the bits of the element, places the results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Reverse Bit order (vector). This instruction reads each vector element from the source SIMD&amp;FP register, reverses the bits of the element, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWCAS":
        case "RCWCASA":
        case "RCWCASAL":
        case "RCWCASL":
            return {
                "tooltip": "Read Check Write Compare and Swap doubleword in memory reads a 64-bit doubleword from memory, and compares it against the value held in a register. If the comparison is equal, the value in a second register is conditionally written to memory. Storing back to memory is conditional on RCW Checks. If the write is performed, the read and the write occur atomically such that no other modification of the memory location can take place between the read and the write. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Compare and Swap doubleword in memory reads a 64-bit doubleword from memory, and compares it against the value held in a register. If the comparison is equal, the value in a second register is conditionally written to memory. Storing back to memory is conditional on RCW Checks. If the write is performed, the read and the write occur atomically such that no other modification of the memory location can take place between the read and the write. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWCASP":
        case "RCWCASPA":
        case "RCWCASPAL":
        case "RCWCASPL":
            return {
                "tooltip": "Read Check Write Compare and Swap quadword in memory reads a 128-bit quadword from memory, and compares it against the value held in a pair of registers. If the comparison is equal, the value in a second pair of registers is conditionally written to memory. Storing back to memory is conditional on RCW Checks. If the write is performed, the read and the write occur atomically such that no other modification of the memory location can take place between the read and the write. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Compare and Swap quadword in memory reads a 128-bit quadword from memory, and compares it against the value held in a pair of registers. If the comparison is equal, the value in a second pair of registers is conditionally written to memory. Storing back to memory is conditional on RCW Checks. If the write is performed, the read and the write occur atomically such that no other modification of the memory location can take place between the read and the write. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWCLR":
        case "RCWCLRA":
        case "RCWCLRAL":
        case "RCWCLRL":
            return {
                "tooltip": "Read Check Write atomic bit Clear on doubleword in memory atomically loads a 64-bit doubleword from memory, performs a bitwise AND with the complement of the value held in a register on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write atomic bit Clear on doubleword in memory atomically loads a 64-bit doubleword from memory, performs a bitwise AND with the complement of the value held in a register on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWCLRP":
        case "RCWCLRPA":
        case "RCWCLRPAL":
        case "RCWCLRPL":
            return {
                "tooltip": "Read Check Write atomic bit Clear on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise AND with the complement of the value held in a pair of registers on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write atomic bit Clear on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise AND with the complement of the value held in a pair of registers on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSCAS":
        case "RCWSCASA":
        case "RCWSCASAL":
        case "RCWSCASL":
            return {
                "tooltip": "Read Check Write Software Compare and Swap doubleword in memory reads a 64-bit doubleword from memory, and compares it against the value held in a register. If the comparison is equal, the value in a second register is conditionally written to memory. Storing back to memory is conditional on RCW Checks and RCWS Checks. If the write is performed, the read and the write occur atomically such that no other modification of the memory location can take place between the read and the write. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Software Compare and Swap doubleword in memory reads a 64-bit doubleword from memory, and compares it against the value held in a register. If the comparison is equal, the value in a second register is conditionally written to memory. Storing back to memory is conditional on RCW Checks and RCWS Checks. If the write is performed, the read and the write occur atomically such that no other modification of the memory location can take place between the read and the write. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSCASP":
        case "RCWSCASPA":
        case "RCWSCASPAL":
        case "RCWSCASPL":
            return {
                "tooltip": "Read Check Write Software Compare and Swap quadword in memory reads a 128-bit quadword from memory, and compares it against the value held in a pair of registers. If the comparison is equal, the value in a second pair of registers is conditionally written to memory. Storing back to memory is conditional on RCW Checks and RCWS Checks. If the write is performed, the read and the write occur atomically such that no other modification of the memory location can take place between the read and the write. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Software Compare and Swap quadword in memory reads a 128-bit quadword from memory, and compares it against the value held in a pair of registers. If the comparison is equal, the value in a second pair of registers is conditionally written to memory. Storing back to memory is conditional on RCW Checks and RCWS Checks. If the write is performed, the read and the write occur atomically such that no other modification of the memory location can take place between the read and the write. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSCLR":
        case "RCWSCLRA":
        case "RCWSCLRAL":
        case "RCWSCLRL":
            return {
                "tooltip": "Read Check Write Software atomic bit Clear on doubleword in memory atomically loads a 64-bit doubleword from memory, performs a bitwise AND with the complement of the value held in a register on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Software atomic bit Clear on doubleword in memory atomically loads a 64-bit doubleword from memory, performs a bitwise AND with the complement of the value held in a register on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSCLRP":
        case "RCWSCLRPA":
        case "RCWSCLRPAL":
        case "RCWSCLRPL":
            return {
                "tooltip": "Read Check Write Software atomic bit Clear on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise AND with the complement of the value held in a pair of registers on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Software atomic bit Clear on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise AND with the complement of the value held in a pair of registers on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSET":
        case "RCWSETA":
        case "RCWSETAL":
        case "RCWSETL":
            return {
                "tooltip": "Read Check Write atomic bit Set on doubleword in memory atomically loads a 64-bit doubleword from memory, performs a bitwise OR with the complement of the value held in a register on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write atomic bit Set on doubleword in memory atomically loads a 64-bit doubleword from memory, performs a bitwise OR with the complement of the value held in a register on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSETP":
        case "RCWSETPA":
        case "RCWSETPAL":
        case "RCWSETPL":
            return {
                "tooltip": "Read Check Write atomic bit Set on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise OR with the value held in a pair of registers on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write atomic bit Set on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise OR with the value held in a pair of registers on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSSET":
        case "RCWSSETA":
        case "RCWSSETAL":
        case "RCWSSETL":
            return {
                "tooltip": "Read Check Write Software atomic bit Set on doubleword in memory atomically loads a 64-bit doubleword from memory, performs a bitwise OR with the complement of the value held in a register on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Software atomic bit Set on doubleword in memory atomically loads a 64-bit doubleword from memory, performs a bitwise OR with the complement of the value held in a register on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSSETP":
        case "RCWSSETPA":
        case "RCWSSETPAL":
        case "RCWSSETPL":
            return {
                "tooltip": "Read Check Write Software atomic bit Set on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise OR with the value held in a pair of registers on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Software atomic bit Set on quadword in memory atomically loads a 128-bit quadword from memory, performs a bitwise OR with the value held in a pair of registers on it, and conditionally stores the result back to memory. Storing of the result back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSSWP":
        case "RCWSSWPA":
        case "RCWSSWPAL":
        case "RCWSSWPL":
            return {
                "tooltip": "Read Check Write Software Swap doubleword in memory atomically loads a 64-bit doubleword from a memory location, and conditionally stores the value held in a register back to the same memory location. Storing back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Software Swap doubleword in memory atomically loads a 64-bit doubleword from a memory location, and conditionally stores the value held in a register back to the same memory location. Storing back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSSWPP":
        case "RCWSSWPPA":
        case "RCWSSWPPAL":
        case "RCWSSWPPL":
            return {
                "tooltip": "Read Check Write Software Swap quadword in memory atomically loads a 128-bit quadword from a memory location, and conditionally stores the value held in a pair of registers back to the same memory location. Storing back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Software Swap quadword in memory atomically loads a 128-bit quadword from a memory location, and conditionally stores the value held in a pair of registers back to the same memory location. Storing back to memory is conditional on RCW Checks and RCWS Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSWP":
        case "RCWSWPA":
        case "RCWSWPAL":
        case "RCWSWPL":
            return {
                "tooltip": "Read Check Write Swap doubleword in memory atomically loads a 64-bit doubleword from a memory location, and conditionally stores the value held in a register back to the same memory location. Storing back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Swap doubleword in memory atomically loads a 64-bit doubleword from a memory location, and conditionally stores the value held in a register back to the same memory location. Storing back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the destination register. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RCWSWPP":
        case "RCWSWPPA":
        case "RCWSWPPAL":
        case "RCWSWPPL":
            return {
                "tooltip": "Read Check Write Swap quadword in memory atomically loads a 128-bit quadword from a memory location, and conditionally stores the value held in a pair of registers back to the same memory location. Storing back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.",
                "html": "<p>Read Check Write Swap quadword in memory atomically loads a 128-bit quadword from a memory location, and conditionally stores the value held in a pair of registers back to the same memory location. Storing back to memory is conditional on RCW Checks. The value initially loaded from memory is returned in the same pair of registers. This instruction updates the condition flags based on the result of the update of memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RDFFR":
            return {
                "tooltip": "Read the first-fault register (FFR) and place in the destination predicate without predication.",
                "html": "<p>Read the first-fault register (<asm-code>FFR</asm-code>) and place in the destination predicate without predication.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RDFFRS":
            return {
                "tooltip": "Read the first-fault register (FFR) and place active elements in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the First (N), None (Z), !Last (C) condition flags based on the predicate result, and the V flag to zero.",
                "html": "<p>Read the first-fault register (<asm-code>FFR</asm-code>) and place active elements in the corresponding elements of the destination predicate. Inactive elements in the destination predicate register are set to zero. Sets the <arm-defined-word>First</arm-defined-word> (N), <arm-defined-word>None</arm-defined-word> (Z), <arm-defined-word>!Last</arm-defined-word> (C) condition flags based on the predicate result, and the V flag to zero.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RDSVL":
            return {
                "tooltip": "Multiply the Streaming SVE vector register size in bytes by an immediate in the range -32 to 31 and place the result in the 64-bit destination general-purpose register.",
                "html": "<p>Multiply the Streaming SVE vector register size in bytes by an immediate in the range -32 to 31 and place the result in the 64-bit destination general-purpose register.</p><p>This instruction does not require the PE to be in Streaming SVE mode.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RDVL":
            return {
                "tooltip": "Multiply the current vector register size in bytes by an immediate in the range -32 to 31 and place the result in the 64-bit destination general-purpose register.",
                "html": "<p>Multiply the current vector register size in bytes by an immediate in the range -32 to 31 and place the result in the 64-bit destination general-purpose register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RET":
            return {
                "tooltip": "Return from subroutine branches unconditionally to an address in a register, with a hint that this is a subroutine return.",
                "html": "<p>Return from subroutine branches unconditionally to an address in a register, with a hint that this is a subroutine return.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RETAA":
        case "RETAB":
            return {
                "tooltip": "Return from subroutine, with pointer authentication. This instruction authenticates the address that is held in LR, using SP as the modifier and the specified key, branches to the authenticated address, with a hint that this instruction is a subroutine return.",
                "html": "<p>Return from subroutine, with pointer authentication. This instruction authenticates the address that is held in LR, using SP as the modifier and the specified key, branches to the authenticated address, with a hint that this instruction is a subroutine return.</p><p>Key A is used for <instruction>RETAA</instruction>. Key B is used for <instruction>RETAB</instruction>.</p><p>If the authentication passes, the PE continues execution at the target of the branch. For information on behavior if the authentication fails, see <xref>Faulting on pointer authentication</xref>.</p><p>The authenticated address is not written back to LR.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "REV":
            return {
                "tooltip": "Reverse Bytes reverses the byte order in a register.",
                "html": "<p>Reverse Bytes reverses the byte order in a register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "REV16":
            return {
                "tooltip": "Reverse elements in 16-bit halfwords (vector). This instruction reverses the order of 8-bit elements in each halfword of the vector in the source SIMD&FP register, places the results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Reverse elements in 16-bit halfwords (vector). This instruction reverses the order of 8-bit elements in each halfword of the vector in the source SIMD&amp;FP register, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "REV32":
            return {
                "tooltip": "Reverse elements in 32-bit words (vector). This instruction reverses the order of 8-bit or 16-bit elements in each word of the vector in the source SIMD&FP register, places the results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Reverse elements in 32-bit words (vector). This instruction reverses the order of 8-bit or 16-bit elements in each word of the vector in the source SIMD&amp;FP register, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "REV64":
            return {
                "tooltip": "Reverse elements in 64-bit doublewords (vector). This instruction reverses the order of 8-bit, 16-bit, or 32-bit elements in each doubleword of the vector in the source SIMD&FP register, places the results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Reverse elements in 64-bit doublewords (vector). This instruction reverses the order of 8-bit, 16-bit, or 32-bit elements in each doubleword of the vector in the source SIMD&amp;FP register, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "REVB":
        case "REVH":
        case "REVW":
            return {
                "tooltip": "Reverse the order of 8-bit bytes, 16-bit halfwords or 32-bit words within each active element of the source vector, and place the results in the corresponding elements of the destination vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Reverse the order of 8-bit bytes, 16-bit halfwords or 32-bit words within each active element of the source vector, and place the results in the corresponding elements of the destination vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "REVD":
            return {
                "tooltip": "Reverse the order of 64-bit doublewords within each active element of the source vector, and place the results in the corresponding elements of the destination vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Reverse the order of 64-bit doublewords within each active element of the source vector, and place the results in the corresponding elements of the destination vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RMIF":
            return {
                "tooltip": "Performs a rotation right of a value held in a general purpose register by an immediate value, and then inserts a selection of the bottom four bits of the result of the rotation into the PSTATE flags, under the control of a second immediate mask.",
                "html": "<p>Performs a rotation right of a value held in a general purpose register by an immediate value, and then inserts a selection of the bottom four bits of the result of the rotation into the PSTATE flags, under the control of a second immediate mask.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ROR":
            return {
                "tooltip": "Rotate right (immediate) provides the value of the contents of a register rotated by a variable number of bits. The bits that are rotated off the right end are inserted into the vacated bit positions on the left.",
                "html": "<p>Rotate right (immediate) provides the value of the contents of a register rotated by a variable number of bits. The bits that are rotated off the right end are inserted into the vacated bit positions on the left.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RORV":
            return {
                "tooltip": "Rotate Right Variable provides the value of the contents of a register rotated by a variable number of bits. The bits that are rotated off the right end are inserted into the vacated bit positions on the left. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is right-shifted.",
                "html": "<p>Rotate Right Variable provides the value of the contents of a register rotated by a variable number of bits. The bits that are rotated off the right end are inserted into the vacated bit positions on the left. The remainder obtained by dividing the second source register by the data size defines the number of bits by which the first source register is right-shifted.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RPRFM":
            return {
                "tooltip": "Range Prefetch Memory signals the memory system that data memory accesses from a specified range of addresses are likely to occur in the near future. The instruction may also signal the memory system about the likelihood of data reuse of the specified range of addresses. The memory system can respond by taking actions that are expected to speed up the memory accesses when they do occur, such as prefetching locations within the specified address ranges into one or more caches. The memory system may also exploit the data reuse hints to decide whether to retain the data in other caches upon eviction from the innermost caches or to discard it.",
                "html": "<p>Range Prefetch Memory signals the memory system that data memory accesses from a specified range of addresses are likely to occur in the near future. The instruction may also signal the memory system about the likelihood of data reuse of the specified range of addresses. The memory system can respond by taking actions that are expected to speed up the memory accesses when they do occur, such as prefetching locations within the specified address ranges into one or more caches. The memory system may also exploit the data reuse hints to decide whether to retain the data in other caches upon eviction from the innermost caches or to discard it.</p><p>The effect of an <instruction>RPRFM</instruction> instruction is <arm-defined-word>implementation defined</arm-defined-word>, but because these signals are only hints, the instruction cannot cause a synchronous Data Abort exception and is guaranteed not to access Device memory. It is valid for the PE to treat this instruction as a NOP.</p><p>An <instruction>RPRFM</instruction> instruction specifies the type of accesses and range of addresses using the following parameters:</p><p>Software is expected to honor the parameters it provides to the <instruction>RPRFM</instruction> instruction, and the same PE should access all locations in the range, in the direction specified by the sign of the 'Length' and 'Stride' parameters. A range prefetch is considered active on a PE until all locations in the range have been accessed by the PE. A range prefetch might also be inactivated by the PE prior to completion, for example due to a software context switch or lack of hardware resources.</p><p>Software should not specify overlapping addresses in multiple active ranges. If a range is expected to be accessed by both load and store instructions (read-modify-write), then a single range with a 'Type' parameter of <syntax>PST</syntax> (prefetch for store) should be specified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RSHRN":
        case "RSHRN2":
            return {
                "tooltip": "Rounding Shift Right Narrow (immediate). This instruction reads each unsigned integer value from the vector in the source SIMD&FP register, right shifts each result by an immediate value, writes the final result to a vector, and writes the vector to the lower or upper half of the destination SIMD&FP register. The destination vector elements are half as long as the source vector elements. The results are rounded. For truncated results, see SHRN.",
                "html": "<p>Rounding Shift Right Narrow (immediate). This instruction reads each unsigned integer value from the vector in the source SIMD&amp;FP register, right shifts each result by an immediate value, writes the final result to a vector, and writes the vector to the lower or upper half of the destination SIMD&amp;FP register. The destination vector elements are half as long as the source vector elements. The results are rounded. For truncated results, see <xref linkend=\"A64.instructions.SHRN_advsimd\">SHRN</xref>.</p><p>The <instruction>RSHRN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>RSHRN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RSHRNB":
            return {
                "tooltip": "Shift each unsigned integer value in the source vector elements right by an immediate value, and place the rounded results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each unsigned integer value in the source vector elements right by an immediate value, and place the rounded results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RSHRNT":
            return {
                "tooltip": "Shift each unsigned integer value in the source vector elements right by an immediate value, and place the rounded results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each unsigned integer value in the source vector elements right by an immediate value, and place the rounded results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RSUBHN":
        case "RSUBHN2":
            return {
                "tooltip": "Rounding Subtract returning High Narrow. This instruction subtracts each vector element of the second source SIMD&FP register from the corresponding vector element of the first source SIMD&FP register, places the most significant half of the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&FP register.",
                "html": "<p>Rounding Subtract returning High Narrow. This instruction subtracts each vector element of the second source SIMD&amp;FP register from the corresponding vector element of the first source SIMD&amp;FP register, places the most significant half of the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&amp;FP register.</p><p>The results are rounded. For truncated results, see <xref linkend=\"A64.instructions.SUBHN_advsimd\">SUBHN</xref>.</p><p>The <instruction>RSUBHN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>RSUBHN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RSUBHNB":
            return {
                "tooltip": "Subtract each vector element of the second source vector from the corresponding vector element in the first source vector, and place the most significant rounded half of the result in the even-numbered half-width destination elements, while setting the odd-numbered half-width destination elements to zero. This instruction is unpredicated.",
                "html": "<p>Subtract each vector element of the second source vector from the corresponding vector element in the first source vector, and place the most significant rounded half of the result in the even-numbered half-width destination elements, while setting the odd-numbered half-width destination elements to zero. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "RSUBHNT":
            return {
                "tooltip": "Subtract each vector element of the second source vector from the corresponding vector element in the first source vector, and place the most significant rounded half of the result in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. This instruction is unpredicated.",
                "html": "<p>Subtract each vector element of the second source vector from the corresponding vector element in the first source vector, and place the most significant rounded half of the result in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SABA":
            return {
                "tooltip": "Signed Absolute difference and Accumulate. This instruction subtracts the elements of the vector of the second source SIMD&FP register from the corresponding elements of the first source SIMD&FP register, and accumulates the absolute values of the results into the elements of the vector of the destination SIMD&FP register.",
                "html": "<p>Signed Absolute difference and Accumulate. This instruction subtracts the elements of the vector of the second source SIMD&amp;FP register from the corresponding elements of the first source SIMD&amp;FP register, and accumulates the absolute values of the results into the elements of the vector of the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SABAL":
        case "SABAL2":
            return {
                "tooltip": "Signed Absolute difference and Accumulate Long. This instruction subtracts the vector elements in the lower or upper half of the second source SIMD&FP register from the corresponding vector elements of the first source SIMD&FP register, and accumulates the absolute values of the results into the vector elements of the destination SIMD&FP register. The destination vector elements are twice as long as the source vector elements.",
                "html": "<p>Signed Absolute difference and Accumulate Long. This instruction subtracts the vector elements in the lower or upper half of the second source SIMD&amp;FP register from the corresponding vector elements of the first source SIMD&amp;FP register, and accumulates the absolute values of the results into the vector elements of the destination SIMD&amp;FP register. The destination vector elements are twice as long as the source vector elements.</p><p>The <instruction>SABAL</instruction> instruction extracts each source vector from the lower half of each source register. The <instruction>SABAL2</instruction> instruction extracts each source vector from the upper half of each source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SABALB":
            return {
                "tooltip": "Compute the absolute difference between even-numbered signed integer values in elements of the second source vector and corresponding elements of the first source vector, and destructively add to the overlapping double-width elements of the addend vector. This instruction is unpredicated.",
                "html": "<p>Compute the absolute difference between even-numbered signed integer values in elements of the second source vector and corresponding elements of the first source vector, and destructively add to the overlapping double-width elements of the addend vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SABALT":
            return {
                "tooltip": "Compute the absolute difference between odd-numbered signed elements of the second source vector and corresponding elements of the first source vector, and destructively add to the overlapping double-width elements of the addend vector. This instruction is unpredicated.",
                "html": "<p>Compute the absolute difference between odd-numbered signed elements of the second source vector and corresponding elements of the first source vector, and destructively add to the overlapping double-width elements of the addend vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SABD":
            return {
                "tooltip": "Signed Absolute Difference. This instruction subtracts the elements of the vector of the second source SIMD&FP register from the corresponding elements of the first source SIMD&FP register, places the absolute values of the results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed Absolute Difference. This instruction subtracts the elements of the vector of the second source SIMD&amp;FP register from the corresponding elements of the first source SIMD&amp;FP register, places the absolute values of the results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SABDL":
        case "SABDL2":
            return {
                "tooltip": "Signed Absolute Difference Long. This instruction subtracts the vector elements in the lower or upper half of the second source SIMD&FP register from the corresponding vector elements of the first source SIMD&FP register, places the absolute value of the results into a vector, and writes the vector to the destination SIMD&FP register. The destination vector elements are twice as long as the source vector elements.",
                "html": "<p>Signed Absolute Difference Long. This instruction subtracts the vector elements in the lower or upper half of the second source SIMD&amp;FP register from the corresponding vector elements of the first source SIMD&amp;FP register, places the absolute value of the results into a vector, and writes the vector to the destination SIMD&amp;FP register. The destination vector elements are twice as long as the source vector elements.</p><p>The <instruction>SABDL</instruction> instruction extracts each source vector from the lower half of each source register. The <instruction>SABDL2</instruction> instruction extracts each source vector from the upper half of each source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SABDLB":
            return {
                "tooltip": "Compute the absolute difference between even-numbered signed integer values in elements of the second source vector and corresponding elements of the first source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Compute the absolute difference between even-numbered signed integer values in elements of the second source vector and corresponding elements of the first source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SABDLT":
            return {
                "tooltip": "Compute the absolute difference between odd-numbered signed integer values in elements of the second source vector and corresponding elements of the first source vector, and place the results in overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Compute the absolute difference between odd-numbered signed integer values in elements of the second source vector and corresponding elements of the first source vector, and place the results in overlapping double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SADALP":
            return {
                "tooltip": "Signed Add and Accumulate Long Pairwise. This instruction adds pairs of adjacent signed integer values from the vector in the source SIMD&FP register and accumulates the results into the vector elements of the destination SIMD&FP register. The destination vector elements are twice as long as the source vector elements.",
                "html": "<p>Signed Add and Accumulate Long Pairwise. This instruction adds pairs of adjacent signed integer values from the vector in the source SIMD&amp;FP register and accumulates the results into the vector elements of the destination SIMD&amp;FP register. The destination vector elements are twice as long as the source vector elements.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SADDL":
        case "SADDL2":
            return {
                "tooltip": "Signed Add Long (vector). This instruction adds each vector element in the lower or upper half of the first source SIMD&FP register to the corresponding vector element of the second source SIMD&FP register, places the results into a vector, and writes the vector to the destination SIMD&FP register.  The destination vector elements are twice as long as the source vector elements. All the values in this instruction are signed integer values.",
                "html": "<p>Signed Add Long (vector). This instruction adds each vector element in the lower or upper half of the first source SIMD&amp;FP register to the corresponding vector element of the second source SIMD&amp;FP register, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register.  The destination vector elements are twice as long as the source vector elements. All the values in this instruction are signed integer values.</p><p>The <instruction>SADDL</instruction> instruction extracts each source vector from the lower half of each source register. The <instruction>SADDL2</instruction> instruction extracts each source vector from the upper half of each source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SADDLB":
            return {
                "tooltip": "Add the corresponding even-numbered signed elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Add the corresponding even-numbered signed elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SADDLBT":
            return {
                "tooltip": "Add the even-numbered signed elements of the first source vector to the odd-numbered signed elements of the second source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Add the even-numbered signed elements of the first source vector to the odd-numbered signed elements of the second source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SADDLP":
            return {
                "tooltip": "Signed Add Long Pairwise. This instruction adds pairs of adjacent signed integer values from the vector in the source SIMD&FP register, places the result into a vector, and writes the vector to the destination SIMD&FP register. The destination vector elements are twice as long as the source vector elements.",
                "html": "<p>Signed Add Long Pairwise. This instruction adds pairs of adjacent signed integer values from the vector in the source SIMD&amp;FP register, places the result into a vector, and writes the vector to the destination SIMD&amp;FP register. The destination vector elements are twice as long as the source vector elements.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SADDLT":
            return {
                "tooltip": "Add the corresponding odd-numbered signed elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Add the corresponding odd-numbered signed elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SADDLV":
            return {
                "tooltip": "Signed Add Long across Vector. This instruction adds every vector element in the source SIMD&FP register together, and writes the scalar result to the destination SIMD&FP register. The destination scalar is twice as long as the source vector elements. All the values in this instruction are signed integer values.",
                "html": "<p>Signed Add Long across Vector. This instruction adds every vector element in the source SIMD&amp;FP register together, and writes the scalar result to the destination SIMD&amp;FP register. The destination scalar is twice as long as the source vector elements. All the values in this instruction are signed integer values.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SADDV":
            return {
                "tooltip": "Signed add horizontally across all lanes of a vector, and place the result in the SIMD&FP scalar destination register. Narrow elements are first sign-extended to 64 bits. Inactive elements in the source vector are treated as zero.",
                "html": "<p>Signed add horizontally across all lanes of a vector, and place the result in the SIMD&amp;FP scalar destination register. Narrow elements are first sign-extended to 64 bits. Inactive elements in the source vector are treated as zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SADDW":
        case "SADDW2":
            return {
                "tooltip": "Signed Add Wide. This instruction adds vector elements of the first source SIMD&FP register to the corresponding vector elements in the lower or upper half of the second source SIMD&FP register, places the results in a vector, and writes the vector to the SIMD&FP destination register.",
                "html": "<p>Signed Add Wide. This instruction adds vector elements of the first source SIMD&amp;FP register to the corresponding vector elements in the lower or upper half of the second source SIMD&amp;FP register, places the results in a vector, and writes the vector to the SIMD&amp;FP destination register.</p><p>The <instruction>SADDW</instruction> instruction extracts the second source vector from the lower half of the second source register. The <instruction>SADDW2</instruction> instruction extracts the second source vector from the upper half of the second source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SADDWB":
            return {
                "tooltip": "Add the even-numbered signed elements of the second source vector to the overlapping double-width elements of the first source vector and place the results in the corresponding double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Add the even-numbered signed elements of the second source vector to the overlapping double-width elements of the first source vector and place the results in the corresponding double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SADDWT":
            return {
                "tooltip": "Add the odd-numbered signed elements of the second source vector to the overlapping double-width elements of the first source vector and place the results in the corresponding double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Add the odd-numbered signed elements of the second source vector to the overlapping double-width elements of the first source vector and place the results in the corresponding double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SB":
            return {
                "tooltip": "Speculation Barrier is a barrier that controls speculation.",
                "html": "<p>Speculation Barrier is a barrier that controls speculation.</p><p>The semantics of the Speculation Barrier are that the execution, until the barrier completes, of any instruction that appears later in the program order than the barrier:</p><p>In particular, any instruction that appears later in the program order than the barrier cannot cause a speculative allocation into any caching structure where the allocation of that entry could be indicative of any data value present in memory or in the registers.</p><p>The SB instruction:</p><p>When the prediction of the instruction stream is not informed by data taken from the register outputs of the speculative execution of instructions appearing in program order after an uncompleted SB instruction, the SB instruction has no effect on the use of prediction resources to predict the instruction stream that is being fetched.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SBC":
            return {
                "tooltip": "Subtract with Carry subtracts a register value and the value of NOT (Carry flag) from a register value, and writes the result to the destination register.",
                "html": "<p>Subtract with Carry subtracts a register value and the value of NOT (Carry flag) from a register value, and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SBCLB":
            return {
                "tooltip": "Subtract the even-numbered elements of the first source vector and the inverted 1-bit carry from the least-significant bit of the odd-numbered elements of the second source vector from the even-numbered elements of the destination and accumulator vector. The 1-bit carry output is placed in the corresponding odd-numbered element of the destination vector.",
                "html": "<p>Subtract the even-numbered elements of the first source vector and the inverted 1-bit carry from the least-significant bit of the odd-numbered elements of the second source vector from the even-numbered elements of the destination and accumulator vector. The 1-bit carry output is placed in the corresponding odd-numbered element of the destination vector.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SBCLT":
            return {
                "tooltip": "Subtract the odd-numbered elements of the first source vector and the inverted 1-bit carry from the least-significant bit of the odd-numbered elements of the second source vector from the even-numbered elements of the destination and accumulator vector. The 1-bit carry output is placed in the corresponding odd-numbered element of the destination vector.",
                "html": "<p>Subtract the odd-numbered elements of the first source vector and the inverted 1-bit carry from the least-significant bit of the odd-numbered elements of the second source vector from the even-numbered elements of the destination and accumulator vector. The 1-bit carry output is placed in the corresponding odd-numbered element of the destination vector.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SBCS":
            return {
                "tooltip": "Subtract with Carry, setting flags, subtracts a register value and the value of NOT (Carry flag) from a register value, and writes the result to the destination register. It updates the condition flags based on the result.",
                "html": "<p>Subtract with Carry, setting flags, subtracts a register value and the value of NOT (Carry flag) from a register value, and writes the result to the destination register. It updates the condition flags based on the result.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SBFIZ":
            return {
                "tooltip": "Signed Bitfield Insert in Zeros copies a bitfield of <width> bits from the least significant bits of the source register to bit position <lsb> of the destination register, setting the destination bits below the bitfield to zero, and the bits above the bitfield to a copy of the most significant bit of the bitfield.",
                "html": "<p>Signed Bitfield Insert in Zeros copies a bitfield of <syntax>&lt;width&gt;</syntax> bits from the least significant bits of the source register to bit position <syntax>&lt;lsb&gt;</syntax> of the destination register, setting the destination bits below the bitfield to zero, and the bits above the bitfield to a copy of the most significant bit of the bitfield.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SBFM":
            return {
                "tooltip": "Signed Bitfield Move is usually accessed via one of its aliases, which are always preferred for disassembly.",
                "html": "<p>Signed Bitfield Move is usually accessed via one of its aliases, which are always preferred for disassembly.</p><p>If <syntax>&lt;imms&gt;</syntax> is greater than or equal to <syntax>&lt;immr&gt;</syntax>, this copies a bitfield of (<syntax>&lt;imms&gt;</syntax>-<syntax>&lt;immr&gt;</syntax>+1) bits starting from bit position <syntax>&lt;immr&gt;</syntax> in the source register to the least significant bits of the destination register.</p><p>If <syntax>&lt;imms&gt;</syntax> is less than <syntax>&lt;immr&gt;</syntax>, this copies a bitfield of (<syntax>&lt;imms&gt;</syntax>+1) bits from the least significant bits of the source register to bit position (regsize-<syntax>&lt;immr&gt;</syntax>) of the destination register, where regsize is the destination register size of 32 or 64 bits.</p><p>In both cases the destination bits below the bitfield are set to zero, and the bits above the bitfield are set to a copy of the most significant bit of the bitfield.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SBFX":
            return {
                "tooltip": "Signed Bitfield Extract copies a bitfield of <width> bits starting from bit position <lsb> in the source register to the least significant bits of the destination register, and sets destination bits above the bitfield to a copy of the most significant bit of the bitfield.",
                "html": "<p>Signed Bitfield Extract copies a bitfield of <syntax>&lt;width&gt;</syntax> bits starting from bit position <syntax>&lt;lsb&gt;</syntax> in the source register to the least significant bits of the destination register, and sets destination bits above the bitfield to a copy of the most significant bit of the bitfield.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SCLAMP":
            return {
                "tooltip": "Clamp each signed element in the two or four destination vectors to between the signed minimum value in the corresponding element of the first source vector and the signed maximum value in the corresponding element of the second source vector and destructively place the clamped results in the corresponding elements of the two or four destination vectors.",
                "html": "<p>Clamp each signed element in the two or four destination vectors to between the signed minimum value in the corresponding element of the first source vector and the signed maximum value in the corresponding element of the second source vector and destructively place the clamped results in the corresponding elements of the two or four destination vectors.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SCVTF":
            return {
                "tooltip": "Signed fixed-point Convert to Floating-point (vector). This instruction converts each element in a vector from fixed-point to floating-point using the rounding mode that is specified by the FPCR, and writes the result to the SIMD&FP destination register.",
                "html": "<p>Signed fixed-point Convert to Floating-point (vector). This instruction converts each element in a vector from fixed-point to floating-point using the rounding mode that is specified by the <xref linkend=\"AArch64.fpcr\">FPCR</xref>, and writes the result to the SIMD&amp;FP destination register.</p><p>A floating-point exception can be generated by this instruction. Depending on the settings in <xref linkend=\"AArch64.fpcr\">FPCR</xref>, the exception results in either a flag being set in <xref linkend=\"AArch64.fpsr\">FPSR</xref>, or a synchronous exception being generated. For more information, see <xref linkend=\"BEIJDDAG\">Floating-point exception traps</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the Security state and Exception level in which the instruction is executed, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SDIV":
            return {
                "tooltip": "Signed Divide divides a signed integer register value by another signed integer register value, and writes the result to the destination register. The condition flags are not affected.",
                "html": "<p>Signed Divide divides a signed integer register value by another signed integer register value, and writes the result to the destination register. The condition flags are not affected.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SDIVR":
            return {
                "tooltip": "Signed reversed divide active elements of the second source vector by corresponding elements of the first source vector and destructively place the quotient in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Signed reversed divide active elements of the second source vector by corresponding elements of the first source vector and destructively place the quotient in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SDOT":
            return {
                "tooltip": "Dot Product signed arithmetic (vector, by element). This instruction performs the dot product of the four 8-bit elements in each 32-bit element of the first source register with the four 8-bit elements of an indexed 32-bit element in the second source register, accumulating the result into the corresponding 32-bit element of the destination register.",
                "html": "<p>Dot Product signed arithmetic (vector, by element). This instruction performs the dot product of the four 8-bit elements in each 32-bit element of the first source register with the four 8-bit elements of an indexed 32-bit element in the second source register, accumulating the result into the corresponding 32-bit element of the destination register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p><p>In Armv8.2 and Armv8.3, this is an <arm-defined-word>optional</arm-defined-word> instruction. From Armv8.4 it is mandatory for all implementations to support it.</p><p><xref linkend=\"AArch64.id_aa64isar0_el1\">ID_AA64ISAR0_EL1</xref>.DP indicates whether this instruction is supported.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SEL":
            return {
                "tooltip": "Read active elements from the two or four first source vectors and inactive elements from the two or four second source vectors and place in the corresponding elements of the two or four destination vectors.",
                "html": "<p>Read active elements from the two or four first source vectors and inactive elements from the two or four second source vectors and place in the corresponding elements of the two or four destination vectors.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SETF16":
        case "SETF8":
            return {
                "tooltip": "Set the PSTATE.NZV flags based on the value in the specified general-purpose register. SETF8 treats the value as an 8 bit value, and SETF16 treats the value as an 16 bit value.",
                "html": "<p>Set the PSTATE.NZV flags based on the value in the specified general-purpose register. <instruction>SETF8</instruction> treats the value as an 8 bit value, and <instruction>SETF16</instruction> treats the value as an 16 bit value.</p><p>The PSTATE.C flag is not affected by these instructions.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SETFFR":
            return {
                "tooltip": "Initialise the first-fault register (FFR) to all true prior to a sequence of first-fault or non-fault loads. This instruction is unpredicated.",
                "html": "<p>Initialise the first-fault register (<asm-code>FFR</asm-code>) to all true prior to a sequence of first-fault or non-fault loads. This instruction is unpredicated.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SETGE":
        case "SETGM":
        case "SETGP":
            return {
                "tooltip": "Memory Set with tag setting. These instructions perform a memory set using the value in the bottom byte of the source register and store an Allocation Tag to memory for each Tag Granule written. The Allocation Tag is calculated from the Logical Address Tag in the register which holds the first address that the set is made to. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETGP, then SETGM, and then SETGE.",
                "html": "<p>Memory Set with tag setting. These instructions perform a memory set using the value in the bottom byte of the source register and store an Allocation Tag to memory for each Tag Granule written. The Allocation Tag is calculated from the Logical Address Tag in the register which holds the first address that the set is made to. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETGP, then SETGM, and then SETGE.</p><p>SETGP performs some preconditioning of the arguments suitable for using the SETGM instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETGM performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETGE performs the last part of the memory set.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory set allows some optimization of the size that can be performed.</p><p>The architecture supports two algorithms for the memory set: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p><p>Portable software should not assume that the choice of algorithm is constant.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SETGEN":
        case "SETGMN":
        case "SETGPN":
            return {
                "tooltip": "Memory Set with tag setting, non-temporal. These instructions perform a memory set using the value in the bottom byte of the source register and store an Allocation Tag to memory for each Tag Granule written. The Allocation Tag is calculated from the Logical Address Tag in the register which holds the first address that the set is made to. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETGPN, then SETGMN, and then SETGEN.",
                "html": "<p>Memory Set with tag setting, non-temporal. These instructions perform a memory set using the value in the bottom byte of the source register and store an Allocation Tag to memory for each Tag Granule written. The Allocation Tag is calculated from the Logical Address Tag in the register which holds the first address that the set is made to. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETGPN, then SETGMN, and then SETGEN.</p><p>SETGPN performs some preconditioning of the arguments suitable for using the SETGMN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETGMN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETGEN performs the last part of the memory set.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory set allows some optimization of the size that can be performed.</p><p>The architecture supports two algorithms for the memory set: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p><p>Portable software should not assume that the choice of algorithm is constant.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SETGET":
        case "SETGMT":
        case "SETGPT":
            return {
                "tooltip": "Memory Set with tag setting, unprivileged. These instructions perform a memory set using the value in the bottom byte of the source register and store an Allocation Tag to memory for each Tag Granule written. The Allocation Tag is calculated from the Logical Address Tag in the register which holds the first address that the set is made to. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETGPT, then SETGMT, and then SETGET.",
                "html": "<p>Memory Set with tag setting, unprivileged. These instructions perform a memory set using the value in the bottom byte of the source register and store an Allocation Tag to memory for each Tag Granule written. The Allocation Tag is calculated from the Logical Address Tag in the register which holds the first address that the set is made to. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETGPT, then SETGMT, and then SETGET.</p><p>SETGPT performs some preconditioning of the arguments suitable for using the SETGMT instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETGMT performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETGET performs the last part of the memory set.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory set allows some optimization of the size that can be performed.</p><p>The architecture supports two algorithms for the memory set: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p><p>Portable software should not assume that the choice of algorithm is constant.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SETGETN":
        case "SETGMTN":
        case "SETGPTN":
            return {
                "tooltip": "Memory Set with tag setting, unprivileged and non-temporal. These instructions perform a memory set using the value in the bottom byte of the source register and store an Allocation Tag to memory for each Tag Granule written. The Allocation Tag is calculated from the Logical Address Tag in the register which holds the first address that the set is made to. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETGPTN, then SETGMTN, and then SETGETN.",
                "html": "<p>Memory Set with tag setting, unprivileged and non-temporal. These instructions perform a memory set using the value in the bottom byte of the source register and store an Allocation Tag to memory for each Tag Granule written. The Allocation Tag is calculated from the Logical Address Tag in the register which holds the first address that the set is made to. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETGPTN, then SETGMTN, and then SETGETN.</p><p>SETGPTN performs some preconditioning of the arguments suitable for using the SETGMTN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETGMTN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETGETN performs the last part of the memory set.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory set allows some optimization of the size that can be performed.</p><p>The architecture supports two algorithms for the memory set: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p><p>Portable software should not assume that the choice of algorithm is constant.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SETE":
        case "SETM":
        case "SETP":
            return {
                "tooltip": "Memory Set. These instructions perform a memory set using the value in the bottom byte of the source register. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETP, then SETM, and then SETE.",
                "html": "<p>Memory Set. These instructions perform a memory set using the value in the bottom byte of the source register. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETP, then SETM, and then SETE.</p><p>SETP performs some preconditioning of the arguments suitable for using the SETM instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETM performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETE performs the last part of the memory set.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory set allows some optimization of the size that can be performed.</p><p>The architecture supports two algorithms for the memory set: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p><p>Portable software should not assume that the choice of algorithm is constant.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SETEN":
        case "SETMN":
        case "SETPN":
            return {
                "tooltip": "Memory Set, non-temporal. These instructions perform a memory set using the value in the bottom byte of the source register. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETPN, then SETMN, and then SETEN.",
                "html": "<p>Memory Set, non-temporal. These instructions perform a memory set using the value in the bottom byte of the source register. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETPN, then SETMN, and then SETEN.</p><p>SETPN performs some preconditioning of the arguments suitable for using the SETMN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETMN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETEN performs the last part of the memory set.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory set allows some optimization of the size that can be performed.</p><p>The architecture supports two algorithms for the memory set: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p><p>Portable software should not assume that the choice of algorithm is constant.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SETET":
        case "SETMT":
        case "SETPT":
            return {
                "tooltip": "Memory Set, unprivileged. These instructions perform a memory set using the value in the bottom byte of the source register. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETPT, then SETMT, and then SETET.",
                "html": "<p>Memory Set, unprivileged. These instructions perform a memory set using the value in the bottom byte of the source register. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETPT, then SETMT, and then SETET.</p><p>SETPT performs some preconditioning of the arguments suitable for using the SETMT instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETMT performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETET performs the last part of the memory set.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory set allows some optimization of the size that can be performed.</p><p>The architecture supports two algorithms for the memory set: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p><p>Portable software should not assume that the choice of algorithm is constant.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SETETN":
        case "SETMTN":
        case "SETPTN":
            return {
                "tooltip": "Memory Set, unprivileged and non-temporal. These instructions perform a memory set using the value in the bottom byte of the source register. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETPTN, then SETMTN, and then SETETN.",
                "html": "<p>Memory Set, unprivileged and non-temporal. These instructions perform a memory set using the value in the bottom byte of the source register. The prologue, main, and epilogue instructions are expected to be run in succession and to appear consecutively in memory: SETPTN, then SETMTN, and then SETETN.</p><p>SETPTN performs some preconditioning of the arguments suitable for using the SETMTN instruction, and performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETMTN performs an <arm-defined-word>implementation defined</arm-defined-word> amount of the memory set. SETETN performs the last part of the memory set.</p><p>The inclusion of <arm-defined-word>implementation defined</arm-defined-word> amounts of memory set allows some optimization of the size that can be performed.</p><p>The architecture supports two algorithms for the memory set: option A and option B. Which algorithm is used is <arm-defined-word>implementation defined</arm-defined-word>.</p><p>Portable software should not assume that the choice of algorithm is constant.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SEV":
            return {
                "tooltip": "Send Event is a hint instruction. It causes an event to be signaled to all PEs in the multiprocessor system. For more information, see Wait for Event mechanism and Send event.",
                "html": "<p>Send Event is a hint instruction. It causes an event to be signaled to all PEs in the multiprocessor system. For more information, see <xref linkend=\"BEIJHBBD\">Wait for Event mechanism and Send event</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SEVL":
            return {
                "tooltip": "Send Event Local is a hint instruction that causes an event to be signaled locally without requiring the event to be signaled to other PEs in the multiprocessor system. It can prime a wait-loop which starts with a WFE instruction.",
                "html": "<p>Send Event Local is a hint instruction that causes an event to be signaled locally without requiring the event to be signaled to other PEs in the multiprocessor system. It can prime a wait-loop which starts with a <instruction>WFE</instruction> instruction.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA1C":
            return {
                "tooltip": "SHA1 hash update (choose).",
                "html": "<p>SHA1 hash update (choose).</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA1H":
            return {
                "tooltip": "SHA1 fixed rotate.",
                "html": "<p>SHA1 fixed rotate.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA1M":
            return {
                "tooltip": "SHA1 hash update (majority).",
                "html": "<p>SHA1 hash update (majority).</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA1P":
            return {
                "tooltip": "SHA1 hash update (parity).",
                "html": "<p>SHA1 hash update (parity).</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA1SU0":
            return {
                "tooltip": "SHA1 schedule update 0.",
                "html": "<p>SHA1 schedule update 0.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA1SU1":
            return {
                "tooltip": "SHA1 schedule update 1.",
                "html": "<p>SHA1 schedule update 1.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA256H2":
            return {
                "tooltip": "SHA256 hash update (part 2).",
                "html": "<p>SHA256 hash update (part 2).</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA256H":
            return {
                "tooltip": "SHA256 hash update (part 1).",
                "html": "<p>SHA256 hash update (part 1).</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA256SU0":
            return {
                "tooltip": "SHA256 schedule update 0.",
                "html": "<p>SHA256 schedule update 0.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA256SU1":
            return {
                "tooltip": "SHA256 schedule update 1.",
                "html": "<p>SHA256 schedule update 1.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA512H2":
            return {
                "tooltip": "SHA512 Hash update part 2 takes the values from the three 128-bit source SIMD&FP registers and produces a 128-bit output value that combines the sigma0 and majority functions of two iterations of the SHA512 computation. It returns this value to the destination SIMD&FP register.",
                "html": "<p>SHA512 Hash update part 2 takes the values from the three 128-bit source SIMD&amp;FP registers and produces a 128-bit output value that combines the sigma0 and majority functions of two iterations of the SHA512 computation. It returns this value to the destination SIMD&amp;FP register.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SHA512\">FEAT_SHA512</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA512H":
            return {
                "tooltip": "SHA512 Hash update part 1 takes the values from the three 128-bit source SIMD&FP registers and produces a 128-bit output value that combines the sigma1 and chi functions of two iterations of the SHA512 computation. It returns this value to the destination SIMD&FP register.",
                "html": "<p>SHA512 Hash update part 1 takes the values from the three 128-bit source SIMD&amp;FP registers and produces a 128-bit output value that combines the sigma1 and chi functions of two iterations of the SHA512 computation. It returns this value to the destination SIMD&amp;FP register.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SHA512\">FEAT_SHA512</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA512SU0":
            return {
                "tooltip": "SHA512 Schedule Update 0 takes the values from the two 128-bit source SIMD&FP registers and produces a 128-bit output value that combines the gamma0 functions of two iterations of the SHA512 schedule update that are performed after the first 16 iterations within a block. It returns this value to the destination SIMD&FP register.",
                "html": "<p>SHA512 Schedule Update 0 takes the values from the two 128-bit source SIMD&amp;FP registers and produces a 128-bit output value that combines the gamma0 functions of two iterations of the SHA512 schedule update that are performed after the first 16 iterations within a block. It returns this value to the destination SIMD&amp;FP register.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SHA512\">FEAT_SHA512</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHA512SU1":
            return {
                "tooltip": "SHA512 Schedule Update 1 takes the values from the three source SIMD&FP registers and produces a 128-bit output value that combines the gamma1 functions of two iterations of the SHA512 schedule update that are performed after the first 16 iterations within a block. It returns this value to the destination SIMD&FP register.",
                "html": "<p>SHA512 Schedule Update 1 takes the values from the three source SIMD&amp;FP registers and produces a 128-bit output value that combines the gamma1 functions of two iterations of the SHA512 schedule update that are performed after the first 16 iterations within a block. It returns this value to the destination SIMD&amp;FP register.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SHA512\">FEAT_SHA512</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHADD":
            return {
                "tooltip": "Signed Halving Add. This instruction adds corresponding signed integer values from the two source SIMD&FP registers, shifts each result right one bit, places the results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed Halving Add. This instruction adds corresponding signed integer values from the two source SIMD&amp;FP registers, shifts each result right one bit, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>The results are truncated. For rounded results, see <xref linkend=\"A64.instructions.SRHADD_advsimd\">SRHADD</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHL":
            return {
                "tooltip": "Shift Left (immediate). This instruction reads each value from a vector, left shifts each result by an immediate value, writes the final result to a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Shift Left (immediate). This instruction reads each value from a vector, left shifts each result by an immediate value, writes the final result to a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHLL":
        case "SHLL2":
            return {
                "tooltip": "Shift Left Long (by element size). This instruction reads each vector element in the lower or upper half of the source SIMD&FP register, left shifts each result by the element size, writes the final result to a vector, and writes the vector to the destination SIMD&FP register. The destination vector elements are twice as long as the source vector elements.",
                "html": "<p>Shift Left Long (by element size). This instruction reads each vector element in the lower or upper half of the source SIMD&amp;FP register, left shifts each result by the element size, writes the final result to a vector, and writes the vector to the destination SIMD&amp;FP register. The destination vector elements are twice as long as the source vector elements.</p><p>The <instruction>SHLL</instruction> instruction extracts vector elements from the lower half of the source register. The <instruction>SHLL2</instruction> instruction extracts vector elements from the upper half of the source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHRN":
        case "SHRN2":
            return {
                "tooltip": "Shift Right Narrow (immediate). This instruction reads each unsigned integer value from the source SIMD&FP register, right shifts each result by an immediate value, puts the final result into a vector, and writes the vector to the lower or upper half of the destination SIMD&FP register. The destination vector elements are half as long as the source vector elements. The results are truncated. For rounded results, see RSHRN.",
                "html": "<p>Shift Right Narrow (immediate). This instruction reads each unsigned integer value from the source SIMD&amp;FP register, right shifts each result by an immediate value, puts the final result into a vector, and writes the vector to the lower or upper half of the destination SIMD&amp;FP register. The destination vector elements are half as long as the source vector elements. The results are truncated. For rounded results, see <xref linkend=\"A64.instructions.RSHRN_advsimd\">RSHRN</xref>.</p><p>The <instruction>RSHRN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>RSHRN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHRNB":
            return {
                "tooltip": "Shift each unsigned integer value in the source vector elements right by an immediate value, and place the truncated results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each unsigned integer value in the source vector elements right by an immediate value, and place the truncated results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHRNT":
            return {
                "tooltip": "Shift each unsigned integer value in the source vector elements right by an immediate value, and place the truncated results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each unsigned integer value in the source vector elements right by an immediate value, and place the truncated results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHSUB":
            return {
                "tooltip": "Signed Halving Subtract. This instruction subtracts the elements in the vector in the second source SIMD&FP register from the corresponding elements in the vector in the first source SIMD&FP register, shifts each result right one bit, places each result into elements of a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed Halving Subtract. This instruction subtracts the elements in the vector in the second source SIMD&amp;FP register from the corresponding elements in the vector in the first source SIMD&amp;FP register, shifts each result right one bit, places each result into elements of a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SHSUBR":
            return {
                "tooltip": "Subtract active signed elements of the first source vector from corresponding signed elements of the second source vector, shift right one bit, and destructively place the results in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Subtract active signed elements of the first source vector from corresponding signed elements of the second source vector, shift right one bit, and destructively place the results in the corresponding elements of the first source vector. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SLI":
            return {
                "tooltip": "Shift Left and Insert (immediate). This instruction reads each vector element in the source SIMD&FP register, left shifts each vector element by an immediate value, and inserts the result into the corresponding vector element in the destination SIMD&FP register such that the new zero bits created by the shift are not inserted but retain their existing value. Bits shifted out of the left of each vector element in the source register are lost.",
                "html": "<p>Shift Left and Insert (immediate). This instruction reads each vector element in the source SIMD&amp;FP register, left shifts each vector element by an immediate value, and inserts the result into the corresponding vector element in the destination SIMD&amp;FP register such that the new zero bits created by the shift are not inserted but retain their existing value. Bits shifted out of the left of each vector element in the source register are lost.</p><p></p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SM3PARTW1":
            return {
                "tooltip": "SM3PARTW1 takes three 128-bit vectors from the three source SIMD&FP registers and returns a 128-bit result in the destination SIMD&FP register. The result is obtained by a three-way exclusive-OR of the elements within the input vectors with some fixed rotations, see the Operation pseudocode for more information.",
                "html": "<p>SM3PARTW1 takes three 128-bit vectors from the three source SIMD&amp;FP registers and returns a 128-bit result in the destination SIMD&amp;FP register. The result is obtained by a three-way exclusive-OR of the elements within the input vectors with some fixed rotations, see the Operation pseudocode for more information.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SM3\">FEAT_SM3</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SM3PARTW2":
            return {
                "tooltip": "SM3PARTW2 takes three 128-bit vectors from three source SIMD&FP registers and returns a 128-bit result in the destination SIMD&FP register. The result is obtained by a three-way exclusive-OR of the elements within the input vectors with some fixed rotations, see the Operation pseudocode for more information.",
                "html": "<p>SM3PARTW2 takes three 128-bit vectors from three source SIMD&amp;FP registers and returns a 128-bit result in the destination SIMD&amp;FP register. The result is obtained by a three-way exclusive-OR of the elements within the input vectors with some fixed rotations, see the Operation pseudocode for more information.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SM3\">FEAT_SM3</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SM3SS1":
            return {
                "tooltip": "SM3SS1 rotates the top 32 bits of the 128-bit vector in the first source SIMD&FP register by 12, and adds that 32-bit value to the two other 32-bit values held in the top 32 bits of each of the 128-bit vectors in the second and third source SIMD&FP registers, rotating this result left by 7 and writing the final result into the top 32 bits of the vector in the destination SIMD&FP register, with the bottom 96 bits of the vector being written to 0.",
                "html": "<p>SM3SS1 rotates the top 32 bits of the 128-bit vector in the first source SIMD&amp;FP register by 12, and adds that 32-bit value to the two other 32-bit values held in the top 32 bits of each of the 128-bit vectors in the second and third source SIMD&amp;FP registers, rotating this result left by 7 and writing the final result into the top 32 bits of the vector in the destination SIMD&amp;FP register, with the bottom 96 bits of the vector being written to 0.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SM3\">FEAT_SM3</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SM3TT1A":
            return {
                "tooltip": "SM3TT1A takes three 128-bit vectors from three source SIMD&FP registers and a 2-bit immediate index value, and returns a 128-bit result in the destination SIMD&FP register.  It performs a three-way exclusive-OR of the three 32-bit fields held in the upper three elements of the first source vector, and adds the resulting 32-bit value and the following three other 32-bit values",
                "html": "<p>SM3TT1A takes three 128-bit vectors from three source SIMD&amp;FP registers and a 2-bit immediate index value, and returns a 128-bit result in the destination SIMD&amp;FP register.  It performs a three-way exclusive-OR of the three 32-bit fields held in the upper three elements of the first source vector, and adds the resulting 32-bit value and the following three other 32-bit values:</p><p>The result of this addition is returned as the top element of the result. The other elements of the result are taken from elements of the first source vector, with the element returned in bits&lt;63:32&gt; being rotated left by 9.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SM3\">FEAT_SM3</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SM3TT1B":
            return {
                "tooltip": "SM3TT1B takes three 128-bit vectors from three source SIMD&FP registers and a 2-bit immediate index value, and returns a 128-bit result in the destination SIMD&FP register. It performs a 32-bit majority function between the three 32-bit fields held in the upper three elements of the first source vector, and adds the resulting 32-bit value and the following three other 32-bit values",
                "html": "<p>SM3TT1B takes three 128-bit vectors from three source SIMD&amp;FP registers and a 2-bit immediate index value, and returns a 128-bit result in the destination SIMD&amp;FP register. It performs a 32-bit majority function between the three 32-bit fields held in the upper three elements of the first source vector, and adds the resulting 32-bit value and the following three other 32-bit values:</p><p>The result of this addition is returned as the top element of the result. The other elements of the result are taken from elements of the first source vector, with the element returned in bits&lt;63:32&gt; being rotated left by 9.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SM3\">FEAT_SM3</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SM3TT2A":
            return {
                "tooltip": "SM3TT2A takes three 128-bit vectors from three source SIMD&FP register and a 2-bit immediate index value, and returns a 128-bit result in the destination SIMD&FP register. It performs a three-way exclusive-OR of the three 32-bit fields held in the upper three elements of the first source vector, and adds the resulting 32-bit value and the following three other 32-bit values",
                "html": "<p>SM3TT2A takes three 128-bit vectors from three source SIMD&amp;FP register and a 2-bit immediate index value, and returns a 128-bit result in the destination SIMD&amp;FP register. It performs a three-way exclusive-OR of the three 32-bit fields held in the upper three elements of the first source vector, and adds the resulting 32-bit value and the following three other 32-bit values:</p><p>A three-way exclusive-OR is performed of the result of this addition, the result of the addition rotated left by 9, and the result of the addition rotated left by 17. The result of this exclusive-OR is returned as the top element of the returned result. The other elements of this result are taken from elements of the first source vector, with the element returned in bits&lt;63:32&gt; being rotated left by 19.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SM3\">FEAT_SM3</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SM3TT2B":
            return {
                "tooltip": "SM3TT2B takes three 128-bit vectors from three source SIMD&FP registers, and a 2-bit immediate index value, and returns a 128-bit result in the destination SIMD&FP register. It performs a 32-bit majority function between the three 32-bit fields held in the upper three elements of the first source vector, and adds the resulting 32-bit value and the following three other 32-bit values",
                "html": "<p>SM3TT2B takes three 128-bit vectors from three source SIMD&amp;FP registers, and a 2-bit immediate index value, and returns a 128-bit result in the destination SIMD&amp;FP register. It performs a 32-bit majority function between the three 32-bit fields held in the upper three elements of the first source vector, and adds the resulting 32-bit value and the following three other 32-bit values:</p><p>A three-way exclusive-OR is performed of the result of this addition, the result of the addition rotated left by 9, and the result of the addition rotated left by 17. The result of this exclusive-OR is returned as the top element of the returned result. The other elements of this result are taken from elements of the first source vector, with the element returned in bits&lt;63:32&gt; being rotated left by 19.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SM3\">FEAT_SM3</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SM4E":
            return {
                "tooltip": "SM4 Encode takes input data as a 128-bit vector from the first source SIMD&FP register, and four iterations of the round key held as the elements of the 128-bit vector in the second source SIMD&FP register. It encrypts the data by four rounds, in accordance with the SM4 standard, returning the 128-bit result to the destination SIMD&FP register.",
                "html": "<p>SM4 Encode takes input data as a 128-bit vector from the first source SIMD&amp;FP register, and four iterations of the round key held as the elements of the 128-bit vector in the second source SIMD&amp;FP register. It encrypts the data by four rounds, in accordance with the SM4 standard, returning the 128-bit result to the destination SIMD&amp;FP register.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SM4\">FEAT_SM4</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SM4EKEY":
            return {
                "tooltip": "SM4 Key takes an input as a 128-bit vector from the first source SIMD&FP register and a 128-bit constant from the second SIMD&FP register. It derives four iterations of the output key, in accordance with the SM4 standard, returning the 128-bit result to the destination SIMD&FP register.",
                "html": "<p>SM4 Key takes an input as a 128-bit vector from the first source SIMD&amp;FP register and a 128-bit constant from the second SIMD&amp;FP register. It derives four iterations of the output key, in accordance with the SM4 standard, returning the 128-bit result to the destination SIMD&amp;FP register.</p><p>This instruction is implemented only when <xref linkend=\"v8.2.SM4\">FEAT_SM4</xref> is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMADDL":
            return {
                "tooltip": "Signed Multiply-Add Long multiplies two 32-bit register values, adds a 64-bit register value, and writes the result to the 64-bit destination register.",
                "html": "<p>Signed Multiply-Add Long multiplies two 32-bit register values, adds a 64-bit register value, and writes the result to the 64-bit destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMAX":
            return {
                "tooltip": "Signed Maximum (vector). This instruction compares corresponding elements in the vectors in the two source SIMD&FP registers, places the larger of each pair of signed integer values into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed Maximum (vector). This instruction compares corresponding elements in the vectors in the two source SIMD&amp;FP registers, places the larger of each pair of signed integer values into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMAXP":
            return {
                "tooltip": "Signed Maximum Pairwise. This instruction creates a vector by concatenating the vector elements of the first source SIMD&FP register after the vector elements of the second source SIMD&FP register, reads each pair of adjacent vector elements in the two source SIMD&FP registers, writes the largest of each pair of signed integer values into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed Maximum Pairwise. This instruction creates a vector by concatenating the vector elements of the first source SIMD&amp;FP register after the vector elements of the second source SIMD&amp;FP register, reads each pair of adjacent vector elements in the two source SIMD&amp;FP registers, writes the largest of each pair of signed integer values into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMAXQV":
            return {
                "tooltip": "Signed maximum of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&FP destination register. Inactive elements in the source vector are treated as the minimum signed integer for the element size.",
                "html": "<p>Signed maximum of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&amp;FP destination register. Inactive elements in the source vector are treated as the minimum signed integer for the element size.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMAXV":
            return {
                "tooltip": "Signed Maximum across Vector. This instruction compares all the vector elements in the source SIMD&FP register, and writes the largest of the values as a scalar to the destination SIMD&FP register. All the values in this instruction are signed integer values.",
                "html": "<p>Signed Maximum across Vector. This instruction compares all the vector elements in the source SIMD&amp;FP register, and writes the largest of the values as a scalar to the destination SIMD&amp;FP register. All the values in this instruction are signed integer values.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMC":
            return {
                "tooltip": "Secure Monitor Call causes an exception to EL3.",
                "html": "<p>Secure Monitor Call causes an exception to EL3.</p><p><instruction>SMC</instruction> is available only for software executing at EL1 or higher. It is <arm-defined-word>undefined</arm-defined-word> in EL0.</p><p>If the values of <xref linkend=\"AArch64.hcr_el2\">HCR_EL2</xref>.TSC and <xref linkend=\"AArch64.scr_el3\">SCR_EL3</xref>.SMD are both 0, execution of an <instruction>SMC</instruction> instruction at EL1 or higher generates a Secure Monitor Call exception, recording it in <xref linkend=\"ESR_ELx\">ESR_ELx</xref>, using the EC value <hexnumber>0x17</hexnumber>, that is taken to EL3.</p><p>If the value of <xref linkend=\"AArch64.hcr_el2\">HCR_EL2</xref>.TSC is 1 and EL2 is enabled in the current Security state, execution of an <instruction>SMC</instruction> instruction at EL1 generates an exception that is taken to EL2, regardless of the value of <xref linkend=\"AArch64.scr_el3\">SCR_EL3</xref>.SMD.</p><p>If the value of <xref linkend=\"AArch64.hcr_el2\">HCR_EL2</xref>.TSC is 0 and the value of <xref linkend=\"AArch64.scr_el3\">SCR_EL3</xref>.SMD is 1, the SMC instruction is <arm-defined-word>undefined</arm-defined-word>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMIN":
            return {
                "tooltip": "Signed Minimum (vector). This instruction compares corresponding elements in the vectors in the two source SIMD&FP registers, places the smaller of each of the two signed integer values into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed Minimum (vector). This instruction compares corresponding elements in the vectors in the two source SIMD&amp;FP registers, places the smaller of each of the two signed integer values into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMINP":
            return {
                "tooltip": "Signed Minimum Pairwise. This instruction creates a vector by concatenating the vector elements of the first source SIMD&FP register after the vector elements of the second source SIMD&FP register, reads each pair of adjacent vector elements in the two source SIMD&FP registers, writes the smallest of each pair of signed integer values into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed Minimum Pairwise. This instruction creates a vector by concatenating the vector elements of the first source SIMD&amp;FP register after the vector elements of the second source SIMD&amp;FP register, reads each pair of adjacent vector elements in the two source SIMD&amp;FP registers, writes the smallest of each pair of signed integer values into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMINQV":
            return {
                "tooltip": "Signed minimum of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&FP destination register. Inactive elements in the source vector are treated as the maximum signed integer for the element size.",
                "html": "<p>Signed minimum of the same element numbers from each 128-bit source vector segment, placing each result into the corresponding element number of the 128-bit SIMD&amp;FP destination register. Inactive elements in the source vector are treated as the maximum signed integer for the element size.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMINV":
            return {
                "tooltip": "Signed Minimum across Vector. This instruction compares all the vector elements in the source SIMD&FP register, and writes the smallest of the values as a scalar to the destination SIMD&FP register. All the values in this instruction are signed integer values.",
                "html": "<p>Signed Minimum across Vector. This instruction compares all the vector elements in the source SIMD&amp;FP register, and writes the smallest of the values as a scalar to the destination SIMD&amp;FP register. All the values in this instruction are signed integer values.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMLAL":
        case "SMLAL2":
            return {
                "tooltip": "Signed Multiply-Add Long (vector, by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&FP register by the specified vector element in the second source SIMD&FP register, and accumulates the results with the vector elements of the destination SIMD&FP register. The destination vector elements are twice as long as the elements that are multiplied. All the values in this instruction are signed integer values.",
                "html": "<p>Signed Multiply-Add Long (vector, by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&amp;FP register by the specified vector element in the second source SIMD&amp;FP register, and accumulates the results with the vector elements of the destination SIMD&amp;FP register. The destination vector elements are twice as long as the elements that are multiplied. All the values in this instruction are signed integer values.</p><p>The <instruction>SMLAL</instruction> instruction extracts vector elements from the lower half of the first source register. The <instruction>SMLAL2</instruction> instruction extracts vector elements from the upper half of the first source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMLALB":
            return {
                "tooltip": "Multiply the corresponding even-numbered signed elements of the first and second source vectors and destructively add to the overlapping double-width elements of the addend vector. This instruction is unpredicated.",
                "html": "<p>Multiply the corresponding even-numbered signed elements of the first and second source vectors and destructively add to the overlapping double-width elements of the addend vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMLALL":
            return {
                "tooltip": "This signed integer multiply-add long-long instruction multiplies each signed 8-bit or 16-bit element in the one, two, or four first source vectors with each signed 8-bit or 16-bit indexed element of second source vector, widens each product to 32-bits or 64-bits and destructively adds these values to the corresponding 32-bit or 64-bit elements of the ZA quad-vector groups.",
                "html": "<p>This signed integer multiply-add long-long instruction multiplies each signed 8-bit or 16-bit element in the one, two, or four first source vectors with each signed 8-bit or 16-bit indexed element of second source vector, widens each product to 32-bits or 64-bits and destructively adds these values to the corresponding 32-bit or 64-bit elements of the ZA quad-vector groups.</p><p>The elements within the second source vector are specified using an immediate element index which selects the same element position within each 128-bit vector segment. The index range is from 0 to one less than the number of elements per 128-bit segment, encoded in 3 to 4 bits depending on the size of the element. The lowest of the four consecutive vector numbers forming the quad-vector group within all of, each half of, or each quarter of the ZA array are selected by the sum of the vector select register and immediate offset, modulo all, half, or quarter the number of ZA array vectors.</p><p>The <arm-defined-word>vector group</arm-defined-word> symbol, <syntax>VGx2</syntax> or <syntax>VGx4</syntax>, indicates that the ZA operand consists of two or four ZA quad-vector groups respectively. The <arm-defined-word>vector group</arm-defined-word> symbol is preferred for disassembly, but optional in assembler source code.</p><p>This instruction is unpredicated.</p><p>ID_AA64SMFR0_EL1.I16I64 indicates whether the 16-bit integer variant is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMLALT":
            return {
                "tooltip": "Multiply the corresponding odd-numbered signed elements of the first and second source vectors and destructively add to the overlapping double-width elements of the addend vector. This instruction is unpredicated.",
                "html": "<p>Multiply the corresponding odd-numbered signed elements of the first and second source vectors and destructively add to the overlapping double-width elements of the addend vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMLSL":
        case "SMLSL2":
            return {
                "tooltip": "Signed Multiply-Subtract Long (vector, by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&FP register by the specified vector element of the second source SIMD&FP register and subtracts the results from the vector elements of the destination SIMD&FP register. The destination vector elements are twice as long as the elements that are multiplied.",
                "html": "<p>Signed Multiply-Subtract Long (vector, by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&amp;FP register by the specified vector element of the second source SIMD&amp;FP register and subtracts the results from the vector elements of the destination SIMD&amp;FP register. The destination vector elements are twice as long as the elements that are multiplied.</p><p>The <instruction>SMLSL</instruction> instruction extracts vector elements from the lower half of the first source register. The <instruction>SMLSL2</instruction> instruction extracts vector elements from the upper half of the first source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMLSLB":
            return {
                "tooltip": "Multiply the corresponding even-numbered signed elements of the first and second source vectors and destructively subtract from the overlapping double-width elements of the addend vector. This instruction is unpredicated.",
                "html": "<p>Multiply the corresponding even-numbered signed elements of the first and second source vectors and destructively subtract from the overlapping double-width elements of the addend vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMLSLL":
            return {
                "tooltip": "This signed integer multiply-subtract long-long instruction multiplies each signed 8-bit or 16-bit element in the one, two, or four first source vectors with each signed 8-bit or 16-bit indexed element of second source vector, widens each product to 32-bits or 64-bits and destructively subtracts these values from the corresponding 32-bit or 64-bit elements of the ZA quad-vector groups.",
                "html": "<p>This signed integer multiply-subtract long-long instruction multiplies each signed 8-bit or 16-bit element in the one, two, or four first source vectors with each signed 8-bit or 16-bit indexed element of second source vector, widens each product to 32-bits or 64-bits and destructively subtracts these values from the corresponding 32-bit or 64-bit elements of the ZA quad-vector groups.</p><p>The elements within the second source vector are specified using an immediate element index which selects the same element position within each 128-bit vector segment. The index range is from 0 to one less than the number of elements per 128-bit segment, encoded in 3 to 4 bits depending on the size of the element. The lowest of the four consecutive vector numbers forming the quad-vector group within all of, each half of, or each quarter of the ZA array are selected by the sum of the vector select register and immediate offset, modulo all, half, or quarter the number of ZA array vectors.</p><p>The <arm-defined-word>vector group</arm-defined-word> symbol, <syntax>VGx2</syntax> or <syntax>VGx4</syntax>, indicates that the ZA operand consists of two or four ZA quad-vector groups respectively. The <arm-defined-word>vector group</arm-defined-word> symbol is preferred for disassembly, but optional in assembler source code.</p><p>This instruction is unpredicated.</p><p>ID_AA64SMFR0_EL1.I16I64 indicates whether the 16-bit integer variant is implemented.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMLSLT":
            return {
                "tooltip": "Multiply the corresponding odd-numbered signed elements of the first and second source vectors and destructively subtract from the overlapping double-width elements of the addend vector. This instruction is unpredicated.",
                "html": "<p>Multiply the corresponding odd-numbered signed elements of the first and second source vectors and destructively subtract from the overlapping double-width elements of the addend vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMMLA":
            return {
                "tooltip": "Signed 8-bit integer matrix multiply-accumulate. This instruction multiplies the 2x8 matrix of signed 8-bit integer values in the first source vector by the 8x2 matrix of signed 8-bit integer values in the second source vector. The resulting 2x2 32-bit integer matrix product is destructively added to the 32-bit integer matrix accumulator in the destination vector. This is equivalent to performing an 8-way dot product per destination element.",
                "html": "<p>Signed 8-bit integer matrix multiply-accumulate. This instruction multiplies the 2x8 matrix of signed 8-bit integer values in the first source vector by the 8x2 matrix of signed 8-bit integer values in the second source vector. The resulting 2x2 32-bit integer matrix product is destructively added to the 32-bit integer matrix accumulator in the destination vector. This is equivalent to performing an 8-way dot product per destination element.</p><p>From Armv8.2 to Armv8.5, this is an <arm-defined-word>optional</arm-defined-word> instruction. From Armv8.6 it is mandatory for implementations that include Advanced SIMD to support it. <xref linkend=\"AArch64.id_aa64isar1_el1\">ID_AA64ISAR1_EL1</xref>.I8MM indicates whether this instruction is supported.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMNEGL":
            return {
                "tooltip": "Signed Multiply-Negate Long multiplies two 32-bit register values, negates the product, and writes the result to the 64-bit destination register.",
                "html": "<p>Signed Multiply-Negate Long multiplies two 32-bit register values, negates the product, and writes the result to the 64-bit destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMOPA":
            return {
                "tooltip": "This instruction works with a 32-bit element ZA tile.",
                "html": "<p>This instruction works with a 32-bit element ZA tile.</p><p>The signed integer sum of outer products and accumulate instructions multiply the sub-matrix in the first source vector by the sub-matrix in the second source vector. The first source holds SVL<sub>S</sub>\u00d72 sub-matrix of signed 16-bit integer values, and the second source holds 2\u00d7SVL<sub>S</sub> sub-matrix of signed 16-bit integer values.</p><p>Each source vector is independently predicated by a corresponding governing predicate. When a 16-bit source element is inactive, it is treated as having the value 0.</p><p>The resulting SVL<sub>S</sub>\u00d7SVL<sub>S</sub> widened 32-bit integer sum of outer products is then destructively added to the 32-bit integer destination tile. This is equivalent to performing a 2-way dot product and accumulate to each of the destination tile elements.</p><p>Each 32-bit container of the first source vector holds 2 consecutive column elements of each row of a SVL<sub>S</sub>\u00d72 sub-matrix, and each 32-bit container of the second source vector holds 2 consecutive row elements of each column of a 2\u00d7SVL<sub>S</sub> sub-matrix.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMOPS":
            return {
                "tooltip": "This instruction works with a 32-bit element ZA tile.",
                "html": "<p>This instruction works with a 32-bit element ZA tile.</p><p>The signed integer sum of outer products and subtract instructions multiply the sub-matrix in the first source vector by the sub-matrix in the second source vector. The first source holds SVL<sub>S</sub>\u00d72 sub-matrix of signed 16-bit integer values, and the second source holds 2\u00d7SVL<sub>S</sub> sub-matrix of signed 16-bit integer values.</p><p>Each source vector is independently predicated by a corresponding governing predicate. When a 16-bit source element is inactive, it is treated as having the value 0.</p><p>The resulting SVL<sub>S</sub>\u00d7SVL<sub>S</sub> widened 32-bit integer sum of outer products is then destructively subtracted from the 32-bit integer destination tile. This is equivalent to performing a 2-way dot product and subtract from each of the destination tile elements.</p><p>Each 32-bit container of the first source vector holds 2 consecutive column elements of each row of a SVL<sub>S</sub>\u00d72 sub-matrix, and each 32-bit container of the second source vector holds 2 consecutive row elements of each column of a 2\u00d7SVL<sub>S</sub> sub-matrix.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMOV":
            return {
                "tooltip": "Signed Move vector element to general-purpose register. This instruction reads the signed integer from the source SIMD&FP register, sign-extends it to form a 32-bit or 64-bit value, and writes the result to destination general-purpose register.",
                "html": "<p>Signed Move vector element to general-purpose register. This instruction reads the signed integer from the source SIMD&amp;FP register, sign-extends it to form a 32-bit or 64-bit value, and writes the result to destination general-purpose register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMSTART":
            return {
                "tooltip": "Enables access to Streaming SVE mode and SME architectural state.",
                "html": "<p>Enables access to Streaming SVE mode and SME architectural state.</p><p>SMSTART enters Streaming SVE mode, and enables the SME ZA storage.</p><p>SMSTART SM enters Streaming SVE mode, but does not enable the SME ZA storage.</p><p>SMSTART ZA enables the SME ZA storage, but does not cause an entry to Streaming SVE mode.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMSTOP":
            return {
                "tooltip": "Disables access to Streaming SVE mode and SME architectural state.",
                "html": "<p>Disables access to Streaming SVE mode and SME architectural state.</p><p>SMSTOP exits Streaming SVE mode, and disables the SME ZA storage.</p><p>SMSTOP SM exits Streaming SVE mode, but does not disable the SME ZA storage.</p><p>SMSTOP ZA disables the SME ZA storage, but does not cause an exit from Streaming SVE mode.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMSUBL":
            return {
                "tooltip": "Signed Multiply-Subtract Long multiplies two 32-bit register values, subtracts the product from a 64-bit register value, and writes the result to the 64-bit destination register.",
                "html": "<p>Signed Multiply-Subtract Long multiplies two 32-bit register values, subtracts the product from a 64-bit register value, and writes the result to the 64-bit destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMULH":
            return {
                "tooltip": "Signed Multiply High multiplies two 64-bit register values, and writes bits[127:64] of the 128-bit result to the 64-bit destination register.",
                "html": "<p>Signed Multiply High multiplies two 64-bit register values, and writes bits[127:64] of the 128-bit result to the 64-bit destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMULL":
        case "SMULL2":
            return {
                "tooltip": "Signed Multiply Long (vector, by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&FP register by the specified vector element of the second source SIMD&FP register, places the result in a vector, and writes the vector to the destination SIMD&FP register. The destination vector elements are twice as long as the elements that are multiplied.",
                "html": "<p>Signed Multiply Long (vector, by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&amp;FP register by the specified vector element of the second source SIMD&amp;FP register, places the result in a vector, and writes the vector to the destination SIMD&amp;FP register. The destination vector elements are twice as long as the elements that are multiplied.</p><p>The <instruction>SMULL</instruction> instruction extracts vector elements from the lower half of the first source register. The <instruction>SMULL2</instruction> instruction extracts vector elements from the upper half of the first source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMULLB":
            return {
                "tooltip": "Multiply the corresponding even-numbered signed elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Multiply the corresponding even-numbered signed elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SMULLT":
            return {
                "tooltip": "Multiply the corresponding odd-numbered signed elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Multiply the corresponding odd-numbered signed elements of the first and second source vectors, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SPLICE":
            return {
                "tooltip": "Select a region from the first source vector and copy it to the lowest-numbered elements of the result. Then set any remaining elements of the result to a copy of the lowest-numbered elements from the second source vector. The region is selected using the first and last true elements in the vector select predicate register. The result is placed destructively in the destination and first source vector, or constructively in the destination vector.",
                "html": "<p>Select a region from the first source vector and copy it to the lowest-numbered elements of the result. Then set any remaining elements of the result to a copy of the lowest-numbered elements from the second source vector. The region is selected using the first and last true elements in the vector select predicate register. The result is placed destructively in the destination and first source vector, or constructively in the destination vector.</p><p>The Destructive encoding of this instruction might be immediately preceded in program order by a MOVPRFX instruction. The MOVPRFX instruction must conform to all of the following requirements, otherwise the behavior of the MOVPRFX and this instruction is UNPREDICTABLE: The MOVPRFX instruction must be unpredicated. The MOVPRFX instruction must specify the same destination register as this instruction. The destination register must not refer to architectural register state referenced by any other source operand register of this instruction.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQABS":
            return {
                "tooltip": "Signed saturating Absolute value. This instruction reads each vector element from the source SIMD&FP register, puts the absolute value of the result into a vector, and writes the vector to the destination SIMD&FP register. All the values in this instruction are signed integer values.",
                "html": "<p>Signed saturating Absolute value. This instruction reads each vector element from the source SIMD&amp;FP register, puts the absolute value of the result into a vector, and writes the vector to the destination SIMD&amp;FP register. All the values in this instruction are signed integer values.</p><p>If overflow occurs with any of the results, those results are saturated. If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQADD":
            return {
                "tooltip": "Signed saturating Add. This instruction adds the values of corresponding elements of the two source SIMD&FP registers, places the results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed saturating Add. This instruction adds the values of corresponding elements of the two source SIMD&amp;FP registers, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>If overflow occurs with any of the results, those results are saturated. If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQCADD":
            return {
                "tooltip": "Add the real and imaginary components of the integral complex numbers from the first source vector to the complex numbers from the second source vector which have first been rotated by 90 or 270 degrees in the direction from the positive real axis towards the positive imaginary axis, when considered in polar representation, equivalent to multiplying the complex numbers in the second source vector by \u00b1j beforehand. Destructively place the results in the corresponding elements of the first source vector. Each result element is saturated to the N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. This instruction is unpredicated.",
                "html": "<p>Add the real and imaginary components of the integral complex numbers from the first source vector to the complex numbers from the second source vector which have first been rotated by 90 or 270 degrees in the direction from the positive real axis towards the positive imaginary axis, when considered in polar representation, equivalent to multiplying the complex numbers in the second source vector by \u00b1<arm-defined-word>j</arm-defined-word> beforehand. Destructively place the results in the corresponding elements of the first source vector. Each result element is saturated to the N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. This instruction is unpredicated.</p><p>Each complex number is represented in a vector register as an even/odd pair of elements with the real part in the even-numbered element and the imaginary part in the odd-numbered element.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQCVT":
            return {
                "tooltip": "Saturate the signed integer value in each element of the two source vectors to half the original source element width, and place the results in the half-width destination elements.",
                "html": "<p>Saturate the signed integer value in each element of the two source vectors to half the original source element width, and place the results in the half-width destination elements.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQCVTN":
            return {
                "tooltip": "Saturate the signed integer value in each element of the group of two source vectors to half the original source element width, and place the two-way interleaved results in the half-width destination elements.",
                "html": "<p>Saturate the signed integer value in each element of the group of two source vectors to half the original source element width, and place the two-way interleaved results in the half-width destination elements.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQCVTU":
            return {
                "tooltip": "Saturate the signed integer value in each element of the two source vectors to unsigned integer value that is half the original source element width, and place the results in the half-width destination elements.",
                "html": "<p>Saturate the signed integer value in each element of the two source vectors to unsigned integer value that is half the original source element width, and place the results in the half-width destination elements.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQCVTUN":
            return {
                "tooltip": "Saturate the signed integer value in each element of the group of two source vectors to unsigned integer value that is half the original source element width, and place the two-way interleaved results in the half-width destination elements.",
                "html": "<p>Saturate the signed integer value in each element of the group of two source vectors to unsigned integer value that is half the original source element width, and place the two-way interleaved results in the half-width destination elements.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDECB":
            return {
                "tooltip": "Determines the number of active 8-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to decrement the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.",
                "html": "<p>Determines the number of active 8-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to decrement the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDECD":
            return {
                "tooltip": "Determines the number of active 64-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to decrement the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.",
                "html": "<p>Determines the number of active 64-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to decrement the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDECH":
            return {
                "tooltip": "Determines the number of active 16-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to decrement the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.",
                "html": "<p>Determines the number of active 16-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to decrement the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDECP":
            return {
                "tooltip": "Counts the number of true elements in the source predicate and then uses the result to decrement the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.",
                "html": "<p>Counts the number of true elements in the source predicate and then uses the result to decrement the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDECW":
            return {
                "tooltip": "Determines the number of active 32-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to decrement the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.",
                "html": "<p>Determines the number of active 32-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to decrement the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMLAL":
        case "SQDMLAL2":
            return {
                "tooltip": "Signed saturating Doubling Multiply-Add Long (by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&FP register by the specified vector element of the second source SIMD&FP register, doubles the results, and accumulates the final results with the vector elements of the destination SIMD&FP register. The destination vector elements are twice as long as the elements that are multiplied.",
                "html": "<p>Signed saturating Doubling Multiply-Add Long (by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&amp;FP register by the specified vector element of the second source SIMD&amp;FP register, doubles the results, and accumulates the final results with the vector elements of the destination SIMD&amp;FP register. The destination vector elements are twice as long as the elements that are multiplied.</p><p>If overflow occurs with any of the results, those results are saturated. If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>The <instruction>SQDMLAL</instruction> instruction extracts vector elements from the lower half of the first source register. The <instruction>SQDMLAL2</instruction> instruction extracts vector elements from the upper half of the first source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMLALB":
            return {
                "tooltip": "Multiply then double the corresponding even-numbered signed elements of the first and second source vectors. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2(N-1)  to (2(N-1) )-1. Then destructively add to the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. This instruction is unpredicated.",
                "html": "<p>Multiply then double the corresponding even-numbered signed elements of the first and second source vectors. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. Then destructively add to the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMLALBT":
            return {
                "tooltip": "Multiply then double the corresponding even-numbered signed elements of the first and odd-numbered signed elements of the second source vector. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2(N-1)  to (2(N-1) )-1. Then destructively add to the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. This instruction is unpredicated.",
                "html": "<p>Multiply then double the corresponding even-numbered signed elements of the first and odd-numbered signed elements of the second source vector. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. Then destructively add to the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMLALT":
            return {
                "tooltip": "Multiply then double the corresponding odd-numbered signed elements of the first and second source vectors. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2(N-1)  to (2(N-1) )-1. Then destructively add to the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. This instruction is unpredicated.",
                "html": "<p>Multiply then double the corresponding odd-numbered signed elements of the first and second source vectors. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. Then destructively add to the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMLSL":
        case "SQDMLSL2":
            return {
                "tooltip": "Signed saturating Doubling Multiply-Subtract Long (by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&FP register by the specified vector element of the second source SIMD&FP register, doubles the results, and subtracts the final results from the vector elements of the destination SIMD&FP register. The destination vector elements are twice as long as the elements that are multiplied. All the values in this instruction are signed integer values.",
                "html": "<p>Signed saturating Doubling Multiply-Subtract Long (by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&amp;FP register by the specified vector element of the second source SIMD&amp;FP register, doubles the results, and subtracts the final results from the vector elements of the destination SIMD&amp;FP register. The destination vector elements are twice as long as the elements that are multiplied. All the values in this instruction are signed integer values.</p><p>If overflow occurs with any of the results, those results are saturated. If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>The <instruction>SQDMLSL</instruction> instruction extracts vector elements from the lower half of the first source register. The <instruction>SQDMLSL2</instruction> instruction extracts vector elements from the upper half of the first source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMLSLB":
            return {
                "tooltip": "Multiply then double the corresponding even-numbered signed elements of the first and second source vectors. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2(N-1)  to (2(N-1) )-1. Then destructively subtract from the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. This instruction is unpredicated.",
                "html": "<p>Multiply then double the corresponding even-numbered signed elements of the first and second source vectors. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. Then destructively subtract from the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMLSLBT":
            return {
                "tooltip": "Multiply then double the corresponding even-numbered signed elements of the first and odd-numbered signed elements of the second source vector. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2(N-1)  to (2(N-1) )-1. Then destructively subtract from the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. This instruction is unpredicated.",
                "html": "<p>Multiply then double the corresponding even-numbered signed elements of the first and odd-numbered signed elements of the second source vector. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. Then destructively subtract from the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMLSLT":
            return {
                "tooltip": "Multiply then double the corresponding odd-numbered signed elements of the first and second source vectors. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2(N-1)  to (2(N-1) )-1. Then destructively subtract from the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. This instruction is unpredicated.",
                "html": "<p>Multiply then double the corresponding odd-numbered signed elements of the first and second source vectors. Each intermediate value is saturated to the double-width N-bit value's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. Then destructively subtract from the overlapping double-width elements of the addend and destination vector. Each destination element is saturated to the double-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMULH":
            return {
                "tooltip": "Signed saturating Doubling Multiply returning High half (by element). This instruction multiplies each vector element in the first source SIMD&FP register by the specified vector element of the second source SIMD&FP register, doubles the results, places the most significant half of the final results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed saturating Doubling Multiply returning High half (by element). This instruction multiplies each vector element in the first source SIMD&amp;FP register by the specified vector element of the second source SIMD&amp;FP register, doubles the results, places the most significant half of the final results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>The results are truncated. For rounded results, see <xref linkend=\"A64.instructions.SQRDMULH_advsimd_elt\">SQRDMULH</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMULL":
        case "SQDMULL2":
            return {
                "tooltip": "Signed saturating Doubling Multiply Long (by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&FP register by the specified vector element of the second source SIMD&FP register, doubles the results, places the final results in a vector, and writes the vector to the destination SIMD&FP register. All the values in this instruction are signed integer values.",
                "html": "<p>Signed saturating Doubling Multiply Long (by element). This instruction multiplies each vector element in the lower or upper half of the first source SIMD&amp;FP register by the specified vector element of the second source SIMD&amp;FP register, doubles the results, places the final results in a vector, and writes the vector to the destination SIMD&amp;FP register. All the values in this instruction are signed integer values.</p><p>If overflow occurs with any of the results, those results are saturated. If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>The <instruction>SQDMULL</instruction> instruction extracts the first source vector from the lower half of the first source register. The <instruction>SQDMULL2</instruction> instruction extracts the first source vector from the upper half of the first source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMULLB":
            return {
                "tooltip": "Multiply the corresponding even-numbered signed elements of the first and second source vectors, double and place the results in the overlapping double-width elements of the destination vector. Each result element is saturated to the double-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. This instruction is unpredicated.",
                "html": "<p>Multiply the corresponding even-numbered signed elements of the first and second source vectors, double and place the results in the overlapping double-width elements of the destination vector. Each result element is saturated to the double-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQDMULLT":
            return {
                "tooltip": "Multiply the corresponding odd-numbered signed elements of the first and second source vectors, double and place the results in the overlapping double-width elements of the destination vector. Each result element is saturated to the double-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. This instruction is unpredicated.",
                "html": "<p>Multiply the corresponding odd-numbered signed elements of the first and second source vectors, double and place the results in the overlapping double-width elements of the destination vector. Each result element is saturated to the double-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQINCB":
            return {
                "tooltip": "Determines the number of active 8-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to increment the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.",
                "html": "<p>Determines the number of active 8-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to increment the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQINCD":
            return {
                "tooltip": "Determines the number of active 64-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to increment the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.",
                "html": "<p>Determines the number of active 64-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to increment the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQINCH":
            return {
                "tooltip": "Determines the number of active 16-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to increment the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.",
                "html": "<p>Determines the number of active 16-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to increment the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQINCP":
            return {
                "tooltip": "Counts the number of true elements in the source predicate and then uses the result to increment the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.",
                "html": "<p>Counts the number of true elements in the source predicate and then uses the result to increment the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQINCW":
            return {
                "tooltip": "Determines the number of active 32-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to increment the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.",
                "html": "<p>Determines the number of active 32-bit elements implied by the named predicate constraint, multiplies that by an immediate in the range 1 to 16 inclusive, and then uses the result to increment the scalar destination. The result is saturated to the source general-purpose register's signed integer range. A 32-bit saturated result is then sign-extended to 64 bits.</p><p>The named predicate constraint limits the number of active elements in a single predicate to:</p><p>Unspecified or out of range constraint encodings generate an empty predicate or zero element count rather than Undefined Instruction exception.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQNEG":
            return {
                "tooltip": "Signed saturating Negate. This instruction reads each vector element from the source SIMD&FP register, negates each value, places the result into a vector, and writes the vector to the destination SIMD&FP register. All the values in this instruction are signed integer values.",
                "html": "<p>Signed saturating Negate. This instruction reads each vector element from the source SIMD&amp;FP register, negates each value, places the result into a vector, and writes the vector to the destination SIMD&amp;FP register. All the values in this instruction are signed integer values.</p><p>If overflow occurs with any of the results, those results are saturated. If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRDCMLAH":
            return {
                "tooltip": "Multiply without saturation the duplicated real components for rotations 0 and 180, or imaginary components for rotations 90 and 270, of the integral numbers in the first source vector by the corresponding complex number in the second source vector rotated by 0, 90, 180 or 270 degrees in the direction from the positive real axis towards the positive imaginary axis, when considered in polar representation.",
                "html": "<p>Multiply without saturation the duplicated real components for rotations 0 and 180, or imaginary components for rotations 90 and 270, of the integral numbers in the first source vector by the corresponding complex number in the second source vector rotated by 0, 90, 180 or 270 degrees in the direction from the positive real axis towards the positive imaginary axis, when considered in polar representation.</p><p>Then double and add the products to the corresponding components of the complex numbers in the addend vector. Destructively place the most significant rounded half of the results in the corresponding elements of the addend vector. Each result element is saturated to the N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. This instruction is unpredicated.</p><p>These transformations permit the creation of a variety of multiply-add and multiply-subtract operations on complex numbers by combining two of these instructions with the same vector operands but with rotations that are 90 degrees apart.</p><p>Each complex number is represented in a vector register as an even/odd pair of elements with the real part in the even-numbered element and the imaginary part in the odd-numbered element.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRDMLAH":
            return {
                "tooltip": "Signed Saturating Rounding Doubling Multiply Accumulate returning High Half (by element). This instruction multiplies the vector elements of the first source SIMD&FP register with the value of a vector element of the second source SIMD&FP register without saturating the multiply results, doubles the results, and accumulates the most significant half of the final results with the vector elements of the destination SIMD&FP register. The results are rounded.",
                "html": "<p>Signed Saturating Rounding Doubling Multiply Accumulate returning High Half (by element). This instruction multiplies the vector elements of the first source SIMD&amp;FP register with the value of a vector element of the second source SIMD&amp;FP register without saturating the multiply results, doubles the results, and accumulates the most significant half of the final results with the vector elements of the destination SIMD&amp;FP register. The results are rounded.</p><p>If any of the results overflow, they are saturated. The cumulative saturation bit, <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC, is set if saturation occurs.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRDMLSH":
            return {
                "tooltip": "Signed Saturating Rounding Doubling Multiply Subtract returning High Half (by element). This instruction multiplies the vector elements of the first source SIMD&FP register with the value of a vector element of the second source SIMD&FP register without saturating the multiply results, doubles the results, and subtracts the most significant half of the final results from the vector elements of the destination SIMD&FP register. The results are rounded.",
                "html": "<p>Signed Saturating Rounding Doubling Multiply Subtract returning High Half (by element). This instruction multiplies the vector elements of the first source SIMD&amp;FP register with the value of a vector element of the second source SIMD&amp;FP register without saturating the multiply results, doubles the results, and subtracts the most significant half of the final results from the vector elements of the destination SIMD&amp;FP register. The results are rounded.</p><p>If any of the results overflow, they are saturated. The cumulative saturation bit, <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC, is set if saturation occurs.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRDMULH":
            return {
                "tooltip": "Signed saturating Rounding Doubling Multiply returning High half (by element). This instruction multiplies each vector element in the first source SIMD&FP register by the specified vector element of the second source SIMD&FP register, doubles the results, places the most significant half of the final results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed saturating Rounding Doubling Multiply returning High half (by element). This instruction multiplies each vector element in the first source SIMD&amp;FP register by the specified vector element of the second source SIMD&amp;FP register, doubles the results, places the most significant half of the final results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>The results are rounded. For truncated results, see <xref linkend=\"A64.instructions.SQDMULH_advsimd_elt\">SQDMULH</xref>.</p><p>If any of the results overflows, they are saturated. If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRSHL":
            return {
                "tooltip": "Signed saturating Rounding Shift Left (register). This instruction takes each vector element in the first source SIMD&FP register, shifts it by a value from the least significant byte of the corresponding vector element of the second source SIMD&FP register, places the results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed saturating Rounding Shift Left (register). This instruction takes each vector element in the first source SIMD&amp;FP register, shifts it by a value from the least significant byte of the corresponding vector element of the second source SIMD&amp;FP register, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>If the shift value is positive, the operation is a left shift. Otherwise, it is a right shift. The results are rounded. For truncated results, see <xref linkend=\"A64.instructions.SQSHL_advsimd_reg\">SQSHL</xref>.</p><p>If overflow occurs with any of the results, those results are saturated. If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRSHLR":
            return {
                "tooltip": "Shift active signed elements of the second source vector by corresponding elements of the first source vector and destructively place the rounded results in the corresponding elements of the first source vector. A positive shift amount performs a left shift, otherwise a right shift by the negated shift amount is performed. Each result element is saturated to the N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Shift active signed elements of the second source vector by corresponding elements of the first source vector and destructively place the rounded results in the corresponding elements of the first source vector. A positive shift amount performs a left shift, otherwise a right shift by the negated shift amount is performed. Each result element is saturated to the N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRSHR":
            return {
                "tooltip": "Shift right by an immediate value, the signed integer value in each element of the two source vectors and place the rounded results in the half-width destination elements. Each result element is saturated to the half-width N-bit element's signed integer range -2(N-1) to (2(N-1))-1. The immediate shift amount is an unsigned value in the range 1 to 16.",
                "html": "<p>Shift right by an immediate value, the signed integer value in each element of the two source vectors and place the rounded results in the half-width destination elements. Each result element is saturated to the half-width N-bit element's signed integer range -2<sup>(N-1)</sup> to (2<sup>(N-1)</sup>)-1. The immediate shift amount is an unsigned value in the range 1 to 16.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRSHRN":
        case "SQRSHRN2":
            return {
                "tooltip": "Signed saturating Rounded Shift Right Narrow (immediate). This instruction reads each vector element in the source SIMD&FP register, right shifts each result by an immediate value, saturates each shifted result to a value that is half the original width, puts the final result into a vector, and writes the vector to the lower or upper half of the destination SIMD&FP register. All the values in this instruction are signed integer values. The destination vector elements are half as long as the source vector elements. The results are rounded. For truncated results, see SQSHRN.",
                "html": "<p>Signed saturating Rounded Shift Right Narrow (immediate). This instruction reads each vector element in the source SIMD&amp;FP register, right shifts each result by an immediate value, saturates each shifted result to a value that is half the original width, puts the final result into a vector, and writes the vector to the lower or upper half of the destination SIMD&amp;FP register. All the values in this instruction are signed integer values. The destination vector elements are half as long as the source vector elements. The results are rounded. For truncated results, see <xref linkend=\"A64.instructions.SQSHRN_advsimd\">SQSHRN</xref>.</p><p>The <instruction>SQRSHRN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>SQRSHRN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRSHRNB":
            return {
                "tooltip": "Shift each signed integer value in the source vector elements right by an immediate value, and place the rounded results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. Each result element is saturated to the half-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each signed integer value in the source vector elements right by an immediate value, and place the rounded results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. Each result element is saturated to the half-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRSHRNT":
            return {
                "tooltip": "Shift each signed integer value in the source vector elements right by an immediate value, and place the rounded results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. Each result element is saturated to the half-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each signed integer value in the source vector elements right by an immediate value, and place the rounded results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. Each result element is saturated to the half-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRSHRU":
            return {
                "tooltip": "Shift right by an immediate value, the signed integer value in each element of the two source vectors and place the rounded results in the half-width destination elements. Each result element is saturated to the half-width N-bit element's unsigned integer range 0 to (2N)-1. The immediate shift amount is an unsigned value in the range 1 to 16.",
                "html": "<p>Shift right by an immediate value, the signed integer value in each element of the two source vectors and place the rounded results in the half-width destination elements. Each result element is saturated to the half-width N-bit element's unsigned integer range 0 to (2<sup>N</sup>)-1. The immediate shift amount is an unsigned value in the range 1 to 16.</p><p>This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRSHRUN":
        case "SQRSHRUN2":
            return {
                "tooltip": "Signed saturating Rounded Shift Right Unsigned Narrow (immediate). This instruction reads each signed integer value in the vector of the source SIMD&FP register, right shifts each value by an immediate value, saturates the result to an unsigned integer value that is half the original width, places the final result into a vector, and writes the vector to the destination SIMD&FP register. The results are rounded. For truncated results, see SQSHRUN.",
                "html": "<p>Signed saturating Rounded Shift Right Unsigned Narrow (immediate). This instruction reads each signed integer value in the vector of the source SIMD&amp;FP register, right shifts each value by an immediate value, saturates the result to an unsigned integer value that is half the original width, places the final result into a vector, and writes the vector to the destination SIMD&amp;FP register. The results are rounded. For truncated results, see <xref linkend=\"A64.instructions.SQSHRUN_advsimd\">SQSHRUN</xref>.</p><p>The <instruction>SQRSHRUN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>SQRSHRUN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRSHRUNB":
            return {
                "tooltip": "Shift each signed integer value in the source vector elements right by an immediate value, and place the rounded results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. Each result element is saturated to the half-width N-bit element's unsigned integer range 0 to (2N)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each signed integer value in the source vector elements right by an immediate value, and place the rounded results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. Each result element is saturated to the half-width N-bit element's unsigned integer range 0 to (2<sup>N</sup>)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQRSHRUNT":
            return {
                "tooltip": "Shift each signed integer value in the source vector elements right by an immediate value, and place the rounded results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. Each result element is saturated to the half-width N-bit element's unsigned integer range 0 to (2N)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each signed integer value in the source vector elements right by an immediate value, and place the rounded results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. Each result element is saturated to the half-width N-bit element's unsigned integer range 0 to (2<sup>N</sup>)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQSHL":
            return {
                "tooltip": "Signed saturating Shift Left (immediate). This instruction reads each vector element in the source SIMD&FP register, shifts each result by an immediate value, places the final result in a vector, and writes the vector to the destination SIMD&FP register. The results are truncated. For rounded results, see UQRSHL.",
                "html": "<p>Signed saturating Shift Left (immediate). This instruction reads each vector element in the source SIMD&amp;FP register, shifts each result by an immediate value, places the final result in a vector, and writes the vector to the destination SIMD&amp;FP register. The results are truncated. For rounded results, see <xref linkend=\"A64.instructions.UQRSHL_advsimd\">UQRSHL</xref>.</p><p>If overflow occurs with any of the results, those results are saturated. If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQSHLR":
            return {
                "tooltip": "Shift active signed elements of the second source vector by corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. A positive shift amount performs a left shift, otherwise a right shift by the negated shift amount is performed. Each result element is saturated to the N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Shift active signed elements of the second source vector by corresponding elements of the first source vector and destructively place the results in the corresponding elements of the first source vector. A positive shift amount performs a left shift, otherwise a right shift by the negated shift amount is performed. Each result element is saturated to the N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQSHLU":
            return {
                "tooltip": "Signed saturating Shift Left Unsigned (immediate). This instruction reads each signed integer value in the vector of the source SIMD&FP register, shifts each value by an immediate value, saturates the shifted result to an unsigned integer value, places the result in a vector, and writes the vector to the destination SIMD&FP register. The results are truncated. For rounded results, see UQRSHL.",
                "html": "<p>Signed saturating Shift Left Unsigned (immediate). This instruction reads each signed integer value in the vector of the source SIMD&amp;FP register, shifts each value by an immediate value, saturates the shifted result to an unsigned integer value, places the result in a vector, and writes the vector to the destination SIMD&amp;FP register. The results are truncated. For rounded results, see <xref linkend=\"A64.instructions.UQRSHL_advsimd\">UQRSHL</xref>.</p><p>If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQSHRN":
        case "SQSHRN2":
            return {
                "tooltip": "Signed saturating Shift Right Narrow (immediate). This instruction reads each vector element in the source SIMD&FP register, right shifts and truncates each result by an immediate value, saturates each shifted result to a value that is half the original width, puts the final result into a vector, and writes the vector to the lower or upper half of the destination SIMD&FP register. All the values in this instruction are signed integer values. The destination vector elements are half as long as the source vector elements. For rounded results, see SQRSHRN.",
                "html": "<p>Signed saturating Shift Right Narrow (immediate). This instruction reads each vector element in the source SIMD&amp;FP register, right shifts and truncates each result by an immediate value, saturates each shifted result to a value that is half the original width, puts the final result into a vector, and writes the vector to the lower or upper half of the destination SIMD&amp;FP register. All the values in this instruction are signed integer values. The destination vector elements are half as long as the source vector elements. For rounded results, see <xref linkend=\"A64.instructions.SQRSHRN_advsimd\">SQRSHRN</xref>.</p><p>The <instruction>SQSHRN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>SQSHRN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQSHRNB":
            return {
                "tooltip": "Shift each signed integer value in the source vector elements right by an immediate value, and place the truncated results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. Each result element is saturated to the half-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each signed integer value in the source vector elements right by an immediate value, and place the truncated results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. Each result element is saturated to the half-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQSHRNT":
            return {
                "tooltip": "Shift each signed integer value in the source vector elements right by an immediate value, and place the truncated results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. Each result element is saturated to the half-width N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each signed integer value in the source vector elements right by an immediate value, and place the truncated results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. Each result element is saturated to the half-width N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQSHRUN":
        case "SQSHRUN2":
            return {
                "tooltip": "Signed saturating Shift Right Unsigned Narrow (immediate). This instruction reads each signed integer value in the vector of the source SIMD&FP register, right shifts each value by an immediate value, saturates the result to an unsigned integer value that is half the original width, places the final result into a vector, and writes the vector to the destination SIMD&FP register. The results are truncated. For rounded results, see SQRSHRUN.",
                "html": "<p>Signed saturating Shift Right Unsigned Narrow (immediate). This instruction reads each signed integer value in the vector of the source SIMD&amp;FP register, right shifts each value by an immediate value, saturates the result to an unsigned integer value that is half the original width, places the final result into a vector, and writes the vector to the destination SIMD&amp;FP register. The results are truncated. For rounded results, see <xref linkend=\"A64.instructions.SQRSHRUN_advsimd\">SQRSHRUN</xref>.</p><p>The <instruction>SQSHRUN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>SQSHRUN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQSHRUNB":
            return {
                "tooltip": "Shift each signed integer value in the source vector elements right by an immediate value, and place the truncated results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. Each result element is saturated to the half-width N-bit element's unsigned integer range 0 to (2N)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each signed integer value in the source vector elements right by an immediate value, and place the truncated results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. Each result element is saturated to the half-width N-bit element's unsigned integer range 0 to (2<sup>N</sup>)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQSHRUNT":
            return {
                "tooltip": "Shift each signed integer value in the source vector elements right by an immediate value, and place the truncated results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. Each result element is saturated to the half-width N-bit element's unsigned integer range 0 to (2N)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.",
                "html": "<p>Shift each signed integer value in the source vector elements right by an immediate value, and place the truncated results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. Each result element is saturated to the half-width N-bit element's unsigned integer range 0 to (2<sup>N</sup>)-1. The immediate shift amount is an unsigned value in the range 1 to number of bits per element. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQSUB":
            return {
                "tooltip": "Signed saturating Subtract. This instruction subtracts the element values of the second source SIMD&FP register from the corresponding element values of the first source SIMD&FP register, places the results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed saturating Subtract. This instruction subtracts the element values of the second source SIMD&amp;FP register from the corresponding element values of the first source SIMD&amp;FP register, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>If overflow occurs with any of the results, those results are saturated. If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQSUBR":
            return {
                "tooltip": "Subtract active signed elements of the first source vector from corresponding signed elements of the second source vector and destructively place the results in the corresponding elements of the first source vector. Each result element is saturated to the N-bit element's signed integer range -2(N-1)  to (2(N-1) )-1. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Subtract active signed elements of the first source vector from corresponding signed elements of the second source vector and destructively place the results in the corresponding elements of the first source vector. Each result element is saturated to the N-bit element's signed integer range -2<sup>(N-1) </sup> to (2<sup>(N-1) </sup>)-1. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQXTN":
        case "SQXTN2":
            return {
                "tooltip": "Signed saturating extract Narrow. This instruction reads each vector element from the source SIMD&FP register, saturates the value to half the original width, places the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&FP register. The destination vector elements are half as long as the source vector elements. All the values in this instruction are signed integer values.",
                "html": "<p>Signed saturating extract Narrow. This instruction reads each vector element from the source SIMD&amp;FP register, saturates the value to half the original width, places the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&amp;FP register. The destination vector elements are half as long as the source vector elements. All the values in this instruction are signed integer values.</p><p>If overflow occurs with any of the results, those results are saturated. If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>The <instruction>SQXTN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>SQXTN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQXTNB":
            return {
                "tooltip": "Saturate the signed integer value in each source element to half the original source element width, and place the results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero.",
                "html": "<p>Saturate the signed integer value in each source element to half the original source element width, and place the results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQXTNT":
            return {
                "tooltip": "Saturate the signed integer value in each source element to half the original source element width, and place the results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged.",
                "html": "<p>Saturate the signed integer value in each source element to half the original source element width, and place the results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQXTUN":
        case "SQXTUN2":
            return {
                "tooltip": "Signed saturating extract Unsigned Narrow. This instruction reads each signed integer value in the vector of the source SIMD&FP register, saturates the value to an unsigned integer value that is half the original width, places the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&FP register. The destination vector elements are half as long as the source vector elements.",
                "html": "<p>Signed saturating extract Unsigned Narrow. This instruction reads each signed integer value in the vector of the source SIMD&amp;FP register, saturates the value to an unsigned integer value that is half the original width, places the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&amp;FP register. The destination vector elements are half as long as the source vector elements.</p><p>If saturation occurs, the cumulative saturation bit <xref linkend=\"AArch64.fpsr\">FPSR</xref>.QC is set.</p><p>The <instruction>SQXTUN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>SQXTUN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQXTUNB":
            return {
                "tooltip": "Saturate the signed integer value in each source element to an unsigned integer value that is half the original source element width, and place the results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero.",
                "html": "<p>Saturate the signed integer value in each source element to an unsigned integer value that is half the original source element width, and place the results in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SQXTUNT":
            return {
                "tooltip": "Saturate the signed integer value in each source element to an unsigned integer value that is half the original source element width, and place the results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged.",
                "html": "<p>Saturate the signed integer value in each source element to an unsigned integer value that is half the original source element width, and place the results in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SRHADD":
            return {
                "tooltip": "Signed Rounding Halving Add. This instruction adds corresponding signed integer values from the two source SIMD&FP registers, shifts each result right one bit, places the results into a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed Rounding Halving Add. This instruction adds corresponding signed integer values from the two source SIMD&amp;FP registers, shifts each result right one bit, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>The results are rounded. For truncated results, see <xref linkend=\"A64.instructions.SHADD_advsimd\">SHADD</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SRI":
            return {
                "tooltip": "Shift Right and Insert (immediate). This instruction reads each vector element in the source SIMD&FP register, right shifts each vector element by an immediate value, and inserts the result into the corresponding vector element in the destination SIMD&FP register such that the new zero bits created by the shift are not inserted but retain their existing value. Bits shifted out of the right of each vector element of the source register are lost.",
                "html": "<p>Shift Right and Insert (immediate). This instruction reads each vector element in the source SIMD&amp;FP register, right shifts each vector element by an immediate value, and inserts the result into the corresponding vector element in the destination SIMD&amp;FP register such that the new zero bits created by the shift are not inserted but retain their existing value. Bits shifted out of the right of each vector element of the source register are lost.</p><p></p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SRSHL":
            return {
                "tooltip": "Signed Rounding Shift Left (register). This instruction takes each signed integer value in the vector of the first source SIMD&FP register, shifts it by a value from the least significant byte of the corresponding element of the second source SIMD&FP register, places the results in a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed Rounding Shift Left (register). This instruction takes each signed integer value in the vector of the first source SIMD&amp;FP register, shifts it by a value from the least significant byte of the corresponding element of the second source SIMD&amp;FP register, places the results in a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>If the shift value is positive, the operation is a left shift. If the shift value is negative, it is a rounding right shift. For a truncating shift, see <xref linkend=\"A64.instructions.SSHL_advsimd\">SSHL</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SRSHLR":
            return {
                "tooltip": "Shift active signed elements of the second source vector by corresponding elements of the first source vector and destructively place the rounded results in the corresponding elements of the first source vector. A positive shift amount performs a left shift, otherwise a right shift by the negated shift amount is performed. Inactive elements in the destination vector register remain unmodified.",
                "html": "<p>Shift active signed elements of the second source vector by corresponding elements of the first source vector and destructively place the rounded results in the corresponding elements of the first source vector. A positive shift amount performs a left shift, otherwise a right shift by the negated shift amount is performed. Inactive elements in the destination vector register remain unmodified.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SRSHR":
            return {
                "tooltip": "Signed Rounding Shift Right (immediate). This instruction reads each vector element in the source SIMD&FP register, right shifts each result by an immediate value, places the final result into a vector, and writes the vector to the destination SIMD&FP register. All the values in this instruction are signed integer values. The results are rounded. For truncated results, see SSHR.",
                "html": "<p>Signed Rounding Shift Right (immediate). This instruction reads each vector element in the source SIMD&amp;FP register, right shifts each result by an immediate value, places the final result into a vector, and writes the vector to the destination SIMD&amp;FP register. All the values in this instruction are signed integer values. The results are rounded. For truncated results, see <xref linkend=\"A64.instructions.SSHR_advsimd\">SSHR</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SRSRA":
            return {
                "tooltip": "Signed Rounding Shift Right and Accumulate (immediate). This instruction reads each vector element in the source SIMD&FP register, right shifts each result by an immediate value, and accumulates the final results with the vector elements of the destination SIMD&FP register. All the values in this instruction are signed integer values. The results are rounded. For truncated results, see SSRA.",
                "html": "<p>Signed Rounding Shift Right and Accumulate (immediate). This instruction reads each vector element in the source SIMD&amp;FP register, right shifts each result by an immediate value, and accumulates the final results with the vector elements of the destination SIMD&amp;FP register. All the values in this instruction are signed integer values. The results are rounded. For truncated results, see <xref linkend=\"A64.instructions.SSRA_advsimd\">SSRA</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSBB":
            return {
                "tooltip": "Speculative Store Bypass Barrier is a memory barrier that prevents speculative loads from bypassing earlier stores to the same virtual address under certain conditions. For more information and details of the semantics, see Speculative Store Bypass Barrier (SSBB).",
                "html": "<p>Speculative Store Bypass Barrier is a memory barrier that prevents speculative loads from bypassing earlier stores to the same virtual address under certain conditions. For more information and details of the semantics, see <xref linkend=\"CHDFGADD\">Speculative Store Bypass Barrier (SSBB)</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSHL":
            return {
                "tooltip": "Signed Shift Left (register). This instruction takes each signed integer value in the vector of the first source SIMD&FP register, shifts each value by a value from the least significant byte of the corresponding element of the second source SIMD&FP register, places the results in a vector, and writes the vector to the destination SIMD&FP register.",
                "html": "<p>Signed Shift Left (register). This instruction takes each signed integer value in the vector of the first source SIMD&amp;FP register, shifts each value by a value from the least significant byte of the corresponding element of the second source SIMD&amp;FP register, places the results in a vector, and writes the vector to the destination SIMD&amp;FP register.</p><p>If the shift value is positive, the operation is a left shift. If the shift value is negative, it is a truncating right shift. For a rounding shift, see <xref linkend=\"A64.instructions.SRSHL_advsimd\">SRSHL</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSHLL":
        case "SSHLL2":
            return {
                "tooltip": "Signed Shift Left Long (immediate). This instruction reads each vector element from the source SIMD&FP register, left shifts each vector element by the specified shift amount, places the result into a vector, and writes the vector to the destination SIMD&FP register. The destination vector elements are twice as long as the source vector elements. All the values in this instruction are signed integer values.",
                "html": "<p>Signed Shift Left Long (immediate). This instruction reads each vector element from the source SIMD&amp;FP register, left shifts each vector element by the specified shift amount, places the result into a vector, and writes the vector to the destination SIMD&amp;FP register. The destination vector elements are twice as long as the source vector elements. All the values in this instruction are signed integer values.</p><p>The <instruction>SSHLL</instruction> instruction extracts vector elements from the lower half of the source register. The <instruction>SSHLL2</instruction> instruction extracts vector elements from the upper half of the source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSHLLB":
            return {
                "tooltip": "Shift left by immediate each even-numbered signed element of the source vector, and place the results in the overlapping double-width elements of the destination vector. The immediate shift amount is an unsigned value in the range 0 to number of bits per element minus 1. This instruction is unpredicated.",
                "html": "<p>Shift left by immediate each even-numbered signed element of the source vector, and place the results in the overlapping double-width elements of the destination vector. The immediate shift amount is an unsigned value in the range 0 to number of bits per element minus 1. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSHLLT":
            return {
                "tooltip": "Shift left by immediate each odd-numbered signed element of the source vector, and place the results in the overlapping double-width elements of the destination vector. The immediate shift amount is an unsigned value in the range 0 to number of bits per element minus 1. This instruction is unpredicated.",
                "html": "<p>Shift left by immediate each odd-numbered signed element of the source vector, and place the results in the overlapping double-width elements of the destination vector. The immediate shift amount is an unsigned value in the range 0 to number of bits per element minus 1. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSHR":
            return {
                "tooltip": "Signed Shift Right (immediate). This instruction reads each vector element in the source SIMD&FP register, right shifts each result by an immediate value, places the final result into a vector, and writes the vector to the destination SIMD&FP register. All the values in this instruction are signed integer values. The results are truncated. For rounded results, see SRSHR.",
                "html": "<p>Signed Shift Right (immediate). This instruction reads each vector element in the source SIMD&amp;FP register, right shifts each result by an immediate value, places the final result into a vector, and writes the vector to the destination SIMD&amp;FP register. All the values in this instruction are signed integer values. The results are truncated. For rounded results, see <xref linkend=\"A64.instructions.SRSHR_advsimd\">SRSHR</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSRA":
            return {
                "tooltip": "Signed Shift Right and Accumulate (immediate). This instruction reads each vector element in the source SIMD&FP register, right shifts each result by an immediate value, and accumulates the final results with the vector elements of the destination SIMD&FP register. All the values in this instruction are signed integer values. The results are truncated. For rounded results, see SRSRA.",
                "html": "<p>Signed Shift Right and Accumulate (immediate). This instruction reads each vector element in the source SIMD&amp;FP register, right shifts each result by an immediate value, and accumulates the final results with the vector elements of the destination SIMD&amp;FP register. All the values in this instruction are signed integer values. The results are truncated. For rounded results, see <xref linkend=\"A64.instructions.SRSRA_advsimd\">SRSRA</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSUBL":
        case "SSUBL2":
            return {
                "tooltip": "Signed Subtract Long. This instruction subtracts each vector element in the lower or upper half of the second source SIMD&FP register from the corresponding vector element of the first source SIMD&FP register, places the results into a vector, and writes the vector to the destination SIMD&FP register. All the values in this instruction are signed integer values. The destination vector elements are twice as long as the source vector elements.",
                "html": "<p>Signed Subtract Long. This instruction subtracts each vector element in the lower or upper half of the second source SIMD&amp;FP register from the corresponding vector element of the first source SIMD&amp;FP register, places the results into a vector, and writes the vector to the destination SIMD&amp;FP register. All the values in this instruction are signed integer values. The destination vector elements are twice as long as the source vector elements.</p><p>The <instruction>SSUBL</instruction> instruction extracts each source vector from the lower half of each source register. The <instruction>SSUBL2</instruction> instruction extracts each source vector from the upper half of each source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSUBLB":
            return {
                "tooltip": "Subtract the even-numbered signed elements of the second source vector from the corresponding signed elements of the first source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Subtract the even-numbered signed elements of the second source vector from the corresponding signed elements of the first source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSUBLBT":
            return {
                "tooltip": "Subtract the odd-numbered signed elements of the second source vector from the even-numbered signed elements of the first source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Subtract the odd-numbered signed elements of the second source vector from the even-numbered signed elements of the first source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSUBLT":
            return {
                "tooltip": "Subtract the odd-numbered signed elements of the second source vector from the corresponding signed elements of the first source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Subtract the odd-numbered signed elements of the second source vector from the corresponding signed elements of the first source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSUBLTB":
            return {
                "tooltip": "Subtract the even-numbered signed elements of the second source vector from the odd-numbered signed elements of the first source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Subtract the even-numbered signed elements of the second source vector from the odd-numbered signed elements of the first source vector, and place the results in the overlapping double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSUBW":
        case "SSUBW2":
            return {
                "tooltip": "Signed Subtract Wide. This instruction subtracts each vector element in the lower or upper half of the second source SIMD&FP register from the corresponding vector element in the first source SIMD&FP register, places the result in a vector, and writes the vector to the SIMD&FP destination register. All the values in this instruction are signed integer values.",
                "html": "<p>Signed Subtract Wide. This instruction subtracts each vector element in the lower or upper half of the second source SIMD&amp;FP register from the corresponding vector element in the first source SIMD&amp;FP register, places the result in a vector, and writes the vector to the SIMD&amp;FP destination register. All the values in this instruction are signed integer values.</p><p>The <instruction>SSUBW</instruction> instruction extracts the second source vector from the lower half of the second source register. The <instruction>SSUBW2</instruction> instruction extracts the second source vector from the upper half of the second source register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSUBWB":
            return {
                "tooltip": "Subtract the even-numbered signed elements of the second source vector from the overlapping double-width elements of the first source vector and place the results in the corresponding double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Subtract the even-numbered signed elements of the second source vector from the overlapping double-width elements of the first source vector and place the results in the corresponding double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SSUBWT":
            return {
                "tooltip": "Subtract the even-numbered signed elements of the second source vector from the overlapping double-width elements of the first source vector and place the results in the corresponding double-width elements of the destination vector. This instruction is unpredicated.",
                "html": "<p>Subtract the even-numbered signed elements of the second source vector from the overlapping double-width elements of the first source vector and place the results in the corresponding double-width elements of the destination vector. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST1":
            return {
                "tooltip": "Store multiple single-element structures from one, two, three, or four registers. This instruction stores elements to memory from one, two, three, or four SIMD&FP registers, without interleaving. Every element of each register is stored.",
                "html": "<p>Store multiple single-element structures from one, two, three, or four registers. This instruction stores elements to memory from one, two, three, or four SIMD&amp;FP registers, without interleaving. Every element of each register is stored.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST1B":
            return {
                "tooltip": "Contiguous store of bytes from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous store of bytes from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST1D":
            return {
                "tooltip": "Contiguous store of doublewords from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous store of doublewords from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST1H":
            return {
                "tooltip": "Contiguous store of halfwords from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous store of halfwords from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST1Q":
            return {
                "tooltip": "Scatter store of quadwords from the active elements of a vector register to the memory addresses generated by a vector base plus a 64-bit unscaled scalar register offset. Inactive elements are not written to memory.",
                "html": "<p>Scatter store of quadwords from the active elements of a vector register to the memory addresses generated by a vector base plus a 64-bit unscaled scalar register offset. Inactive elements are not written to memory.</p><p>This instruction is illegal when executed in Streaming SVE mode, unless FEAT_SME_FA64 is implemented and enabled.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST1W":
            return {
                "tooltip": "Contiguous store of words from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous store of words from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST2":
            return {
                "tooltip": "Store multiple 2-element structures from two registers. This instruction stores multiple 2-element structures from two SIMD&FP registers to memory, with interleaving. Every element of each register is stored.",
                "html": "<p>Store multiple 2-element structures from two registers. This instruction stores multiple 2-element structures from two SIMD&amp;FP registers to memory, with interleaving. Every element of each register is stored.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST2B":
            return {
                "tooltip": "Contiguous store two-byte structures, each from the same element number in two vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store two-byte structures, each from the same element number in two vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the two vector registers, or equivalently to the two consecutive bytes in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST2D":
            return {
                "tooltip": "Contiguous store two-doubleword structures, each from the same element number in two vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store two-doubleword structures, each from the same element number in two vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the two vector registers, or equivalently to the two consecutive doublewords in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST2G":
            return {
                "tooltip": "Store Allocation Tags stores an Allocation Tag to two Tag granules of memory. The address used for the store is calculated from the base register and an immediate signed offset scaled by the Tag granule. The Allocation Tag is calculated from the Logical Address Tag in the source register.",
                "html": "<p>Store Allocation Tags stores an Allocation Tag to two Tag granules of memory. The address used for the store is calculated from the base register and an immediate signed offset scaled by the Tag granule. The Allocation Tag is calculated from the Logical Address Tag in the source register.</p><p>This instruction generates an Unchecked access.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST2H":
            return {
                "tooltip": "Contiguous store two-halfword structures, each from the same element number in two vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store two-halfword structures, each from the same element number in two vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the two vector registers, or equivalently to the two consecutive halfwords in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST2Q":
            return {
                "tooltip": "Contiguous store two-quadword structures, each from the same element number in two vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store two-quadword structures, each from the same element number in two vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the two vector registers, or equivalently to the two consecutive quadwords in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST2W":
            return {
                "tooltip": "Contiguous store two-word structures, each from the same element number in two vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store two-word structures, each from the same element number in two vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 2 in the range -16 to 14 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the two vector registers, or equivalently to the two consecutive words in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST3":
            return {
                "tooltip": "Store multiple 3-element structures from three registers. This instruction stores multiple 3-element structures to memory from three SIMD&FP registers, with interleaving. Every element of each register is stored.",
                "html": "<p>Store multiple 3-element structures from three registers. This instruction stores multiple 3-element structures to memory from three SIMD&amp;FP registers, with interleaving. Every element of each register is stored.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST3B":
            return {
                "tooltip": "Contiguous store three-byte structures, each from the same element number in three vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store three-byte structures, each from the same element number in three vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the three vector registers, or equivalently to the three consecutive bytes in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST3D":
            return {
                "tooltip": "Contiguous store three-doubleword structures, each from the same element number in three vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store three-doubleword structures, each from the same element number in three vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the three vector registers, or equivalently to the three consecutive doublewords in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST3H":
            return {
                "tooltip": "Contiguous store three-halfword structures, each from the same element number in three vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store three-halfword structures, each from the same element number in three vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the three vector registers, or equivalently to the three consecutive halfwords in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST3Q":
            return {
                "tooltip": "Contiguous store three-quadword structures, each from the same element number in three vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store three-quadword structures, each from the same element number in three vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the three vector registers, or equivalently to the three consecutive quadwords in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST3W":
            return {
                "tooltip": "Contiguous store three-word structures, each from the same element number in three vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store three-word structures, each from the same element number in three vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 3 in the range -24 to 21 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the three vector registers, or equivalently to the three consecutive words in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST4":
            return {
                "tooltip": "Store multiple 4-element structures from four registers. This instruction stores multiple 4-element structures to memory from four SIMD&FP registers, with interleaving. Every element of each register is stored.",
                "html": "<p>Store multiple 4-element structures from four registers. This instruction stores multiple 4-element structures to memory from four SIMD&amp;FP registers, with interleaving. Every element of each register is stored.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST4B":
            return {
                "tooltip": "Contiguous store four-byte structures, each from the same element number in four vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store four-byte structures, each from the same element number in four vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the four vector registers, or equivalently to the four consecutive bytes in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST4D":
            return {
                "tooltip": "Contiguous store four-doubleword structures, each from the same element number in four vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store four-doubleword structures, each from the same element number in four vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the four vector registers, or equivalently to the four consecutive doublewords in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST4H":
            return {
                "tooltip": "Contiguous store four-halfword structures, each from the same element number in four vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store four-halfword structures, each from the same element number in four vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the four vector registers, or equivalently to the four consecutive halfwords in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST4Q":
            return {
                "tooltip": "Contiguous store four-quadword structures, each from the same element number in four vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store four-quadword structures, each from the same element number in four vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the four vector registers, or equivalently to the four consecutive quadwords in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST4W":
            return {
                "tooltip": "Contiguous store four-word structures, each from the same element number in four vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication",
                "html": "<p>Contiguous store four-word structures, each from the same element number in four vector registers to the memory address generated by a 64-bit scalar base and an immediate index which is a multiple of 4 in the range -32 to 28 that is multiplied by the vector's in-memory size, irrespective of predication,</p><p>Each predicate element applies to the same element number in each of the four vector registers, or equivalently to the four consecutive words in memory which make up each structure. Inactive structures are not written to memory.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST64B":
            return {
                "tooltip": "Single-copy Atomic 64-byte Store without Return stores eight 64-bit doublewords from consecutive registers, Xt to X(t+7), to a memory location. The data that is stored is atomic and is required to be 64-byte-aligned.",
                "html": "<p>Single-copy Atomic 64-byte Store without Return stores eight 64-bit doublewords from consecutive registers, <syntax>Xt</syntax> to <syntax>X(t+7)</syntax>, to a memory location. The data that is stored is atomic and is required to be 64-byte-aligned.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST64BV":
            return {
                "tooltip": "Single-copy Atomic 64-byte Store with Return stores eight 64-bit doublewords from consecutive registers, Xt to X(t+7), to a memory location, and writes the status result of the store to a register. The data that is stored is atomic and is required to be 64-byte aligned.",
                "html": "<p>Single-copy Atomic 64-byte Store with Return stores eight 64-bit doublewords from consecutive registers, <syntax>Xt</syntax> to <syntax>X(t+7)</syntax>, to a memory location, and writes the status result of the store to a register. The data that is stored is atomic and is required to be 64-byte aligned.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "ST64BV0":
            return {
                "tooltip": "Single-copy Atomic 64-byte EL0 Store with Return stores eight 64-bit doublewords from consecutive registers, Xt to X(t+7), to a memory location, with the bottom 32 bits taken from ACCDATA_EL1, and writes the status result of the store to a register. The data that is stored is atomic and is required to be 64-byte aligned.",
                "html": "<p>Single-copy Atomic 64-byte EL0 Store with Return stores eight 64-bit doublewords from consecutive registers, <syntax>Xt</syntax> to <syntax>X(t+7)</syntax>, to a memory location, with the bottom 32 bits taken from <xref linkend=\"AArch64.accdata_el1\">ACCDATA_EL1</xref>, and writes the status result of the store to a register. The data that is stored is atomic and is required to be 64-byte aligned.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STADD":
        case "STADDL":
            return {
                "tooltip": "Atomic add on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, adds the value held in a register to it, and stores the result back to memory.",
                "html": "<p>Atomic add on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, adds the value held in a register to it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STADDB":
        case "STADDLB":
            return {
                "tooltip": "Atomic add on byte in memory, without return, atomically loads an 8-bit byte from memory, adds the value held in a register to it, and stores the result back to memory.",
                "html": "<p>Atomic add on byte in memory, without return, atomically loads an 8-bit byte from memory, adds the value held in a register to it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STADDH":
        case "STADDLH":
            return {
                "tooltip": "Atomic add on halfword in memory, without return, atomically loads a 16-bit halfword from memory, adds the value held in a register to it, and stores the result back to memory.",
                "html": "<p>Atomic add on halfword in memory, without return, atomically loads a 16-bit halfword from memory, adds the value held in a register to it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STCLR":
        case "STCLRL":
            return {
                "tooltip": "Atomic bit clear on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory.",
                "html": "<p>Atomic bit clear on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STCLRB":
        case "STCLRLB":
            return {
                "tooltip": "Atomic bit clear on byte in memory, without return, atomically loads an 8-bit byte from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory.",
                "html": "<p>Atomic bit clear on byte in memory, without return, atomically loads an 8-bit byte from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STCLRH":
        case "STCLRLH":
            return {
                "tooltip": "Atomic bit clear on halfword in memory, without return, atomically loads a 16-bit halfword from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory.",
                "html": "<p>Atomic bit clear on halfword in memory, without return, atomically loads a 16-bit halfword from memory, performs a bitwise AND with the complement of the value held in a register on it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STEOR":
        case "STEORL":
            return {
                "tooltip": "Atomic Exclusive-OR on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory.",
                "html": "<p>Atomic Exclusive-OR on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STEORB":
        case "STEORLB":
            return {
                "tooltip": "Atomic Exclusive-OR on byte in memory, without return, atomically loads an 8-bit byte from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory.",
                "html": "<p>Atomic Exclusive-OR on byte in memory, without return, atomically loads an 8-bit byte from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STEORH":
        case "STEORLH":
            return {
                "tooltip": "Atomic Exclusive-OR on halfword in memory, without return, atomically loads a 16-bit halfword from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory.",
                "html": "<p>Atomic Exclusive-OR on halfword in memory, without return, atomically loads a 16-bit halfword from memory, performs an exclusive-OR with the value held in a register on it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STG":
            return {
                "tooltip": "Store Allocation Tag stores an Allocation Tag to memory. The address used for the store is calculated from the base register and an immediate signed offset scaled by the Tag granule. The Allocation Tag is calculated from the Logical Address Tag in the source register.",
                "html": "<p>Store Allocation Tag stores an Allocation Tag to memory. The address used for the store is calculated from the base register and an immediate signed offset scaled by the Tag granule. The Allocation Tag is calculated from the Logical Address Tag in the source register.</p><p>This instruction generates an Unchecked access.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STGM":
            return {
                "tooltip": "Store Tag Multiple writes a naturally aligned block of N Allocation Tags, where the size of N is identified in GMID_EL1.BS, and the Allocation Tag written to address A is taken from the source register at 4*A<7:4>+3:4*A<7:4>.",
                "html": "<p>Store Tag Multiple writes a naturally aligned block of N Allocation Tags, where the size of N is identified in GMID_EL1.BS, and the Allocation Tag written to address A is taken from the source register at 4*A&lt;7:4&gt;+3:4*A&lt;7:4&gt;.</p><p>This instruction is <arm-defined-word>undefined</arm-defined-word> at EL0.</p><p>This instruction generates an Unchecked access.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STGP":
            return {
                "tooltip": "Store Allocation Tag and Pair of registers stores an Allocation Tag and two 64-bit doublewords to memory, from two registers. The address used for the store is calculated from the base register and an immediate signed offset scaled by the Tag granule. The Allocation Tag is calculated from the Logical Address Tag in the base register.",
                "html": "<p>Store Allocation Tag and Pair of registers stores an Allocation Tag and two 64-bit doublewords to memory, from two registers. The address used for the store is calculated from the base register and an immediate signed offset scaled by the Tag granule. The Allocation Tag is calculated from the Logical Address Tag in the base register.</p><p>This instruction generates an Unchecked access.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STILP":
            return {
                "tooltip": "Store-Release ordered Pair of registers calculates an address from a base register value and an optional offset, and stores two 32-bit words or two 64-bit doublewords to the calculated address, from two registers. For information on single-copy atomicity and alignment requirements, see Requirements for single-copy atomicity and Alignment of data accesses. The instruction also has memory ordering semantics, as described in Load-Acquire, Load-AcquirePC, and Store-Release, with the additional requirement that",
                "html": "<p>Store-Release ordered Pair of registers calculates an address from a base register value and an optional offset, and stores two 32-bit words or two 64-bit doublewords to the calculated address, from two registers. For information on single-copy atomicity and alignment requirements, see <xref linkend=\"CHDDCBCC\">Requirements for single-copy atomicity</xref> and <xref linkend=\"CHDFFEGJ\">Alignment of data accesses</xref>. The instruction also has memory ordering semantics, as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>, with the additional requirement that:</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STL1":
            return {
                "tooltip": "Store-Release a single-element structure from one lane of one register. This instruction stores the specified element of a SIMD&FP register to memory.",
                "html": "<p>Store-Release a single-element structure from one lane of one register. This instruction stores the specified element of a SIMD&amp;FP register to memory.</p><p>The instruction also has memory ordering semantics, as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLLR":
            return {
                "tooltip": "Store LORelease Register stores a 32-bit word or a 64-bit doubleword to a memory location, from a register. The instruction also has memory ordering semantics as described in Load LOAcquire, Store LORelease. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store LORelease Register stores a 32-bit word or a 64-bit doubleword to a memory location, from a register. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIJDIJG\">Load LOAcquire, Store LORelease</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLLRB":
            return {
                "tooltip": "Store LORelease Register Byte stores a byte from a 32-bit register to a memory location. The instruction also has memory ordering semantics as described in Load LOAcquire, Store LORelease. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store LORelease Register Byte stores a byte from a 32-bit register to a memory location. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIJDIJG\">Load LOAcquire, Store LORelease</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLLRH":
            return {
                "tooltip": "Store LORelease Register Halfword stores a halfword from a 32-bit register to a memory location. The instruction also has memory ordering semantics as described in Load LOAcquire, Store LORelease. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store LORelease Register Halfword stores a halfword from a 32-bit register to a memory location. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIJDIJG\">Load LOAcquire, Store LORelease</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLR":
            return {
                "tooltip": "Store-Release Register stores a 32-bit word or a 64-bit doubleword to a memory location, from a register. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store-Release Register stores a 32-bit word or a 64-bit doubleword to a memory location, from a register. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLRB":
            return {
                "tooltip": "Store-Release Register Byte stores a byte from a 32-bit register to a memory location. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store-Release Register Byte stores a byte from a 32-bit register to a memory location. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLRH":
            return {
                "tooltip": "Store-Release Register Halfword stores a halfword from a 32-bit register to a memory location. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store-Release Register Halfword stores a halfword from a 32-bit register to a memory location. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLUR":
            return {
                "tooltip": "Store-Release SIMD&FP Register (unscaled offset). This instruction stores a single SIMD&FP register to memory. The address that is used for the store is calculated from a base register value and an optional immediate offset.",
                "html": "<p>Store-Release SIMD&amp;FP Register (unscaled offset). This instruction stores a single SIMD&amp;FP register to memory. The address that is used for the store is calculated from a base register value and an optional immediate offset.</p><p>The instruction has memory ordering semantics, as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLURB":
            return {
                "tooltip": "Store-Release Register Byte (unscaled) calculates an address from a base register value and an immediate offset, and stores a byte to the calculated address, from a 32-bit register.",
                "html": "<p>Store-Release Register Byte (unscaled) calculates an address from a base register value and an immediate offset, and stores a byte to the calculated address, from a 32-bit register.</p><p>The instruction has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref></p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLURH":
            return {
                "tooltip": "Store-Release Register Halfword (unscaled) calculates an address from a base register value and an immediate offset, and stores a halfword to the calculated address, from a 32-bit register.",
                "html": "<p>Store-Release Register Halfword (unscaled) calculates an address from a base register value and an immediate offset, and stores a halfword to the calculated address, from a 32-bit register.</p><p>The instruction has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Load-AcquirePC, and Store-Release</xref></p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLXP":
            return {
                "tooltip": "Store-Release Exclusive Pair of registers stores two 32-bit words or two 64-bit doublewords to a memory location if the PE has exclusive access to the memory address, from two registers, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See Synchronization and semaphores. For information on single-copy atomicity and alignment requirements, see Requirements for single-copy atomicity and Alignment of data accesses. If a 64-bit pair Store-Exclusive succeeds, it causes a single-copy atomic update of the 128-bit memory location being updated. The instruction also has memory ordering semantics, as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store-Release Exclusive Pair of registers stores two 32-bit words or two 64-bit doublewords to a memory location if the PE has exclusive access to the memory address, from two registers, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. For information on single-copy atomicity and alignment requirements, see <xref linkend=\"CHDDCBCC\">Requirements for single-copy atomicity</xref> and <xref linkend=\"CHDFFEGJ\">Alignment of data accesses</xref>. If a 64-bit pair Store-Exclusive succeeds, it causes a single-copy atomic update of the 128-bit memory location being updated. The instruction also has memory ordering semantics, as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLXR":
            return {
                "tooltip": "Store-Release Exclusive Register stores a 32-bit word or a 64-bit doubleword to memory if the PE has exclusive access to the memory address, from two registers, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See Synchronization and semaphores. The memory access is atomic. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store-Release Exclusive Register stores a 32-bit word or a 64-bit doubleword to memory if the PE has exclusive access to the memory address, from two registers, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. The memory access is atomic. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLXRB":
            return {
                "tooltip": "Store-Release Exclusive Register Byte stores a byte from a 32-bit register to memory if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See Synchronization and semaphores. The memory access is atomic. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store-Release Exclusive Register Byte stores a byte from a 32-bit register to memory if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. The memory access is atomic. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STLXRH":
            return {
                "tooltip": "Store-Release Exclusive Register Halfword stores a halfword from a 32-bit register to memory if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See Synchronization and semaphores. The memory access is atomic. The instruction also has memory ordering semantics as described in Load-Acquire, Store-Release. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store-Release Exclusive Register Halfword stores a halfword from a 32-bit register to memory if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. The memory access is atomic. The instruction also has memory ordering semantics as described in <xref linkend=\"BEIHCHEF\">Load-Acquire, Store-Release</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STNP":
            return {
                "tooltip": "Store Pair of SIMD&FP registers, with Non-temporal hint. This instruction stores a pair of SIMD&FP registers to memory, issuing a hint to the memory system that the access is non-temporal. The address used for the store is calculated from an address from a base register value and an immediate offset. For information about non-temporal pair instructions, see Load/Store SIMD and Floating-point Non-temporal pair.",
                "html": "<p>Store Pair of SIMD&amp;FP registers, with Non-temporal hint. This instruction stores a pair of SIMD&amp;FP registers to memory, issuing a hint to the memory system that the access is non-temporal. The address used for the store is calculated from an address from a base register value and an immediate offset. For information about non-temporal pair instructions, see <xref linkend=\"BABJADHH\">Load/Store SIMD and Floating-point Non-temporal pair</xref>.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STNT1B":
            return {
                "tooltip": "Contiguous store non-temporal of bytes from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous store non-temporal of bytes from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements are not written to memory.</p><p>A non-temporal store is a hint to the system that this data is unlikely to be referenced again soon.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STNT1D":
            return {
                "tooltip": "Contiguous store non-temporal of doublewords from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous store non-temporal of doublewords from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements are not written to memory.</p><p>A non-temporal store is a hint to the system that this data is unlikely to be referenced again soon.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STNT1H":
            return {
                "tooltip": "Contiguous store non-temporal of halfwords from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous store non-temporal of halfwords from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements are not written to memory.</p><p>A non-temporal store is a hint to the system that this data is unlikely to be referenced again soon.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STNT1W":
            return {
                "tooltip": "Contiguous store non-temporal of words from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.",
                "html": "<p>Contiguous store non-temporal of words from elements of two or four consecutive vector registers to the memory address generated by a 64-bit scalar base and immediate index which is multiplied by the vector's in-memory size, irrespective of predication, and added to the base address.</p><p>Inactive elements are not written to memory.</p><p>A non-temporal store is a hint to the system that this data is unlikely to be referenced again soon.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STP":
            return {
                "tooltip": "Store Pair of SIMD&FP registers. This instruction stores a pair of SIMD&FP registers to memory. The address used for the store is calculated from a base register value and an immediate offset.",
                "html": "<p>Store Pair of SIMD&amp;FP registers. This instruction stores a pair of SIMD&amp;FP registers to memory. The address used for the store is calculated from a base register value and an immediate offset.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STR":
            return {
                "tooltip": "Store SIMD&FP register (immediate offset). This instruction stores a single SIMD&FP register to memory. The address that is used for the store is calculated from a base register value and an immediate offset.",
                "html": "<p>Store SIMD&amp;FP register (immediate offset). This instruction stores a single SIMD&amp;FP register to memory. The address that is used for the store is calculated from a base register value and an immediate offset.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STRB":
            return {
                "tooltip": "Store Register Byte (immediate) stores the least significant byte of a 32-bit register to memory. The address that is used for the store is calculated from a base register and an immediate offset. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store Register Byte (immediate) stores the least significant byte of a 32-bit register to memory. The address that is used for the store is calculated from a base register and an immediate offset. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STRH":
            return {
                "tooltip": "Store Register Halfword (immediate) stores the least significant halfword of a 32-bit register to memory. The address that is used for the store is calculated from a base register and an immediate offset. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store Register Halfword (immediate) stores the least significant halfword of a 32-bit register to memory. The address that is used for the store is calculated from a base register and an immediate offset. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STSET":
        case "STSETL":
            return {
                "tooltip": "Atomic bit set on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory.",
                "html": "<p>Atomic bit set on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STSETB":
        case "STSETLB":
            return {
                "tooltip": "Atomic bit set on byte in memory, without return, atomically loads an 8-bit byte from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory.",
                "html": "<p>Atomic bit set on byte in memory, without return, atomically loads an 8-bit byte from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STSETH":
        case "STSETLH":
            return {
                "tooltip": "Atomic bit set on halfword in memory, without return, atomically loads a 16-bit halfword from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory.",
                "html": "<p>Atomic bit set on halfword in memory, without return, atomically loads a 16-bit halfword from memory, performs a bitwise OR with the value held in a register on it, and stores the result back to memory.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STSMAX":
        case "STSMAXL":
            return {
                "tooltip": "Atomic signed maximum on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers.",
                "html": "<p>Atomic signed maximum on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STSMAXB":
        case "STSMAXLB":
            return {
                "tooltip": "Atomic signed maximum on byte in memory, without return, atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers.",
                "html": "<p>Atomic signed maximum on byte in memory, without return, atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STSMAXH":
        case "STSMAXLH":
            return {
                "tooltip": "Atomic signed maximum on halfword in memory, without return, atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers.",
                "html": "<p>Atomic signed maximum on halfword in memory, without return, atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as signed numbers.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STSMIN":
        case "STSMINL":
            return {
                "tooltip": "Atomic signed minimum on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers.",
                "html": "<p>Atomic signed minimum on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STSMINB":
        case "STSMINLB":
            return {
                "tooltip": "Atomic signed minimum on byte in memory, without return, atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers.",
                "html": "<p>Atomic signed minimum on byte in memory, without return, atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STSMINH":
        case "STSMINLH":
            return {
                "tooltip": "Atomic signed minimum on halfword in memory, without return, atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers.",
                "html": "<p>Atomic signed minimum on halfword in memory, without return, atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as signed numbers.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STTR":
            return {
                "tooltip": "Store Register (unprivileged) stores a word or doubleword from a register to memory. The address that is used for the store is calculated from a base register and an immediate offset.",
                "html": "<p>Store Register (unprivileged) stores a word or doubleword from a register to memory. The address that is used for the store is calculated from a base register and an immediate offset.</p><p>Memory accesses made by the instruction behave as if the instruction was executed at EL0 if the <xref linkend=\"CBAICCHA\">Effective value</xref> of PSTATE.UAO is 0 and either:</p><p>Otherwise, the memory access operates with the restrictions determined by the Exception level at which the instruction is executed. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STTRB":
            return {
                "tooltip": "Store Register Byte (unprivileged) stores a byte from a 32-bit register to memory. The address that is used for the store is calculated from a base register and an immediate offset.",
                "html": "<p>Store Register Byte (unprivileged) stores a byte from a 32-bit register to memory. The address that is used for the store is calculated from a base register and an immediate offset.</p><p>Memory accesses made by the instruction behave as if the instruction was executed at EL0 if the <xref linkend=\"CBAICCHA\">Effective value</xref> of PSTATE.UAO is 0 and either:</p><p>Otherwise, the memory access operates with the restrictions determined by the Exception level at which the instruction is executed. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STTRH":
            return {
                "tooltip": "Store Register Halfword (unprivileged) stores a halfword from a 32-bit register to memory. The address that is used for the store is calculated from a base register and an immediate offset.",
                "html": "<p>Store Register Halfword (unprivileged) stores a halfword from a 32-bit register to memory. The address that is used for the store is calculated from a base register and an immediate offset.</p><p>Memory accesses made by the instruction behave as if the instruction was executed at EL0 if the <xref linkend=\"CBAICCHA\">Effective value</xref> of PSTATE.UAO is 0 and either:</p><p>Otherwise, the memory access operates with the restrictions determined by the Exception level at which the instruction is executed. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STUMAX":
        case "STUMAXL":
            return {
                "tooltip": "Atomic unsigned maximum on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers.",
                "html": "<p>Atomic unsigned maximum on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STUMAXB":
        case "STUMAXLB":
            return {
                "tooltip": "Atomic unsigned maximum on byte in memory, without return, atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers.",
                "html": "<p>Atomic unsigned maximum on byte in memory, without return, atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STUMAXH":
        case "STUMAXLH":
            return {
                "tooltip": "Atomic unsigned maximum on halfword in memory, without return, atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers.",
                "html": "<p>Atomic unsigned maximum on halfword in memory, without return, atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the larger value back to memory, treating the values as unsigned numbers.</p><p>For information about memory accesses see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STUMIN":
        case "STUMINL":
            return {
                "tooltip": "Atomic unsigned minimum on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers.",
                "html": "<p>Atomic unsigned minimum on word or doubleword in memory, without return, atomically loads a 32-bit word or 64-bit doubleword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STUMINB":
        case "STUMINLB":
            return {
                "tooltip": "Atomic unsigned minimum on byte in memory, without return, atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers.",
                "html": "<p>Atomic unsigned minimum on byte in memory, without return, atomically loads an 8-bit byte from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STUMINH":
        case "STUMINLH":
            return {
                "tooltip": "Atomic unsigned minimum on halfword in memory, without return, atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers.",
                "html": "<p>Atomic unsigned minimum on halfword in memory, without return, atomically loads a 16-bit halfword from memory, compares it against the value held in a register, and stores the smaller value back to memory, treating the values as unsigned numbers.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STUR":
            return {
                "tooltip": "Store SIMD&FP register (unscaled offset). This instruction stores a single SIMD&FP register to memory. The address that is used for the store is calculated from a base register value and an optional immediate offset.",
                "html": "<p>Store SIMD&amp;FP register (unscaled offset). This instruction stores a single SIMD&amp;FP register to memory. The address that is used for the store is calculated from a base register value and an optional immediate offset.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STURB":
            return {
                "tooltip": "Store Register Byte (unscaled) calculates an address from a base register value and an immediate offset, and stores a byte to the calculated address, from a 32-bit register. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store Register Byte (unscaled) calculates an address from a base register value and an immediate offset, and stores a byte to the calculated address, from a 32-bit register. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STURH":
            return {
                "tooltip": "Store Register Halfword (unscaled) calculates an address from a base register value and an immediate offset, and stores a halfword to the calculated address, from a 32-bit register. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store Register Halfword (unscaled) calculates an address from a base register value and an immediate offset, and stores a halfword to the calculated address, from a 32-bit register. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STXP":
            return {
                "tooltip": "Store Exclusive Pair of registers stores two 32-bit words or two 64-bit doublewords from two registers to a memory location if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See Synchronization and semaphores. For information on single-copy atomicity and alignment requirements, see Requirements for single-copy atomicity and Alignment of data accesses. If a 64-bit pair Store-Exclusive succeeds, it causes a single-copy atomic update of the 128-bit memory location being updated. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store Exclusive Pair of registers stores two 32-bit words or two 64-bit doublewords from two registers to a memory location if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. For information on single-copy atomicity and alignment requirements, see <xref linkend=\"CHDDCBCC\">Requirements for single-copy atomicity</xref> and <xref linkend=\"CHDFFEGJ\">Alignment of data accesses</xref>. If a 64-bit pair Store-Exclusive succeeds, it causes a single-copy atomic update of the 128-bit memory location being updated. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STXR":
            return {
                "tooltip": "Store Exclusive Register stores a 32-bit word or a 64-bit doubleword from a register to memory if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See Synchronization and semaphores. For information about memory accesses, see Load/Store addressing modes.",
                "html": "<p>Store Exclusive Register stores a 32-bit word or a 64-bit doubleword from a register to memory if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STXRB":
            return {
                "tooltip": "Store Exclusive Register Byte stores a byte from a register to memory if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See Synchronization and semaphores. The memory access is atomic.",
                "html": "<p>Store Exclusive Register Byte stores a byte from a register to memory if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. The memory access is atomic.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STXRH":
            return {
                "tooltip": "Store Exclusive Register Halfword stores a halfword from a register to memory if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See Synchronization and semaphores. The memory access is atomic.",
                "html": "<p>Store Exclusive Register Halfword stores a halfword from a register to memory if the PE has exclusive access to the memory address, and returns a status value of 0 if the store was successful, or of 1 if no store was performed. See <xref linkend=\"Chdcgdja\">Synchronization and semaphores</xref>. The memory access is atomic.</p><p>For information about memory accesses, see <xref linkend=\"CHDIIIBB\">Load/Store addressing modes</xref>.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STZ2G":
            return {
                "tooltip": "Store Allocation Tags, Zeroing stores an Allocation Tag to two Tag granules of memory, zeroing the associated data locations. The address used for the store is calculated from the base register and an immediate signed offset scaled by the Tag granule. The Allocation Tag is calculated from the Logical Address Tag in the source register.",
                "html": "<p>Store Allocation Tags, Zeroing stores an Allocation Tag to two Tag granules of memory, zeroing the associated data locations. The address used for the store is calculated from the base register and an immediate signed offset scaled by the Tag granule. The Allocation Tag is calculated from the Logical Address Tag in the source register.</p><p>This instruction generates an Unchecked access.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STZG":
            return {
                "tooltip": "Store Allocation Tag, Zeroing stores an Allocation Tag to memory, zeroing the associated data location. The address used for the store is calculated from the base register and an immediate signed offset scaled by the Tag granule. The Allocation Tag is calculated from the Logical Address Tag in the source register.",
                "html": "<p>Store Allocation Tag, Zeroing stores an Allocation Tag to memory, zeroing the associated data location. The address used for the store is calculated from the base register and an immediate signed offset scaled by the Tag granule. The Allocation Tag is calculated from the Logical Address Tag in the source register.</p><p>This instruction generates an Unchecked access.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "STZGM":
            return {
                "tooltip": "Store Tag and Zero Multiple writes a naturally aligned block of N Allocation Tags and stores zero to the associated data locations, where the size of N is identified in DCZID_EL0.BS, and the Allocation Tag is taken from the source register bits<3:0>.",
                "html": "<p>Store Tag and Zero Multiple writes a naturally aligned block of N Allocation Tags and stores zero to the associated data locations, where the size of N is identified in DCZID_EL0.BS, and the Allocation Tag is taken from the source register bits&lt;3:0&gt;.</p><p>This instruction is <arm-defined-word>undefined</arm-defined-word> at EL0.</p><p>This instruction generates an Unchecked access.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SUB":
            return {
                "tooltip": "Subtract (extended register) subtracts a sign or zero-extended register value, followed by an optional left shift amount, from a register value, and writes the result to the destination register. The argument that is extended from the <Rm> register can be a byte, halfword, word, or doubleword.",
                "html": "<p>Subtract (extended register) subtracts a sign or zero-extended register value, followed by an optional left shift amount, from a register value, and writes the result to the destination register. The argument that is extended from the <syntax>&lt;Rm&gt;</syntax> register can be a byte, halfword, word, or doubleword.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SUBG":
            return {
                "tooltip": "Subtract with Tag subtracts an immediate value scaled by the Tag granule from the address in the source register, modifies the Logical Address Tag of the address using an immediate value, and writes the result to the destination register. Tags specified in GCR_EL1.Exclude are excluded from the possible outputs when modifying the Logical Address Tag.",
                "html": "<p>Subtract with Tag subtracts an immediate value scaled by the Tag granule from the address in the source register, modifies the Logical Address Tag of the address using an immediate value, and writes the result to the destination register. Tags specified in GCR_EL1.Exclude are excluded from the possible outputs when modifying the Logical Address Tag.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SUBHN":
        case "SUBHN2":
            return {
                "tooltip": "Subtract returning High Narrow. This instruction subtracts each vector element in the second source SIMD&FP register from the corresponding vector element in the first source SIMD&FP register, places the most significant half of the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&FP register. All the values in this instruction are signed integer values.",
                "html": "<p>Subtract returning High Narrow. This instruction subtracts each vector element in the second source SIMD&amp;FP register from the corresponding vector element in the first source SIMD&amp;FP register, places the most significant half of the result into a vector, and writes the vector to the lower or upper half of the destination SIMD&amp;FP register. All the values in this instruction are signed integer values.</p><p>The results are truncated. For rounded results, see <xref linkend=\"A64.instructions.RSUBHN_advsimd\">RSUBHN</xref>.</p><p>The <instruction>SUBHN</instruction> instruction writes the vector to the lower half of the destination register and clears the upper half, while the <instruction>SUBHN2</instruction> instruction writes the vector to the upper half of the destination register without affecting the other bits of the register.</p><p>Depending on the settings in the <xref linkend=\"AArch64.cpacr_el1\">CPACR_EL1</xref>, <xref linkend=\"AArch64.cptr_el2\">CPTR_EL2</xref>, and <xref linkend=\"AArch64.cptr_el3\">CPTR_EL3</xref> registers, and the current Security state and Exception level, an attempt to execute the instruction might be trapped.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SUBHNB":
            return {
                "tooltip": "Subtract each vector element of the second source vector from the corresponding vector element in the first source vector, and place the most significant half of the result in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. This instruction is unpredicated.",
                "html": "<p>Subtract each vector element of the second source vector from the corresponding vector element in the first source vector, and place the most significant half of the result in the even-numbered half-width destination elements, while setting the odd-numbered elements to zero. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SUBHNT":
            return {
                "tooltip": "Subtract each vector element of the second source vector from the corresponding vector element in the first source vector, and place the most significant half of the result in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. This instruction is unpredicated.",
                "html": "<p>Subtract each vector element of the second source vector from the corresponding vector element in the first source vector, and place the most significant half of the result in the odd-numbered half-width destination elements, leaving the even-numbered elements unchanged. This instruction is unpredicated.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SUBP":
            return {
                "tooltip": "Subtract Pointer subtracts the 56-bit address held in the second source register from the 56-bit address held in the first source register, sign-extends the result to 64-bits, and writes the result to the destination register.",
                "html": "<p>Subtract Pointer subtracts the 56-bit address held in the second source register from the 56-bit address held in the first source register, sign-extends the result to 64-bits, and writes the result to the destination register.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Base-Instructions/"
            };

        case "SUBPS":
            return {
                "tooltip": "Subtract Pointer, setting Flags subtracts the 56-bit address held in the second source register from the 56-bit address held in the first source register, sign-extends the result to 64-bits, and writes the result to the destination register. It updates the condition flags based on the result of the subtraction.",
                "html": "<p>Subtract Pointer, setting Flags subtracts the 56-bit address held in the second source register from the 56-bit address held in the first source register, sign-extends the result to 64-bits, and writes the result to the destination register. It updates the condition flags based on the result of the subtraction.</p>",
                "url": "https://developer.arm.com/documentation/ddi0602/latest/Ba