In an [ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core) [Web API](https://en.wikipedia.org/wiki/Web_API),
controller actions can optionally return a result value. If a controller action returns a value in the [happy path](https://en.wikipedia.org/wiki/Happy_path), for example [ControllerBase.Ok(Object)](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.controllerbase.ok#microsoft-aspnetcore-mvc-controllerbase-ok%28system-object%29),
annotating the action with one of the [`\[ProducesResponseType\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.producesresponsetypeattribute)
overloads that describe the type is recommended.

## Why is this an issue?

If an ASP.NET Core Web API uses [Swagger](https://swagger.io/), the API documentation will be generated based on the input/output types
of the controller actions, as well as the attributes annotating the actions. If an action returns [IActionResult](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.iactionresult) or [IResult](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.iresult), Swagger cannot infer the type of the response. From
the consumer’s perspective, this can be confusing and lead to unexpected results and bugs in the long run without the API provider’s awareness.

This rule raises an issue on a controller action when:

- The action returns a value in the happy path. This can be either:
    - [200 OK](https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/200)
    - [201 Created](https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/201)
    - [202 Accepted](https://developer.mozilla.org/en-US/docs/Web/HTTP/Status/202)
- There is no `[ProducesResponseType]` attribute containing the return type, either at controller or action level.
- There is no `[SwaggerResponse]` attribute containing the return type, either at controller or action level.
- The controller is annotated with the [`\[ApiController\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.apicontrollerattribute) attribute.
- The controller action returns either [IActionResult](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.iactionresult) or [IResult](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.iresult).
- The application has enabled the [Swagger
  middleware](https://learn.microsoft.com/en-us/aspnet/core/tutorials/getting-started-with-swashbuckle#add-and-configure-swagger-middleware).

## How to fix it

There are multiple ways to fix this issue:

- Annotate the action with [`\[ProducesResponseType\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.producesresponsetypeattribute)
  containing the return type.
- Annotate the action with [SwaggerResponse Class](https://github.com/domaindrivendev/Swashbuckle.AspNetCore/blob/master/README.md#enrich-response-metadata) containing
  the return type.
- Return [ActionResult&lt;TValue&gt;](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.actionresult-1) instead of
  `[IActionResult]` or `[IResult]`.
- Return [Results&lt;TResult1,
  TResult2&gt;](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.httpresults.results-2) instead of `[IActionResult]` or `[IResult]`.

### Code examples

#### Noncompliant code example

    [HttpGet("foo")]
    // Noncompliant: Annotate this method with ProducesResponseType containing the return type for succesful responses.
    public IActionResult MagicNumber() => Ok(42);

    [HttpGet("foo")]
    // Noncompliant: Use the ProducesResponseType overload containing the return type for succesful responses.
    [ProducesResponseType(StatusCodes.Status200OK)]
    public IActionResult MagicNumber() => Ok(42);

#### Compliant solution

    [HttpGet("foo")]
    [ProducesResponseType<int>(StatusCodes.Status200OK)]
    public IActionResult MagicNumber() => Ok(42);

    [HttpGet("foo")]
    [ProducesResponseType(typeof(int), StatusCodes.Status200OK)]
    public IActionResult MagicNumber() => Ok(42);

## Resources

### Documentation

- Wikipedia - [Web API](https://en.wikipedia.org/wiki/Web_API)
- Wikipedia - [Happy path](https://en.wikipedia.org/wiki/Happy_path)
- Microsoft Learn - [ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core)
- Microsoft Learn - [Get started with
  Swashbuckle and ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core/tutorials/getting-started-with-swashbuckle)
- Microsoft Learn - [ApiControllerAttribute
  Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.apicontrollerattribute)
- Microsoft Learn - [ProducesResponseTypeAttribute Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.producesresponsetypeattribute)
- Microsoft Learn - [ProducesResponseTypeAttribute&lt;T&gt;
  Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.producesresponsetypeattribute-1)
- Microsoft Learn - [ActionResult&lt;TValue&gt;
  Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.actionresult-1)
- Microsoft Learn - [Results&lt;TResult1,
  TResult2&gt; Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.httpresults.results-2)
- Microsoft Learn - [HttpResults type](https://learn.microsoft.com/en-us/aspnet/core/web-api/action-return-types#httpresults-type)
- GitHub - [SwaggerResponse
  Class](https://github.com/domaindrivendev/Swashbuckle.AspNetCore/blob/master/README.md#enrich-response-metadata)
- SmartBear - [Swagger](https://swagger.io/)