In the context of ASP.NET Core MVC web applications, both model binding and model validation are processes that take place prior to the execution
of a controller action. It is imperative for the application to examine the `ModelState.IsValid` and respond accordingly.

This rule enforces the developer to validate the model within a controller action, ensuring the application’s robustness and reliability.

## Why is this an issue?

Querying the `ModelState.IsValid` property is necessary because it checks if the submitted data in the HTTP request is valid or not.
This property evaluates all the validation attributes applied on your model properties and determines whether the data provided satisfies those
validation rules.

### What is the potential impact?

Skipping model validation can lead to:

- Data Integrity Issues: Without validation, incorrect or inconsistent data can be saved to your database, leading to potential data corruption
  or loss.
- Security Vulnerabilities: Skipping validation can expose your application to security risks.
- Application Errors: Invalid data can lead to unexpected application errors or crashes, which can disrupt the user experience and potentially
  lead to data loss.
- Poor User Experience: Without validation, users may not receive appropriate feedback about any mistakes in the data they have entered, leading
  to confusion and frustration.
- Increased Debugging Time: If invalid data causes issues in your application and was not validatated at the entry point, it can take
  significantly more time to debug and fix these issues.

Therefore, it’s highly recommended to always validate models in your application to ensure data integrity, application stability, and a good user
experience.

While client-side validation enhances user experience by providing immediate feedback, it’s not sufficient due to potential manipulation of
client-side code, browser compatibility issues, and dependence on JavaScript. Users can bypass or disable it, leading to invalid or malicious data
being submitted. Therefore, server-side validation is essential to ensure data integrity and security, making it a best practice to use both
client-side and server-side validation in your application.

### Exceptions

- Web API controllers don’t have to check `ModelState.IsValid` if they have the `[ApiController]` attribute. In that case,
  an automatic HTTP `400` response containing error details is returned when model state is invalid.
- When action filters are used for controller actions, the analyzer will skip the model validation detection to avoid false positives since the
  model state could be verified by the action filer.
- `TryValidateModel` can also be used for model validation.
- The project references a 3rd-party library for validation, e.g. FluentValidation.
- The rule will not raise issues if the model, or the model members, are not decorated with validation attributes, or if it does not implement
  custom validation.

## How to fix it

If `ModelState.IsValid` returns true, it means that the data is valid and the process can continue. If it returns false, it means that
the validation failed, indicating that the data is not in the expected format or is missing required information.

In such cases, the controller action should handle this by returning an appropriate response, such as re-displaying the form with error messages.
This helps maintain the integrity of the data and provides feedback to the user, enhancing the overall user experience and security of your
application.

### Code examples

#### Noncompliant code example

    public async Task<IActionResult> Create(Movie movie) // Noncompliant: model validity check is missing
    {
        _context.Movies.Add(movie);
        await _context.SaveChangesAsync();
    
        return RedirectToAction(nameof(Index));
    }

#### Compliant solution

    public async Task<IActionResult> Create(Movie movie)
    {
        if (!ModelState.IsValid)
        {
            return View(movie);
        }
    
        _context.Movies.Add(movie);
        await _context.SaveChangesAsync();
    
        return RedirectToAction(nameof(Index));
    }

## Resources

### Documentation

- Microsoft Learn - [Model Validation](https://learn.microsoft.com/en-us/aspnet/core/mvc/models/validation)
- Microsoft Learn - [IsValid property](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.modelbinding.modelstatedictionary.isvalid)
- Microsoft Learn - [ValidationAttribute](https://learn.microsoft.com/en-us/dotnet/api/system.componentmodel.dataannotations.validationattribute)
- Fluent Validation - [ASP.NET Core integration](https://docs.fluentvalidation.net/en/latest/aspnet.html)