In an `async` method, any blocking operations should be avoided.

## Why is this an issue?

Using a synchronous method instead of its [asynchronous](https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/)
counterpart in an `async` method blocks the execution and is considered bad practice for several reasons:

  - Resource Utilization

      - Each thread consumes system resources, such as memory. When a thread is blocked, it’s not doing any useful work, but it’s still consuming these
    resources. This can lead to inefficient use of system resources.

  - Scalability

      - Blocking threads can limit the scalability of your application. In a high-load scenario where many operations are happening concurrently, each
    blocked thread represents a missed opportunity to do useful work. This can prevent your application from effectively handling increased load.

  - Performance

      - Blocking threads can degrade the performance of your application. If all threads in the thread pool become blocked, new tasks can’t start
    executing until an existing task completes and frees up a thread. This can lead to delays and poor responsiveness.

Instead of blocking, it’s recommended to use the [`async` operator](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/await) with async methods. This
allows the system to release the current thread back to the thread pool until the awaited task is complete, improving scalability and
responsiveness.

## How to fix it

### Code examples

#### Noncompliant code example

    public async Task Examples(Stream stream, DbSet<Person> dbSet)
    {
        stream.Read(array, 0, 1024);            // Noncompliant
        File.ReadAllLines("path");              // Noncompliant
        dbSet.ToList();                         // Noncompliant in Entity Framework Core queries
        dbSet.FirstOrDefault(x => x.Age >= 18); // Noncompliant in Entity Framework Core queries
    }

#### Compliant solution

    public async Task Examples(Stream stream, DbSet<Person> dbSet)
    {
        await stream.ReadAsync(array, 0, 1024);
        await File.ReadAllLinesAsync("path");
        await dbSet.ToListAsync();
        await dbSet.FirstOrDefaultAsync(x => x.Age >= 18);
    }

## Resources

### Documentation

- Microsoft Learn - [async (C# Reference)](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/async)
- Microsoft Learn - [await operator - asynchronously
  await for a task to complete](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/await)

### Articles & blog posts

- Microsoft Learn - [Asynchronous
  programming with async and await - Don’t block, await instead](https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/#dont-block-await-instead)
- Microsoft Learn - [Async/Await - Best
  Practices in Asynchronous Programming](https://learn.microsoft.com/en-us/archive/msdn-magazine/2013/march/async-await-best-practices-in-asynchronous-programming)
- Microsoft Developer Blog - [Async/Await FAQ](https://devblogs.microsoft.com/pfxteam/asyncawait-faq/)