When building a [REST API](https://learn.microsoft.com/en-us/aspnet/core/tutorials/first-web-api), [it’s
recommended](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/routing?view=aspnetcore-8.0#attribute-routing-with-http-verb-attributes) to annotate the controller actions with the available [HTTP attributes](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.routing.httpmethodattribute) to be precise about what
your API supports.

## Why is this an issue?

- **Ambiguity**: Without HttpAttributes, it’s unclear which HTTP methods an action method should respond to. This can lead to
  confusion and make the code harder to understand and maintain.
- **Unsupported HTTP Methods**: If an action is not annotated at all or is annotated only with the [Route attribute](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.routeattribute), it accepts all HTTP methods even if
  they are not supported by that action, which leads to further confusion.
- **Problems with Swagger**: [Swagger](https://learn.microsoft.com/en-us/aspnet/core/tutorials/web-api-help-pages-using-swagger) relies on HttpAttributes to generate
  parts of the API documentation. These attributes are necessary for the generated documentation to be complete.
- **Route path conflicts**: Without HttpAttributes, it’s possible to accidentally create action methods that respond to the same
  route and HTTP method. This can lead to unexpected behavior and hard-to-diagnose bugs.
- **Lack of routing flexibility**: The HTTP attributes allow you to define multiple action methods in the same controller that
  respond to the same route but different HTTP methods. If you don’t use them, you might have limited flexibility when designing your API.

## How to fix it

You should annotate the controller actions with the available HttpMethod attributes. You can still use them in conjunction with the Route
attribute, in case there are multiple templates for one action and you need to [set the order](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.routeattribute.order?view=aspnetcore-8.0). This allows
you to clearly define the HTTP methods each action method should respond to, while still being able to customize your routes.

## Exceptions

This rule does not raise if the controller or the action is annotated with `[ApiExplorerSettings(IgnoreApi =
true)]` or [`AcceptsVerbs`
attribute](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.acceptverbsattribute).

### Code examples

#### Noncompliant code example

    [Route("Customer")]                                                        // This route conflicts with GetCustomers action route
    public async Task<IResult> ChangeCustomer([FromBody] CustomerData data)   // Noncompliant
    {
        // ...
        return Results.Ok();
    }
    
    [Route("Customer")]                         // This route conflicts with ChangeCustomer action route
    public async Task<string> GetCustomers()    // Noncompliant
    {
        return _customerRepository.GetAll();
    }

#### Compliant solution

    [Route("Customer")]
    [HttpPost]
    public async Task<IResult> ChangeCustomer([FromBody] CustomerData data)    // Compliant
    {
        // ...
        return Results.Ok();
    }
    
    [HttpGet("Customer")]
    public async Task<string> GetCustomers()    // Compliant
    {
        return _customerRepository.GetAll();
    }

## Resources

### Documentation

- Microsoft Learn - [Routing to controller actions in ASP.NET
  Core](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/routing)
- Microsoft Learn - [Attribute routing with Http
  verb attributes](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/routing#attribute-routing-with-http-verb-attributes)
- Microsoft Learn - [Get started with
  Swashbuckle and ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core/tutorials/getting-started-with-swashbuckle)
- Microsoft Learn - [ASP.NET Core Exception
  handler](https://learn.microsoft.com/en-us/aspnet/core/web-api/handle-errors#exception-handler)
- Microsoft Learn - [RouteAttribute Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.routeattribute)