["Under-posting"](https://learn.microsoft.com/en-us/aspnet/web-api/overview/formats-and-model-binding/model-validation-in-aspnet-web-api#data-annotations)
refers to a situation where a client sends less data than expected to the server during an HTTP request, for example when the client omits some
properties from the request body that the server expects to receive.

## Why is this an issue?

One of the main issues that under-posting can cause is data inconsistency. If the client sends less data than expected, the application might fill
any value type properties with their default values, leading to inaccurate or inconsistent data in your database. Additionally, there might be
unexpected behavior if there are certain data expected that are not provided and even security issues; for example, if a user omits a role or
permission field from a POST request, and the server fills in a default value, it could inadvertently grant more access than intended.

A [model class](https://learn.microsoft.com/en-us/aspnet/core/tutorials/first-mvc-app/adding-model) (in this case the
`Product` class) can be an input of an HTTP handler method:

    public class ProductsController : Controller
    {
        [HttpPost]
        public IActionResult Create([FromBody]Product product)
        {
            // Process product data...
        }
    }

### Exceptions

This rule does not raise an issue when properties are decorated with the following attributes:

- [ValidateNever](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.modelbinding.validation.validateneverattribute)
- [JsonProperty(Required = Required.Always)](https://www.newtonsoft.com/json/help/html/JsonPropertyRequired.htm)
- [JsonProperty(Required = Required.AllowNull)](https://www.newtonsoft.com/json/help/html/JsonPropertyRequired.htm)
- [Newtonsoft.Json.JsonIgnore](https://www.newtonsoft.com/json/help/html/PropertyJsonIgnore.htm)
- [Newtonsoft.Json.JsonRequired](https://www.newtonsoft.com/json/help/html/t_newtonsoft_json_jsonrequiredattribute.htm)
- [System.Text.Json.Serialization.JsonRequired](https://learn.microsoft.com/en-us/dotnet/api/system.text.json.serialization.jsonrequiredattribute)
- [System.Text.Json.Serialization.JsonIgnore](https://learn.microsoft.com/en-us/dotnet/api/system.text.json.serialization.jsonignoreattribute)
- [Range](https://learn.microsoft.com/en-us/dotnet/api/system.componentmodel.dataannotations.rangeattribute)
- [BindNever](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.modelbinding.bindneverattribute)

Additionally, this rule does not raise for properties in model classes that are not in the same project as the Controller class that references
them. This is due to a limitation of Roslyn (see [here](https://github.com/SonarSource/sonar-dotnet/issues/9243)).

## How to fix it

You should mark any model value-type property as [nullable](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/nullable-value-types), [required](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/required) or [JsonRequired](https://learn.microsoft.com/en-us/dotnet/api/system.text.json.serialization.jsonrequiredattribute). Thus when a client
underposts, you ensure that the missing properties can be detected on the server side rather than being auto-filled, and therefore, incoming data
meets the application’s expectations.

### Code examples

#### Noncompliant code example

    public class Product
    {
        public int Id { get; set; }             // Noncompliant
        public string Name { get; set; }
        public int NumberOfItems { get; set; }  // Noncompliant
        public decimal Price { get; set; }      // Noncompliant
    }

If the client sends a request without setting the `NumberOfItems` or `Price` properties, they will default to `0`.
In the request handler method, there’s no way to determine whether they were intentionally set to `0` or omitted by mistake.

#### Compliant solution

    public class Product
    {
        public required int Id { get; set; }
        public string Name { get; set; }
        public int? NumberOfItems { get; set; }            // Compliant - property is optional
        [JsonRequired] public decimal Price { get; set; }  // Compliant - property must have a value
    }

In this example, the request handler method can

- manually check whether `NumberOfItems` was filled out through the `HasValue` property
- rely on Model Validation to make sure `Price` is not missing

    public class ProductsController : Controller
    {
        [HttpPost]
        public IActionResult Create(Product product)
        {
            if (!ModelState.IsValid)    // if product.Price is missing then the model state will not be valid
            {
                return View(product);
            }
    
            if (product.NumberOfItems.HasValue)
            {
                // ...
            }
            // Process input...
        }
    }

## Recommended Secure Coding Practices

- Client-Side Validation: While server-side validation is crucial, implementing client-side validation can provide immediate feedback to the user
  when certain fields are not filled out, which helps to avoid under-posting.
- Comprehensive Testing: Include testing scenarios that specifically check for under-posting vulnerabilities to ensure that all required data is
  correctly validated and processed.

## Resources

### Documentation

- Microsoft Learn - [Overview of ASP.NET Core MVC](https://learn.microsoft.com/en-us/aspnet/core/mvc/overview)
- Microsoft Learn - [Overview of
  ASP.NET MVC 5](https://learn.microsoft.com/en-us/aspnet/mvc/overview/getting-started/introduction/getting-started)
- Microsoft Learn - [Overview of ASP.NET Razor Pages](https://learn.microsoft.com/en-us/aspnet/core/razor-pages)
- Microsoft Learn - [Model Classes in ASP.NET MVC](https://learn.microsoft.com/en-us/aspnet/core/tutorials/first-mvc-app/adding-model)
- Microsoft Learn - [Model Binding in ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core/mvc/models/model-binding)
- Microsoft Learn - [Model Validation in
  ASP.NET Web API](https://learn.microsoft.com/en-us/aspnet/web-api/overview/formats-and-model-binding/model-validation-in-aspnet-web-api)
- Microsoft Learn - [Nullable
  Value Types in .NET](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/nullable-value-types)
- Microsoft Learn - [Data
  Annotations](https://learn.microsoft.com/en-us/aspnet/web-api/overview/formats-and-model-binding/model-validation-in-aspnet-web-api#data-annotations)
- Microsoft Learn - [RequiredAttribute
  Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.routing.httpmethodattribute)
- Microsoft Learn - [ValidateNeverAttribute
  Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.modelbinding.validation.validateneverattribute)