In [ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core), controllers usually inherit either from [`ControllerBase`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.controllerbase) or [`Controller`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.controller). If a controller does not use any
View-specific functionality, it is recommended to inherit from `ControllerBase`.

## Why is this an issue?

The `ControllerBase` class contains all the necessary functionality to handle [API](https://en.wikipedia.org/wiki/Web_API)
requests and responses. The `Controller` class inherits from `ControllerBase` and adds support for [Views](https://learn.microsoft.com/en-us/aspnet/core/mvc/views/overview), [PartialViews](https://learn.microsoft.com/en-us/aspnet/core/mvc/views/partial) and [ViewComponents](https://learn.microsoft.com/en-us/aspnet/core/mvc/views/view-components).

Inheriting from `Controller` when not using any View-specific functionality exposes unnecessary methods and can lead to confusion about
the intention of the class.

Furthermore, inheriting from `Controller` may come with a performance cost. Even though the controller might only deal with API
operations, the support for Views might introduce some overhead from the MVC framework during the [request processing pipeline](https://learn.microsoft.com/en-us/aspnet/core/fundamentals/middleware).

An issue is raised when:

- The class is marked with the [`\[ApiController\]`](https://learn.microsoft.com/en-us/aspnet/core/web-api#apicontroller-attribute) attribute.
- The class inherits *directly* from `Controller`.
- No View-specific functionality is used in the class.

### Exceptions

- If a class is marked with the [`\[NonController\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.noncontrollerattribute) attribute.
- If a class does not have [public](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/public) accessibility.

## How to fix it

Change the base type of the controller from `Controller` to `ControllerBase`.

### Code examples

#### Noncompliant code example

    [ApiController]
    public class MyController : Controller // Noncompliant: Inherit from ControllerBase instead of Controller.
    //                          ^^^^^^^^^^
    {
        // ..
    }

#### Compliant solution

    [ApiController]
    public class MyController : ControllerBase
    {
        // ..
    }

## Resources

### Documentation

- Wikipedia - [Web API](https://en.wikipedia.org/wiki/Web_API)
- Microsoft Learn - [ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core)
- Microsoft Learn - [Controller Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.controller)
- Microsoft Learn - [ControllerBase Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.controllerbase)
- Microsoft Learn - [ApiControllerAttribute
  Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.apicontrollerattribute)
- Microsoft Learn - [NonControllerAttribute
  Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.noncontrollerattribute)
- Microsoft Learn - [Views in ASP.NET Core MVC](https://learn.microsoft.com/en-us/aspnet/core/mvc/views/overview)
- Microsoft Learn - [Partial Views in ASP.NET Core MVC](https://learn.microsoft.com/en-us/aspnet/core/mvc/views/partial)
- Microsoft Learn - [View Components in ASP.NET Core MVC](https://learn.microsoft.com/en-us/aspnet/core/mvc/views/view-components)
- Microsoft Learn - [ASP.NET Core Middleware](https://learn.microsoft.com/en-us/aspnet/core/fundamentals/middleware)