[ASP.NET controllers](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/actions) should not have mixed responsibilities.
Following the [Single Responsibility Principle (SRP)](https://en.wikipedia.org/wiki/Single_responsibility_principle), they should be kept
lean and [focused
on a single, separate concern](https://learn.microsoft.com/en-us/dotnet/architecture/modern-web-apps-azure/architectural-principles#separation-of-concerns). In short, they should have a *single reason to change*.

The rule identifies different responsibilities by looking at groups of actions that use different sets of services defined in the controller.

Basic services that are typically required by most controllers are not considered:

- [`ILogger`](https://learn.microsoft.com/en-us/aspnet/core/fundamentals/logging/)
- [`IMediator`](https://en.wikipedia.org/wiki/Mediator_pattern)
- [`IMapper`](https://medium.com/@sumit.kharche/how-to-integrate-automapper-in-asp-net-core-web-api-b765b5bed35c)
- [`IConfiguration`](https://learn.microsoft.com/en-us/aspnet/core/fundamentals/configuration/)
- [`IBus`](https://masstransit.io/documentation/configuration#configuration)
- [`IMessageBus`](https://wolverinefx.io/guide/messaging/message-bus.html)

The rule currently applies to ASP.NET Core only, and doesn’t cover [ASP.NET MVC 4.x](https://learn.microsoft.com/en-us/aspnet/core/fundamentals/choose-aspnet-framework).

It also only takes into account web APIs controllers, i.e. the ones marked with the [`ApiController` attribute](https://learn.microsoft.com/en-us/aspnet/core/web-api/#apicontroller-attribute). MVC controllers are
not in scope.

## Why is this an issue?

Multiple issues can appear when the Single Responsibility Principle (SRP) is violated.

### Harder to read and understand

A controller violating SRP is **harder to read and understand** since its Cognitive Complexity is generally above average (see
{rule:csharpsquid:S3776}).

*For example, a controller `MediaController` that is in charge of both the "movies" and "photos" APIs would need to define all theactions dealing with movies, alongside the ones dealing with photos, all defined in the same controller class.*

*The alternative is to define two controllers: a `MovieController` and a `PhotoController`, each in charge of a smallernumber of actions.*

### Harder to maintain and modify

Such complexity makes the controller **harder to maintain and modify**, slowing down new development and [increasing the likelihood of bugs](https://arxiv.org/ftp/arxiv/papers/1912/1912.01142.pdf).

*For example, a change in `MediaController` made for the movies APIs may inadvertently have an impact on the photos APIs as well.Because the change was made in the context of movies, tests on photos may be overlooked, resulting in bugs in production.*

*That would not be likely to happen when two distinct controllers, `MovieController` and a `PhotoController`, aredefined.*

### Harder to test

The controller also becomes **harder to test** since the test suite would need to define a set of tests for each of the
responsibilities of the controller, resulting in a large and complex test suite.

*For example, the `MediaController` introduced above would need to be tested on all movies-related actions, as well as on allphotos-related actions.*

*All those tests would be defined in the same test suite for `MediaController`, which would be affected by the same issues ofcognitive complexity as the controller under test by the suite.*

### Harder to reuse

A controller that has multiple responsibilities is **less likely to be reusable**. Lack of reuse can result in code duplication.

*For example, when a new controller wants to derive from an existing one, it’s less probable that the new controller requires all the behaviorsexposed by the reused controller.*

*Rather, it’s much more common that the new controller only needs to reuse a fraction of the existing one. When reuse is not possible, the onlyvalid alternative is to duplicate part of the logic in the new controller.*

### Higher likelihood of performance issues

A controller that has multiple responsibilities may end up doing more than strictly necessary, resulting in a **higher likelihood ofperformance issues**.

To understand why, it’s important to consider the difference between ASP.NET application vs non-ASP.NET applications.

In a non-ASP.NET application, controllers may be defined as [Singletons](https://en.wikipedia.org/wiki/Singleton_pattern) via a [Dependency Injection](https://learn.microsoft.com/en-us/aspnet/core/fundamentals/dependency-injection) library.

In such a scenario, they would typically be instantiated only once, lazily or eagerly, at application startup.

In ASP.NET applications, however, the default is that controllers are instantiated *as many times as the number of requests that are served bythe web server*. Each instance of the controller would need to resolve services independently.

While **service instantiation** is typically handled at application startup, **service resolution** happens every time an
instance of controller needs to be built, for each service declared in the controller.

Whether the resolution is done via Dependency Injection, direct static access (in the case of a Singleton), or a [Service Locator](https://en.wikipedia.org/wiki/Service_locator_pattern), the cost of resolution needs to be paid at every single
instantiation.

*For example, the movies-related APIs of the `MediaController` mentioned above may require the instantiation of an`IStreamingService`, typically done via [dependency injection](https://learn.microsoft.com/en-us/aspnet/core/fundamentals/dependency-injection). Such a service may not be relevantfor photos-related APIs.*

*Similarly, some of the photos-related APIs may require the instantiation of an `IRedEyeRemovalService`, which may not work at allwith movies.*

*Having a single controller would force the developer to deal with both instantiations, even though a given instance of the controller may beused only for photos, or only for movies.*

### More complex routing

A controller that deals with multiple concerns often has unnecessarily complex routing: the route template at controller level cannot factorize the
route identifying the concern, so the full route template needs to be defined at the action level.

*For example, the `MediaController` would have an empty route (or equivalent, e.g. `/` or `~/`) and the actionswould need to define themselves the `movie` or `photo` prefix, depending on the type of media they deal with.*

*On the other hand, `MovieController` and `PhotoController` can factorize the `movie` and `photo`route respectively, so that the route on the action would only contain action-specific information.*

### What is the potential impact?

As the size and the responsibilities of the controller increase, the issues that come with such an increase will have a further impact on the
code.

- The increased complexity of reading and understanding the code may require the introduction of [regions](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/preprocessor-directives#defining-regions) or [partial classes](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/classes-and-structs/partial-classes-and-methods), to be
  able to visually separate the actions related to the different concerns. Those are patches, that don’t address the underlying issue.
- The increased complexity to maintain and modify will not give incentives to keep the architecture clean, leading to a **more tightly  coupled and less modular system**.
- The reduced reusability of the code may bring **code duplication**, which breaks the [Don’t repeat
  yourself (DRY)](https://learn.microsoft.com/en-us/dotnet/architecture/modern-web-apps-azure/architectural-principles#dont-repeat-yourself-dry) principle and which itself comes with a whole lot of issues, such as **lack of maintainability and consistency**.
- The performance penalty of conflating multiple responsibilities into a single controller may induce the use of techniques such as lazy service
  resolution (you can find an example of this approach [here](https://medium.com/@jayeshtambe/lazy-t-in-dependency-injection-with-c-net-core-c418cc80cd13)). Those **increase the  complexity** of the code and make the **runtime behavior less predictable**.

### Why MVC controllers are not in scope

Alongside [attribute
routing](https://learn.microsoft.com/en-us/aspnet/web-api/overview/web-api-routing-and-actions/attribute-routing-in-web-api-2), which is typical of web APIs, MVC controllers also come with [conventional routing].

In MVC, the file structure of controllers is important, since it drives [conventional routing](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/routing#conventional-routing), which is specific to MVC,
as well as [default view mapping](https://learn.microsoft.com/en-us/aspnet/core/mvc/views/overview#how-controllers-specify-views).

For those reasons, splitting an MVC controller into smaller pieces may break core behaviors of the web application such as routing and views,
triggering a large refactor of the whole project.

## How to fix it in ASP.NET Core

Split the controller into multiple controllers, each dealing with a single responsibility.

### Code examples

#### Noncompliant code example

    [Route("media")]
    public class MediaController( // Noncompliant: This controller has multiple responsibilities and could be split into 2 smaller units.
        // Used by all actions
        ILogger<MediaController> logger,
        // Movie-specific dependencies
        IStreamingService streamingService, ISubtitlesService subtitlesService,
        // Photo-specific dependencies
        IRedEyeRemovalService redEyeRemovalService, IPhotoEnhancementService photoEnhancementService) : Controller
    {
        [Route("movie/stream")]
        public IActionResult MovieStream([FromQuery] StreamRequest request) // Belongs to responsibility #1.
        {
            logger.LogInformation("Requesting movie stream for {MovieId}", request.MovieId);
            return File(streamingService.GetStream(request.MovieId), "video/mp4");
        }
    
        [Route("movie/subtitles")]
        public IActionResult MovieSubtitles([FromQuery] SubtitlesRequest request) // Belongs to responsibility #1.
        {
            logger.LogInformation("Requesting movie subtitles for {MovieId}", request.MovieId);
            return File(subtitlesService.GetSubtitles(request.MovieId, request.Language), "text/vtt");
        }
    
        [Route("photo/remove-red-eye")]
        public IActionResult RemoveRedEye([FromQuery] RedEyeRemovalRequest request) // Belongs to responsibility #2.
        {
            logger.LogInformation("Removing red-eye from photo {PhotoId}", request.PhotoId);
            return File(redEyeRemovalService.RemoveRedEye(request.PhotoId, request.Sensitivity), "image/jpeg");
        }
    
        [Route("photo/enhance")]
        public IActionResult EnhancePhoto([FromQuery] PhotoEnhancementRequest request) // Belongs to responsibility #2.
        {
            logger.LogInformation("Enhancing photo {PhotoId}", request.PhotoId);
            return File(photoEnhancementService.EnhancePhoto(request.PhotoId, request.ColorGrading), "image/jpeg");
        }
    }

#### Compliant solution

    [Route("media/[controller]")]
    public class MovieController(
        ILogger<MovieController> logger,
        IStreamingService streamingService, ISubtitlesService subtitlesService) : Controller
    {
        [Route("stream")]
        public IActionResult MovieStream([FromQuery] StreamRequest request)
        {
            logger.LogInformation("Requesting movie stream for {MovieId}", request.MovieId);
            return File(streamingService.GetStream(request.MovieId), "video/mp4");
        }
    
        [Route("subtitles")]
        public IActionResult MovieSubtitles([FromQuery] SubtitlesRequest request)
        {
            logger.LogInformation("Requesting movie subtitles for {MovieId}", request.MovieId);
            return File(subtitlesService.GetSubtitles(request.MovieId, request.Language), "text/vtt");
        }
    }
    
    [Route("media/[controller]")]
    public class PhotoController(
        ILogger<PhotoController> logger,
        IRedEyeRemovalService redEyeRemovalService, IPhotoEnhancementService photoEnhancementService) : Controller
    {
        [Route("remove-red-eye")]
        public IActionResult RemoveRedEye([FromQuery] RedEyeRemovalRequest request)
        {
            logger.LogInformation("Removing red-eye from photo {PhotoId}", request.PhotoId);
            return File(redEyeRemovalService.RemoveRedEye(request.PhotoId, request.Sensitivity), "image/jpeg");
        }
    
        [Route("enhance")]
        public IActionResult EnhancePhoto([FromQuery] PhotoEnhancementRequest request)
        {
            logger.LogInformation("Enhancing photo {PhotoId}", request.PhotoId);
            return File(photoEnhancementService.EnhancePhoto(request.PhotoId, request.ColorGrading), "image/jpeg");
        }
    }

## Resources

### Documentation

- Microsoft Learn - [Create web APIs with ASP.NET Core:
  `ApiController` attribute](https://learn.microsoft.com/en-us/aspnet/core/web-api/#apicontroller-attribute)
- Microsoft Learn - [Web API Routing](https://learn.microsoft.com/en-us/aspnet/web-api/overview/web-api-routing-and-actions/)
- Microsoft Learn - [Architectural
  principles: Separation of concerns](https://learn.microsoft.com/en-us/dotnet/architecture/modern-web-apps-azure/architectural-principles#separation-of-concerns)
- Microsoft Learn - [Architectural
  principles: Single responsibility](https://learn.microsoft.com/en-us/dotnet/architecture/modern-web-apps-azure/architectural-principles#single-responsibility)
- Microsoft Learn - [ASP.NET Core: Handle requests with
  controllers in ASP.NET Core MVC](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/actions)
- Microsoft Learn - [C# Best Practices: Dangers of Violating SOLID Principles in C#](https://learn.microsoft.com/en-us/archive/msdn-magazine/2014/may/csharp-best-practices-dangers-of-violating-solid-principles-in-csharp#the-single-responsibility-principle)
- Microsoft Learn - [Choose between ASP.NET 4.x and
  ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core/fundamentals/choose-aspnet-framework)
- Microsoft Learn - [Dependency injection in ASP.NET
  Core](https://learn.microsoft.com/en-us/aspnet/core/fundamentals/dependency-injection)
- Microsoft Learn - [Implement the command process pipeline with a mediator pattern (MediatR)](https://learn.microsoft.com/en-us/dotnet/architecture/microservices/microservice-ddd-cqrs-patterns/microservice-application-layer-implementation-web-api#implement-the-command-process-pipeline-with-a-mediator-pattern-mediatr)
- Microsoft Learn - [Lazy&lt;T&gt; Class](https://learn.microsoft.com/en-us/dotnet/api/system.lazy-1)
- MassTransit - [Concepts](https://masstransit.io/documentation/concepts)
- Sonar - [Cognitive Complexity](https://www.sonarsource.com/docs/CognitiveComplexity.pdf)
- Wikipedia - [Single responsibility principle](https://en.wikipedia.org/wiki/Single_responsibility_principle)
- Wikipedia - [Mediator pattern](https://en.wikipedia.org/wiki/Mediator_pattern)
- Wolverine - [Getting Started](https://wolverinefx.io/tutorials/getting-started.html)

### Articles & blog posts

- Sonar Blog - [5 Clean Code Tips for Reducing
  Cognitive Complexity](https://www.sonarsource.com/blog/5-clean-code-tips-for-reducing-cognitive-complexity/)
- Medium - [Lazy&lt;T&gt; in Dependency
  Injection with C# .Net Core](https://medium.com/@jayeshtambe/lazy-t-in-dependency-injection-with-c-net-core-c418cc80cd13)
- Medium - [How to integrate
  AutoMapper in ASP.NET Core Web API](https://medium.com/@sumit.kharche/how-to-integrate-automapper-in-asp-net-core-web-api-b765b5bed35c)

### Conference presentations

- Cornell University arxiv.org - [Changqi Chen: An Empirical Investigation of
  Correlation between Code Complexity and Bugs](https://arxiv.org/ftp/arxiv/papers/1912/1912.01142.pdf)

### Related rules

- {rule:csharpsquid:S3776} - Cognitive Complexity of functions should not be too high