The `HttpRequest` class provides access to the raw request data through the `QueryString`, `Headers`, and
`Forms` properties. However, whenever possible it is recommended to use model binding instead of directly accessing the input data.

## Why is this an issue?

Both ASP.Net MVC implementations - [Core](https://learn.microsoft.com/en-us/aspnet/core) and [Framework](https://learn.microsoft.com/en-us/aspnet/overview) - support model binding in a comparable fashion. Model binding streamlines the
process by automatically aligning data from HTTP requests with action method parameters, providing numerous benefits compared to manually parsing raw
incoming request data:

  - Simplicity

      - Model binding simplifies the code by automatically mapping data from HTTP requests to action method parameters. You don’t need to write any
    code to manually extract values from the request.

  - Type Safety

      - Model binding provides type safety by automatically converting the incoming data into the appropriate .NET types. If the conversion fails, the
    model state becomes invalid, which you can easily check using `ModelState.IsValid`.

  - Validation

      - With model binding, you can easily apply validation rules to your models using data annotations. If the incoming data doesn’t comply with these
    rules, the model state becomes invalid.

  - Security

      - Model binding helps protect against over-posting attacks by only including properties in the model that you explicitly bind using the
    `[Bind]` attribute or by using view models that only contain the properties you want to update.

  - Maintainability

      - By using model binding, your code becomes cleaner, easier to read, and maintain. It promotes the use of strongly typed views, which can provide
    compile-time checking of your views.

## How to fix it in ASP.NET Core

[`Request.Form`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.httprequest.form), [`Request.Form.Files`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.iformcollection.files), [`Request.Headers`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.httprequest.headers), [`Request.Query`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.httprequest.query) and [`Request.RouteValues`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.httprequest.routeValues) are keyed
collections that expose data from the incoming HTTP request:

- `Request.Form` - [`application/x-www-form-urlencoded`](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/POST#:~:text=%3Cbutton%3E%20elements%3A-,application/x%2Dwww%2Dform%2Durlencoded,-%3A%20the%20keys%20and) form data from the HTTP request body
- `Request.Form.Files` - [`multipart/form-data`](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/POST#:~:text=form%2Ddata%20instead%29-,multipart/form%2Ddata,-%3A%20each%20value%20is) file data from the HTTP request body
- `Request.Headers` - [HTTP Header values](https://developer.mozilla.org/en-US/docs/Glossary/Request_header)
- `Request.Query` - [URL parameter values](https://developer.mozilla.org/en-US/docs/Learn/Common_questions/Web_mechanics/What_is_a_URL#parameters)
- `Request.RouteValues` - Values extracted from the [path portion of the URL](https://developer.mozilla.org/en-US/docs/Learn/Common_questions/Web_mechanics/What_is_a_URL#path_to_resource)

Model binding can bind these keyed collections to

- action method parameters by matching the key to the parameter name or
- the property of a complex type by matching the key to the property name.

To replace the keyed collection access, you can:

| Replace | with parameter binding | or complex type binding | or route binding |
| --- | --- | --- | --- |
| `Request.Form["id"]` | optional [`\[FromForm\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.fromformattribute)<br>      attribute on the parameter or a [`FormCollection`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.formcollection) parameter | optional [`\[FromForm\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.fromformattribute)<br>      attribute on the property |  |
| [`Request.Form.Files`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.iformcollection.files) | [`IFormFile`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.iformfile), [`IFormFileCollection`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.iformfilecollection), or<br>      `IEnumerable<IFormFile>` parameter |  |  |
| `Request.Headers["id"]` | [`\[FromHeader\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.fromheaderattribute)<br>      attribute on the parameter | [`\[FromHeader\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.fromheaderattribute)<br>      attribute on the property |  |
| `Request.Query["id"]` | optional [`\[FromQuery\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.fromqueryattribute)<br>      attribute on the parameter | optional [`\[FromQuery\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.fromqueryattribute)<br>      attribute on the property |  |
| `Request.RouteValues["id"]` | optional [`\[FromRoute\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.fromrouteattribute)<br>      attribute on the parameter |  | optional [`\[Route("{id}")\]`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.fromrouteattribute)attribute on the<br>      action method/controller or via conventional routing |

The [Model Binding in ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core/mvc/models/model-binding) article describes the
mechanisms, conventions, and customization options for model binding in more detail. Route-based binding is described in the [Routing to controller actions in ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/routing) document.

### Code examples

#### Noncompliant code example

    public IActionResult Post()
    {
        var name = Request.Form["name"];                           // Noncompliant: Request.Form
        var birthdate = DateTime.Parse(Request.Form["Birthdate"]); // Noncompliant: Request.Form
    
        var origin = Request.Headers[HeaderNames.Origin];          // Noncompliant: Request.Headers
        var locale = Request.Query.TryGetValue("locale", out var locales)
            ? locales.ToString()
            : "en-US";                                             // Noncompliant: Request.Query
        // ..
    }

#### Compliant solution

    public record User
    {
        [Required, StringLength(100)]
        public required string Name { get; init; }
        [DataType(DataType.Date)]
        public DateTime? Birthdate { get; init; }
    }
    
    public IActionResult Post(User user, [FromHeader] string origin, [FromQuery] string locale = "en-US")
    {
        if (ModelState.IsValid)
        {
            // ...
        }
    }

### How does this work?

Model binding in ASP.NET Core MVC and ASP.NET MVC 4.x works by automatically mapping data from HTTP requests to action method parameters. Here’s a
step-by-step breakdown of how it works:

1. **Request Data** When a user submits a form or sends a request to an ASP.NET application, the request data might include form
  data, query string parameters, request body, and HTTP headers.
2. **Model Binder** The model binder’s job is to create .NET objects from the request data. It looks at each parameter in the action
  method and attempts to populate it with the incoming data.
3. **Value Providers** The model binder uses Value Providers to get data from various parts of the request, such as the query string,
  form data, or route data. Each value provider tells the model binder where to find values in the request.
4. **Binding** The model binder tries to match the keys from the incoming data with the properties of the action method’s parameters.
  If a match is found, it attempts to convert the incoming data into the appropriate .NET type and assigns it to the parameter.
5. **Validation** If the model binder can’t convert the value or if the converted value doesn’t pass any specified validation rules,
  it adds an error to the `ModelState.Errors` collection. You can check `ModelState.IsValid` in your action method to see if any
  errors occurred during model binding.
6. **Action Method Execution** The action method is executed with the bound parameters. If `ModelState.IsValid` is
  `false`, you can handle the errors in your action method and return an appropriate response.

See the links in the Resources section for more information.

## How to fix it in ASP.NET MVC 4.x

[`Request.Form`](https://learn.microsoft.com/en-us/dotnet/api/system.web.httprequestbase.form) and [`Request.QueryString`](https://learn.microsoft.com/en-us/dotnet/api/system.web.httprequestbase.querystring) are keyed collections
that expose data from the incoming HTTP request:

- `Request.Form` - [`application/x-www-form-urlencoded`](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/POST#:~:text=%3Cbutton%3E%20elements%3A-,application/x%2Dwww%2Dform%2Durlencoded,-%3A%20the%20keys%20and) form data from the HTTP request body
- `Request.QueryString` - [URL parameter values](https://developer.mozilla.org/en-US/docs/Learn/Common_questions/Web_mechanics/What_is_a_URL#parameters)

Model binding can bind these keyed collections to

- action method parameters by matching the key to the parameter name or
- the property of a complex type by matching the key to the property name.

To replace the keyed collection access, you can:

| Replace | with parameter binding | or complex type binding |
| --- | --- | --- |
| `Request.Form["id"]` | optional [`\[Bind\]`](https://learn.microsoft.com/en-us/dotnet/api/system.web.mvc.bindattribute) attribute on the<br>      parameter or a [`FormCollection`](https://learn.microsoft.com/en-us/dotnet/api/system.web.mvc.formcollection)<br>      parameter | optional [`\[Bind\]`](https://learn.microsoft.com/en-us/dotnet/api/system.web.mvc.bindattribute) attribute on the<br>      parameter or type |
| `Request.QueryString["id"]` | optional [`\[Bind\]`](https://learn.microsoft.com/en-us/dotnet/api/system.web.mvc.bindattribute) attribute on the<br>      parameter | property name must match query parameter key |

### Code examples

#### Noncompliant code example

    public ActionResult Post()
    {
        var name = Request.Form["name"];                            // Noncompliant: Request.Form
        Debug.WriteLine(Request.Form[0]);                           // Compliant: Binding by index is not supported.
        var birthdate = DateTime.Parse(Request.Form["Birthdate"]);  // Noncompliant: Request.Form
    
        var cultureName = Request.QueryString["locale"] ?? "en-US"; // Noncompliant: Request.QueryString
        // ..
    }

#### Compliant solution

    public class User
    {
        [Required, StringLength(100)]
        public string Name { get; set; }
        [DataType(DataType.Date)]
        public DateTime? Birthdate { get; set; }
    }
    
    public ActionResult Post(User user, [Bind(Prefix = "locale")] string cultureName = "en-US")
    {
        if (ModelState.IsValid)
        {
            // ...
        }
    }

### How does this work?

Model binding in ASP.NET Core MVC and ASP.NET MVC 4.x works by automatically mapping data from HTTP requests to action method parameters. Here’s a
step-by-step breakdown of how it works:

1. **Request Data** When a user submits a form or sends a request to an ASP.NET application, the request data might include form
  data, query string parameters, request body, and HTTP headers.
2. **Model Binder** The model binder’s job is to create .NET objects from the request data. It looks at each parameter in the action
  method and attempts to populate it with the incoming data.
3. **Value Providers** The model binder uses Value Providers to get data from various parts of the request, such as the query string,
  form data, or route data. Each value provider tells the model binder where to find values in the request.
4. **Binding** The model binder tries to match the keys from the incoming data with the properties of the action method’s parameters.
  If a match is found, it attempts to convert the incoming data into the appropriate .NET type and assigns it to the parameter.
5. **Validation** If the model binder can’t convert the value or if the converted value doesn’t pass any specified validation rules,
  it adds an error to the `ModelState.Errors` collection. You can check `ModelState.IsValid` in your action method to see if any
  errors occurred during model binding.
6. **Action Method Execution** The action method is executed with the bound parameters. If `ModelState.IsValid` is
  `false`, you can handle the errors in your action method and return an appropriate response.

See the links in the Resources section for more information.

## Resources

### Documentation

- Microsoft Learn - Asp.Net Core - [Model Binding in ASP.NET
  Core](https://learn.microsoft.com/en-us/aspnet/core/mvc/models/model-binding)
- Microsoft Learn - Asp.Net Core - [Model validation in ASP.NET Core
  MVC and Razor Pages](https://learn.microsoft.com/en-us/aspnet/core/mvc/models/validation)
- Microsoft Learn - Asp.Net Core - [Custom Model Binding
  in ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core/mvc/advanced/custom-model-binding)
- Microsoft Learn - Asp.Net Core - [HttpRequest.Form Property](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.httprequest.form)
- Microsoft Learn - Asp.Net Core - [IFormCollection.Files Property](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.iformcollection.files)
- Microsoft Learn - Asp.Net Core - [HttpRequest.Headers Property](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.httprequest.headers)
- Microsoft Learn - Asp.Net Core - [HttpRequest.Query Property](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.httprequest.query)
- Microsoft Learn - Asp.Net Core - [HttpRequest.RouteValues Property](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.httprequest.routevalues)
- Microsoft Learn - Asp.Net Core - [FormCollection
  Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.formcollection)
- Microsoft Learn - Asp.Net Core - [IFormFile
  Interface](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.iformfile)
- Microsoft Learn - Asp.Net Core - [IFormFileCollection Interface](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.http.iformfilecollection)
- Microsoft Learn - Asp.Net Core - [BindAttribute
  Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.mvc.bindattribute)
- Microsoft Learn - ASP.NET MVC 4.x - [Parameter Binding in
  ASP.NET Web API](https://learn.microsoft.com/en-us/aspnet/web-api/overview/formats-and-model-binding/parameter-binding-in-aspnet-web-api)
- Microsoft Learn - ASP.NET MVC 4.x - [Adding a New Controller](https://learn.microsoft.com/en-us/aspnet/mvc/overview/getting-started/introduction/adding-a-controller)
- Microsoft Learn - ASP.NET MVC 4.x - [Adding a New Model](https://learn.microsoft.com/en-us/aspnet/mvc/overview/getting-started/introduction/adding-a-model)
- Microsoft Learn - ASP.NET MVC 4.x - [Adding Validation](https://learn.microsoft.com/en-us/aspnet/mvc/overview/getting-started/introduction/adding-validation)
- Microsoft Learn - ASP.NET MVC 4.x - [Model Validation in
  ASP.NET Web API](https://learn.microsoft.com/en-us/aspnet/web-api/overview/formats-and-model-binding/model-validation-in-aspnet-web-api)
- Microsoft Learn - ASP.NET MVC 4.x - [HttpRequest.Form
  Property](https://learn.microsoft.com/en-us/dotnet/api/system.web.httprequest.form)
- Microsoft Learn - ASP.NET MVC 4.x - [HttpRequest.QueryString Property](https://learn.microsoft.com/en-us/dotnet/api/system.web.httprequest.querystring)
- Microsoft Learn - ASP.NET MVC 4.x - [BindAttribute Class](https://learn.microsoft.com/en-us/dotnet/api/system.web.mvc.bindattribute)
- MDN - [HTTP request methods &gt; POST](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/POST)
- MDN - [Request header](https://developer.mozilla.org/en-US/docs/Glossary/Request_header)
- MDN - [What is a URL?](https://developer.mozilla.org/en-US/docs/Learn/Common_questions/Web_mechanics/What_is_a_URL)