Backslash characters (`\`) should be avoided in [route templates](https://learn.microsoft.com/en-us/aspnet/core/fundamentals/routing#route-templates).

## Why is this an issue?

[Routing](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/routing) in ASP.NET MVC maps [controllers](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/actions#what-is-a-controller) and [actions](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/actions#defining-actions) to paths in request [URIs](https://en.wikipedia.org/wiki/Uniform_Resource_Identifier).

In the former syntax specification of URIs, backslash characters (`\`) were not allowed at all (see [section "2.4.3. Excluded US-ASCII Characters" of RFC 2396](https://datatracker.ietf.org/doc/html/rfc2396/#section-2.4.3)). While the current
specification ([RFC 3986](https://datatracker.ietf.org/doc/html/rfc3986)) doesn’t include anymore the "Excluded US-ASCII Characters"
section, most URL processors still don’t support backslash properly.

For instance, a backslash in the ["path" part](https://datatracker.ietf.org/doc/html/rfc3986#section-3.3) of a [URL](https://en.wikipedia.org/wiki/URL#Syntax) is automatically converted to a forward slash (`/`) both by Chrome and Internet
Explorer (see [here](https://stackoverflow.com/q/10438008)).

As an example, `\Calculator\Evaluate?expression=3\4` is converted on the fly into `/Calculator/Evaluate?expression=3\4`
before the HTTP request is made to the server.

While backslashes are allowed in the "query" part of a URL, and it’s common to have them as part of a complex query expression, the route of a
controller is always part of the "path".

That is why the use of backslashes in controller templates should be avoided in general.

### What is the potential impact?

A backslash in the route pattern of a controller would only make sense if the developer intended the backslash in the route to be explicitly
escaped by the user, using [`%5C`](https://en.wikipedia.org/wiki/Percent-encoding#Character_data).

For example, the route `Something\[controller]` for the `HomeController` would need to be called as
`Something%5CHome`.

The validity of such a scenario is unlikely and the resulting behavior is surprising.

## How to fix it

### Code examples

#### Noncompliant code example

    [Route(@"Something\[controller]")] // Noncompliant: Replace '\' with '/'.
    public class HomeController : Controller
    {
        [HttpGet]
        public ActionResult Index() => View();
    }

#### Compliant solution

    [Route(@"Something/[controller]")] // '\' replaced with '/'
    public class HomeController : Controller
    {
        [HttpGet]
        public ActionResult Index() => View();
    }

#### Noncompliant code example

    app.MapControllerRoute(
        name: "default",
        pattern: "{controller=Home}\\{action=Index}"); // Noncompliant: Replace '\' with '/'.

#### Compliant solution

    app.MapControllerRoute(
        name: "default",
        pattern: "{controller=Home}/{action=Index}"); // '\' replaced with '/'

## Resources

### Documentation

- Microsoft Learn - [Routing in ASP.NET Core](https://learn.microsoft.com/en-us/aspnet/core/fundamentals/routing)
- Microsoft Learn - [Routing to controller actions in ASP.NET
  Core](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/routing)
- Microsoft Learn - [Handle requests with controllers in ASP.NET
  Core MVC](https://learn.microsoft.com/en-us/aspnet/core/mvc/controllers/actions)
- Wikipedia - [Uniform Resource Identifier](https://en.wikipedia.org/wiki/Uniform_Resource_Identifier)
- Wikipedia - [URL - Syntax](https://en.wikipedia.org/wiki/URL#Syntax)
- Wikipedia - [Percent-encoding](https://en.wikipedia.org/wiki/Percent-encoding)

### Articles & blog posts

- StackOverflow - [Different behaviours of treating \ (backslash) in the url by FireFox and
  Chrome](https://stackoverflow.com/questions/10438008)

### Standards

- IETF.org - [RFC 3986 - Uniform Resource Identifier (URI): Generic Syntax](https://datatracker.ietf.org/doc/html/rfc3986)
- IETF.org - [RFC 2396 - Uniform Resource Identifiers (URI): Generic Syntax (OBSOLETE,
  replaced by RFC 3986)](https://datatracker.ietf.org/doc/html/rfc2396)