## Why is this an issue?

Named placeholders in [message templates](https://messagetemplates.org) should be unique. The meaning of the named placeholders is to
store the value of the provided argument under that name, enabling easier log querying. Since the named placeholder is used multiple times, it cannot
store the different values uniquely with each name hence not serving its original purpose. There can be different behaviours when using the same named
placeholder multiple times:

- [Microsoft.Extensions.Logging](https://www.nuget.org/packages/Microsoft.Extensions.Logging) saves the different values under the
  same name
- [Serilog](https://www.nuget.org/packages/Serilog) stores only the latest assigned value
- [Nlog](https://www.nuget.org/packages/NLog) makes the name unique by suffixing it with `_index`

The rule covers the following logging frameworks:

- [Microsoft.Extensions.Logging](https://www.nuget.org/packages/Microsoft.Extensions.Logging)
- [Serilog](https://www.nuget.org/packages/Serilog)
- [Nlog](https://www.nuget.org/packages/NLog)

## How to fix it

Assign unique names to each template placeholder.

### Code examples

#### Noncompliant code example

    public void Checkout(ILogger logger, User user, Order order)
    {
        logger.LogDebug("User {Id} purchased order {Id}", user.Id, order.Id);
    }

#### Compliant solution

    public void Checkout(ILogger logger, User user, Order order)
    {
        logger.LogDebug("User {UserId} purchased order {OrderId}", user.Id, order.Id);
    }

## Resources

### Documentation

- Message Templates - [Message template specification](https://messagetemplates.org/)
- Microsoft Learn - [Log message template
  formatting](https://learn.microsoft.com/en-us/dotnet/core/extensions/logging?tabs=command-line#log-message-template-formatting)
- NLog - [How to use structured logging](https://github.com/NLog/NLog/wiki/How-to-use-structured-logging)
- Serilog - [Structured Data](https://github.com/serilog/serilog/wiki/Structured-Data)
- Serilog - [`Serilog005`: Unique
  Property Name Verifier](https://github.com/Suchiman/SerilogAnalyzer/blob/master/README.md#serilog005-unique-property-name-verifier)