## Why is this an issue?

The [`Trace.WriteLineIf` Method](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.writelineif) from the
[`System.Diagnostic.Trace`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace) facility writes a trace if
the condition passed as the first parameter is `true`.

[`TraceSwitch`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.traceswitch) allows trace control via
`bool` properties for each relevant [`TraceLevel`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.tracelevel), such as [`TraceSwitch.TraceError`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.traceswitch.traceerror).

Using `Trace.WriteLineIf` with such properties should be avoided since it can lead to misinterpretation and produce confusion.

In particular, `Trace.WriteLineIf` may appear as equivalent to the level-specific tracing methods provided by `Trace`, such
as [`Trace.Error`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.traceerror), but it is not.

The difference is that `Trace.WriteLineIf(switch.TraceError, …​)` conditionally writes the trace, based on the switch, whereas
`Trace.TraceError` always writes the trace, no matter whether `switch.TraceError` is `true` or
`false`.

Moreover, unlike `Trace.TraceError`, `Trace.WriteLineIf(switch.TraceError, …​)` would behave like
`Trace.WriteLine(…​)` when `switch.TraceError` is `true`, writing unfiltered to the underlying trace listeners and
not categorizing the log entry by level, as described more in detail in {rule:csharpsquid:S6670}.

## How to fix it

The fix depends on the intent behind the use of [`TraceSwitch`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.traceswitch) levels with [`Trace.WriteLineIf`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.writelineif).

If it is **trace categorization**, level-specific tracing methods, such as [`Trace.TraceError`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.traceerror) or [`Trace.TraceWarning`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.tracewarning), should be used
instead.

If it is **trace filtering**, [`TraceSource`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.tracesource) should be used instead.

If it is **log filtering**, `Trace` should be replaced by logging APIs, such as the [`ILogger` API](https://learn.microsoft.com/en-us/dotnet/core/diagnostics/logging-tracing#net-logging-apis).

Modern logging APIs are also more suitable than `Trace` when [high-performance logging](https://learn.microsoft.com/en-us/dotnet/core/extensions/high-performance-logging) is required.

## Resources

### Documentation

- Microsoft Learn - [`Trace.WriteLineIf`
  Method](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.writelineif)
- Microsoft Learn - [`TraceSwitch`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.traceswitch)
- Microsoft Learn - [`TraceSource`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.tracesource)
- Microsoft Learn - [`Trace.WriteLine`
  Method](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.writeline)
- Microsoft Learn - [High-performance logging in
  .NET](https://learn.microsoft.com/en-us/dotnet/core/extensions/high-performance-logging)

### Articles & blog posts

- StackOverflow - [Difference between Trace.WriteLineIf and Trace.Error](https://stackoverflow.com/a/5118040)
- StackOverflow - [Difference between TraceSwitch and SourceSwitch](https://stackoverflow.com/a/3691841)
- InfoSupport Blogs - [Please be careful when using
  Trace.WriteLineIf()](https://blogs.infosupport.com/please-be-careful-when-using-trace-writelineif/)