## Why is this an issue?

In most logging frameworks, it’s good practice to set the logger name to match its enclosing type, as enforced by {rule:csharpsquid:S3416}.

Logging frameworks can define or use [Generic interfaces](https://learn.microsoft.com/en-us/dotnet/standard/generics/interfaces) for the
logger, such as [`ILogger<TCategoryName>`](https://learn.microsoft.com/en-us/dotnet/api/microsoft.extensions.logging.ilogger-1).

The use of a logger of a generic type parameter `A` (e.g. `ILogger<A>`) in a type different than `A`, say
`B`, goes against the convention.

Because the instance of type `A` would log with a logger named after `B`, log items would appear as if they were logged by
`B` instead, resulting in confusion and logging misconfiguration:

- overriding defaults for the logger named after `A` would not take effect for entries logged in the type `A`
- fine-graned logging configuration would not be possible, since there would be no way to distinguish entries logged in the type `A`
  from entries logged in the type `B`

Further details and examples are provided in {rule:csharpsquid:S3416}.

This rule specifically targets the generic logging interface [`ILogger<TCategoryName>` Interface](https://learn.microsoft.com/en-us/dotnet/api/microsoft.extensions.logging.ilogger-1)
defined by [Microsoft Extensions Logging](https://learn.microsoft.com/en-us/dotnet/core/extensions/logging).

## How to fix it

Change the generic type parameter of the `ILogger` interface to match the enclosing type.

### Noncompliant code example

    class EnclosingType
    {
        public EnclosingType(ILogger<AnotherType> logger) // Noncompliant
        {
            // ...
        }
    }

### Compliant solution

    class EnclosingType
    {
        public EnclosingType(ILogger<EnclosingType> logger) // Compliant
        {
            // ...
        }
    }

## Resources

### Documentation

- Microsoft Learn - [.NET logging and tracing](https://learn.microsoft.com/en-us/dotnet/core/diagnostics/logging-tracing)
- Microsoft Learn - [Generic interface](https://learn.microsoft.com/en-us/dotnet/standard/generics/interfaces)
- Microsoft Learn - [`ILogger<TCategoryName>` Interface](https://learn.microsoft.com/en-us/dotnet/api/microsoft.extensions.logging.ilogger-1)
- Microsoft Learn - [Logging in C# and
  .NET - Log category](https://learn.microsoft.com/en-us/dotnet/core/extensions/logging?tabs=command-line#log-category)