## Why is this an issue?

[Trace.Write](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.write) and [Trace.WriteLine](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.writeline) methods are writing to the underlying
output stream directly, bypassing the trace formatting and filtering performed by [TraceListener.TraceEvent](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.tracelistener.traceevent) implementations. It is
preferred to use [Trace.TraceError](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.traceerror), [Trace.TraceWarning](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.tracewarning), and [Trace.TraceInformation](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.traceinformation) methods instead because they
call the [TraceEvent method](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.tracelistener.traceevent) which filters the
trace output according to the [TraceEventType](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.traceeventtype) (Error,
Warning or Information) and enhance the output with additional information.

## How to fix it

Use the [Trace.TraceError](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.traceerror), [Trace.TraceWarning](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.tracewarning), or [Trace.TraceInformation](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.traceinformation) methods.

### Noncompliant code example

    try
    {
        var message = RetrieveMessage();
        Trace.Write($"Message received: {message}"); // Noncompliant
    }
    catch (Exception ex)
    {
        Trace.WriteLine(ex); // Noncompliant
    }

### Compliant solution

    try
    {
        var message = RetrieveMessage();
        Trace.TraceInformation($"Message received: {message}");
    }
    catch (Exception ex)
    {
        Trace.TraceError(ex);
    }

## Resources

### Documentation

- Microsoft Learn - [Trace.TraceError Method](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.traceerror)
- Microsoft Learn - [Trace.TraceInformation
  Method](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.traceinformation)
- Microsoft Learn - [Trace.TraceWarning Method](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.tracewarning)
- Microsoft Learn - [Trace.Write Method](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.write)
- Microsoft Learn - [Trace.WriteLine Method](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.trace.writeline)
- Microsoft Learn - [TraceListener.TraceEvent
  Method](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.tracelistener.traceevent)

### Articles & blog posts

- Stackoverflow - [Difference between Trace.Write() and Trace.TraceInformation()](https://stackoverflow.com/q/26350620)