This rule raises an issue on logging calls inside a `catch` clause that does not pass the raised `Exception`.

## Why is this an issue?

A log entry should contain all the relevant information about the current execution context. The [Exception](https://learn.microsoft.com/en-us/dotnet/api/system.exception) raised in a catch block not only provides the message but also:

- the exception type
- the [stack trace](https://learn.microsoft.com/en-us/dotnet/api/system.exception.stacktrace)
- any [inner exceptions](https://learn.microsoft.com/en-us/dotnet/api/system.exception.innerexception)
- and more about the cause of the error.

[Logging methods](https://learn.microsoft.com/en-us/dotnet/api/microsoft.extensions.logging.loggerextensions) provide overloads that
accept an `Exception` as a parameter and [logging
providers](https://learn.microsoft.com/en-us/dotnet/core/extensions/logging-providers) persist the `Exception` in a structured way to facilitate the tracking of system failures. Therefore `Exceptions`
should be passed to the logger.

The rule covers the following logging frameworks:

- Nuget package - [Castle.Core](https://www.nuget.org/packages/Castle.Core)
- Nuget package - [Common.Core](https://www.nuget.org/packages/Common.Logging.Core)
- Nuget package - [log4net](https://www.nuget.org/packages/log4net)
- Nuget package - [NLog](https://www.nuget.org/packages/NLog)
- Nuget package - [Microsoft.Extensions.Logging](https://www.nuget.org/packages/Microsoft.Extensions.Logging)

## How to fix it

### Code examples

#### Noncompliant code example

    public bool Save()
    {
        try
        {
            DoSave();
            return true;
        }
        catch(IOException)
        {
            logger.LogError("Saving failed.");             // Noncompliant: No specifics about the error are logged
            return false;
        }
    }

#### Compliant solution

    public bool Save()
    {
        try
        {
            DoSave();
            return true;
        }
        catch(IOException exception)
        {
            logger.LogError(exception, "Saving failed.");  // Compliant: Exception details are logged
            return false;
        }
    }

## Resources

### Documentation

- Microsoft Learn - [Log
  exceptions](https://learn.microsoft.com/en-us/dotnet/core/extensions/logging?tabs=command-line#log-exceptions)
- Microsoft Learn - [LoggerExtensions
  Class](https://learn.microsoft.com/en-us/dotnet/api/microsoft.extensions.logging.loggerextensions)
- Microsoft Learn - [Logging providers](https://learn.microsoft.com/en-us/dotnet/core/extensions/logging-providers)
- Microsoft Learn - [The
  `try-catch` statement](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/exception-handling-statements#the-try-catch-statement)
- Serilog - [Example application](https://github.com/serilog/serilog/wiki/Getting-Started#example-application)
- Serilog Analyzer - [`Serilog001`: Exception
  Usage](https://github.com/Suchiman/SerilogAnalyzer#serilog001-exception-usage)