A code block should not contain too many logging statements of a specific level.

## Why is this an issue?

Excessive logging within a code block can lead to several problems:

- **Log file overload**: generating an overwhelming number of log entries can fill up disk space quickly (thus increasing the
  storage space cost) and make it challenging to identify important log events promptly.
- **Performance degradation**: writing a large number of log statements can impact the performance of an application, especially
  when the logs are placed in frequently executed paths.
- **Code readability and maintainability**: excessive logging can clutter the code and increase the code’s complexity, making it
  difficult for developers to identify essential logic.

Only the logging statements that are directly within the [code block](https://learn.microsoft.com/en-us/dotnet/csharp/tour-of-csharp/program-building-blocks#statements) will be counted, and any
logging statements within nested blocks will count towards their own. For example consider the snippet below:

    void MyMethod(List<MyObject> items)
    {
        logger.Debug("The operation started");
        foreach(var item in items)
        {
            logger.Debug($"Evaluating {item.Name}");
            var result = Evaluate(item);
            logger.Debug($"Evaluating resulted in {result}");
        }
        logger.Debug("The operation ended");
    }

The rule will count 2 logging statements that are within the method block (namely `logger.Debug("The operation started")` and
`logger.Debug("The operation ended")`). Any statements within nested blocks, such as the `foreach` block will be counted
separately. The rule considers the log level of the calls, as follows:

- **Debug**, **Trace** and **Verbose** logging level statements will count together and raise when the
  ***Debug threshold*** parameter is exceeded (default value: *4*);
- **Information** logging level statements will raise when the ***Information threshold*** parameter is exceeded
  (default value: *2*);
- **Warning** logging level statements will raise when the ***Warning threshold*** parameter is exceeded
  (default value: *1*);
- **Error** and **Fatal** logging level statements will count together and raise when the ***Error  threshold*** parameter is exceeded (default value: *1*);

The most popular logging frameworks are supported:

- Nuget package - [Microsoft.Extensions.Logging](https://www.nuget.org/packages/Microsoft.Extensions.Logging)
- Nuget package - [Serilog](https://www.nuget.org/packages/Serilog)
- Nuget package - [Castle.Core](https://www.nuget.org/packages/Castle.Core)
- Nuget package - [NLog](https://www.nuget.org/packages/NLog)
- Nuget package - [log4net](https://www.nuget.org/packages/log4net)

## How to fix it

Reduce the number of specific logging level calls within the code block by identifying and selecting essential log statements with relevant
information, necessary for understanding the flow of execution or diagnosing issues.

### Code examples

#### Noncompliant code example

With the default Information threshold parameter value 2:

    void MyMethod(List<MyObject> items)
    {
        logger.Debug("The operation started");
        foreach(var item in items)
        {
            logger.Information($"Evaluating {item.Name}"); // Noncompliant
            var result = Evaluate(item);
            logger.Information($"Evaluating resulted in {result}"); // Secondary 1
            if (item.Name is string.Empty)
            {
                logger.Error("Invalid item name");
            }
            logger.Information("End item evaluation"); // Secondary 2
        }
        logger.Debug("The operation ended");
    }

#### Compliant solution

With the default Information threshold parameter value 2:

    void MyMethod(List<MyObject> items)
    {
        logger.Debug("The operation started");
        foreach(var item in items)
        {
            logger.Information($"Evaluating {item.Name}");
            var result = Evaluate(item);
            if (item.Name is string.Empty)
            {
                logger.Error("Invalid item name");
            }
            logger.Information($"End item evaluation with result: {result}");
        }
        logger.Debug("The operation ended");
    }

## Resources

### Documentation

- Microsoft Learn - [Code blocks](https://learn.microsoft.com/en-us/dotnet/csharp/tour-of-csharp/program-building-blocks#statements)
- Microsoft Learn - [Exception-handling statements](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/exception-handling-statements)