## Why is this an issue?

Both the `Enumerable.First` extension method and the `LinkedList<T>.First` property can be used to find the first value
in a `LinkedList<T>`. However, `LinkedList<T>.First` is much faster than `Enumerable.First`. For small
collections, the performance difference may be minor, but for large collections, it can be noticeable. The same applies for the `Last`
property as well.

**Applies to:**

- [LinkedList&lt;T&gt;.First](https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.linkedlist-1.first)
- [LinkedList&lt;T&gt;.Last](https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.linkedlist-1.last)

### What is the potential impact?

We measured a significant improvement both in execution time and memory allocation. For more details see the `Benchmarks` section from
the `More info` tab.

## How to fix it

The `First` and `Last` properties are defined on the `LinkedList` class, and the extension method call can be
replaced by calling the propery instead.

### Code examples

#### Noncompliant code example

    int GetFirst(LinkedList<int> data) =>
        data.First();

    int GetLast(LinkedList<int> data) =>
        data.Last();

#### Compliant solution

    int GetFirst(LinkedList<int> data) =>
        data.First.Value;

    int GetLast(LinkedList<int> data) =>
        data.Last.Value;

## Resources

### Documentation

- [LinkedList&lt;T&gt;](https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.linkedlist-1)

### Benchmarks

| Method | Runtime | Mean | Standard Deviation | Allocated |
| --- | --- | --- | --- | --- |
| LastMethod | .NET 7.0 | 919,577,629.0 ns | 44,299,688.61 ns | 48504 B |
| LastProperty | .NET 7.0 | 271.8 ns | 15.63 ns | - |
| LastMethod | .NET Framework 4.6.2 | 810,316,427.1 ns | 47,768,482.31 ns | 57344 B |
| LastProperty | .NET Framework 4.6.2 | 372.0 ns | 13.38 ns | - |

#### Glossary

- [Mean](https://en.wikipedia.org/wiki/Arithmetic_mean)
- [Standard Deviation](https://en.wikipedia.org/wiki/Standard_deviation)
- [Allocated](https://en.wikipedia.org/wiki/Memory_management)

The results were generated by running the following snippet with [BenchmarkDotNet](https://github.com/dotnet/BenchmarkDotNet):

    private LinkedList<int> data;
    private Random random = new Random();
    
    [Params(100_000)]
    public int Size { get; set; }
    
    [Params(1_000)]
    public int Runs { get; set; }
    
    [GlobalSetup]
    public void Setup() =>
        data = new LinkedList<int>(Enumerable.Range(0, Size).Select(x => random.Next()));
    
    [Benchmark(Baseline = true)]
    public void LastMethod()
    {
        for (var i = 0; i < Runs; i++)
        {
            _ = data.Last();                // Enumerable.Last()
        }
    }
    
    [Benchmark]
    public void LastProperty()
    {
        for (var i = 0; i < Runs; i++)
        {
            _ = data.Last;                  // Last property
        }
    }

Hardware configuration:

    BenchmarkDotNet=v0.13.5, OS=Windows 10 (10.0.19045.2846/22H2/2022Update)
    11th Gen Intel Core i7-11850H 2.50GHz, 1 CPU, 16 logical and 8 physical cores
      [Host]               : .NET Framework 4.8 (4.8.4614.0), X64 RyuJIT VectorSize=256
      .NET 7.0             : .NET 7.0.5 (7.0.523.17405), X64 RyuJIT AVX2
      .NET Framework 4.6.2 : .NET Framework 4.8 (4.8.4614.0), X64 RyuJIT VectorSize=256