## Why is this an issue?

Both the `List.TrueForAll` method and the `IEnumerable.All` method can be used to check if all list elements satisfy a given
condition in a collection. However, `List.TrueForAll` can be faster than `IEnumerable.All` for `List` objects. The
performance difference may be minor for small collections, but for large collections, it can be noticeable.

**Applies to**

- [List](https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.list-1.trueforall)
- [Array](https://learn.microsoft.com/en-us/dotnet/api/system.array.trueforall)
- [ImmutableList](https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.trueforall)
- [ImmutableList.Builder](https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.builder.trueforall)

### What is the potential impact?

We measured at least 4x improvement both in execution time. For more details see the `Benchmarks` section from the `More
info` tab.

## How to fix it

The `TrueForAll` method is defined on the collection class, and it has the same signature as the `All` extension method. The
method can be replaced in place.

### Code examples

#### Noncompliant code example

    public bool AreAllEven(List<int> data) =>
        data.All(x => x % 2 == 0);

    public bool AreAllEven(int[] data) =>
        data.All(x => x % 2 == 0);

#### Compliant solution

    public bool AreAllEven(List<int> data) =>
        data.TrueForAll(x => x % 2 == 0);

    public bool AreAllEven(int[] data) =>
        Array.TrueForAll(data, x => x % 2 == 0);

## Resources

### Documentation

- [List&lt;T&gt;.TrueForAll(Predicate&lt;T&gt;)](https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.list-1.trueforall)
- [Array.TrueForAll&lt;T&gt;(T\[\], Predicate&lt;T&gt;)](https://learn.microsoft.com/en-us/dotnet/api/system.array.trueforall)
- [ImmutableList&lt;T&gt;.TrueForAll(Predicate&lt;T&gt;)](https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.trueforall)
- [ImmutableList&lt;T&gt;.Builder.TrueForAll(Predicate&lt;T&gt;)](https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.builder.trueforall)
- [Enumerable.All&lt;TSource&gt;](https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.all)

### Benchmarks

| Method | Runtime | Mean | Standard Deviation | Allocated |
| --- | --- | --- | --- | --- |
| TrueForAll | .NET 7.0 | 1.302 ms | 0.0027 ms | 1 B |
| All | .NET 7.0 | 6.279 ms | 0.0181 ms | 40004 B |
| TrueForAll | .NET Framework 4.6.2 | 1.105 ms | 0.0142 ms | - |
| All | .NET Framework 4.6.2 | 4.968 ms | 0.0143 ms | 40128 B |

#### Glossary

- [Mean](https://en.wikipedia.org/wiki/Arithmetic_mean)
- [Standard Deviation](https://en.wikipedia.org/wiki/Standard_deviation)
- [Allocated](https://en.wikipedia.org/wiki/Memory_management)

The results were generated by running the following snippet with [BenchmarkDotNet](https://github.com/dotnet/BenchmarkDotNet):

    private List<int> data;
    
    [Params(10_000)]
    public int N { get; set; }
    
    [GlobalSetup]
    public void Setup() =>
        data = Enumerable.Range(0, N).Select(x => 42).ToList();
    
    [Benchmark]
    public void TrueForAll()
    {
        for (var i = 0; i < N; i++)
        {
            _ = data.TrueForAll(x => x == 42);  // List<T>.TrueForAll
        }
    }
    
    [Benchmark(Baseline = true)]
    public void All()
    {
        for (var i = 0; i < N; i++)
        {
            _ = data.All(x => x == 42);         // Enumerable.All<TSource>
        }
    }

Hardware configuration:

    BenchmarkDotNet=v0.13.5, OS=Windows 10 (10.0.19045.2846/22H2/2022Update)
    12th Gen Intel Core i7-12800H, 1 CPU, 20 logical and 14 physical cores
    .NET SDK=7.0.203
      [Host]               : .NET 7.0.5 (7.0.523.17405), X64 RyuJIT AVX2
      .NET 7.0             : .NET 7.0.5 (7.0.523.17405), X64 RyuJIT AVX2
      .NET Framework 4.6.2 : .NET Framework 4.8 (4.8.4614.0), X64 RyuJIT VectorSize=256