Since .NET 6 you don’t have to use the `TimeZoneConverter` library to manually do the conversion between IANA and Windows timezones. The
.NET 6.0 introduced new Time Zone enhancements, one being the `TimeZoneInfo.FindSystemTimeZoneById(string timezone)` method now accepts as
input both IANA and Windows time zone IDs on any operating system with installed time zone data. `TimeZoneInfo.FindSystemTimeZoneById` will
automatically convert its input from IANA to Windows and vice versa if the requested time zone is not found on the system.

## Why is this an issue?

The method `TimeZoneInfo.FindSystemTimeZoneById(string timezone)` can get both IANA and Windows timezones as input and automatically
convert one to the other if the requested time zone is not found on the system. Because one does not need to handle the conversion, the code will be
less complex and easier to maintain.

## How to fix it

There’s no need to translate manually between time zones; it is enough to call `TimeZoneInfo.FindSystemTimeZoneById(string timezone)`,
where the timezone can be IANA or Windows format. Depending on the OS, the equivalent time zone will be returned (Windows Time Zones for Windows and
IANA timezones for Linux, macOS).

### Code examples

#### Noncompliant code example

    // Assuming we are in Windows OS and we need to get the Tokyo Time Zone.
    var ianaTimeZone = "Asia/Tokyo";
    var windowsTimeZone = TZConvert.IanaToWindows(ianaTimeZone);
    TimeZoneInfo tokyoWindowsTimeZone = TimeZoneInfo.FindSystemTimeZoneById(windowsTimeZone);

#### Compliant solution

    // Assuming we are in Windows OS and we need to get the Tokyo Time Zone.
    var ianaTimeZone = "Asia/Tokyo";
    TimeZoneInfo tokyoWindowsTimeZone = TimeZoneInfo.FindSystemTimeZoneById(ianaTimeZone);

## Resources

### Documentation

- [TimeZoneInfo.FindSystemTimeZoneById
  documentation](https://learn.microsoft.com/en-us/dotnet/api/system.timezoneinfo.findsystemtimezonebyid)
- [Date, Time, and Time Zone Enhancements in .NET
  6](https://devblogs.microsoft.com/dotnet/date-time-and-time-zone-enhancements-in-net-6/)
- [TimeZoneConverter](https://github.com/mattjohnsonpint/TimeZoneConverter)

### Articles & blog posts

- [What’s up with TimeZoneInfo on .NET 6?](https://codeblog.jonskeet.uk/2022/02/05/whats-up-with-timezoneinfo-on-net-6-part-1/)

### Standards

- [IANA Time Zone Database](https://www.iana.org/time-zones)
- [Windows Time Zones](https://learn.microsoft.com/en-us/windows-hardware/manufacture/desktop/default-time-zones?view=windows-11)