Not specifying a timeout for regular expressions can lead to a Denial-of-Service attack. Pass a timeout when using
`System.Text.RegularExpressions` to process untrusted input because a malicious user might craft a value for which the evaluation lasts
excessively long.

## Ask Yourself Whether

- the input passed to the regular expression is untrusted.
- the regular expression contains patterns vulnerable to [catastrophic
  backtracking](https://www.regular-expressions.info/catastrophic.html).

There is a risk if you answered yes to any of those questions.

## Recommended Secure Coding Practices

- It is recommended to specify a [`matchTimeout`](https://learn.microsoft.com/dotnet/standard/base-types/best-practices#use-time-out-values) when executing a
  regular expression.
- Make sure regular expressions are not vulnerable to Denial-of-Service attacks by reviewing the patterns.
- Consider using a non-backtracking algorithm by specifying [`RegexOptions.NonBacktracking`](https://learn.microsoft.com/dotnet/api/system.text.regularexpressions.regexoptions?view=net-7.0).

## Sensitive Code Example

    public void RegexPattern(string input)
    {
        var emailPattern = new Regex(".+@.+", RegexOptions.None);
        var isNumber = Regex.IsMatch(input, "[0-9]+");
        var isLetterA = Regex.IsMatch(input, "(a+)+");
    }

## Compliant Solution

    public void RegexPattern(string input)
    {
        var emailPattern = new Regex(".+@.+", RegexOptions.None, TimeSpan.FromMilliseconds(100));
        var isNumber = Regex.IsMatch(input, "[0-9]+", RegexOptions.None, TimeSpan.FromMilliseconds(100));
        var isLetterA = Regex.IsMatch(input, "(a+)+", RegexOptions.NonBacktracking); // .Net 7 and above
        AppDomain.CurrentDomain.SetData("REGEX_DEFAULT_MATCH_TIMEOUT", TimeSpan.FromMilliseconds(100)); // process-wide setting
    }

## See

- OWASP - [Top 10 2017 Category A1 - Injection](https://owasp.org/www-project-top-ten/2017/A1_2017-Injection)
- CWE - [CWE-400 - Uncontrolled Resource Consumption](https://cwe.mitre.org/data/definitions/400)
- CWE - [CWE-1333 - Inefficient Regular Expression Complexity](https://cwe.mitre.org/data/definitions/1333)
- [regular-expressions.info](https://www.regular-expressions.info/catastrophic.html) - Runaway Regular Expressions: Catastrophic
  Backtracking
- [owasp.org](https://owasp.org/www-community/attacks/Regular_expression_Denial_of_Service_-_ReDoS) - Regular expression Denial of
  Service - ReDoS
- CWE - [CWE-1333 - Inefficient Regular Expression Complexity](https://cwe.mitre.org/data/definitions/1333)
- [docs.microsoft.com](https://docs.microsoft.com/dotnet/standard/base-types/best-practices) - Best practices for regular expressions
  in .NET
- [docs.microsoft.com](https://docs.microsoft.com/dotnet/standard/base-types/backtracking-in-regular-expressions) - Backtracking in
  Regular Expressions
- [devblogs.microsoft.com](https://devblogs.microsoft.com/dotnet/regular-expression-improvements-in-dotnet-7/#backtracking-and-regexoptions-nonbacktracking) - Regular Expression Improvements in .NET 7: Backtracking (and RegexOptions.NonBacktracking)
- [docs.microsoft.com](https://docs.microsoft.com/dotnet/api/system.text.regularexpressions.regex.matchtimeout) - Regex.MatchTimeout
  Property
- [docs.microsoft.com](https://docs.microsoft.com/dotnet/api/system.text.regularexpressions.regexoptions?view=net-7.0) - RegexOptions
  Enum (NonBacktracking option)