## Why is this an issue?

[Regular expressions](https://learn.microsoft.com/en-us/dotnet/standard/base-types/regular-expressions) have their own syntax that is
understood by regular expression engines. Those engines will throw an exception at runtime if they are given a regular expression that does not
conform to that syntax.

To avoid syntax errors, special characters should be [escaped with backslashes](https://learn.microsoft.com/en-us/dotnet/standard/base-types/character-escapes-in-regular-expressions) when they
are intended to be matched literally and references to capturing groups should use the correctly spelled name or number of the group.

Negative [lookaround](https://learn.microsoft.com/en-us/dotnet/standard/base-types/regular-expression-language-quick-reference#lookarounds-at-a-glance)
groups cannot be combined with [RegexOptions.NonBacktracking](https://learn.microsoft.com/en-us/dotnet/standard/base-types/backtracking-in-regular-expressions). Such
combination would throw an exception during runtime.

## How to fix it

### Code examples

#### Noncompliant code example

    void Regexes(string input)
    {
        var regex = new Regex("[A");                                                    // Noncompliant
        var match = Regex.Match(input, "[A");                                           // Noncompliant
        var negativeLookahead = new Regex("a(?!b)", RegexOptions.NonBacktracking);      // Noncompliant
        var negativeLookbehind = new Regex("(?<!a)b", RegexOptions.NonBacktracking);    // Noncompliant
    }

#### Compliant solution

    void Regexes(string input)
    {
        var regex = new Regex("[A-Z]");
        var match = Regex.Match(input, "[A-Z]");
        var negativeLookahead = new Regex("a(?!b)");
        var negativeLookbehind = new Regex("(?<!a)b");
    }

## Resources

### Documentation

- [.NET Regular expressions](https://learn.microsoft.com/en-us/dotnet/standard/base-types/regular-expressions)
- [Lookarounds
  at a glance](https://learn.microsoft.com/en-us/dotnet/standard/base-types/regular-expression-language-quick-reference#lookarounds-at-a-glance)
- [Backtracking in Regular
  Expressions](https://learn.microsoft.com/en-us/dotnet/standard/base-types/backtracking-in-regular-expressions)
- [Character Escapes in Regular
  Expressions](https://learn.microsoft.com/en-us/dotnet/standard/base-types/character-escapes-in-regular-expressions)