Deserialization is the process of converting serialized data (such as objects or data structures) back into their original form. Types allowed to
be unserialized should be strictly controlled.

## Why is this an issue?

During the deserialization process, the state of an object will be reconstructed from the serialized data stream. By allowing unrestricted
deserialization of types, the application makes it possible for attackers to use types with dangerous or otherwise sensitive behavior during the
deserialization process.

### What is the potential impact?

When an application deserializes untrusted data without proper restrictions, an attacker can craft malicious serialized objects. Depending on the
affected objects and properties, the consequences can vary.

### Remote Code Execution

If attackers can craft malicious serialized objects that contain executable code, this code will run within the application’s context, potentially
gaining full control over the system. This can lead to unauthorized access, data breaches, or even complete system compromise.

For example, a well-known attack vector consists in serializing an object of type `TempFileCollection`
with arbitrary files (defined by an attacker) which will be deleted on the application deserializing this object (when the [finalize()](https://docs.microsoft.com/en-us/dotnet/api/system.codedom.compiler.tempfilecollection.finalize?view=netframework-4.8) method of
the TempFileCollection object is called). These kinds of specially crafted serialized objects are called "[gadgets](https://github.com/pwntester/ysoserial.net)".

### Privilege escalation

Unrestricted deserialization can also enable attackers to escalate their privileges within the application. By manipulating the serialized data, an
attacker can modify object properties or bypass security checks, granting them elevated privileges that they should not have. This can result in
unauthorized access to sensitive data, unauthorized actions, or even administrative control over the application.

### Denial of Service

In some cases, an attacker can abuse the deserialization process to cause a denial of service (DoS) condition. By providing specially crafted
serialized data, the attacker can trigger excessive resource consumption, leading to system instability or unresponsiveness. This can disrupt the
availability of the application, impacting its functionality and causing inconvenience to users.

## How to fix it

### Code examples

#### Noncompliant code example

With [`BinaryFormatter`](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.formatters.binary.binaryformatter),
[`NetDataContractSerializer`](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.netdatacontractserializer)
or [`SoapFormatter`](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.formatters.soap.soapformatter):

    var myBinaryFormatter = new BinaryFormatter();
    myBinaryFormatter.Deserialize(stream); // Noncompliant

With [`JavaScriptSerializer`](https://learn.microsoft.com/en-us/dotnet/api/system.web.script.serialization.javascriptserializer):

    JavaScriptSerializer serializer1 = new JavaScriptSerializer(new SimpleTypeResolver()); // Noncompliant
    serializer1.Deserialize<ExpectedType>(json);

#### Compliant solution

With [`BinaryFormatter`](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.formatters.binary.binaryformatter),
[`NetDataContractSerializer`](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.netdatacontractserializer)
or [`SoapFormatter`](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.formatters.soap.soapformatter):

    sealed class CustomBinder : SerializationBinder
    {
       public override Type BindToType(string assemblyName, string typeName)
       {
           if (!(typeName == "type1" || typeName == "type2" || typeName == "type3"))
           {
              throw new SerializationException("Only type1, type2 and type3 are allowed");
           }
           return Assembly.Load(assemblyName).GetType(typeName);
       }
    }
    
    var myBinaryFormatter = new BinaryFormatter();
    myBinaryFormatter.Binder = new CustomBinder();
    myBinaryFormatter.Deserialize(stream);

With [`JavaScriptSerializer`](https://learn.microsoft.com/en-us/dotnet/api/system.web.script.serialization.javascriptserializer):

    public class CustomSafeTypeResolver : JavaScriptTypeResolver
    {
       public override Type ResolveType(string id)
       {
          if(id != "ExpectedType") {
             throw new ArgumentNullException("Only ExpectedType is allowed during deserialization");
          }
          return Type.GetType(id);
       }
    }
    
    JavaScriptSerializer serializer = new JavaScriptSerializer(new CustomSafeTypeResolver());
    serializer.Deserialize<ExpectedType>(json);

### Going the extra mile

Instead of using [`BinaryFormatter`](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.formatters.binary.binaryformatter)
and similar serializers, it is recommended to use safer alternatives in most of the cases, such as [`XmlSerializer`](https://learn.microsoft.com/en-us/dotnet/api/system.xml.serialization.xmlserializer) or [`DataContractSerializer`](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.datacontractserializer).

If it’s not possible then try to mitigate the risk by restricting the types allowed to be deserialized:

- by implementing an "allow-list" of types, but keep in mind that novel dangerous types are regularly discovered and this protection could be
  insufficient over time.
- or/and implementing a tamper protection, such as [message authentication codes](https://en.wikipedia.org/wiki/HMAC) (MAC). This way
  only objects serialized with the correct MAC hash will be deserialized.

## Resources

### Documentation

- Microsoft Learn - [BinaryFormatter Class](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.formatters.binary.binaryformatter)
- Microsoft Learn - [NetDataContractSerializer Class](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.netdatacontractserializer)
- Microsoft Learn - [SoapFormatter Class](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.formatters.soap.soapformatter)
- Microsoft Learn - [JavaScriptSerializer Class](https://learn.microsoft.com/en-us/dotnet/api/system.web.script.serialization.javascriptserializer)
- Microsoft Learn - [XmlSerializer Class](https://learn.microsoft.com/en-us/dotnet/api/system.xml.serialization.xmlserializer)
- Microsoft Learn - [DataContractSerializer Class](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.serialization.datacontractserializer)

### Articles & blog posts

- Microsoft Learn - [Deserialization
  risks in use of BinaryFormatter and related types](https://docs.microsoft.com/en-us/dotnet/standard/serialization/binaryformatter-security-guide?s=03)
- OWASP - [Deserialization Cheat
  Sheet](https://github.com/OWASP/CheatSheetSeries/blob/master/cheatsheets/Deserialization_Cheat_Sheet.md)
- Wikipedia - [Message Authentication Codes (MAC)](https://en.wikipedia.org/wiki/HMAC)

### Standards

- OWASP - [Top 10 2021 Category A8 - Software and Data Integrity
  Failures](https://owasp.org/Top10/A08_2021-Software_and_Data_Integrity_Failures/)
- OWASP - [Top 10 2017 Category A8 - Insecure
  Deserialization](https://owasp.org/www-project-top-ten/2017/A8_2017-Insecure_Deserialization)
- CWE - [CWE-134 - Use of Externally-Controlled Format String](https://cwe.mitre.org/data/definitions/134)
- CWE - [CWE-502 - Deserialization of Untrusted Data](https://cwe.mitre.org/data/definitions/502)