Deserialization process extracts data from the serialized representation of an object and reconstruct it directly, without calling constructors.
Thus, data validation implemented in constructors can be bypassed if serialized objects are controlled by an attacker.

## Ask Yourself Whether

- The data validation implemented in constructors enforces a relevant security check.
- Objects instantiated via deserialization don’t run the same security checks as the ones executed when objects are created through constructors.

There is a risk if you answered yes to any of those questions.

## Recommended Secure Coding Practices

- At the end of the deserialization process it is recommended to perform the same validation checks as the ones performed in constructors,
  especially when the serialized object can be controlled by an attacker.

## Sensitive Code Example

When a `serializable` class doesn’t inherit from [ISerializable](https://docs.microsoft.com/en-us/dotnet/api/system.runtime.serialization.iserializable?view=netframework-4.8) or [IDeserializationCallback](https://docs.microsoft.com/fr-fr/dotnet/api/system.runtime.serialization.ideserializationcallback?view=netframework-4.8)
types and has a constructor using its parameters in conditions:

    [Serializable]
    public class InternalUrl
    {
        private string url;
    
        public InternalUrl(string tmpUrl) // Sensitive
        {
           if(!tmpUrl.StartsWith("http://localhost/")) // there is some input validation
           {
              url= "http://localhost/default";
           }
           else
           {
              url= tmpUrl;
           }
        }
    }

When a class inherit from [ISerializable](https://docs.microsoft.com/en-us/dotnet/api/system.runtime.serialization.iserializable?view=netframework-4.8) type, has a
regular constructor using its parameters in conditions, but doesn’t perform the same validation after deserialization:

    [Serializable]
    public class InternalUrl : ISerializable
    {
        private string url;
    
        public InternalUrl(string tmpUrl) // Sensitive
        {
            if(!tmpUrl.StartsWith("http://localhost/")) // there is some input validation
            {
                url= "http://localhost/default";
            }
            else
            {
                url= tmpUrl;
            }
        }
    
        // special constructor used during deserialization
        protected InternalUrl(SerializationInfo info, StreamingContext context) // Sensitive
        {
           url= (string) info.GetValue("url", typeof(string));
           // the same validation as seen in the regular constructor is not performed
         }
    
        void ISerializable.GetObjectData(SerializationInfo info, StreamingContext context)
        {
            info.AddValue("url", url);
        }
    }

When a class inherit from [IDeserializationCallback](https://docs.microsoft.com/fr-fr/dotnet/api/system.runtime.serialization.ideserializationcallback?view=netframework-4.8)
type, has a constructor using its parameters in conditions but the `IDeserializationCallback.OnDeserialization` method doesn’t perform any
conditional checks:

    [Serializable]
    public class InternalUrl : IDeserializationCallback
    {
        private string url;
    
        public InternalUrl(string tmpUrl) // Sensitive
        {
            if(!tmpUrl.StartsWith("http://localhost/")) // there is some input validation
            {
                url= "http://localhost/default";
            }
            else
            {
                url= tmpUrl;
            }
        }
    
        void IDeserializationCallback.OnDeserialization(object sender) // Sensitive
        {
           // the same validation as seen in the constructor is not performed
        }
    }

## Compliant Solution

When using [ISerializable](https://docs.microsoft.com/en-us/dotnet/api/system.runtime.serialization.iserializable?view=netframework-4.8)
type to control deserialization, perform the same checks inside regular constructors than in the special constructor `SerializationInfo info,
StreamingContext context` used during deserialization:

    [Serializable]
    public class InternalUrl : ISerializable
    {
        private string url;
    
        public InternalUrl(string tmpUrl)
        {
            if(!tmpUrl.StartsWith("http://localhost/")) // there is some input validation
            {
                url= "http://localhost/default";
            }
            else
            {
                url= tmpUrl;
            }
        }
    
        // special constructor used during deserialization
        protected InternalUrl(SerializationInfo info, StreamingContext context)
        {
           string tmpUrl= (string) info.GetValue("url", typeof(string));
    
           if(!tmpUrl.StartsWith("http://localhost/") { // Compliant
              url= "http://localhost/default";
           }
           else {
              url= tmpUrl;
           }
         }
    
        void ISerializable.GetObjectData(SerializationInfo info, StreamingContext context)
        {
            info.AddValue("url", url);
        }
    }

When using [IDeserializationCallback](https://docs.microsoft.com/fr-fr/dotnet/api/system.runtime.serialization.ideserializationcallback?view=netframework-4.8)
type to control deserialization, perform the same checks inside regular constructors than after deserialization with
`IDeserializationCallback.OnDeserialization` method:

    [Serializable]
    public class InternalUrl : IDeserializationCallback
    {
        private string url;
    
        public InternalUrl(string tmpUrl)
        {
           if(!tmpUrl.StartsWith("http://localhost/")) // there is some input validation
           {
              url= "http://localhost/default";
           }
           else
           {
              url= tmpUrl;
           }
        }
    
        void IDeserializationCallback.OnDeserialization(object sender) // Compliant
        {
            if(!url.StartsWith("http://localhost/"))
            {
                url= "http://localhost/default";
            }
            else
            {
            }
        }
    }

## See

- OWASP - [Top 10 2021 Category A8 - Software and Data Integrity
  Failures](https://owasp.org/Top10/A08_2021-Software_and_Data_Integrity_Failures/)
- OWASP - [Top 10 2017 Category A8 - Insecure
  Deserialization](https://owasp.org/www-project-top-ten/2017/A8_2017-Insecure_Deserialization)
- [docs.microsoft.com](https://docs.microsoft.com/en-us/dotnet/framework/misc/security-and-serialization) - security-and-serialization
- CWE - [CWE-502 - Deserialization of Untrusted Data](https://cwe.mitre.org/data/definitions/502)