This vulnerability allows forging of JSON Web Tokens to impersonate other users.

## Why is this an issue?

JSON Web Tokens (JWTs), a popular method of securely transmitting information between parties as a JSON object, can become a significant security
risk when they are not properly signed with a robust cipher algorithm, left unsigned altogether, or if the signature is not verified. This
vulnerability class allows malicious actors to craft fraudulent tokens, effectively impersonating user identities. In essence, the integrity of a JWT
hinges on the strength and presence of its signature.

### What is the potential impact?

When a JSON Web Token is not appropriately signed with a strong cipher algorithm or if the signature is not verified, it becomes a significant
threat to data security and the privacy of user identities.

#### Impersonation of users

JWTs are commonly used to represent user authorization claims. They contain information about the user’s identity, user roles, and access rights.
When these tokens are not securely signed, it allows an attacker to forge them. In essence, a weak or missing signature gives an attacker the power to
craft a token that could impersonate any user. For instance, they could create a token for an administrator account, gaining access to high-level
permissions and sensitive data.

#### Unauthorized data access

When a JWT is not securely signed, it can be tampered with by an attacker, and the integrity of the data it carries cannot be trusted. An attacker
can manipulate the content of the token and grant themselves permissions they should not have, leading to unauthorized data access.

## How to fix it in Jwt.Net

### Code examples

The following code contains an example of JWT decoding without verification of the signature.

#### Noncompliant code example

    using JWT;
    
    public static void decode(IJwtDecoder decoder)
    {
        decoder.Decode(token, secret, verify: false); // Noncompliant
    }

    using JWT;
    
    public static void decode()
    {
        var jwt = new JwtBuilder()
            .WithSecret(secret)
            .Decode(token); // Noncompliant
    }

#### Compliant solution

    using JWT;
    
    public static void decode(IJwtDecoder decoder)
    {
        decoder.Decode(token, secret, verify: true);
    }

When using `JwtBuilder`, make sure to call `MustVerifySignature()`.

    using JWT;
    
    public static void decode()
    {
        var jwt = new JwtBuilder()
            .WithSecret(secret)
            .MustVerifySignature()
            .Decode(token);
    }

### How does this work?

#### Verify the signature of your tokens

Resolving a vulnerability concerning the validation of JWT token signatures is mainly about incorporating a critical step into your process:
validating the signature every time a token is decoded. Just having a signed token using a secure algorithm is not enough. If you are not validating
signatures, they are not serving their purpose.

Every time your application receives a JWT, it needs to decode the token to extract the information contained within. It is during this decoding
process that the signature of the JWT should also be checked.

To resolve the issue, follow these instructions:

1. Use framework-specific functions for signature verification: Most programming frameworks that support JWTs provide specific functions to not
  only decode a token but also validate its signature simultaneously. Make sure to use these functions when handling incoming tokens.
2. Handle invalid signatures appropriately: If a JWT’s signature does not validate correctly, it means the token is not trustworthy, indicating
  potential tampering. The action to take when encountering an invalid token should be denying the request carrying it and logging the event for
  further investigation.
3. Incorporate signature validation in your tests: When you are writing tests for your application, include tests that check the signature
  validation functionality. This can help you catch any instances where signature verification might be unintentionally skipped or bypassed.

By following these practices, you can ensure the security of your application’s JWT handling process, making it resistant to attacks that rely on
tampering with tokens. Validation of the signature needs to be an integral and non-negotiable part of your token handling process.

### Going the extra mile

#### Securely store your secret keys

Ensure that your secret keys are stored securely. They should not be hard-coded into your application code or checked into your version control
system. Instead, consider using environment variables, secure key management systems, or vault services.

#### Rotate your secret keys

Even with the strongest cipher algorithms, there is a risk that your secret keys may be compromised. Therefore, it is a good practice to
periodically rotate your secret keys. By doing so, you limit the amount of time that an attacker can misuse a stolen key. When you rotate keys, be
sure to allow a grace period where tokens signed with the old key are still accepted to prevent service disruptions.

## Resources

### Standards

- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- CWE - [CWE-347 - Improper Verification of Cryptographic Signature](https://cwe.mitre.org/data/definitions/347)