This vulnerability makes it possible that the cleartext of the encrypted message might be recoverable without prior knowledge of the key.

## Why is this an issue?

Encryption algorithms are essential for protecting sensitive information and ensuring secure communication in various domains. They are used for
several important reasons:

- Confidentiality, privacy, and intellectual property protection
- Security during transmission or on storage devices
- Data integrity, general trust, and authentication

When selecting encryption algorithms, tools, or combinations, you should also consider two things:

1. No encryption is unbreakable.
2. The strength of an encryption algorithm is usually measured by the effort required to crack it within a reasonable time frame.

For these reasons, as soon as cryptography is included in a project, it is important to choose encryption algorithms that are considered strong and
secure by the cryptography community.

### What is the potential impact?

The cleartext of an encrypted message might be recoverable. Additionally, it might be possible to modify the cleartext of an encrypted message.

Below are some real-world scenarios that illustrate some impacts of an attacker exploiting the vulnerability.

#### Theft of sensitive data

The encrypted message might contain data that is considered sensitive and should not be known to third parties.

By using a weak algorithm the likelihood that an attacker might be able to recover the cleartext drastically increases.

#### Additional attack surface

By modifying the cleartext of the encrypted message it might be possible for an attacker to trigger other vulnerabilities in the code. Encrypted
values are often considered trusted, since under normal circumstances it would not be possible for a third party to modify them.

## How to fix it in .NET

### Code examples

The following code contains examples of algorithms that are not considered highly resistant to cryptanalysis and thus should be avoided.

#### Noncompliant code example

    using System.Security.Cryptography;
    
    public void encrypt()
    {
        var simpleDES = new DESCryptoServiceProvider(); // Noncompliant
    }

#### Compliant solution

    using System.Security.Cryptography;
    
    public void encrypt()
    {
        using (Aes aes = Aes.Create())
        {
            // ...
        }
    }

### How does this work?

#### Use a secure algorithm

It is highly recommended to use an algorithm that is currently considered secure by the cryptographic community. A common choice for such an
algorithm is the Advanced Encryption Standard (AES).

For block ciphers, it is not recommended to use algorithms with a block size that is smaller than 128 bits.

## How to fix it in BouncyCastle

### Code examples

The following code contains examples of algorithms that are not considered highly resistant to cryptanalysis and thus should be avoided.

#### Noncompliant code example

    using Org.BouncyCastle.Crypto.Engines;
    using Org.BouncyCastle.Crypto.Parameters;
    
    public void encrypt()
    {
        AesFastEngine aesFast = new AesFastEngine(); // Noncompliant
    }

#### Compliant solution

    using Org.BouncyCastle.Crypto.Engines;
    using Org.BouncyCastle.Crypto.Parameters;
    
    public void encrypt()
    {
        var AES = new AESEngine();
    }

### How does this work?

#### Use a secure algorithm

It is highly recommended to use an algorithm that is currently considered secure by the cryptographic community. A common choice for such an
algorithm is the Advanced Encryption Standard (AES).

For block ciphers, it is not recommended to use algorithms with a block size that is smaller than 128 bits.

## Resources

### Standards

- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- OWASP - [Top 10 2017 Category A6 - Security
  Misconfiguration](https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration)
- CWE - [CWE-327 - Use of a Broken or Risky Cryptographic Algorithm](https://cwe.mitre.org/data/definitions/327)
- STIG Viewer - [Application Security and
  Development: V-222396](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222396) - The application must implement DoD-approved encryption to protect the confidentiality of remote access sessions.