The improper storage of passwords poses a significant security risk to software applications. This vulnerability arises when passwords are stored
in plaintext or with a fast hashing algorithm. To exploit this vulnerability, an attacker typically requires access to the stored passwords.

## Why is this an issue?

Attackers who would get access to the stored passwords could reuse them without further attacks or with little additional effort.  
 Obtaining the
plaintext passwords, they could then gain unauthorized access to user accounts, potentially leading to various malicious activities.

### What is the potential impact?

Plaintext or weakly hashed password storage poses a significant security risk to software applications.

#### Unauthorized Access

When passwords are stored in plaintext or with weak hashing algorithms, an attacker who gains access to the password database can easily retrieve
and use the passwords to gain unauthorized access to user accounts. This can lead to various malicious activities, such as unauthorized data access,
identity theft, or even financial fraud.

#### Credential Reuse

Many users tend to reuse passwords across multiple platforms. If an attacker obtains plaintext or weakly hashed passwords, they can potentially use
these credentials to gain unauthorized access to other accounts held by the same user. This can have far-reaching consequences, as sensitive personal
information or critical systems may be compromised.

#### Regulatory Compliance

Many industries and jurisdictions have specific regulations and standards to protect user data and ensure its confidentiality. Storing passwords in
plaintext or with weak hashing algorithms can lead to non-compliance with these regulations, potentially resulting in legal consequences, financial
penalties, and damage to the reputation of the software application and its developers.

## How to fix it in ASP.NET Core

### Code examples

#### Noncompliant code example

Using `Microsoft.AspNetCore.Cryptography.KeyDerivation`:

    using Microsoft.AspNetCore.Cryptography.KeyDerivation;
    using System.Security.Cryptography;
    
    string password = Request.Query["password"];
    byte[] salt = RandomNumberGenerator.GetBytes(128 / 8);
    
    string hashed = Convert.ToBase64String(KeyDerivation.Pbkdf2(
        password: password!,
        salt: salt,
        prf: KeyDerivationPrf.HMACSHA256,
        iterationCount: 1, // Noncompliant
        numBytesRequested: 256 / 8));

Using `System.Security.Cryptography`:

    using System.Security.Cryptography;
    
    string password = Request.Query["password"];
    byte[] salt = RandomNumberGenerator.GetBytes(128 / 8);
    Rfc2898DeriveBytes kdf = new Rfc2898DeriveBytes(password, 128/8); // Noncompliant
    string hashed = Convert.ToBase64String(kdf.GetBytes(256 / 8));

Using `Microsoft.AspNetCore.Identity`:

    using Microsoft.AspNetCore.Identity;
    using Microsoft.Extensions.Options;
    
    string password = Request.Query["password"];
    IOptions<PasswordHasherOptions> options = Options.Create(new PasswordHasherOptions() {
        IterationCount = 1 // Noncompliant
    });
    PasswordHasher<User> hasher = new PasswordHasher<User>(options);
    string hash = hasher.HashPassword(new User("test"), password);

#### Compliant solution

Using `Microsoft.AspNetCore.Cryptography.KeyDerivation`:

    using Microsoft.AspNetCore.Cryptography.KeyDerivation;
    using System.Security.Cryptography;
    
    string password = Request.Query["password"];
    byte[] salt = RandomNumberGenerator.GetBytes(128 / 8);
    
    string hashed = Convert.ToBase64String(KeyDerivation.Pbkdf2(
        password: password!,
        salt: salt,
        prf: KeyDerivationPrf.HMACSHA256,
        iterationCount: 100_000,
        numBytesRequested: 256 / 8));

Using `System.Security.Cryptography`

    using System.Security.Cryptography;
    
    string password = Request.Query["password"];
    byte[] salt = RandomNumberGenerator.GetBytes(128 / 8);
    Rfc2898DeriveBytes kdf = new Rfc2898DeriveBytes(password, salt, 100_000, HashAlgorithmName.SHA256);
    string hashed = Convert.ToBase64String(kdf.GetBytes(256 / 8));

Using `Microsoft.AspNetCore.Identity`:

    using Microsoft.AspNetCore.Identity;
    using Microsoft.Extensions.Options;
    
    string password = Request.Query["password"];
    PasswordHasher<User> hasher = new PasswordHasher<User>();
    string hash = hasher.HashPassword(new User("test"), password);

### How does this work?

#### Select the correct PBKDF2 parameters

If PBKDF2 must be used, be aware that default values might not be considered secure.  
 Depending on the algorithm used, the number of iterations
should be adjusted to ensure that the derived key is secure. The following are the recommended number of iterations for PBKDF2:

- PBKDF2-HMAC-SHA1: 1,300,000 iterations
- PBKDF2-HMAC-SHA256: 600,000 iterations
- PBKDF2-HMAC-SHA512: 210,000 iterations

Note that PBKDF2-HMAC-SHA256 is recommended by NIST.  
 Iterations are also called "rounds" depending on the library used.

When recommended cost factors are too high in the context of the application or if the performance cost is unacceptable, a cost factor reduction
might be considered. In that case, it should not be chosen under 100,000.

### Going the extra mile

#### Pepper

In a defense-in-depth security approach, **peppering** can also be used. This is a security technique where an external secret value
is added to a password before it is hashed.  
 This makes it more difficult for an attacker to crack the hashed passwords, as they would need to know
the secret value to generate the correct hash.  
 [Learn more here](https://cheatsheetseries.owasp.org/cheatsheets/Password_Storage_Cheat_Sheet.html#peppering).

## How to fix it in ASP.NET

### Code examples

#### Noncompliant code example

    using System.Security.Cryptography;
    
    RNGCryptoServiceProvider rngCsp = new RNGCryptoServiceProvider();
    byte[] salt = new byte[32];
    rngCsp.GetBytes(salt);
    Rfc2898DeriveBytes kdf = new Rfc2898DeriveBytes(password, salt, 100, HashAlgorithmName.SHA256); // Noncompliant
    string hashed = Convert.ToBase64String(kdf.GetBytes(256 / 8));

Using `using Microsoft.AspNet.Identity`:

    using Microsoft.AspNet.Identity;
    
    string password = "NotSoSecure";
    PasswordHasher hasher = new PasswordHasher();
    string hash = hasher.HashPassword(password); // Noncompliant

#### Compliant solution

    using System.Security.Cryptography;
    
    RNGCryptoServiceProvider rngCsp = new RNGCryptoServiceProvider();
    byte[] salt = new byte[32];
    rngCsp.GetBytes(salt);
    Rfc2898DeriveBytes kdf = new Rfc2898DeriveBytes(password, salt, 100_000, HashAlgorithmName.SHA256); // Compliant
    string hashed = Convert.ToBase64String(kdf.GetBytes(256 / 8));

### How does this work?

#### Select the correct PBKDF2 parameters

If PBKDF2 must be used, be aware that default values might not be considered secure.  
 Depending on the algorithm used, the number of iterations
should be adjusted to ensure that the derived key is secure. The following are the recommended number of iterations for PBKDF2:

- PBKDF2-HMAC-SHA1: 1,300,000 iterations
- PBKDF2-HMAC-SHA256: 600,000 iterations
- PBKDF2-HMAC-SHA512: 210,000 iterations

Note that PBKDF2-HMAC-SHA256 is recommended by NIST.  
 Iterations are also called "rounds" depending on the library used.

When recommended cost factors are too high in the context of the application or if the performance cost is unacceptable, a cost factor reduction
might be considered. In that case, it should not be chosen under 100,000.

### Going the extra mile

#### Pepper

In a defense-in-depth security approach, **peppering** can also be used. This is a security technique where an external secret value
is added to a password before it is hashed.  
 This makes it more difficult for an attacker to crack the hashed passwords, as they would need to know
the secret value to generate the correct hash.  
 [Learn more here](https://cheatsheetseries.owasp.org/cheatsheets/Password_Storage_Cheat_Sheet.html#peppering).

## How to fix it in BouncyCastle

### Code examples

#### Noncompliant code example

Using SCrypt:

    using Org.BouncyCastle.Crypto.Generators;
    
    string password = Request.Query["password"];
    byte[] salt = RandomNumberGenerator.GetBytes(128 / 8); // divide by 8 to convert bits to bytes
    
    string hashed = Convert.ToBase64String(SCrypt.Generate(Encoding.Unicode.GetBytes(password), salt, 4, 2, 1, 32));  // Noncompliant

Using BCrypt:

    using Org.BouncyCastle.Crypto.Generators;
    using Org.BouncyCastle.Crypto.Parameters;
    
    string password = Request.Query["password"];
    byte[] salt = RandomNumberGenerator.GetBytes(128 / 8);
    
    string hashed = OpenBsdBCrypt.Generate(password.ToCharArray(), salt, 4); // Noncompliant

Using PBKDF2:

    using Org.BouncyCastle.Crypto.Generators;
    using Org.BouncyCastle.Crypto.Parameters;
    using System.Security.Cryptography;
    
    string password = Request.Query["password"];
    byte[] salt = RandomNumberGenerator.GetBytes(128 / 8);
    Pkcs5S2ParametersGenerator gen = new Pkcs5S2ParametersGenerator();
    gen.Init(Encoding.Unicode.GetBytes(password), salt, 100);  // Noncompliant
    KeyParameter keyParam = (KeyParameter) gen.GenerateDerivedMacParameters(256);
    string hashed = Convert.ToBase64String(keyParam.GetKey());

#### Compliant solution

Using SCrypt:

    using Org.BouncyCastle.Crypto.Generators;
    
    string password = Request.Query["password"];
    byte[] salt = RandomNumberGenerator.GetBytes(128 / 8); // divide by 8 to convert bits to bytes
    
    string hashed = Convert.ToBase64String(SCrypt.Generate(Encoding.Unicode.GetBytes(password), salt, 1<<12, 8, 1, 32));  // Noncompliant

Using BCrypt:

    using Org.BouncyCastle.Crypto.Generators;
    using Org.BouncyCastle.Crypto.Parameters;
    
    string password = Request.Query["password"];
    byte[] salt = RandomNumberGenerator.GetBytes(128 / 8);
    
    string hashed = OpenBsdBCrypt.Generate(password.ToCharArray(), salt, 14); // Noncompliant

Using PBKDF2:

    using Org.BouncyCastle.Crypto.Generators;
    using Org.BouncyCastle.Crypto.Parameters;
    using System.Security.Cryptography;
    
    string password = Request.Query["password"];
    byte[] salt = RandomNumberGenerator.GetBytes(128 / 8);
    Pkcs5S2ParametersGenerator gen = new Pkcs5S2ParametersGenerator();
    gen.Init(Encoding.Unicode.GetBytes(password), salt, 100_000);  // Noncompliant
    KeyParameter keyParam = (KeyParameter) gen.GenerateDerivedMacParameters(256);
    string hashed = Convert.ToBase64String(keyParam.GetKey());

### How does this work?

#### Select the correct Bcrypt parameters

When bcrypt’s hashing function is used, it is important to select a round count that is high enough to make the function slow enough to prevent
brute force: More than 12 rounds.

For bcrypt’s key derivation function, the number of rounds should likewise be high enough to make the function slow enough to prevent brute force:
More than 4096 rounds `(2^12)`.  
 This number is not the same coefficient as the first one because it uses a different algorithm.

#### Select the correct Scrypt parameters

If scrypt must be used, the default values of scrypt are considered secure.

Like Argon2id, scrypt has three different parameters that can be configured. N is the CPU/memory cost parameter and must be a power of two. r is
the block size and p is the parallelization factor.

All three parameters affect the memory and CPU usage of the algorithm. Higher values of N, r and p result in safer hashes, but come at the cost of
higher resource usage.

For scrypt, OWASP recommends to have a hash length of at least 64 bytes, and to set N, p and r to the values of one of the following rows:

| N (cost parameter) | p (parallelization factor) | r (block size) |
| --- | --- | --- |
| 2^17^ (`1 << 17`) | 1 | 8 |
| 2^16^ (`1 << 16`) | 2 | 8 |
| 2^15^ (`1 << 15`) | 3 | 8 |
| 2^14^ (`1 << 14`) | 5 | 8 |
| 2^13^ (`1 << 13`) | 10 | 8 |

Every row provides the same level of defense. They only differ in the amount of CPU and RAM used: the top row has low CPU usage and high memory
usage, while the bottom row has high CPU usage and low memory usage.

#### Select the correct PBKDF2 parameters

If PBKDF2 must be used, be aware that default values might not be considered secure.  
 Depending on the algorithm used, the number of iterations
should be adjusted to ensure that the derived key is secure. The following are the recommended number of iterations for PBKDF2:

- PBKDF2-HMAC-SHA1: 1,300,000 iterations
- PBKDF2-HMAC-SHA256: 600,000 iterations
- PBKDF2-HMAC-SHA512: 210,000 iterations

Note that PBKDF2-HMAC-SHA256 is recommended by NIST.  
 Iterations are also called "rounds" depending on the library used.

When recommended cost factors are too high in the context of the application or if the performance cost is unacceptable, a cost factor reduction
might be considered. In that case, it should not be chosen under 100,000.

### Going the extra mile

#### Pepper

In a defense-in-depth security approach, **peppering** can also be used. This is a security technique where an external secret value
is added to a password before it is hashed.  
 This makes it more difficult for an attacker to crack the hashed passwords, as they would need to know
the secret value to generate the correct hash.  
 [Learn more here](https://cheatsheetseries.owasp.org/cheatsheets/Password_Storage_Cheat_Sheet.html#peppering).

## Resources

### Documentation

- OWASP CheatSheet - [Password Storage Cheat Sheet](https://cheatsheetseries.owasp.org/cheatsheets/Password_Storage_Cheat_Sheet.html)

### Standards

- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Top 10 2021 Category A4 - Insecure Design](https://owasp.org/Top10/A04_2021-Insecure_Design/)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- CWE - [CWE-256 - Plaintext Storage of a Password](https://cwe.mitre.org/data/definitions/256)
- CWE - [CWE-916 - Use of Password Hash With Insufficient Computational Effort](https://cwe.mitre.org/data/definitions/916)
- STIG Viewer - [Application Security and
  Development: V-222542](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222542) - The application must only store cryptographic representations of passwords.