Clear-text protocols such as `ftp`, `telnet`, or `http` lack encryption of transported data, as well as the
capability to build an authenticated connection. It means that an attacker able to sniff traffic from the network can read, modify, or corrupt the
transported content. These protocols are not secure as they expose applications to an extensive range of risks:

- sensitive data exposure
- traffic redirected to a malicious endpoint
- malware-infected software update or installer
- execution of client-side code
- corruption of critical information

Even in the context of isolated networks like offline environments or segmented cloud environments, the insider threat exists. Thus, attacks
involving communications being sniffed or tampered with can still happen.

For example, attackers could successfully compromise prior security layers by:

- bypassing isolation mechanisms
- compromising a component of the network
- getting the credentials of an internal IAM account (either from a service account or an actual person)

In such cases, encrypting communications would decrease the chances of attackers to successfully leak data or steal credentials from other network
components. By layering various security practices (segmentation and encryption, for example), the application will follow the
*defense-in-depth* principle.

Note that using the `http` protocol is being deprecated by [major web browsers](https://blog.mozilla.org/security/2015/04/30/deprecating-non-secure-http).

In the past, it has led to the following vulnerabilities:

- [CVE-2019-6169](https://nvd.nist.gov/vuln/detail/CVE-2019-6169)
- [CVE-2019-12327](https://nvd.nist.gov/vuln/detail/CVE-2019-12327)
- [CVE-2019-11065](https://nvd.nist.gov/vuln/detail/CVE-2019-11065)

## Ask Yourself Whether

- Application data needs to be protected against falsifications or leaks when transiting over the network.
- Application data transits over an untrusted network.
- Compliance rules require the service to encrypt data in transit.
- Your application renders web pages with a relaxed mixed content policy.
- OS-level protections against clear-text traffic are deactivated.

There is a risk if you answered yes to any of those questions.

## Recommended Secure Coding Practices

- Make application data transit over a secure, authenticated and encrypted protocol like TLS or SSH. Here are a few alternatives to the most
  common clear-text protocols:
    - Use `ssh` as an alternative to `telnet`.
    - Use `sftp`, `scp`, or `ftps` instead of `ftp`.
    - Use `https` instead of `http`.
    - Use `SMTP` over `SSL/TLS` or `SMTP` with `STARTTLS` instead of clear-text SMTP.
- Enable encryption of cloud components communications whenever it is possible.
- Configure your application to block mixed content when rendering web pages.
- If available, enforce OS-level deactivation of all clear-text traffic.

It is recommended to secure all transport channels, even on local networks, as it can take a single non-secure connection to compromise an entire
application or system.

## Sensitive Code Example

    var urlHttp = "http://example.com";                 // Noncompliant
    var urlFtp = "ftp://anonymous@example.com";         // Noncompliant
    var urlTelnet = "telnet://anonymous@example.com";   // Noncompliant

    using var smtp = new SmtpClient("host", 25); // Noncompliant, EnableSsl is not set
    using var telnet = new MyTelnet.Client("host", port); // Noncompliant, rule raises Security Hotspot on any member containing "Telnet"

## Compliant Solution

    var urlHttps = "https://example.com";
    var urlSftp = "sftp://anonymous@example.com";
    var urlSsh = "ssh://anonymous@example.com";

    using var smtp = new SmtpClient("host", 25) { EnableSsl = true };
    using var ssh = new MySsh.Client("host", port);

## Exceptions

No issue is reported for the following cases because they are not considered sensitive:

- Insecure protocol scheme followed by loopback addresses like 127.0.0.1 or `localhost`.

## See

### Documentation

- AWS Documentation - [Listeners for
  your Application Load Balancers](https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html)
- AWS Documentation - [Stream Encryption](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-kinesis-stream-streamencryption.html)

### Articles & blog posts

- Google - [Moving towards more secure web](https://security.googleblog.com/2016/09/moving-towards-more-secure-web.html)
- Mozilla - [Deprecating non secure http](https://blog.mozilla.org/security/2015/04/30/deprecating-non-secure-http/)

### Standards

- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Mobile AppSec Verification Standard - Network Communication Requirements](https://mas.owasp.org/checklists/MASVS-NETWORK/)
- OWASP - [Mobile Top 10 2016 Category M3 - Insecure
  Communication](https://owasp.org/www-project-mobile-top-10/2016-risks/m3-insecure-communication)
- CWE - [CWE-200 - Exposure of Sensitive Information to an Unauthorized Actor](https://cwe.mitre.org/data/definitions/200)
- CWE - [CWE-319 - Cleartext Transmission of Sensitive Information](https://cwe.mitre.org/data/definitions/319)
- STIG Viewer - [Application Security and
  Development: V-222397](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222397) - The application must implement cryptographic mechanisms to protect the integrity of remote access sessions.
- STIG Viewer - [Application Security and
  Development: V-222534](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222534) - Service-Oriented Applications handling non-releasable data must authenticate endpoint devices via mutual SSL/TLS.
- STIG Viewer - [Application Security and
  Development: V-222562](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222562) - Applications used for non-local maintenance must implement cryptographic mechanisms to protect the integrity of
  maintenance and diagnostic communications.
- STIG Viewer - [Application Security and
  Development: V-222563](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222563) - Applications used for non-local maintenance must implement cryptographic mechanisms to protect the confidentiality of
  maintenance and diagnostic communications.
- STIG Viewer - [Application Security and
  Development: V-222577](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222577) - The application must not expose session IDs.
- STIG Viewer - [Application Security and
  Development: V-222596](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222596) - The application must protect the confidentiality and integrity of transmitted information.
- STIG Viewer - [Application Security and
  Development: V-222597](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222597) - The application must implement cryptographic mechanisms to prevent unauthorized disclosure of information and/or detect
  changes to information during transmission.
- STIG Viewer - [Application Security and
  Development: V-222598](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222598) - The application must maintain the confidentiality and integrity of information during preparation for transmission.
- STIG Viewer - [Application Security and
  Development: V-222599](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222599) - The application must maintain the confidentiality and integrity of information during reception.