The access control of an application must be properly implemented in order to restrict access to resources to authorized entities otherwise this could lead to vulnerabilities:
 
-	    [CVE-2018-12999](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2018-12999)
-	    [CVE-2018-10285](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2018-10285)
-	    [CVE-2017-7455](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2017-7455)

Granting correct permissions to users, applications, groups or roles and defining required permissions that allow access to a resource is sensitive, must therefore be done with care. For instance, it is obvious that only users with administrator privilege should be authorized to add/remove the administrator permission of another user.
 
## Ask Yourself Whether

-	    Granted permission to an entity (user, application) allow access to information or functionalities not needed by this entity.
-	    Privileges are easily acquired (eg: based on the location of the user, type of device used, defined by third parties, does not require approval
  …​).
-	    Inherited permission, default permission, no privileges (eg: anonymous user) is authorized to access to a protected resource.

There is a risk if you answered yes to any of those questions.
 
## Recommended Secure Coding Practices
 
At minimum, an access control system should:

-	    Use a well-defined access control model like [RBAC](https://en.wikipedia.org/wiki/Role-based_access_control) or [ACL](https://en.wikipedia.org/wiki/Access-control_list).
-	    Entities' permissions should be reviewed regularly to remove permissions that are no longer needed.
-	    Respect [the principle of least privilege](https://en.wikipedia.org/wiki/Principle_of_least_privilege) ("*an entity has access  only the information and resources that are necessary for its legitimate purpose*").

## Sensitive Code Example

    using System.Threading;
    using System.Security.Permissions;
    using System.Security.Principal;
    using System.IdentityModel.Tokens;
    
    class SecurityPrincipalDemo
    {
        class MyIdentity : IIdentity // Sensitive, custom IIdentity implementations should be reviewed
        {
            // ...
        }
    
        class MyPrincipal : IPrincipal // Sensitive, custom IPrincipal implementations should be reviewed
        {
            // ...
        }
        [System.Security.Permissions.PrincipalPermission(SecurityAction.Demand, Role = "Administrators")] // Sensitive. The access restrictions enforced by this attribute should be reviewed.
        static void CheckAdministrator()
        {
            WindowsIdentity MyIdentity = WindowsIdentity.GetCurrent(); // Sensitive
            HttpContext.User = ...; // Sensitive: review all reference (set and get) to System.Web HttpContext.User
            AppDomain domain = AppDomain.CurrentDomain;
            domain.SetPrincipalPolicy(PrincipalPolicy.WindowsPrincipal); // Sensitive
            MyIdentity identity = new MyIdentity(); // Sensitive
            MyPrincipal MyPrincipal = new MyPrincipal(MyIdentity); // Sensitive
            Thread.CurrentPrincipal = MyPrincipal; // Sensitive
            domain.SetThreadPrincipal(MyPrincipal); // Sensitive
    
            // All instantiation of PrincipalPermission should be reviewed.
            PrincipalPermission principalPerm = new PrincipalPermission(null, "Administrators"); // Sensitive
            principalPerm.Demand();
    
            SecurityTokenHandler handler = ...;
            // Sensitive: this creates an identity.
            ReadOnlyCollection<ClaimsIdentity> identities = handler.ValidateToken(…);
        }
    
         // Sensitive: review how this function uses the identity and principal.
        void modifyPrincipal(MyIdentity identity, MyPrincipal principal)
        {
            // ...
        }
    }

## See

-	    [OWASP Top 10 2017 Category A5](https://www.owasp.org/index.php/Top_10-2017_A5-Broken_Access_Control) - Broken Access Control
-	    [SANS Top 25](https://www.sans.org/top25-software-errors/#cat3) - Porous Defenses
-	    [MITRE, CWE-276](https://cwe.mitre.org/data/definitions/276) - Incorrect Default Permissions
-	    [MITRE, CWE-732](https://cwe.mitre.org/data/definitions/732) - Incorrect Permission Assignment for Critical Resource
-	    [MITRE, CWE-668](https://cwe.mitre.org/data/definitions/668) - Exposure of Resource to Wrong Sphere
-	    [MITRE, CWE-277](https://cwe.mitre.org/data/definitions/277) - Insecure Inherited Permissions

## Deprecated
 
This rule is deprecated, and will eventually be removed.