This vulnerability makes it possible that an encrypted communication is intercepted.

## Why is this an issue?

Transport Layer Security (TLS) provides secure communication between systems over the internet by encrypting the data sent between them.
Certificate validation adds an extra layer of trust and security to this process to ensure that a system is indeed the one it claims to be.

When certificate validation is disabled, the client skips a critical security check. This creates an opportunity for attackers to pose as a trusted
entity and intercept, manipulate, or steal the data being transmitted.

### What is the potential impact?

Establishing trust in a secure way is a non-trivial task. When you disable certificate validation, you are removing a key mechanism designed to
build this trust in internet communication, opening your system up to a number of potential threats.

#### Identity spoofing

If a system does not validate certificates, it cannot confirm the identity of the other party involved in the communication. An attacker can
exploit this by creating a fake server and masquerading as a legitimate one. For example, they might set up a server that looks like your bank’s
server, tricking your system into thinking it is communicating with the bank. This scenario, called identity spoofing, allows the attacker to collect
any data your system sends to them, potentially leading to significant data breaches.

#### Loss of data integrity

When TLS certificate validation is disabled, the integrity of the data you send and receive cannot be guaranteed. An attacker could modify the data
in transit, and you would have no way of knowing. This could range from subtle manipulations of the data you receive to the injection of malicious
code or malware into your system. The consequences of such breaches of data integrity can be severe, depending on the nature of the data and the
system.

## How to fix it in .NET

### Code examples

In the following example, the callback change impacts the entirety of HTTP requests made by the application.

The certificate validation gets disabled by overriding `ServerCertificateValidationCallback` with an empty implementation. It is highly
recommended to use the original implementation.

#### Noncompliant code example

    using System.Net;
    using System.Net.Http;
    
    public static void connect()
    {
        ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, errors) => {     return true; // Noncompliant
        };
    
        HttpClient httpClient = new HttpClient();
        HttpResponseMessage response = httpClient.GetAsync("https://example.com").Result;
    }

### How does this work?

Addressing the vulnerability of disabled TLS certificate validation primarily involves re-enabling the default validation.

To avoid running into problems with invalid certificates, consider the following sections.

#### Using trusted certificates

If possible, always use a certificate issued by a well-known, trusted CA for your server. Most programming environments come with a predefined list
of trusted root CAs, and certificates issued by these authorities are validated automatically. This is the best practice, and it requires no
additional code or configuration.

#### Working with self-signed certificates or non-standard CAs

In some cases, you might need to work with a server using a self-signed certificate, or a certificate issued by a CA not included in your trusted
roots. Rather than disabling certificate validation in your code, you can add the necessary certificates to your trust store.

## Resources

### Standards

- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Top 10 2021 Category A5 - Security Misconfiguration](https://owasp.org/Top10/A05_2021-Security_Misconfiguration/)
- OWASP - [Top 10 2021 Category A7 - Identification and
  Authentication Failures](https://owasp.org/Top10/A07_2021-Identification_and_Authentication_Failures/)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- OWASP - [Top 10 2017 Category A6 - Security
  Misconfiguration](https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration)
- OWASP - [Mobile Top 10 2016 Category M3 - Insecure
  Communication](https://owasp.org/www-project-mobile-top-10/2016-risks/m3-insecure-communication)
- OWASP - [Mobile AppSec Verification Standard - Network Communication Requirements](https://mas.owasp.org/checklists/MASVS-NETWORK/)
- CWE - [CWE-295 - Improper Certificate Validation](https://cwe.mitre.org/data/definitions/295)
- STIG Viewer - [Application Security and
  Development: V-222550](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222550) - The application must validate certificates by constructing a certification path to an accepted trust anchor.