## Why is this an issue?

Returning `null` from a non-`async` `Task`/`Task<TResult>` method will cause a
`NullReferenceException` at runtime if the method is awaited. This problem can be avoided by returning [`Task.CompletedTask`](https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.completedtask) or [`Task.FromResult<TResult>(null)`](https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.fromresult)
respectively.

    public Task DoFooAsync()
    {
        return null;               // Noncompliant: Causes a NullReferenceException if awaited.
    }
    
    public async Task Main()
    {
        await DoFooAsync();        // NullReferenceException
    }

## How to fix it

Instead of `null` [`Task.CompletedTask`](https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.completedtask) or [`Task.FromResult<TResult>(null)`](https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.fromresult)
should be returned.

### Code examples

A `Task` returning method can be fixed like so:

#### Noncompliant code example

    public Task DoFooAsync()
    {
        return null;               // Noncompliant: Causes a NullReferenceException if awaited.
    }

#### Compliant solution

    public Task DoFooAsync()
    {
        return Task.CompletedTask; // Compliant: Method can be awaited.
    }

A `Task<TResult>` returning method can be fixed like so:

#### Noncompliant code example

    public Task<object> GetFooAsync()
    {
        return null;                          // Noncompliant: Causes a NullReferenceException if awaited.
    }

#### Compliant solution

    public Task<object> GetFooAsync()
    {
        return Task.FromResult<object>(null); // Compliant: Method can be awaited.
    }

## Resources

### Documentation

- Microsoft Learn - [`Task.CompletedTask` Property](https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.completedtask)
- Microsoft Learn - [`Task.FromResult<TResult>(TResult)`
  Method](https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task.fromresult)

### Articles & blog posts

- StackOverflow - Answer by Stephen Cleary for [Is it better to return an empty task or null?](https://stackoverflow.com/a/45350108)
- StackOverflow - Answer by Stephen Cleary for [Best way to handle null task inside async
  method?](https://stackoverflow.com/a/27551261)
- C# Language Design - [Proposal Champion "Null-conditional await"](https://github.com/dotnet/csharplang/issues/35)