Development tools and frameworks usually have options to make debugging easier for developers. Although these features are useful during
development, they should never be enabled for applications deployed in production. Debug instructions or error messages can leak detailed information
about the system, like the application’s path or file names.

## Ask Yourself Whether

- The code or configuration enabling the application debug features is deployed on production servers or distributed to end users.
- The application runs by default with debug features activated.

There is a risk if you answered yes to any of those questions.

## Recommended Secure Coding Practices

Do not enable debugging features on production servers.

The .Net Core framework offers multiple features which help during debug.
`Microsoft.AspNetCore.Builder.IApplicationBuilder.UseDeveloperExceptionPage` and
`Microsoft.AspNetCore.Builder.IApplicationBuilder.UseDatabaseErrorPage` are two of them. Make sure that those features are disabled in
production.

Use `if (env.IsDevelopment())` to disable debug code.

## Sensitive Code Example

This rule raises issues when the following .Net Core methods are called:
`Microsoft.AspNetCore.Builder.IApplicationBuilder.UseDeveloperExceptionPage`,
`Microsoft.AspNetCore.Builder.IApplicationBuilder.UseDatabaseErrorPage`.

    using Microsoft.AspNetCore.Builder;
    using Microsoft.AspNetCore.Hosting;
    
    namespace mvcApp
    {
        public class Startup2
        {
            public void Configure(IApplicationBuilder app, IHostingEnvironment env)
            {
                // Those calls are Sensitive because it seems that they will run in production
                app.UseDeveloperExceptionPage(); // Sensitive
                app.UseDatabaseErrorPage(); // Sensitive
            }
        }
    }

## Compliant Solution

    using Microsoft.AspNetCore.Builder;
    using Microsoft.AspNetCore.Hosting;
    
    namespace mvcApp
    {
        public class Startup2
        {
            public void Configure(IApplicationBuilder app, IHostingEnvironment env)
            {
                if (env.IsDevelopment())
                {
                    // The following calls are ok because they are disabled in production
                    app.UseDeveloperExceptionPage(); // Compliant
                    app.UseDatabaseErrorPage(); // Compliant
                }
            }
        }
    }

## Exceptions

This rule does not analyze configuration files. Make sure that debug mode is not enabled by default in those files.

## See

- OWASP - [Top 10 2021 Category A5 - Security Misconfiguration](https://owasp.org/Top10/A05_2021-Security_Misconfiguration/)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- CWE - [CWE-489 - Active Debug Code](https://cwe.mitre.org/data/definitions/489)
- CWE - [CWE-215 - Information Exposure Through Debug Information](https://cwe.mitre.org/data/definitions/215)