A cross-site request forgery (CSRF) attack occurs when a trusted user of a web application can be forced, by an attacker, to perform sensitive
actions that he didn’t intend, such as updating his profile or sending a message, more generally anything that can change the state of the
application.

The attacker can trick the user/victim to click on a link, corresponding to the privileged action, or to visit a malicious web site that embeds a
hidden web request and as web browsers automatically include cookies, the actions can be authenticated and sensitive.

## Ask Yourself Whether

- The web application uses cookies to authenticate users.
- There exist sensitive operations in the web application that can be performed when the user is authenticated.
- The state / resources of the web application can be modified by doing HTTP POST or HTTP DELETE requests for example.

There is a risk if you answered yes to any of those questions.

## Recommended Secure Coding Practices

- Protection against CSRF attacks is strongly recommended:
    - to be activated by default for all [unsafe HTTP
      methods](https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Safe_methods).
    - implemented, for example, with an unguessable CSRF token
- Of course all sensitive operations should not be performed with [safe HTTP](https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol#Safe_methods) methods like `GET` which are designed to be
  used only for information retrieval.

## Sensitive Code Example

    public void ConfigureServices(IServiceCollection services)
    {
        // ...
        services.AddControllersWithViews(options => options.Filters.Add(new IgnoreAntiforgeryTokenAttribute())); // Sensitive
        // ...
    }

    [HttpPost, IgnoreAntiforgeryToken] // Sensitive
    public IActionResult ChangeEmail(ChangeEmailModel model) => View("~/Views/...");

## Compliant Solution

    public void ConfigureServices(IServiceCollection services)
    {
        // ...
        services.AddControllersWithViews(options => options.Filters.Add(new AutoValidateAntiforgeryTokenAttribute()));
        // or
        services.AddControllersWithViews(options => options.Filters.Add(new ValidateAntiForgeryTokenAttribute()));
        // ...
    }

    [HttpPost]
    [AutoValidateAntiforgeryToken]
    public IActionResult ChangeEmail(ChangeEmailModel model) => View("~/Views/...");

## See

- OWASP - [Top 10 2021 Category A1 - Broken Access Control](https://owasp.org/Top10/A01_2021-Broken_Access_Control/)
- CWE - [CWE-352 - Cross-Site Request Forgery (CSRF)](https://cwe.mitre.org/data/definitions/352)
- OWASP - [Top 10 2017 Category A6 - Security
  Misconfiguration](https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration)
- [OWASP: Cross-Site Request Forgery](https://owasp.org/www-community/attacks/csrf)
- STIG Viewer - [Application Security and
  Development: V-222603](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222603) - The application must protect from Cross-Site Request Forgery (CSRF) vulnerabilities.