This vulnerability exposes encrypted data to attacks whose goal is to recover the plaintext.

## Why is this an issue?

Encryption algorithms are essential for protecting sensitive information and ensuring secure communications in a variety of domains. They are used
for several important reasons:

- Confidentiality, privacy, and intellectual property protection
- Security during transmission or on storage devices
- Data integrity, general trust, and authentication

When selecting encryption algorithms, tools, or combinations, you should also consider two things:

1. No encryption is unbreakable.
2. The strength of an encryption algorithm is usually measured by the effort required to crack it within a reasonable time frame.

In today’s cryptography, the length of the **key** directly affects the security level of cryptographic algorithms.

Note that depending on the algorithm, the term **key** refers to a different mathematical property. For example:

- For RSA, the key is the product of two large prime numbers, also called the **modulus**.
- For AES and Elliptic Curve Cryptography (ECC), the key is only a sequence of randomly generated bytes.
    - In some cases, AES keys are derived from a master key or a passphrase using a Key Derivation Function (KDF) like PBKDF2 (Password-Based Key
      Derivation Function 2)

If an application uses a key that is considered short and **insecure**, the encrypted data is exposed to attacks aimed at getting at
the plaintext.

In general, it is best practice to expect a breach: that a user or organization with malicious intent will perform cryptographic attacks on this
data after obtaining it by other means.

### What is the potential impact?

After retrieving encrypted data and performing cryptographic attacks on it on a given timeframe, attackers can recover the plaintext that
encryption was supposed to protect.

Depending on the recovered data, the impact may vary.

Below are some real-world scenarios that illustrate the potential impact of an attacker exploiting the vulnerability.

#### Additional attack surface

By modifying the plaintext of the encrypted message, an attacker may be able to trigger additional vulnerabilities in the code. An attacker can
further exploit a system to obtain more information.  
 Encrypted values are often considered trustworthy because it would not be possible for a
third party to modify them under normal circumstances.

#### Breach of confidentiality and privacy

When encrypted data contains personal or sensitive information, its retrieval by an attacker can lead to privacy violations, identity theft,
financial loss, reputational damage, or unauthorized access to confidential systems.

In this scenario, the company, its employees, users, and partners could be seriously affected.

The impact is twofold, as data breaches and exposure of encrypted data can undermine trust in the organization, as customers, clients and
stakeholders may lose confidence in the organization’s ability to protect their sensitive data.

#### Legal and compliance issues

In many industries and locations, there are legal and compliance requirements to protect sensitive data. If encrypted data is compromised and the
plaintext can be recovered, companies face legal consequences, penalties, or violations of privacy laws.

## How to fix it in .NET

### Code examples

The following code examples either explicitly or implicitly generate keys. Note that there are differences in the size of the keys depending on the
algorithm.

Due to the mathematical properties of the algorithms, the security requirements for the key size vary depending on the algorithm.  
 For example,
a 256-bit ECC key provides about the same level of security as a 3072-bit RSA key and a 128-bit symmetric key.

#### Noncompliant code example

Here is an example of a private key generation with RSA:

    using System;
    using System.Security.Cryptography;
    
    public void encrypt()
    {
        var RsaCsp = new RSACryptoServiceProvider(); // Noncompliant
    }

Here is an example of a key generation with the Digital Signature Algorithm (DSA):

    using System;
    using System.Security.Cryptography;
    
    public void encrypt()
    {
        var DsaCsp = new DSACryptoServiceProvider(); // Noncompliant
    }

Here is an example of an Elliptic Curve (EC) initialization. It implicitly generates a private key whose size is indicated in the elliptic curve
name:

    using System;
    using System.Security.Cryptography;
    
    public void encrypt()
    {
        ECDsa ecdsa = ECDsa.Create(ECCurve.NamedCurves.brainpoolP160t1); // Noncompliant
    }

#### Compliant solution

    using System;
    using System.Security.Cryptography;
    
    public void encrypt()
    {
        var RsaCsp = new RSACryptoServiceProvider(2048);
    }

    using System;
    using System.Security.Cryptography;
    
    public void encrypt()
    {
        var Dsa = new DSACng(2048);
    }

    using System;
    using System.Security.Cryptography;
    
    public void encrypt()
    {
        ECDsa ecdsa = ECDsa.Create(ECCurve.NamedCurves.nistP256);
    }

### How does this work?

As a rule of thumb, use the cryptographic algorithms and mechanisms that are considered strong by the cryptography community.

The appropriate choices are the following.

#### RSA (Rivest-Shamir-Adleman) and DSA (Digital Signature Algorithm)

The security of these algorithms depends on the difficulty of attacks attempting to solve their underlying mathematical problem.

In general, a minimum key size of **2048** bits is recommended for both. It provides 112 bits of security. A key length of
**3072** or **4092** should be preferred when possible.

#### AES (Advanced Encryption Standard)

AES supports three key sizes: 128 bits, 192 bits and 256 bits. The security of the AES algorithm is based on the computational complexity of trying
all possible keys.  
 A larger key size increases the number of possible keys and makes exhaustive search attacks computationally infeasible.
Therefore, a 256-bit key provides a higher level of security than a 128-bit or 192-bit key.

Currently, a minimum key size of **128 bits** is recommended for AES.

#### Elliptic Curve Cryptography (ECC)

Elliptic curve cryptography is also used in various algorithms, such as ECDSA, ECDH, or ECMQV. The length of keys generated with elliptic curve
algorithms is mentioned directly in their names. For example, `secp256k1` generates a 256-bits long private key.

Currently, a minimum key size of **224 bits** is recommended for EC-based algorithms.

Additionally, some curves that theoretically provide sufficiently long keys are still discouraged. This can be because of a flaw in the curve
parameters, a bad overall design, or poor performance. It is generally advised to use a NIST-approved elliptic curve wherever possible. Such curves
currently include:

- NIST P curves with a size of at least 224 bits, e.g. secp256r1.
- Curve25519, generally known as ed25519 or x25519 depending on its application.
- Curve448.
- Brainpool curves with a size of at least 224 bits, e.g. brainpoolP224r1

### Pitfalls

#### The KeySize Property is not a setter

The following code is invalid:

     ----
         var RsaCsp = new RSACryptoServiceProvider();
         RsaCsp.KeySize = 2048;
    ----

The KeySize property of CryptoServiceProviders cannot be updated because the setter simply does not exist. This means that this line will not
perform any update on `KeySize`, and the compiler won’t raise an Exception when compiling it. This should not be considered a
workaround.  
 To change the key size, use one of the overloaded constructors with the desired key size instead.

### Going the extra mile

#### Pre-Quantum Cryptography

Encrypted data and communications recorded today could be decrypted in the future by an attack from a quantum computer.  
 It is important to keep
in mind that NIST-approved digital signature schemes, key agreement, and key transport may need to be replaced with secure quantum-resistant (or
"post-quantum") counterpart.

Thus, if data is to remain secure beyond 2030, proactive measures should be taken now to ensure its safety.

[Learn more here](https://www.enisa.europa.eu/publications/post-quantum-cryptography-current-state-and-quantum-mitigation).

## Resources

- Documentation
    - NIST Documentation - [NIST SP 800-186: Recommendations
      for Discrete Logarithm-based Cryptography: Elliptic Curve Domain Parameters](https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-186.pdf)
    - IETF - [rfc5639: Elliptic Curve Cryptography (ECC) Brainpool Standard Curves and
      Curve Generation](https://datatracker.ietf.org/doc/html/rfc5639)

### Articles & blog posts

- [Microsoft, Timing vulnerabilities with CBC-mode
  symmetric decryption using padding](https://learn.microsoft.com/en-us/dotnet/standard/security/vulnerabilities-cbc-mode)
- [Wikipedia, Padding Oracle Attack](https://en.wikipedia.org/wiki/Padding_oracle_attack)
- [Wikipedia, Chosen-Ciphertext Attack](https://en.wikipedia.org/wiki/Chosen-ciphertext_attack)
- [Wikipedia, Chosen-Plaintext Attack](https://en.wikipedia.org/wiki/Chosen-plaintext_attack)
- [Wikipedia, Semantically Secure Cryptosystems](https://en.wikipedia.org/wiki/Semantic_security)
- [Wikipedia, OAEP](https://en.wikipedia.org/wiki/Optimal_asymmetric_encryption_padding)
- [Wikipedia, Galois/Counter Mode](https://en.wikipedia.org/wiki/Galois/Counter_Mode)

### Standards

- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://www.owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- OWASP - [Top 10 2017 Category A6 - Security
  Misconfiguration](https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration)
- OWASP - [Mobile AppSec Verification Standard - Cryptography Requirements](https://mas.owasp.org/checklists/MASVS-CRYPTO/)
- OWASP - [Mobile Top 10 2016 Category M5 -
  Insufficient Cryptography](https://owasp.org/www-project-mobile-top-10/2016-risks/m5-insufficient-cryptography)
- [NIST 800-131A](https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-131Ar1.pdf) - Recommendation for Transitioning the
  Use of Cryptographic Algorithms and Key Lengths
- CWE - [CWE-326 - Inadequate Encryption Strength](https://cwe.mitre.org/data/definitions/326)
- CWE - [CWE-327 - Use of a Broken or Risky Cryptographic Algorithm](https://cwe.mitre.org/data/definitions/327)
- [CERT, MSC61-J.](https://wiki.sei.cmu.edu/confluence/x/hDdGBQ) - Do not use insecure or weak cryptographic algorithms