This vulnerability exposes encrypted data to a number of attacks whose goal is to recover the plaintext.

## Why is this an issue?

Encryption algorithms are essential for protecting sensitive information and ensuring secure communications in a variety of domains. They are used
for several important reasons:

- Confidentiality, privacy, and intellectual property protection
- Security during transmission or on storage devices
- Data integrity, general trust, and authentication

When selecting encryption algorithms, tools, or combinations, you should also consider two things:

1. No encryption is unbreakable.
2. The strength of an encryption algorithm is usually measured by the effort required to crack it within a reasonable time frame.

For these reasons, as soon as cryptography is included in a project, it is important to choose encryption algorithms that are considered strong and
secure by the cryptography community.

To provide communication security over a network, SSL and TLS are generally used. However, it is important to note that the following protocols are
all considered weak by the cryptographic community, and are officially deprecated:

- SSL versions 1.0, 2.0 and 3.0
- TLS versions 1.0 and 1.1

When these unsecured protocols are used, it is best practice to expect a breach: that a user or organization with malicious intent will perform
mathematical attacks on this data after obtaining it by other means.

### What is the potential impact?

After retrieving encrypted data and performing cryptographic attacks on it on a given timeframe, attackers can recover the plaintext that
encryption was supposed to protect.

Depending on the recovered data, the impact may vary.

Below are some real-world scenarios that illustrate the potential impact of an attacker exploiting the vulnerability.

#### Additional attack surface

By modifying the plaintext of the encrypted message, an attacker may be able to trigger additional vulnerabilities in the code. An attacker can
further exploit a system to obtain more information.  
 Encrypted values are often considered trustworthy because it would not be possible for a
third party to modify them under normal circumstances.

#### Breach of confidentiality and privacy

When encrypted data contains personal or sensitive information, its retrieval by an attacker can lead to privacy violations, identity theft,
financial loss, reputational damage, or unauthorized access to confidential systems.

In this scenario, the company, its employees, users, and partners could be seriously affected.

The impact is twofold, as data breaches and exposure of encrypted data can undermine trust in the organization, as customers, clients and
stakeholders may lose confidence in the organization’s ability to protect their sensitive data.

#### Legal and compliance issues

In many industries and locations, there are legal and compliance requirements to protect sensitive data. If encrypted data is compromised and the
plaintext can be recovered, companies face legal consequences, penalties, or violations of privacy laws.

## How to fix it in .NET

### Code examples

#### Noncompliant code example

These samples use TLSv1.0 as the default TLS algorithm, which is cryptographically weak.

    using System.Net;
    
    public void encrypt()
    {
        ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls; // Noncompliant
    }

    using System.Net.Http;
    using System.Security.Authentication;
    
    public void encrypt()
    {
        new HttpClientHandler
        {
            SslProtocols = SslProtocols.Tls // Noncompliant
        };
    }

#### Compliant solution

    Using System.Net;
    
    public void encrypt()
    {
        ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12 | SecurityProtocolType.Tls13;
    }

    using System.Net.Http;
    using System.Security.Authentication;
    
    public void encrypt()
    {
        new HttpClientHandler
        {
            SslProtocols = SslProtocols.Tls12
        };
    }

### How does this work?

As a rule of thumb, by default you should use the cryptographic algorithms and mechanisms that are considered strong by the cryptographic
community.

The best choices at the moment are the following.

#### Use TLS v1.2 or TLS v1.3

Even though TLS V1.3 is available, using TLS v1.2 is still considered good and secure practice by the cryptography community.  

The use of TLS v1.2 ensures compatibility with a wide range of platforms and enables seamless communication between different systems that do not
yet have TLS v1.3 support.

The only drawback depends on whether the framework used is outdated: its TLS v1.2 settings may enable older and insecure cipher suites that are
deprecated as insecure.

On the other hand, TLS v1.3 removes support for older and weaker cryptographic algorithms, eliminates known vulnerabilities from previous TLS
versions, and improves performance.

## Resources

### Articles & blog posts

- [Wikipedia, Padding Oracle Attack](https://en.wikipedia.org/wiki/Padding_oracle_attack)
- [Wikipedia, Chosen-Ciphertext Attack](https://en.wikipedia.org/wiki/Chosen-ciphertext_attack)
- [Wikipedia, Chosen-Plaintext Attack](https://en.wikipedia.org/wiki/Chosen-plaintext_attack)
- [Wikipedia, Semantically Secure Cryptosystems](https://en.wikipedia.org/wiki/Semantic_security)
- [Wikipedia, OAEP](https://en.wikipedia.org/wiki/Optimal_asymmetric_encryption_padding)
- [Wikipedia, Galois/Counter Mode](https://en.wikipedia.org/wiki/Galois/Counter_Mode)

### Standards

- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Top 10 2021 Category A7 - Identification and
  Authentication Failures](https://owasp.org/Top10/A07_2021-Identification_and_Authentication_Failures/)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- OWASP - [Top 10 2017 Category A6 - Security
  Misconfiguration](https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration)
- CWE - [CWE-327 - Use of a Broken or Risky Cryptographic Algorithm](https://cwe.mitre.org/data/definitions/327)