Cryptographic operations often rely on unpredictable random numbers to enhance security. These random numbers are created by cryptographically
secure pseudo-random number generators (CSPRNG). It is important not to use a predictable seed with these random number generators otherwise the
random numbers will also become predictable.

## Why is this an issue?

Random number generators are often used to generate random values for cryptographic algorithms. When a random number generator is used for
cryptographic purposes, the generated numbers must be as random and unpredictable as possible. When the random number generator is improperly seeded
with a constant or a predictable value, its output will also be predictable.

This can have severe security implications for cryptographic operations that rely on the randomness of the generated numbers. By using a
predictable seed, an attacker can potentially guess or deduce the generated numbers, compromising the security of whatever cryptographic algorithm
relies on the random number generator.

### What is the potential impact?

It is crucial to understand that the strength of cryptographic algorithms heavily relies on the quality of the random numbers used. By improperly
seeding a CSPRNG, we introduce a significant weakness that can be exploited by attackers.

#### Insecure cryptographic keys

One of the primary use cases for CSPRNGs is generating cryptographic keys. If an attacker can predict the seed used to initialize the random number
generator, they may be able to derive the same keys. Depending on the use case, this can lead to multiple severe outcomes, such as:

- Being able to decrypt sensitive documents, leading to privacy breaches or identity theft.
- Gaining access to a private key used for signing, allowing an attacker to forge digital signatures and impersonate legitimate entities.
- Bypassing authentication mechanisms that rely on public-key infrastructure (PKI), which can be abused to gain unauthorized access to systems or
  networks.

#### Session hijacking and man-in-the-middle attack

Another scenario where this vulnerability can be exploited is in the generation of session tokens or nonces for secure communication protocols. If
an attacker can predict the seed used to generate these tokens, they can impersonate legitimate users or intercept sensitive information.

## How to fix it in BouncyCastle

BouncyCastle provides several random number generators implementations. Most of these will automatically create unpredictable output.

The remaining random number generators require seeding with an unpredictable value before they will produce unpredictable outputs. These should be
seeded with at least 16 bytes of random data to ensure unpredictable output and that the random seed cannot be guessed using a brute-force attack.

### Code examples

#### Noncompliant code example

`SecureRandom` instances created with `GetInstance()` are seeded by default. Disabling seeding results in predictable
output.

    using Org.BouncyCastle.Security;
    
    byte[] random = new byte[8];
    
    SecureRandom sr = SecureRandom.GetInstance("SHA256PRNG", false);
    sr.NextBytes(random); // Noncompliant

`DigestRandomGenerator` and `VmpcRandomGenerator` instances require seeding before use. Predictable seed values will result
in predictable outputs.

    using Org.BouncyCastle.Crypto.Digest;
    using Org.BouncyCastle.Crypto.Prng;
    
    byte[] random = new byte[8];
    
    IRandomGenerator digest = new DigestRandomGenerator(new Sha256Digest());
    digest.AddSeedMaterial(Encoding.UTF8.GetBytes("predictable seed value"));
    digest.NextBytes(random); // Noncompliant
    
    IRandomGenerator vmpc = new VmpcRandomGenerator();
    vmpc.AddSeedMaterial(Convert.FromBase64String("2hq9pkyqLQJkrYTrEv1eNw=="));
    vmpc.NextBytes(random); // Noncompliant

When a `SecureRandom` is created using an unseeded `DigestRandomGenerator` and `VmpcRandomGenerator` instance, the
`SecureRandom` will create predictable outputs.

    using Org.BouncyCastle.Crypto.Digest;
    using Org.BouncyCastle.Crypto.Prng;
    using Org.BouncyCastle.Security;
    
    byte[] random = new byte[8];
    
    IRandomGenerator digest = new DigestRandomGenerator(new Sha256Digest());
    SecureRandom sr = new SecureRandom(digest);
    sr.NextBytes(random); // Noncompliant

#### Compliant solution

Allow `SecureRandom.GetInstance()` to automatically seed new `SecureRandom` instances.

    using Org.BouncyCastle.Security;
    
    byte[] random = new byte[8];
    
    SecureRandom sr = SecureRandom.GetInstance("SHA256PRNG");
    sr.NextBytes(random);

Use unpredictable values to seed `DigestRandomGenerator` and `VmpcRandomGenerator` instances. The
`SecureRandom.GenerateSeed()` method is designed for this purpose.

    using Org.BouncyCastle.Crypto.Digest;
    using Org.BouncyCastle.Crypto.Prng;
    using Org.BouncyCastle.Security;
    
    byte[] random = new byte[8];
    
    IRandomGenerator digest = new DigestRandomGenerator(new Sha256Digest());
    digest.AddSeedMaterial(SecureRandom.GenerateSeed(16));
    digest.NextBytes(random);
    
    IRandomGenerator vmpc = new VmpcRandomGenerator();
    vmpc.AddSeedMaterial(SecureRandom.GenerateSeed(16));
    vmpc.NextBytes(random);

An overload of the `SecureRandom` constructor will automatically seed the underlying `IRandomGenerator` with an unpredictable
value.

    using Org.BouncyCastle.Crypto.Digest;
    using Org.BouncyCastle.Crypto.Prng;
    using Org.BouncyCastle.Security;
    
    byte[] random = new byte[8];
    
    IRandomGenerator digest = new DigestRandomGenerator(new Sha256Digest());
    SecureRandom sr = new SecureRandom(digest, 16);
    sr.NextBytes(random);

## Resources

### Documentation

- Bouncy Castle - [The BouncyCastle.NET User Guide](https://downloads.bouncycastle.org/csharp/docs/BC-CSharpDotNet-UserGuide.pdf)

### Standards

- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Top 10 2017 Category A6 - Security
  Misconfiguration](https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration)
- CWE - [CWE-330 - Use of Insufficiently Random Values](https://cwe.mitre.org/data/definitions/330)
- CWE - [CWE-332 - Insufficient Entropy in PRNG](https://cwe.mitre.org/data/definitions/332)
- CWE - [CWE-336 - Same Seed in Pseudo-Random Number Generator (PRNG)](https://cwe.mitre.org/data/definitions/336)
- CWE - [CWE-337 - Predictable Seed in Pseudo-Random Number Generator (PRNG)](https://cwe.mitre.org/data/definitions/337)
- [CERT, MSC63J.](https://wiki.sei.cmu.edu/confluence/display/java/MSC63-J.+Ensure+that+SecureRandom+is+properly+seeded) - Ensure that
  SecureRandom is properly seeded

## Implementation Specification

(visible only on this page)

### Message

When the random number generator’s output **is not** predictable by default:

> 
>   
> Change this seed value to something unpredictable, or remove the seed.
> 

When the random number generator’s output **is** predictable by default:

> 
>   
> Set an unpredictable seed before generating random values.
> 

### Highlighting

When the random number generator’s output **is not** predictable by default:

- The most recent function call that sets a seed. For example:
    - The factory method that returns the RNG, where the seed is passed as a parameter.
    - The RNG constructor, where the seed is a parameter.
    - The function call on the RNG that sets the seed.

When the random number generator’s output **is** predictable by default:

- The function call on the RNG that returns a random value.

If the factory method or constructor is not already highlighted, it should become a secondary highlight.