Transparency attributes in the .NET Framework, designed to protect security-critical operations, can lead to ambiguities and vulnerabilities when
declared at different levels such as both for the class and a method.

## Why is this an issue?

Transparency attributes can be declared at several levels. If two different attributes are declared at two different levels, the attribute that
prevails is the one in the highest level. For example, you can declare that a class is `SecuritySafeCritical` and that a method of this
class is `SecurityCritical`. In this case, the method will be `SecuritySafeCritical` and the `SecurityCritical`
attribute attached to it is ignored.

### What is the potential impact?

Below are some real-world scenarios that illustrate some impacts of an attacker exploiting the vulnerability.

#### Elevation of Privileges

An attacker could potentially exploit conflicting transparency attributes to perform actions with higher privileges than intended.

#### Data Exposure

If a member with conflicting attributes is involved in handling sensitive data, an attacker could exploit the vulnerability to gain unauthorized
access to this data. This could lead to breaches of confidentiality and potential data loss.

## How to fix it

### Code examples

#### Noncompliant code example

    using System;
    using System.Security;
    
    namespace MyLibrary
    {
        [SecuritySafeCritical]
        public class Foo
        {
            [SecurityCritical] // Noncompliant
            public void Bar()
            {
            }
        }
    }

#### Compliant solution

    using System;
    using System.Security;
    
    namespace MyLibrary
    {
        public class Foo
        {
            [SecurityCritical]
            public void Bar()
            {
            }
        }
    }

### How does this work?

#### Never set class-level annotations

A class should never have class-level annotations if some functions have different permission levels. Instead, make sure every function has its own
correct annotation.

If no function needs a particularly distinct security annotation in a class, just set a class-level `[SecurityCritical]`.

## Resources

### Articles & blog posts

- Redgate Hub - [What’s New
  in Code Access Security in .NET Framework 4.0 – Part I](https://www.red-gate.com/simple-talk/development/dotnet-development/whats-new-in-code-access-security-in-net-framework-4-0-part-i/)

### Standards

- OWASP - [Top 10 2021 Category A5 - Security Misconfiguration](https://owasp.org/Top10/A05_2021-Security_Misconfiguration/)
- OWASP - [Top 10 2017 Category A6 - Security
  Misconfiguration](https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration)