## Why is this an issue?

Certain characters, once normalized to lowercase, cannot make a round trip. That is, they can not be converted from one locale to another and then
accurately restored to their original characters.

It is therefore strongly recommended to normalize characters and strings to uppercase instead.

### Noncompliant code example

    Thread.CurrentThread.CurrentCulture = new CultureInfo("tr-TR");
    var areStringEqual = "INTEGER".ToLower() == "integer"; // Noncompliant, the result is false as the ToLower will resolve to "ınteger"
    var areCharEqual = char.ToLower('I') == 'i'; // Noncompliant, the result is false as the ToLower will resolve to "ı"
    
    var incorrectRoundtrip = "İ".ToLowerInvariant().ToUpper() == "I".ToLowerInvariant().ToUpper(); // Noncompliant, because of the lower we lose the information about the correct uppercase character

### Compliant solution

    Thread.CurrentThread.CurrentCulture = new CultureInfo("tr-TR");
    var areStringEqual = "ınteger".ToUpperInvariant() == "ıNTEGER";
    var areCharEqual = char.ToUpperInvariant('ı') == 'ı';
    var correctRoundtrip = "İ".ToUpperInvariant().ToLower() != "I".ToUpperInvariant().ToLower();

## Resources

- [Internationalization for Turkish](http://www.i18nguy.com/unicode/turkish-i18n.html)
- [How to correctly normalize
  strings](https://gingter.org/2018/07/10/how-to-correctly-normalize-strings-and-how-to-compare-them-in-net/)
- [Best Practices
  for Using Strings in .NET](https://docs.microsoft.com/en-us/dotnet/standard/base-types/best-practices-strings#recommendations-for-string-usage)