## Why is this an issue?

The [AssemblyVersion](https://learn.microsoft.com/en-us/dotnet/api/system.reflection.assemblyversionattribute) attribute is used to
specify the version number of an assembly. An assembly is a compiled unit of code, which can be marked with a version number by applying the attribute
to an assembly’s source code file.

The `AssemblyVersion` attribute is useful for many reasons:

- **Versioning**: The attribute allows developers to track and manage different versions of an assembly. By incrementing the version
  number for each new release, you can easily identify and differentiate between different versions of the same assembly. This is particularly useful
  when distributing and deploying software, as it helps manage updates and compatibility between different versions.
- **Dependency management**: When an assembly references another assembly, it can specify the specific version of the dependency it
  requires. By using the `AssemblyVersion` attribute, you can ensure that the correct version of the referenced assembly is used. This
  helps avoid compatibility issues and ensures that the expected behavior and functionality are maintained.
- **GAC management**: The [GAC](https://learn.microsoft.com/en-us/dotnet/framework/app-domains/gac), also known as Global
  Assembly Cache, is a central repository for storing shared assemblies on a system. The AssemblyVersion attribute plays a crucial role in managing
  assemblies in the GAC. Different versions of an assembly can coexist in the GAC, allowing applications to use the specific version they require.

If no `AssemblyVersion` is provided, the same default version will be used for every build. Since the version number is used by .NET
Framework to uniquely identify an assembly, this can lead to broken dependencies.

## How to fix it

### Code examples

#### Noncompliant code example

    using System.Reflection;
    
    [assembly: AssemblyTitle("MyAssembly")] // Noncompliant
    namespace MyLibrary
    {
        // ...
    }

#### Compliant solution

    using System.Reflection;
    
    [assembly: AssemblyTitle("MyAssembly")]
    [assembly: AssemblyVersion("42.1.125.0")]
    namespace MyLibrary
    {
        // ...
    }

## Resources

### Documentation

- [Assembly Versioning](https://docs.microsoft.com/en-us/dotnet/standard/assembly/versioning)
- [Global Assembly Cache](https://learn.microsoft.com/en-us/dotnet/framework/app-domains/gac)